/* ***************************************************************************************
* FILE:          ToString.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ToString.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "ToString.h"

#include <iomanip>
#include <ostream>

namespace hmibase {
namespace trace {
namespace util {


/************************************************************************/
#define _HMIBASE_TO_STRING_DEF(TVal)\
const char* ToString::c_str(const TVal& value)\
{\
   ::std::stringstream ss;\
   ss << '[' << value << ']';\
   _HMIBASE_EXTRACT_STRING(ss, _string)\
   return _string.c_str();\
}\
const char* ToString::c_str(const TVal* value)\
{\
   ::std::stringstream ss;\
   ss << '[' << static_cast<const void*>(value);\
   if (value != NULL)\
   {\
      ss << '.' << *value;\
   }\
   ss << ']';\
   _HMIBASE_EXTRACT_STRING(ss, _string)\
   return _string.c_str();\
}\
const char* ToString::c_str(TVal* value)\
{\
   ::std::stringstream ss;\
   ss << '[' << static_cast<const void*>(value);\
   if (value != NULL)\
   {\
      ss << '.' << *value;\
   }\
   ss << ']';\
   _HMIBASE_EXTRACT_STRING(ss, _string)\
   return _string.c_str();\
}


/************************************************************************/
#define _HMIBASE_TO_STRING_DEF_HINT(TVal)\
const char* ToString::c_str(const TVal& value, HintType hint)\
{\
   ::std::stringstream ss;\
   ss << '[' << Manipulator<TVal>(value, hint) << ']';\
   _HMIBASE_EXTRACT_STRING(ss, _string)\
   return _string.c_str();\
}\
const char* ToString::c_str(const TVal* value, HintType hint)\
{\
   ::std::stringstream ss;\
   ss << '[' << Manipulator<TVal>(value, hint) << ']';\
   _HMIBASE_EXTRACT_STRING(ss, _string)\
   return _string.c_str();\
}


/************************************************************************/
_HMIBASE_TO_STRING_DEF(::Candera::Bitmap)
_HMIBASE_TO_STRING_DEF_HINT(::Candera::CanderaObject)
_HMIBASE_TO_STRING_DEF(::Candera::Color)
_HMIBASE_TO_STRING_DEF_HINT(::Candera::GraphicDeviceUnit)
_HMIBASE_TO_STRING_DEF(::Candera::Image2D)
_HMIBASE_TO_STRING_DEF(::Candera::Margin)
_HMIBASE_TO_STRING_DEF(::Candera::Matrix3)
_HMIBASE_TO_STRING_DEF(::Candera::Matrix3x2)
_HMIBASE_TO_STRING_DEF(::Candera::Matrix4)
_HMIBASE_TO_STRING_DEF_HINT(::Candera::Node)
_HMIBASE_TO_STRING_DEF_HINT(::Candera::Node2D)
_HMIBASE_TO_STRING_DEF(::Candera::Vector2)
_HMIBASE_TO_STRING_DEF(::Candera::Vector3)
_HMIBASE_TO_STRING_DEF(::Candera::Vector4)
_HMIBASE_TO_STRING_DEF(::Candera::Rectangle)

_HMIBASE_TO_STRING_DEF(::Courier::Identifier);
_HMIBASE_TO_STRING_DEF(::Courier::ItemId);
_HMIBASE_TO_STRING_DEF(::Courier::Message);
_HMIBASE_TO_STRING_DEF(::Courier::TouchInfo);
_HMIBASE_TO_STRING_DEF_HINT(::Courier::View)
_HMIBASE_TO_STRING_DEF(::Courier::ViewId);

_HMIBASE_TO_STRING_DEF(::FeatStd::String);

_HMIBASE_TO_STRING_DEF(::hmibase::util::geometry::Point)
_HMIBASE_TO_STRING_DEF(::hmibase::util::geometry::Rect)
_HMIBASE_TO_STRING_DEF(::hmibase::util::geometry::Size)

_HMIBASE_TO_STRING_DEF(::hmibase::input::gesture::GestureEvent)
_HMIBASE_TO_STRING_DEF_HINT(::hmibase::widget::Widget)


/************************************************************************/
static bool insertPointer(::std::ostream& os, const void* ptr, HintType hint)
{
   if ((hint & tsh::Pointer) != 0)
   {
      os << ptr;
      return true;
   }
   return false;
}


/************************************************************************/
template <> void Manipulator< ::Candera::CanderaObject> ::insertInto(::std::ostream& os) const
{
   bool dotRequired = insertPointer(os, _value, _hint);

   if ((_value != NULL) && ((_hint & tsh::Object) != 0))
   {
      if (dotRequired)
      {
         os << '.';
      }
      os << *_value;
   }
}


/************************************************************************/
template <> void Manipulator< ::Candera::GraphicDeviceUnit> ::insertInto(::std::ostream& os) const
{
   bool dotRequired = insertPointer(os, _value, _hint);

   if ((_value != NULL) && ((_hint & tsh::Object) != 0))
   {
      if (dotRequired)
      {
         os << '.';
      }
      os << *_value;
   }
}


/************************************************************************/
template <typename TScene>
static bool insertScene(::std::ostream& os, const TScene& scene, HintType hint, bool dotRequired)
{
   if ((hint & tsh::Container) != 0)
   {
      if (dotRequired)
      {
         os << '.';
      }
      os << scene;
      return true;
   }
   return dotRequired;
}


/************************************************************************/
template <typename TScene, typename TNode>
static bool insertNodePath(::std::ostream& os, const TNode& parent, HintType hint, bool dotRequired)
{
   if ((parent.GetParent() == NULL) && parent.IsTypeOf(TScene::GetTypeId()))
   {
      return insertScene(os, static_cast<const TScene&>(parent), hint, dotRequired);
   }

   if (parent.GetParent() != NULL)
   {
      dotRequired = insertNodePath<TScene, TNode>(os, *(parent.GetParent()), hint, dotRequired);
   }

   if ((hint & tsh::Path) != 0)
   {
      if (dotRequired)
      {
         os << '.';
      }
      os << parent << '/';
   }
   return false;
}


/************************************************************************/
template <typename TScene, typename TNode>
static void insertNode(::std::ostream& os, const TNode* node, HintType hint)
{
   bool dotRequired = insertPointer(os, node, hint);

   if ((node != NULL) && (hint != 0))
   {
      if (((hint & tsh::ContainerPath) != 0) && (node->GetParent() != NULL))
      {
         dotRequired = insertNodePath<TScene, TNode>(os, *(node->GetParent()), hint, dotRequired);
      }

      if ((hint & tsh::Object) != 0)
      {
         if (dotRequired)
         {
            os << '.';
         }
         os << *node;
      }
   }
}


/************************************************************************/
template <> void Manipulator< ::Candera::Node2D> ::insertInto(::std::ostream& os) const
{
   insertNode< ::Candera::Scene2D, ::Candera::Node2D>(os, _value, _hint);
}


/************************************************************************/
template <> void Manipulator< ::Candera::Node> ::insertInto(::std::ostream& os) const
{
   insertNode< ::Candera::Scene, ::Candera::Node>(os, _value, _hint);
}


/************************************************************************/
static bool insertView(::std::ostream& os, const ::Courier::View* view, HintType hint, bool dotRequired)
{
   if ((hint & tsh::Container) != 0)
   {
      if (dotRequired)
      {
         os << '.';
      }
      if (view != NULL)
      {
         os << *view;
      }
      else
      {
         os << "<view_not_set>";
      }
      return true;
   }
   return dotRequired;
}


/************************************************************************/
template <> void Manipulator< ::Courier::View> ::insertInto(::std::ostream& os) const
{
   bool dotRequired = insertPointer(os, _value, _hint);

   if ((_value != NULL) && ((_hint & tsh::Object) != 0))
   {
      if (dotRequired)
      {
         os << '.';
      }
      os << *_value;
   }
}


/************************************************************************/
template <> void Manipulator< ::hmibase::widget::Widget> ::insertInto(::std::ostream& os) const
{
   bool dotRequired = insertPointer(os, _value, _hint);

   if ((_value != NULL) && (_hint != 0))
   {
      dotRequired = insertView(os, _value->GetParentView(), _hint, dotRequired);

      if ((_hint & tsh::Object) != 0)
      {
         if (dotRequired)
         {
            os << '.';
         }
         os << *_value;
      }
   }
}


}
}


}
