/* ***************************************************************************************
* FILE:          MemoryStatistics.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MemoryStatistics.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "MemoryStatistics.h"

#ifndef WIN32

#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
#include "FeatStd/Diagnostics/SystemMemoryStatistic.h"
using namespace FeatStd::Diagnostics;

#include "hmibase/util/Mutex.h"
#endif

#ifdef FEATSTD_VIDEO_MEMORY_STATISTIC_ENABLED
#include "FeatStd/Config.h"
#include "Candera/System/Diagnostics/VideoMemoryStatistic.h"
using namespace Candera::Diagnostics;
#endif
#endif


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MemoryStatistics.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


namespace hmibase {
namespace trace {

MemoryStatistics::MemoryStatistics()
{
}


MemoryStatistics::~MemoryStatistics()
{
}


void MemoryStatistics::print()
{
#ifdef WIN32
   ETG_TRACE_FATAL_THR(("Memory statistics not enabled in WIN32"));
#else

#if defined (FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED) || defined (FEATSTD_VIDEO_MEMORY_STATISTIC_ENABLED)
   hmibase::util::cCriticalSection cs;
   cs.Enter();
#endif

#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
   int max = SystemMemoryStatistic::GetAllocationCount();
   ETG_TRACE_FATAL_THR((" #### [%25s] RAM statistics ####", hmibase::util::trace::getAppName()));
   ETG_TRACE_FATAL_THR(("  FEAT RAM Statistics[%20s]: Current memory usage %u bytes", hmibase::util::trace::getAppName(), SystemMemoryStatistic::GetCurrentMemoryUsage()));
   ETG_TRACE_FATAL_THR(("  FEAT RAM Statistics[%20s]: Peak memory usage %u bytes", hmibase::util::trace::getAppName(), SystemMemoryStatistic::GetPeakMemoryUsage()));
   ETG_TRACE_FATAL_THR(("  FEAT RAM Statistics[%20s]: Allocation count %u bytes", hmibase::util::trace::getAppName(), max));
   ETG_TRACE_FATAL_THR(("  FEAT RAM Statistics[%20s]: Maximum item size %u bytes", hmibase::util::trace::getAppName(), SystemMemoryStatistic::GetMaximumItemSize()));
   ETG_TRACE_FATAL_THR((" ################"));

   char fn[100];
   static unsigned int count = 0;
   memset(fn, 0, sizeof(fn));
#ifdef WIN32
   sprintf(fn, "MemStat_%s_%02d_Win32.txt", hmibase::util::trace::getAppName(), ++count);
#else
   sprintf(fn, "/tmp/MemStat_%s_%02d.txt", hmibase::util::trace::getAppName(), ++count);
#endif


   FILE* fp = fopen(fn, "w");
   if (fp)
   {
      fprintf(fp, "Current memory usage %u bytes\n", SystemMemoryStatistic::GetCurrentMemoryUsage());
      fprintf(fp, "Peak memory usage %u bytes\n", SystemMemoryStatistic::GetPeakMemoryUsage());
      fprintf(fp, "Allocation count %d bytes\n", max);
      fprintf(fp, "Maximum item size %u bytes\n\n", SystemMemoryStatistic::GetMaximumItemSize());

      for (int i = 0; i < max; i++)
      {
         const SystemMemoryStatistic::AllocationInfo* p = SystemMemoryStatistic::GetAllocationInfo(i);
         if (p != 0)
         {
#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED
            if (p->m_line != 0)
            {
               std::string s = p->m_fileName;
               int pos = s.rfind("ai_hmi_base");
               unsigned int offset = (pos != std::string::npos) ? pos : 0;
               fprintf(fp, "AllocationInfo size %5d bytes in %s(%d)\n", p->m_size, p->m_fileName + offset, p->m_line);
            }
            else
            {
               fprintf(fp, "AllocationInfo size %5d bytes in ???\n", p->m_size);
            }
#else
            fprintf(fp, "AllocationInfo size %5d\n", p->m_size);
#endif
         }
      }
      fclose(fp);
   }

   ETG_TRACE_FATAL_THR(("FeatMemStatistics[%20s]: Full statistic available under %s", hmibase::util::trace::getAppName(), fn));

   //SystemMemoryStatistic::Reset();
#else
   // additional defines in CGI/gensrc/FeatStd/Config.h required
   // enable FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED to get access to FEAT memory statistics,
   // enable FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED to get information about the location that allocates memory
   ETG_TRACE_FATAL_THR(("Memory statistics not enabled in the code, get in contact with HMIBase team"));
#endif

#ifdef FEATSTD_VIDEO_MEMORY_STATISTIC_ENABLED
   ETG_TRACE_FATAL_THR((" #### [%25s] VRAM statistics ####", hmibase::util::trace::getAppName()));
   ETG_TRACE_FATAL_THR(("  FEAT VRAM Statistics[%20s] (2D bitmaps): current = %u bytes, max = %u bytes", hmibase::util::trace::getAppName(), VideoMemoryStatistic::GetCurrentBitmapImage2DSizeUploaded(), VideoMemoryStatistic::GetMaxBitmapImage2DSizeUploaded()));
   ETG_TRACE_FATAL_THR(("  FEAT VRAM Statistics[%20s] (3D textures): current = %u bytes, max = %u bytes", hmibase::util::trace::getAppName(), VideoMemoryStatistic::GetCurrentTextureImageSizeUploaded(), VideoMemoryStatistic::GetMaxTextureImageSizeUploaded()));
   ETG_TRACE_FATAL_THR(("  FEAT VRAM Statistics[%20s] (3D vertices): current = %u bytes, max = %u bytes", hmibase::util::trace::getAppName(), VideoMemoryStatistic::GetCurrentVertexBufferSizeUploaded(), VideoMemoryStatistic::GetMaxVertexBufferSizeUploaded()));
   ETG_TRACE_FATAL_THR(("  FEAT VRAM Statistics[%20s] (window surfaces): current = %u bytes, max = %u bytes", hmibase::util::trace::getAppName(), VideoMemoryStatistic::GetCurrentWindowSurfaceSizeUploaded(), VideoMemoryStatistic::GetMaxWindowSurfaceSizeUploaded()));
   ETG_TRACE_FATAL_THR(("  FEAT VRAM Statistics[%20s] (frame buffers): current = %u bytes, max = %u bytes", hmibase::util::trace::getAppName(), VideoMemoryStatistic::GetCurrentFrameBufferObjectSizeUploaded(), VideoMemoryStatistic::GetMaxWindowSurfaceSizeUploaded()));
   ETG_TRACE_FATAL_THR((" ################"));
#else
   // additional defines in CGI/gensrc/FeatStd/Config.h required
   // enable FEATSTD_VIDEO_MEMORY_STATISTIC_ENABLED to get access to FEAT VRAM statistics
   ETG_TRACE_FATAL_THR(("VRAM statistics not enabled in the code, get in contact with HMIBase team"));
#endif

#if defined (FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED) || defined (FEATSTD_VIDEO_MEMORY_STATISTIC_ENABLED)
   cs.Leave();
#endif
#endif // WIN32
}


}
}
