/* ***************************************************************************************
* FILE:          IOSExtensions.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IOSExtensions.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "IOSExtensions.h"

#include <iomanip>
#include <ostream>

#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/Engine2D/Core/BitmapImage2D.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Layout/Margin.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/System/Mathematics/Matrix3.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Courier/Messaging/Message.h>
#include <Courier/Util/Identifier.h>
#include <Courier/Visualization/IViewHandler.h>
#include <Courier/Visualization/View.h>
#include <Courier/Visualization/ViewId.h>
#include <FeatStd/Util/String.h>
#include <TouchGestures/GestureBasetypes.h>
#include <hmibase/util/Geometry.h>
#include <View/CGI/Widget/Widget.h>

namespace Candera {

/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Bitmap& value)
{
   os << "w=" << value.GetWidth() << ",h=" << value.GetHeight();
   if ((value.GetName() != NULL) && (value.GetName()[0] != '\0'))
   {
      os << ',' << _HMIBASE_TO_STRING_NON_NULL(value.GetName());
   }
   return os;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const CanderaObject& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(value.GetName());
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Color& value)
{
   return os << value.GetRed() << ',' << value.GetGreen() << ',' << value.GetBlue() << ',' << value.GetAlpha();
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const GraphicDeviceUnit& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(value.GetName());
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Image2D& value)
{
   const Candera::BitmapImage2D* bitmapImage = Candera::Dynamic_Cast<const Candera::BitmapImage2D*>(&value);
   if ((bitmapImage != NULL) && (!bitmapImage->GetBitmap().PointsToNull()))
   {
      return os << *(bitmapImage->GetBitmap().GetPointerToSharedInstance());
   }
   else
   {
      os << "w=" << value.GetWidth() << ",h=" << value.GetHeight();
      if ((value.GetName() != NULL) && (value.GetName()[0] != '\0'))
      {
         os << ',' << _HMIBASE_TO_STRING_NON_NULL(value.GetName());
      }
      return os;
   }
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Margin& value)
{
   return os << value.GetLeft() << ',' << value.GetTop() << ',' << value.GetBottom() << ',' << value.GetRight();
}


/************************************************************************/
template <typename TMatrix, ::FeatStd::UInt8 ROW_COUNT, ::FeatStd::UInt8 COLUMN_COUNT>
static ::std::ostream& insertMatrix(::std::ostream& os, const TMatrix& matrix)
{
   for (::FeatStd::UInt8 row = 0; row < ROW_COUNT; ++row)
   {
      if (row > 0)
      {
         os << '|';
      }
      for (::FeatStd::UInt8 column = 0; column < COLUMN_COUNT; ++column)
      {
         if (column > 0)
         {
            os << ',';
         }
         os << matrix.Get(row, column);
      }
   }
   return os;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Matrix3& value)
{
   return insertMatrix<Matrix3, 3, 3>(os, value);
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Matrix3x2& value)
{
   return insertMatrix<Matrix3x2, 3, 2>(os, value);
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Matrix4& value)
{
   return insertMatrix<Matrix4, 4, 4>(os, value);
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Node& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(value.GetName());
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Node2D& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(value.GetName());
}


/************************************************************************/
template <typename TVector, FeatStd::UInt8 SIZE>
static ::std::ostream& insertVector(::std::ostream& os, const TVector& v)
{
   for (::FeatStd::UInt8 index = 0; index < SIZE; ++index)
   {
      if (index > 0)
      {
         os << ',';
      }
      os << v[index];
   }
   return os;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Vector2& value)
{
   return insertVector<Vector2, 2>(os, value);
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Vector3& value)
{
   return insertVector<Vector3, 3>(os, value);
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Vector4& value)
{
   return insertVector<Vector4, 4>(os, value);
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Rectangle& value)
{
   return os << value.GetLeft() << ',' << value.GetTop() << ',' << value.GetWidth() << ',' << value.GetHeight();
}


}

namespace Courier {
/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Identifier& value)
{
   ::std::ios::fmtflags outFlags(os.flags());
   os << "0x" << ::std::hex << value.HashValue();
   os.flags(outFlags);
   return os;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const ItemId& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(value.CStr());
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Message& value)
{
   ::std::ios::fmtflags outFlags(os.flags());
   os << "0x" << ::std::hex << value.GetId() << '.' << _HMIBASE_TO_STRING_NON_NULL(value.GetName());
   os.flags(outFlags);
   return os;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const TouchInfo& value)
{
   return os << "x=" << value.mX << ",y=" << value.mY << ",t=" << value.mTimestamp << ",p=" << static_cast<FeatStd::UInt16>(value.mPointerId) << ",src=" << value.mSourceId;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const View& value)
{
   return os << value.GetId();
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const ViewId& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(value.CStr());
}


}

namespace FeatStd {
/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const String& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(value.GetCString());
}


}

namespace hmibase {
namespace input {
namespace gesture {
::std::ostream& operator<<(::std::ostream& os, const GestureEvent& value)
{
   os << "type=" << value._gestureType << ",state=" << value._gestureState << ",dir=" << value._gestureDirection;

   if (value._isPoint1Valid)
   {
      os << ",p1=" << value._pt1;
   }
   if (value._isPoint1VelocityValid)
   {
      os << ",v1=" << value._velocity1;
   }

   if (value._isPoint2Valid)
   {
      os << ",p2=" << value._pt2;
   }
   if (value._isPoint2VelocityValid)
   {
      os << ",v2=" << value._velocity2;
   }

   if (value._isPoint3Valid)
   {
      os << ",p3=" << value._pt3;
   }
   if (value._isPoint3VelocityValid)
   {
      os << ",v3=" << value._velocity3;
   }

   os << ",r=" << value._rotateAngle;
   os << ",t=" << value._nTimestamp;

   return os;
}


}
}


namespace widget {

/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Widget& value)
{
   return os << _HMIBASE_TO_STRING_NON_NULL(const_cast<Widget&>(value).GetLegacyName());
}


}

namespace util {
namespace geometry {
/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Point& value)
{
   return os << value.x << ',' << value.y;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Rect& value)
{
   return os << value.ptMin.x << ',' << value.ptMin.y << ',' << value.ptMax.x << ',' << value.ptMax.y;
}


/************************************************************************/
::std::ostream& operator<<(::std::ostream& os, const Size& value)
{
   return os << value.width << ',' << value.height;
}


}
}


}
