/* ***************************************************************************************
* FILE:          GuiInfo.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GuiInfo.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "GuiInfo.h"

#include <Candera/Engine2D/Core/Node2D.h>
#include <Courier/Visualization/IViewHandler.h>
#include <Courier/Visualization/ViewScene2D.h>

#include "Courier/Platform/CriticalSectionLocker.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GuiInfo.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


namespace hmibase {
namespace trace {

Courier::IViewHandler* GuiInfo::s_viewHandler = NULL;
Courier::Platform::CriticalSection GuiInfo::_critical;

void GuiInfo::setViewHandler(Courier::IViewHandler* viewHandler)
{
   s_viewHandler = viewHandler;
}


static void processNode(Candera::Node2D& node, std::vector<std::string>& output, size_t level = 0)
{
   /*static */char buffer[1024];
   if (level < sizeof(buffer))
   {
      Candera::Rectangle effectiveBoundingRectangle;
      node.GetEffectiveBoundingRectangle(effectiveBoundingRectangle);

      memset(buffer, ' ', level);//indentation based on node depth
      SNPRINTF(buffer + level,
               sizeof(buffer) - level,
               "<Node name=\"%s%s\" worldPos=\"%.0f,%.0f\" pos=\"%.0f,%.0f\" bounds=\"%.0f,%.0f,%.0f,%.0f\" effectiveRank=\"%d\" address=\"%p\" %s>",
               node.IsRenderingEnabled() ? "" : "*",
               node.GetName(),
               node.GetWorldPosition().GetX(),
               node.GetWorldPosition().GetY(),
               node.GetPosition().GetX(),
               node.GetPosition().GetY(),
               effectiveBoundingRectangle.GetLeft(),
               effectiveBoundingRectangle.GetTop(),
               effectiveBoundingRectangle.GetWidth(),
               effectiveBoundingRectangle.GetHeight(),
               node.GetEffectiveRenderOrderRank(),
               &node,
               (node.GetChildCount() > 0) ? "" : "/"
              );
      buffer[sizeof(buffer) - 1] = '\0';
      output.push_back(std::string(buffer));

      Candera::Node2D* child = node.GetFirstChild();
      while (child != NULL)
      {
         processNode(*child, output, level + 1);
         child = child->GetNextSibling();
      }

      if (node.GetChildCount() > 0)
      {
         memset(buffer, ' ', level);//indentation based on node depth
         strncpy(buffer + level, "</Node>", sizeof(buffer) - level);
         output.push_back(std::string(buffer));
      }
   }
}


void GuiInfo::printScene(Candera::Scene2D& scene)
{
   std::vector<std::string> output;
   processNode(scene, output);

   for (std::vector<std::string>::iterator it = output.begin(); it != output.end(); ++it)
   {
      ETG_TRACE_USR1_THR(("%s", (*it).c_str()));
   }
}


void GuiInfo::printScene(const char* sceneName)
{
   if (s_viewHandler != NULL)
   {
      Courier::Platform::CriticalSectionLocker csLock(&_critical);
      Courier::ViewScene2D* viewScene = dynamic_cast<Courier::ViewScene2D*>(s_viewHandler->FindView(Courier::ViewId(sceneName)));
      if ((viewScene != NULL)
            && (viewScene->GetScene2DContext() != NULL)
            && (viewScene->GetScene2DContext()->GetScene() != NULL))
      {
         printScene(*viewScene->GetScene2DContext()->GetScene());
      }
      else
      {
         ETG_TRACE_ERR_THR(("printScene - scene '%s' not available", sceneName));
      }
   }
   else
   {
      ETG_TRACE_FATAL_THR(("printScene - ViewHandler was not set with setViewHandler()"));
   }
}


}
}
