/* ***************************************************************************************
* FILE:          GetViewMetaInfo.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GetViewMetaInfo.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#ifdef WIN32
#include "hmibase/util/win32/oss/tinyxml/include/tinyxml.h"
#else
#include "tinyxml.h"
#endif

#include "CanderaAssetLoader/AssetLoaderBase/AssetContext.h"
#include "CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h"

#include "CanderaAssetLoader/AssetLoaderBase/AssetDescriptor.h"

#include "CanderaPlatform/Device/Common/Effects/TextBrushBlend.h"

#include "View/CGI/CgiExtensions/AppViewHandler.h"
#include "View/CGI/CgiExtensions/PopupConfig.h"
#include "Trace/GetViewMetaInfo.h"

#include "AppBase/ILM_Accessor.h"

#include <set>

#include "TraceQueryHelper.hpp"
#include "Courier/Messaging/MessageReferrer.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GetViewMetaInfo.cpp.trc.h"
#endif

namespace hmibase {
namespace trace {

static std::map<std::string, char*> oldViews;

class SceneDumper : public Courier::ViewVisitor, public hmibase::view::RenderJobStrategy::RenderJobListener
{
   private:
      std::string _viewName;
      Courier::ViewId _viewId;
      bool _surfaceOnly;

      std::set<Candera::GraphicDeviceUnit*> _gdusToObserve;

   public:
      SceneDumper() : _surfaceOnly(false)
      {
      }

      virtual ~SceneDumper()
      {
      }

      void SetScreenShotData(const char* viewName, bool surfaceOnly = false)
      {
         _viewName = viewName;
         _viewId = Courier::ViewId(_viewName.c_str());
         _surfaceOnly = surfaceOnly;
      }

      virtual bool OnBegin()
      {
         return true;
      }

      virtual bool Visit(Courier::ViewContainer* /*viewContainer*/)
      {
         return true;
      }

#if defined(CANDERA_2D_ENABLED)
      virtual void Visit(Courier::ViewScene2D* view)
      {
         if (view && view->IsRenderingEnabled() && (_viewId == view->GetId()))
         {
            // let's make a screenshot
            Courier::ViewScene2D::CameraPtrVector cameras = view->GetCameraPtrVector();
            for (Courier::ViewScene2D::CameraPtrVector::Iterator it = cameras.Begin(); it != cameras.End(); ++it)
            {
               if ((*it)->IsEffectiveRenderingEnabled())
               {
                  Candera::GraphicDeviceUnit* gdu = (*it)->GetRenderTarget()->GetGraphicDeviceUnit();
                  _gdusToObserve.insert(gdu);
               }
            }

            if (_gdusToObserve.empty() == false)
            {
               hmibase::view::RenderJobStrategy::RegisterListener(this);
            }
         }
      }
#endif

#if defined(CANDERA_3D_ENABLED)
      virtual void Visit(Courier::ViewScene3D* /*view*/)
      {
      }
#endif

      virtual void OnEnd()
      {
      }

      virtual void OnPostSwapBuffer(Courier::Gdu& gdu)
      {
         std::set<Candera::GraphicDeviceUnit*>::iterator it = _gdusToObserve.begin();
         while (it != _gdusToObserve.end())
         {
            if (gdu.GetRenderTarget()->GetGraphicDeviceUnit() == *it)
            {
               if (_surfaceOnly)
               {
                  hmibase::ILM_Accessor::takeSnapshot(_viewName.c_str());
               }
               else
               {
                  unsigned int surfaceId;
                  if (HMIRenderViewVisitor::getSurfaceId(gdu.GetRenderTarget(), surfaceId))
                  {
                     hmibase::ILM_Accessor::takeSurfaceSnapshot(_viewName.c_str(), surfaceId);
                  }
               }

               _gdusToObserve.erase(it);
               break;
            }
            else
            {
               ++it;
            }
         }

         if (_gdusToObserve.empty())
         {
            hmibase::view::RenderJobStrategy::DeregisterListener(this);
            _viewName = "";
            _viewId = Courier::ViewId();
         }
      }
};


class WidgetInfoCallback : public WidgetCheckCallback
{
   public:
      explicit WidgetInfoCallback(GetViewMetaInfo* mi)
      {
         _mi = mi;
      }
      virtual bool CheckWidget(Candera::Widget2D* w)
      {
         if (_mi)
         {
            _mi->onWidgetInfo(w);
         }
         return false;
      }
   private:
      GetViewMetaInfo* _mi;
};


bool isValidUtf8(const char* input)
{
   int nb = 0;

   for (const char* c = input; *c;)
   {
      if (!(*c & 0x80))
      {
         nb = 0;
      }
      else if ((*c & 0xc0) == 0x80)
      {
         return false;
      }
      else if ((*c & 0xe0) == 0xc0)
      {
         nb = 1;
      }
      else if ((*c & 0xf0) == 0xe0)
      {
         nb = 2;
      }
      else if ((*c & 0xf8) == 0xf0)
      {
         nb = 3;
      }
      else
      {
         return false;
      }

      ++c;
      for (; nb > 0; --nb, ++c)
      {
         if ((*c & 0xc0) != 0x80)
         {
            return false;
         }
      }
   }

   return true;
}


GetViewMetaInfo::GetViewMetaInfo() :
   _viewHandler(NULL),
   _timer(0),
   _isViewIdRecordingRunning(true),
   _textMetaDataCollectionRunning(false),
   _sceneCounter(0),
   _placeHolderRequired(false)
{
   Courier::ViewResMsg::Subscribe(::Courier::ComponentId(::Courier::ComponentType::View));
}


GetViewMetaInfo::~GetViewMetaInfo()
{
   timerStop(true);
   _viewHandler = 0;
   _scenes.clear();
   _sceneList.clear();
   _sceneCounter = 0;
} //lint !e1579 Pointer _timer is freed in timerStop()

void GetViewMetaInfo::setViewHandler(AppViewHandler* viewHandler)
{
   _viewHandler = viewHandler;
}


void GetViewMetaInfo::setupSurface()
{
   //call screen broker to show the surface
   ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(), ScreenBrokerClient::GetInstance().SurfaceId());

   if (_viewHandler)
   {
      Courier::RenderReqMsg* renderReqMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)();
      if (NULL != renderReqMsg)
      {
         ::Courier::MessageReferrer tMsg(renderReqMsg);
         _viewHandler->OnMessage(*renderReqMsg);
      }
   }
}


bool GetViewMetaInfo::sendPopupShowReqMsg(::Courier::ViewId const& viewId)
{
   if (_viewHandler)
   {
      ::PopupReqMsg* popupReqMsg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::ForceShow, viewId, static_cast<Courier::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY));
      if (NULL != popupReqMsg)
      {
         ::Courier::MessageReferrer tMsg(popupReqMsg);
         _viewHandler->OnMessage(*popupReqMsg);
      }
      return true;
   }
   return false;
}


bool GetViewMetaInfo::sendPopupHideReqMsg(::Courier::ViewId const& viewId)
{
   if (_viewHandler)
   {
      ::PopupReqMsg* popupReqMsg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, viewId, static_cast<Courier::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY));
      if (NULL != popupReqMsg)
      {
         ::Courier::MessageReferrer tMsg(popupReqMsg);
         _viewHandler->OnMessage(*popupReqMsg);
      }
      return true;
   }
   return false;
}


bool GetViewMetaInfo::onCourierMessage(const CreateTextMetaDataCollectionReqMsg& /*msg*/)
{
   ETG_TRACE_FATAL_THR(("Start text meta data collect"));

   _isViewIdRecordingRunning = false;
   _textMetaDataCollectionRunning = true;

   setupSurface();
   unsigned int surfaceid = ScreenBrokerClient::GetInstance().SurfaceId();

   Courier::Message* msg = COURIER_MESSAGE_NEW(ApplicationStateUpdMsg)(hmibase::FORCE_IN_FOREGROUND, ScreenBrokerClient::GetInstance().GetDisplayID(surfaceid));
   if (msg)
   {
      msg->Post();
   }

   //request for view placeholder to avoid unload of intermediate rendertarget
   _placeHolderRequired = _createdViewIds.size() == 0 ? true : false;

   nextScene(true);
   return true;
}


bool GetViewMetaInfo::onCourierMessage(const DumpScreenReqMsg& msg)
{
   _isViewIdRecordingRunning = false;
   static char* pOldView = NULL;
   static char* pCurrentView = NULL;

   static SceneDumper s_SceneDumper;

   if (_viewHandler)
   {
      _viewHandler->RegisterViewVisitor(s_SceneDumper);
   }

   unsigned int surfaceid = ScreenBrokerClient::GetInstance().SurfaceId();
   ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(), surfaceid);

   Courier::Message* show = COURIER_MESSAGE_NEW(ApplicationStateUpdMsg)(hmibase::FORCE_IN_FOREGROUND, ScreenBrokerClient::GetInstance().GetDisplayID(surfaceid));
   if (show)
   {
      show->Post();
   }
   delete [] pOldView;
   pOldView = pCurrentView;
   size_t sz = strlen(msg.GetViewName().GetCString());
   pCurrentView = new char[sz + 1];
   memcpy(pCurrentView, msg.GetViewName().GetCString(), sz + 1);

   _placeHolderRequired = false;
   if (msg.GetViewName().IsEmpty() == false)
   {
      s_SceneDumper.SetScreenShotData(pCurrentView, msg.GetDumpAll() ? false : true);
      switchToView(pCurrentView);
   }
   return true;
}


bool GetViewMetaInfo::onCourierMessage(const Courier::ViewResMsg& msg)
{
   Courier::ViewId id = msg.GetViewId();
   if (_isViewIdRecordingRunning)
   {
      switch (msg.GetViewAction())
      {
         case Courier::ViewAction::Create:
         {
            if (id.CStr() != 0)
            {
               _createdViewIds.insert(id);
            }
         }
         break;
         case Courier::ViewAction::CreateAll:
            _createdViewIds.clear();
            if (_sceneList.size() == 0)
            {
               getScenes();
            }

            for (std::vector<std::string>::iterator it = _sceneList.begin(); it != _sceneList.end(); ++it)
            {
               _createdViewIds.insert(Courier::ViewId(it->c_str()));
            }
            break;
         case Courier::ViewAction::Destroy:
         {
            if (id.CStr() != 0)
            {
               _createdViewIds.erase(id);

               std::map<std::string, char*>::iterator it;
               std::string viewName(id.CStr());
               it = oldViews.find(viewName);
               if (it != oldViews.end())
               {
                  delete [](it->second);
                  oldViews.erase(it);
               }
            }
         }
         break;
         case Courier::ViewAction::DestroyAll:
         {
            _createdViewIds.clear();
            for (std::map<std::string, char*>::iterator it = oldViews.begin(); it != oldViews.end(); ++it)
            {
               delete [](it->second);
            }
            oldViews.clear();
         }
         break;
         default:
            break;
      }
   }
   return false;
}


bool GetViewMetaInfo::onCourierMessage(const Courier::ActivationResMsg& msg)
{
   // the last requested view is activated
   _lastActViewId = msg.GetViewId();
   if ((_lastReqViewStr.length() != 0) && (_textMetaDataCollectionRunning))
   {
      if (_lastActViewId == _lastReqViewId)
      {
         if (_viewHandler)
         {
            Courier::RenderReqMsg* renderReqMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)();
            if (NULL != renderReqMsg)
            {
               ::Courier::MessageReferrer tMsg(renderReqMsg);
               _viewHandler->OnMessage(*renderReqMsg);
            }

            Courier::View* view = _viewHandler->FindView(_lastActViewId);
            if (view != NULL)
            {
               ETG_TRACE_FATAL_THR(("Collect text meta data from scene '%s'", view->GetId().CStr()));
               WidgetInfoCallback callback(this);
               WidgetCheckReqMsg msg(&callback);

               TiXmlElement element_Scene("Scene");

               element_Scene.SetAttribute("Name", _lastReqViewStr.c_str());
               _scenes.push_back(element_Scene);

               view->OnMessage(msg);
            }
         }
      }
      else
      {
         ETG_TRACE_FATAL_THR(("ActivationResMsg differs from last requested: '%50s' != '%s'", _lastActViewId.CStr(), _lastReqViewStr.c_str()));
      }
      timerStart();
   }
   return false;
}


size_t GetViewMetaInfo::getScenes()
{
   std::vector<std::string> popups;
   std::vector<std::string> offscreen;

   if (_viewHandler == 0)
   {
      return 0;
   }

   Candera::DefaultAssetProvider* myassetProvider = dynamic_cast<Candera::DefaultAssetProvider*>(_viewHandler->GetAssetProvider());
   if (myassetProvider == 0)
   {
      return 0;
   }

   const Candera::AssetDescriptor& assetDescriptor = myassetProvider->GetAssetDescriptor();
   for (Candera::AssetDescriptor::AssetIdIterator scene2DIdList = assetDescriptor.GetAssetIdIterator(Candera::Scene2DLib); scene2DIdList.IsValid(); ++scene2DIdList)

   {
      const char* scene2D = myassetProvider->GetNameById(Candera::Scene2DLib, *scene2DIdList, 0);

      std::string str(scene2D);
      if (isPopup(scene2D))
      {
         popups.push_back(str);
      }
      else if (str.find("OffScreen") != std::string::npos)
      {
         offscreen.push_back(str);
      }
      else
      {
         _sceneList.push_back(str);
      }
   }

   if (popups.size() > 0)
   {
      _sceneList.insert(_sceneList.end(), popups.begin(), popups.end());
   }
   if (offscreen.size() > 0)
   {
      _sceneList.insert(_sceneList.end(), offscreen.begin(), offscreen.end());
   }

   return _sceneList.size();
}


bool GetViewMetaInfo::isPopup(const char* view)
{
   return isPopup(::hmibase::view::ViewIdentifier(view));
}


bool GetViewMetaInfo::isPopup(const ::hmibase::view::ViewIdentifier& viewId)
{
   ::hmibase::view::ST_POPUP_CONFIG* pp = ::hmibase::view::PopUpConfig::FindPopupConfig(viewId, static_cast<Courier::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY));
   if (pp)
   {
      return true;
   }
   return false;
}


bool GetViewMetaInfo::isNotValid(const char* view)
{
   const char* s1 = strstr(view, "LAYOUT");
   if (s1)
   {
      return true;
   }
   return false;
}


void GetViewMetaInfo::switchToView(const char* view)
{
   Courier::ViewId viewId(view);
   Courier::ViewPlaceholderReqMsg viewPlaceholderReqMsg(_lastActViewId, true);
   Courier::ViewFacade viewFacade;
   viewFacade.Init(_viewHandler);
   viewFacade.OnViewComponentMessage(viewPlaceholderReqMsg);

   if ((_viewHandler) && (_lastActViewId != viewId))
   {
      if (isPopup(::hmibase::view::ViewIdentifier(_lastActViewId.CStr())))
      {
         sendPopupHideReqMsg(_lastActViewId);

         Courier::RenderReqMsg* renderReqMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)(true);
         if (NULL != renderReqMsg)
         {
            ::Courier::MessageReferrer tRenderMsg(renderReqMsg);
            _viewHandler->OnMessage(*renderReqMsg);
         }
      }
      // destroy old (main) scene if it was not part of the list of views to recover
      else if ((_createdViewIds.find(Courier::ViewId(_lastActViewId)) != _createdViewIds.end()) && (_lastActViewId != Courier::ViewId()))
      {
         ExtendedViewReqMsg* extendedViewReqMsg = COURIER_MESSAGE_NEW(ExtendedViewReqMsg)(hmibase::views::HideAndDestroy, _lastActViewId, 0);
         if (NULL != extendedViewReqMsg)
         {
            ::Courier::MessageReferrer tExtView(extendedViewReqMsg);
            _viewHandler->OnMessage(*extendedViewReqMsg);
         }
      }

      if (viewId != Courier::ViewId())
      {
         if (isPopup(::hmibase::view::ViewIdentifier(viewId.CStr())))
         {
            sendPopupShowReqMsg(viewId);

            Courier::RenderReqMsg* renderReqMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)(true);
            if (NULL != renderReqMsg)
            {
               ::Courier::MessageReferrer tMsg(renderReqMsg);
               _viewHandler->OnMessage(*renderReqMsg);
            }
         }
         else
         {
            ExtendedViewReqMsg* extendedViewReqMsg = COURIER_MESSAGE_NEW(ExtendedViewReqMsg)(hmibase::views::CreateAndShow, viewId, 0);
            if (NULL != extendedViewReqMsg)
            {
               ::Courier::MessageReferrer tExtMsg(extendedViewReqMsg);
               _viewHandler->OnMessage(*extendedViewReqMsg);
            }
         }
      }
      _lastReqViewId = viewId;
      _lastReqViewStr = view;
   }

   if (_viewHandler)
   {
      Courier::RenderReqMsg* renderReqMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)(true);
      if (NULL != renderReqMsg)
      {
         ::Courier::MessageReferrer tRenderMsg(renderReqMsg);
         _viewHandler->OnMessage(*renderReqMsg);
      }
   }
}


void GetViewMetaInfo::HideView(const char* view)
{
   std::string viewName(view);
   Courier::ViewId viewId(view);
   Courier::ViewPlaceholderReqMsg viewPlaceholderReqMsg(viewId, true);
   Courier::ViewFacade viewFacade;
   viewFacade.Init(_viewHandler);
   viewFacade.OnViewComponentMessage(viewPlaceholderReqMsg);

   if (viewName == "DELETE_ALL")
   {
      _lastActViewId = Courier::ViewId();
      _lastReqViewId = Courier::ViewId();
      _lastReqViewStr = "";
      for (std::set<Courier::ViewId>::iterator it = _createdViewIds.begin(); it != _createdViewIds.end(); ++it)
      {
         ExtendedViewReqMsg* extendedViewReqMsg = COURIER_MESSAGE_NEW(ExtendedViewReqMsg)(hmibase::views::HideAndDestroy, *it, 0);
         if (NULL != extendedViewReqMsg)
         {
            ::Courier::MessageReferrer tExtViewMsg(extendedViewReqMsg);
            _viewHandler->OnMessage(*extendedViewReqMsg);
         }
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("GetViewMetaInfo::HideView "));
      if (_lastActViewId == viewId)
      {
         if (isPopup(::hmibase::view::ViewIdentifier(viewId.CStr())))
         {
            sendPopupHideReqMsg(viewId);

            Courier::RenderReqMsg* renderReqMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)(true);
            if (NULL != renderReqMsg)
            {
               ::Courier::MessageReferrer tRenderMsg(renderReqMsg);
               _viewHandler->OnMessage(*renderReqMsg);
            }
         }
         // destroy old (main) scene if it was not part of the list of views to recover
         else if ((_createdViewIds.find(Courier::ViewId(viewId)) != _createdViewIds.end()) && (viewId != Courier::ViewId()))
         {
            ETG_TRACE_USR4_THR(("Last Activated ViewId is destroyed"));
            ExtendedViewReqMsg* extendedViewReqMsg = COURIER_MESSAGE_NEW(ExtendedViewReqMsg)(hmibase::views::HideAndDestroy, viewId, 0);
            if (NULL != extendedViewReqMsg)
            {
               ::Courier::MessageReferrer tExtMsg(extendedViewReqMsg);
               _viewHandler->OnMessage(*extendedViewReqMsg);
            }
         }
         //Clear the Last Activated ViewID as it is destroyed
         _lastActViewId = Courier::ViewId();
         _lastReqViewId = Courier::ViewId();
         _lastReqViewStr = "";
      }
   }
   Courier::RenderReqMsg* renderReqMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)(true);
   if (NULL != renderReqMsg)
   {
      ::Courier::MessageReferrer tMsg(renderReqMsg);
      _viewHandler->OnMessage(*renderReqMsg);
   }
}


void GetViewMetaInfo::printAllScenes()
{
   if (getScenes() == 0)
   {
      return;
   }

   int ii = 1;
   for (_it = _sceneList.begin(); _it != _sceneList.end(); ++_it)
   {
      ETG_TRACE_FATAL_THR((" SCENE%02d  %s", ii++, _it->c_str()));
   }
}


bool GetViewMetaInfo::nextScene(bool restart)
{
   _lastReqViewStr = "";

   if (restart)
   {
      _scenes.clear();
      _sceneList.clear();
      _sceneCounter = 0;

      if (getScenes() > 0)
      {
         _it = _sceneList.begin();
      }
   }

   if (_sceneList.size())
   {
      _sceneCounter++;
      while (_it != _sceneList.end() && isNotValid(_it->c_str()))
      {
         ++_it;
         _sceneCounter++;
      }
      if (_it != _sceneList.end())
      {
         ETG_TRACE_FATAL_THR(("Switch to scene [%u/%u] %s", _sceneCounter, _sceneList.size(), _it->c_str()));
         switchToView(_it->c_str());
         ++_it;
         return true;
      }
      else
      {
         switchToView("");
      }
   }

   ETG_TRACE_FATAL_THR(("Text meta data collect finished"));

   timerStop();

   _isViewIdRecordingRunning = true;

   TiXmlDocument doc;
   doc.InsertEndChild(TiXmlDeclaration("1.0", "utf-8", ""));

   TiXmlElement element("TextMetaData");
   element.SetAttribute("AppName", hmibase::trace::getAppName().c_str());
   element.SetAttribute("xsi:noNamespaceSchemaLocation", "TextMetaData.xsd");
   element.SetAttribute("xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");

   for (std::vector<TiXmlElement>::iterator it = _scenes.begin(); it != _scenes.end(); ++it)
   {
      element.InsertEndChild(*it);
   }

   doc.InsertEndChild(element);

#ifdef WIN32
   const char* format = "d:\\TextMetaData_%s.xml";
#else
   const char* format = "/tmp/TextMetaData_%s.xml";
#endif
   char metaDataXml[100];
   SNPRINTF(metaDataXml, sizeof metaDataXml, format, hmibase::trace::getAppName().c_str());

   if (doc.SaveFile(metaDataXml))
   {
      ETG_TRACE_FATAL_THR(("Text meta data xml '%s'", metaDataXml));
   }

   _textMetaDataCollectionRunning = false;

   return false;
}


void GetViewMetaInfo::dumpSceneOnSurface(const char* name, unsigned int surface)
{
   hmibase::ILM_Accessor::takeSurfaceSnapshot(name, surface);
}


void GetViewMetaInfo::dumpScreen(const char* name)
{
   hmibase::ILM_Accessor::takeSnapshot(name);
}


bool GetViewMetaInfo::onCourierMessage(const TimerExpiredMsg& msg)
{
   if (msg.GetTimer()  == _timer)
   {
      nextScene();
      return true;
   }
   return false;
}


void GetViewMetaInfo::timerStart()
{
   if (_timer == NULL)
   {
      _timer = CANDERA_NEW(Util::Timer);
      FEATSTD_DEBUG_ASSERT(_timer != 0);
      if (_timer == 0)
      {
         return;
      }
      _timer->setName("GetViewMetaInfo", "");
   }

   if (!_timer->stopped())
   {
      _timer->stop();
   }
   _timer->setTimeout(0, 1000);
   _timer->start();
}


void GetViewMetaInfo::timerStop(bool deleteInstance)
{
   if (_timer != NULL)
   {
      if (!_timer->stopped())
      {
         _timer->stop();
      }
      if (deleteInstance)
      {
         CANDERA_DELETE(_timer);
         _timer = NULL;
      }
   }
}


bool GetViewMetaInfo::onWidgetInfo(Candera::Widget2D* w)
{
   static unsigned int invalidStrings = 0;
   if (w == NULL)
   {
      return false;
   }

   Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(w);
   ::hmibase::trace::TraceQueryHelper::IWidgetTraceQueryHelper::TextProperties textProperties;
   ::hmibase::trace::TraceQueryHelper::GetInstance().GetTextWidgetProperties(w, textProperties);

   if (textProperties.isValid)
   {
      TiXmlElement element_WidgetInstance("WidgetInstance");

      const FeatStd::Char* name = w->GetLegacyName();

      element_WidgetInstance.SetAttribute("Name", name);
      element_WidgetInstance.SetAttribute("Type", wb->GetTypeName());

      if (textProperties.style == NULL)
      {
         return false;
      }

      {
         Candera::TextRendering::Font fnt = textProperties.style->GetDefaultFont();
         Candera::TextRendering::Metrics mt = textProperties.style->GetMetrics();

         TiXmlElement element_Text("Text");

         if (isValidUtf8(textProperties.text.GetCString()))
         {
            element_Text.SetAttribute("Value", textProperties.text.GetCString());
         }
         else
         {
            char buf[100];
            TiXmlComment comment;

            SNPRINTF(buf, sizeof(buf), "Invalid UTF-8 character found, string replaced, identifier in trace %u", ++invalidStrings);

            comment.SetValue(buf);
            element_WidgetInstance.InsertEndChild(comment);
            element_Text.SetAttribute("Value", "");

            ETG_TRACE_FATAL_THR(("Invalid UTF-8 string detected, invalid string counter %u, string '%s' ", invalidStrings, textProperties.text.GetCString()));
         }

         char textID[40];
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
         SNPRINTF(textID, sizeof(textID), "0x%016lX", textProperties.text.GetId());  // 0, if a free text or hash_id of identifier in the translation table
#else
         SNPRINTF(textID, sizeof(textID), "0x%08X", textProperties.text.GetId());  // 0, if a free text or hash_id of identifier in the translation table
#endif
         element_Text.SetAttribute("ID", textID);
         element_WidgetInstance.InsertEndChild(element_Text);

         TiXmlElement element_Dimension("Dimension");

         element_Dimension.SetAttribute("Width", static_cast<int>(textProperties.dimension.GetX()));
         element_Dimension.SetAttribute("Height", static_cast<int>(textProperties.dimension.GetY()));
         element_Dimension.SetAttribute("MaxNumOfLines", static_cast<int>(textProperties.maxNumOfLines));
         element_WidgetInstance.InsertEndChild(element_Dimension);

         TiXmlElement element_Style("Style");

         element_Style.SetAttribute("Name", (textProperties.style.PointsToNull()) ? "" : (textProperties.style->GetName() != 0 ? textProperties.style->GetName() : ""));
         element_Style.SetAttribute("LineHeight", mt.lineHeight);
         element_WidgetInstance.InsertEndChild(element_Style);

         TiXmlElement element_Font("Font");

         element_Font.SetAttribute("Name", fnt.GetFaceName());
         element_Font.SetAttribute("Height", fnt.GetHeight());
         element_WidgetInstance.InsertEndChild(element_Font);

         ////////////////////////////////////////////////////////////

         Candera::Node2D* node = w->GetNode();
         if (node != 0)
         {
            float width_WithScale = 0.0f;
            float heigth_WithScale = 0.0f;
            Candera::RenderNode* rNode = dynamic_cast<Candera::RenderNode*>(node);
            if (rNode)
            {
               Candera::Vector2 v2OriginalScale = rNode->GetScale();
               Candera::Effect2D* eNodeEffect = rNode->GetEffect(0);

               if (eNodeEffect && eNodeEffect->IsTypeOf(Candera::TextBrushBlend::GetTypeId()))
               {
                  Candera::TextBrushBlend* sEffect = dynamic_cast<Candera::TextBrushBlend*>(eNodeEffect);
                  if (sEffect != NULL)
                  {
                     Candera::Rectangle  r;
                     sEffect->GetBoundingRectangle(r);
                     width_WithScale = v2OriginalScale.GetX() * r.GetWidth(); // daselbe wie wt->cofGetWidth()
                     heigth_WithScale = v2OriginalScale.GetY() * r.GetHeight(); // daselbe wie wt->cofGetHeigth()
                  }
               }
               else
               {
                  ETG_TRACE_FATAL_THR(("----!!!Brush-------------------------------------------"));
                  ETG_TRACE_FATAL_THR(("----!!!Brush-------------------------------------------"));
                  ETG_TRACE_FATAL_THR(("----!!!Brush-------------------------------------------"));
               }
            }

            Candera::Vector2 pt = node->GetWorldPosition();

            Candera::Rectangle r;
            node->GetEffectiveBoundingRectangle(r);

            TiXmlElement element_Position("Position");

            element_Position.SetAttribute("x", static_cast<int>(pt.GetX()));
            element_Position.SetAttribute("y", static_cast<int>(pt.GetY()));
            element_WidgetInstance.InsertEndChild(element_Position);

            //if (r.GetWidth() != width || (r.GetHeight() != height))
            {
               // EffectiveBoundingRectangle
               TiXmlElement element_BoundingRect("BoundingRect");

               element_BoundingRect.SetAttribute("Width", static_cast<int>(r.GetWidth()));
               element_BoundingRect.SetAttribute("Height", static_cast<int>(r.GetHeight()));
               element_WidgetInstance.InsertEndChild(element_BoundingRect);
            }

            if ((width_WithScale > 0.0f)
                  || (heigth_WithScale > 0.0f))
            {
               // Scaled dimension
               TiXmlElement element_ScaledDimension("ScaledDimension");

               element_ScaledDimension.SetAttribute("Width", static_cast<int>(width_WithScale));
               element_ScaledDimension.SetAttribute("Height", static_cast<int>(heigth_WithScale));
               element_WidgetInstance.InsertEndChild(element_ScaledDimension);
            }

            // get the maximum size
            TiXmlElement element_MaxSize("MaximumSize");
            element_MaxSize.SetAttribute("Width", static_cast<int>(textProperties.maxDimension.GetX()));
            element_MaxSize.SetAttribute("Height", static_cast<int>(textProperties.maxDimension.GetY()));
            element_WidgetInstance.InsertEndChild(element_MaxSize);

            // get referred text ids
            TiXmlElement element_ID("ID");
            element_ID.SetAttribute("ReferedIds", textProperties.referredTextId.GetCString());
            element_WidgetInstance.InsertEndChild(element_ID);
         }
      }

      // Widget properties dump
#if 0
      Candera::MetaInfo::WidgetMetaInfo* metaInfo = wb->GetMetaInfo();
      if (metaInfo != 0)
      {
         for (int i = 0; i < metaInfo->GetItemCount(); ++i)
         {
            const char* propertyName = metaInfo->GetItem(i)->GetName();
            Candera::MetaInfo::PropertyMetaInfo<Candera::WidgetBase>* propertyMetaInfo = metaInfo->LookupItem(propertyName);
            if (propertyMetaInfo)
            {
               char buf[1000];
               buf[0] = '\0';

               if (propertyMetaInfo->Get(wb, buf, sizeof(buf)))
               {
                  TiXmlElement element_WidgetProperty("Widgetproperty");

                  element_WidgetProperty.SetAttribute("Name", propertyName);
                  element_WidgetProperty.SetAttribute("Value", buf);
                  element_WidgetInstance.InsertEndChild(element_WidgetProperty);
               }
            }
         }
      }
#endif

      // add the WidgetInstance to the last element on the stack what is the current scene
      _scenes.back().InsertEndChild(element_WidgetInstance);
   }

   return true;
}


bool GetViewMetaInfo::onCourierMessage(const ManualSceneActionReqMsg& msg)
{
   if ((msg.GetAction() == hmibase::SHOW_SCENE) || (msg.GetAction() == hmibase::HIDE_SCENE))
   {
      char* pCurrentView = NULL;

      std::map<std::string, char*>::iterator it;
      std::string viewName(msg.GetSceneName().CStr());

      it = oldViews.find(viewName);
      if (it == oldViews.end())
      {
         size_t sz = strlen(msg.GetSceneName().CStr());
         pCurrentView = new char[sz + 1];
         memcpy(pCurrentView, msg.GetSceneName().CStr(), sz + 1);

         oldViews.insert(std::pair<std::string, char*>(viewName, pCurrentView));
      }
      else
      {
         pCurrentView = it->second;
      }

      if (pCurrentView != NULL)
      {
         if (msg.GetAction() == hmibase::SHOW_SCENE)
         {
            switchToView(pCurrentView);
         }
         else
         {
            HideView(pCurrentView);
         }
      }
   }
   else
   {
      ETG_TRACE_FATAL_THR(("Invalid Scene Change Operation"));
   }
   return true;
}


}
}
