/* ***************************************************************************************
* FILE:          TimerObjectPool.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TimerObjectPool is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SpreadGestureDetector.cpp.trc.h"
#endif

#include "TimerObjectPool.h"


TimerObjectPool::TimerObjectPool() :
   _isInitialized(false),
   _firstFree(0),
   _lastFree(0),
   _objectSize(0),
   _numGrow(0),
   _numAllocated(0),
   _bytesAllocated(0),
   _numCurrentlyUsed(0),
   _blocks(NULL),
   _headBlk(NULL)
{
   // empty
}


TimerObjectPool::~TimerObjectPool()
{
   for (Buffer* pDel = _blocks; pDel != NULL;)
   {
      Buffer* ptmp = pDel->next;
      delete[] pDel->data;
      delete pDel;
      pDel = ptmp;
   }
   _blocks = NULL;
   _headBlk = NULL;
   _firstFree = 0;
   _lastFree = 0;
}


void TimerObjectPool::addFreeObject(Link* object)
{
   //reset next index
   object->next = 0;
   if (_lastFree == 0)
   {
      //set object to first position
      _firstFree = object;
      _lastFree = object;
   }
   else
   {
      //set object to next free position
      _lastFree->next = object;
      _lastFree = object;
   }
}


void TimerObjectPool::allocate(int num)
{
   _numAllocated += num;
   int nBytes = _objectSize * num;
   _bytesAllocated += nBytes;

   Buffer* pNew = new Buffer;
   if (pNew != NULL)
   {
      pNew->data = new char[(static_cast<unsigned int>(nBytes))];
      //GESTURE_ASSERT(pNew->data != 0);
      pNew->next = NULL;
      char* pcBuffer = pNew->data;

      if (_blocks == NULL)
      {
         _blocks = pNew;
      }
      if (_headBlk != NULL)
      {
         _headBlk->next = pNew;
      }
      _headBlk = pNew;

      for (int i = 0; i < num; i++)
      {
         // cast to void* first to satisfy lint
         addFreeObject(reinterpret_cast<Link*>(reinterpret_cast<void*>(pcBuffer)));
         pcBuffer += _objectSize; //lint !e613 GMNGGUILINT-613-1 Pointer fatal asserted above
      }
   }

   /* pcBuffer has been added as free object (initial free space for the object pool).
   This buffer is maintained by the objectPool classs and cannot be deleted here. */
}


bool TimerObjectPool::initialize(int objectSize, int numInitial, int numGrow)
{
   if (_isInitialized || objectSize < 1 || numInitial < 0 || numGrow < 1)
   {
      return false;
   }

   //calculate object size
   _objectSize = (objectSize + (static_cast<int>(ALIGNMENT)) - 1) & (-(static_cast<int>(ALIGNMENT)));
   //assign NumGrow
   _numGrow = numGrow;
   //set initalize flag
   _isInitialized = true;

   if (numInitial > 0)
   {
      allocate(numInitial);
   }

   return true;
}


void* TimerObjectPool::getObject()
{
   //exit if object pool is not initialized
   if (!_isInitialized)
   {
      return 0;
   }

   if (_firstFree == 0)
   {
      allocate(_numGrow);
   }
   //get first object
   void* object = _firstFree;
   //correct next pointer
   if (_firstFree)
   {
      _firstFree = _firstFree->next;
      if (_firstFree == 0)
      {
         _lastFree = 0;
      }
   }

   _numCurrentlyUsed++;
   return object;
}


void TimerObjectPool::releaseObject(void* object)
{
   if (!_isInitialized || object == 0)
   {
      return;
   }
   _numCurrentlyUsed--;
   // free Object
   addFreeObject(reinterpret_cast<Link*>(object));
}
