/* ***************************************************************************************
* FILE:          TimerManager.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TimerManager is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#ifndef _TimerManager_H
#define _TimerManager_H

#include  "TimerObjectPool.h"
#include  "GestureDetector.h"


#ifdef WIN32
#include <windows.h>
#else
#include <pthread.h>
#include <unistd.h>
//#include <errno.h>
#include <sys/time.h>
#endif

namespace hmibase {
namespace input {
namespace gesture {

class TimerManager
{
   public:
      /**
      *  @brief Initialize the class.
      *
      *  This function should be called before using any other function of this class.
      *  It creates an instance of the TimerManager class, and initializes it.
      *
      *  @return  false in case of an error that should be considered fatal
      *
      */
      static bool initializeTimerManager();

      /**
      *  @brief Get instance.
      *
      *  Please make sure to call initializeTimerManager() before using this function.
      *
      *  @return  The one and only instance of the TimerManager class.
      *
      */
      static TimerManager& s_getInstance()
      {
         return *_TimerMngInstance;
      }

      /**
      *  @brief Start or restart a timer.
      *
      *  Please make sure to call initializeTimerManager() before using this function.
      *
      *  First, this function will automatically stop a timer with the same
      *  receiver and user id. Then, a new one-shot timer is created. When
      *  this timer expires, the call-back function processTimerEvent() of
      *  receiver is called.
      *
      *  @param[in]  receiver  Receiver that is called when this timer expires
      *  @param[in]  userId    User id that will be passed to the timer call-back function
      *  @param[in]  time      Time in milliseconds (must be greater than 0)
      *
      *  @return  true if successful
      *
      */
      bool startTimer(hmibase::input::gesture::GestureDetector* receiver, int userId, int time);

      /**
      *  @brief Stop a timer.
      *
      *  Please make sure to call initializeTimerManager() before using this function.
      *
      *  @param[in]  receiver    Receiver of the timer to be stopped
      *  @param[in]  userId      User id of the timer to be stopped
      *  @param[in]  allUserIds  true means to ignore the userId parameter and stop
      *                           all timers for the specified receiver
      *
      *  @return  true if successful (i.e. at least one timer has been stopped)
      *
      */
      bool stopTimer(const hmibase::input::gesture::GestureDetector* receiver, int userId, bool allUserIds);

      /**
      *  @brief Notify all expired timers.
      *
      *  Please make sure to call initializeTimerManager() before using this function.
      *
      *  This function is called from the GUI main loop.
      *
      */
      static void notifyTimers(void* point);

      /**
      *  @brief Notify all expired timers.
      *
      *  Please make sure to call initializeTimerManager() before using this function.
      *
      *  This function is called from the GUI main loop.
      */
      void notifySoftTimersTimers();

      /*****************************************************************************
      *
      * FUNCTION:    clockGetElapsedTime
      *
      * DESCRIPTION: this function returns the current time .
      *
      * RETURNVALUE: time in milliseconds
      *
      *****************************************************************************/
      static uint32_t clockGetElapsedTime();

      /**
      *  @brief The destructor is normally not called.
      *
      */
      virtual ~TimerManager();

      /**
      *  @brief Check when the next timer will expire.
      *
      *  Please make sure to call initializeTimerManager() before using this function.
      *
      *  @return  Milliseconds when the next timer will expire,
      *           or 0 if the next timer already has expired,
      *           or -1 if there are no timers.
      */
      int getWaitTime() const;

      /**
      *  @brief The destructor is normally not called.
      *
      */
      void createMasterthread();

      void pollThread();

#ifdef WIN32
      static DWORD WINAPI timerThreadStart(LPVOID lpParam);
#else
      static void* timerThreadStart(void* pParam);
#endif


   private:
      /**
      *  @brief The constructor is private and only called by initializeTimerManager().
      */
      TimerManager();

      /**
      *  @brief Initializes the object.
      *
      *  This function is called be initializeTimerManager().
      *
      *  @return  true if successful, false otherwise
      *
      */
      bool initialize();

      /**
      *  @brief The copy constructor shall never be called.
      *
      *  Therefore, it is defined private.
      *
      *  @param[in]  rhs  The object to copy
      *
      */
#if __cplusplus >= 201103L // c++11 support
      TimerManager(const TimerManager& rhs) = delete;
#else
      TimerManager(const TimerManager& rhs);
#endif

      /**
      *  @brief The assignment operator shall never be called.
      *
      *  Therefore, it is defined private.
      *
      *  @param[in]  rhs  The object to copy
      *
      */
#if __cplusplus >= 201103L // c++11 support
      TimerManager& operator=(const TimerManager& rhs) = delete;
#else
      TimerManager& operator=(const TimerManager& rhs);
#endif

      enum PrivateIntegerConstants
      {
         NUM_INIT_TIMERS = 20,       ///< number of timer structures initially allocated
         NUM_GROW_TIMERS = 5       ///< number of timer structures additionally allocated when needed
      };

      struct TimerInfo
      {
         TimerInfo* next;
         hmibase::input::gesture::GestureDetector* receiver;
         int userId;
         int expireTime;
      };

      static TimerManager* _TimerMngInstance;     //< the one and only instance of this class
      TimerObjectPool _TimerPool;            //< object pool for TimerInfo structures
      TimerInfo*      _firstTimer;          //< single-chained list of timers
};


}
}


}

#endif // #ifndef _TimerManager_H
