/* ***************************************************************************************
* FILE:          TapGestureDetector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TapGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TapGestureDetector.cpp.trc.h"
#endif

#include "TapGestureDetector.h"
#include "GestureHandler.h"
#ifdef GESTURE_SIMULATION
#include "TimerManager.h"
#else

#include "hmibase/util/Ticker.h"
#include "BaseContract/generated/BaseGesturesMsgs.h"
#include "hmibase/util/TimerThreaded.h"
#endif
#include "IGestureListener.h"

namespace hmibase {
namespace input {
namespace gesture {
unsigned long TapGestureDetector::_LastTapEndedTime = 0;
IGestureListener* TapGestureDetector::_pLastGestureReceiver = 0;

TapGestureDetector::TapGestureDetector(IGestureListener* receiver,
                                       int gesturePriority,
                                       bool doubleTapEnabled,
                                       bool parallelGestureRecognitionEnabled,
                                       GestureEvent::GestureType parallelRecognizableGestureType,
                                       unsigned int doubleTapDetectionTime,
                                       int maxDistance,
                                       unsigned int holdTime,
                                       unsigned int repeatTime
                                      ) :
   GestureDetector(receiver, parallelGestureRecognitionEnabled, parallelRecognizableGestureType),
   _activeTimerId(TIMERID_UNDEF),
   _tapHoldTime(holdTime),
   _tapPrimaryRepeatTime(repeatTime),
   _tapMaxDistance(maxDistance),
   _doubleTapDetectionTime(doubleTapDetectionTime),
   _gesturePriority(gesturePriority),
   _isDoubleTapEnabled(doubleTapEnabled),
   _isTapDetected(false),
   _isDoubleTapDetected(false),
   _isInitialActionsExecuted(false),
   _isFirstTouch(false),
   _isStartEvSent(false),
   _isTapHoldRepeatLogged(false)
{
   //set gesture
   setGestureType(GestureEvent::GT_PressHoldRepeatTap);
   setDirection(GestureEvent::DIR_NONE);

   setGestureDetectorType(getGesture()._gestureType);
}


TapGestureDetector::~TapGestureDetector()
{
}


bool TapGestureDetector::registerGesture(IGestureListener* receiver,
      int  gesturePriority,
      bool doubleTapEnabled,
      bool parallelGestureRecognitionEnabled,
      GestureEvent::GestureType  parallelRecognizableGestureType,
      unsigned int  doubleTapDetectionTime,
      int  maxDistance,
      unsigned int holdTime,
      unsigned int repeatTime)
{
   GESTURE_ASSERT(receiver != 0);
   //init result flag
   bool result = false;

   if (GestureHandler::s_getInstance().getRegistrationEnabled())
   {
      //get detector
      GestureDetector* detector = new TapGestureDetector(receiver,
            gesturePriority,
            doubleTapEnabled,
            parallelGestureRecognitionEnabled,
            parallelRecognizableGestureType,
            doubleTapDetectionTime,
            maxDistance,
            holdTime,
            repeatTime);

      GESTURE_ASSERT(detector != 0);
      //register detector
      result = GestureHandler::s_getInstance().insertDetector(detector);
      if (!result)
      {
         //free detector if registration failed
         delete detector;
      }
   }

   return result;
}


void TapGestureDetector::cancel()
{
   _timer.stop();

   // Just send cancel to registered component
   if ((getReceiver() != 0) && _isTapDetected)
   {
      _isTapDetected = false;
      _isDoubleTapDetected = false;

      //don't send ET_ABORT event to widget if ET_START event is not sent to widget
      if (_isStartEvSent)
      {
         //set parameters
         setEventType(GestureEvent::ET_ABORT);
         setPoint1Valid(false);
         ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                             ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                             ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
         getReceiver()->processGestureEvent(getGesture());
      }
      _LastTapEndedTime = 0;
      //clear here
      _isFirstTouch = false;
   }
}


bool TapGestureDetector::detect()
{
   // we should Revisit once Wayland Sending wl_touch::motion instead of Depending on Distance Alone
   bool isPoint1Press = GestureHandler::s_getInstance().getPosition1Valid();
   bool isPoint2Press = GestureHandler::s_getInstance().getPosition2Valid();
   bool detected = false;

   IGestureListener* receiver = getReceiver();
   if (receiver != 0)
   {
      // Set the initial touch position and touch valid flag
      setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
      setPoint1Valid(isPoint1Press);
      bool isDistanceExceeded = false;
      //need to send tap events only if there is no velocity(user is intended to fling not tap-release).
      if (isPoint1Press && !isPoint2Press && (_tapMaxDistance >= 0))
      {
         // If the press is recognized, check if the current position didn't exceed the allowed move distance
         isDistanceExceeded = distanceExceeded(GestureHandler::s_getInstance().getStartPoint1(), GestureHandler::s_getInstance().getLastTouchPoint1(), _tapMaxDistance);
      }

      if (isDistanceExceeded)
      {
         //Cancel gesture if the distance was exceeded
         setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
         return false;
      }
      else
      {
         if (isPoint1Press && !isPoint2Press && !_isInitialActionsExecuted)
         {
            // If this is an initial press, perform the initial actions
            // Set the gesture type to START and inform receiver
            //compare the previous release time with current time.
            //if less and double tap was registered to, then send the double tap event.
#ifdef GESTURE_SIMULATION
            unsigned int elapsed = TimerManager::clockGetElapsedTime();
#else
            unsigned long elapsed = hmibase::util::Ticker::getTickCountMsec();
#endif
            if ((_pLastGestureReceiver == receiver) && _isDoubleTapEnabled && (elapsed >= _LastTapEndedTime) && (elapsed - _LastTapEndedTime) <= _doubleTapDetectionTime)
            {
               setEventType(GestureEvent::ET_DT_START);
               //ET_DT_START is also like ET_START, so set _isStartEvSent to TRUE.
               _isStartEvSent = true;
               _isDoubleTapDetected = true;

               ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                   ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                   ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

               receiver->processGestureEvent(getGesture());
               //to avoid detection of multiple double taps
               _LastTapEndedTime = 0;
               _pLastGestureReceiver = 0;
            }
            else
            {
               //  int nTapPressDelay = 0;  //if we need a Delay To send Tap Start Enable this Code
               //set if points is first touch
               if (!_isFirstTouch)
               {
                  _isFirstTouch = true;
                  setEventType(GestureEvent::ET_INITIATE);

                  ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                      ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                      ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

                  receiver->processGestureEvent(getGesture());
               }
               //  if (nTapPressDelay > 0)
               // {
               //   ETG_TRACE_USR4_THR(("TapGestureDetector.detect:starting tap-press timer to delay tap start"));
#ifdef GESTURE_SIMULATION
               //   TimerManager::s_getInstance().startTimer(this, TIMERID_TOUCH_STARTEVDELAY, nTapPressDelay);
#else
               //   _timer.setName("TapGestureDetector", "");
               //   _activeTimerId = TIMERID_TOUCH_STARTEVDELAY;
               //   _timer.setTimeout(nTapPressDelay, COURIER_MESSAGE_NEW(hmibase::input::gesture::GestureTimerExpiredMsg)(GestureEvent::GT_PressHoldRepeatTap, TIMERID_TOUCH_STARTEVDELAY));
               //   _timer.start();
#endif
               // }
               //else
               {
                  //ETG_TRACE_USR4_THR(("TapGestureDetector.detect:tap-press timer delay is 0, sending tap start immediately"));
                  _isStartEvSent = true;
                  setEventType(GestureEvent::ET_START);
                  ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                      ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                      ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
                  receiver->processGestureEvent(getGesture());
               }
               if (_pLastGestureReceiver != receiver)
               {
                  _pLastGestureReceiver = receiver;
#ifdef GESTURE_SIMULATION
                  _LastTapEndedTime = TimerManager::clockGetElapsedTime();
#else
                  _LastTapEndedTime = hmibase::util::Ticker::getTickCountMsec();
#endif
               }
            }

            // start timers (previously running timers with same ID will automatically be stopped)
            if (_tapHoldTime > 0)
            {
#ifdef GESTURE_SIMULATION
               TimerManager::s_getInstance().startTimer(this, static_cast<int>(TIMERID_TOUCH_LONG), _tapHoldTime);
#else
               _timer.setName("TapGestureDetector", "");
               _timer.setTimerId(GestureEvent::GT_PressHoldRepeatTap);
               _activeTimerId = TIMERID_TOUCH_LONG;
               _timer.setTimeout(_tapHoldTime, &(GestureHandler::s_getInstance()));
               _timer.start();
#endif
            }
            else if (_tapPrimaryRepeatTime > 0)
            {
#ifdef GESTURE_SIMULATION
               TimerManager::s_getInstance().startTimer(this, static_cast<int>(TIMERID_TOUCH_REPEAT), _tapPrimaryRepeatTime);
#else
               _timer.setName("TapGestureDetector->PrimaryRepeatTimer", "");
               _timer.setTimerId(GestureEvent::GT_PressHoldRepeatTap);
               _activeTimerId = TIMERID_TOUCH_REPEAT;
               _timer.setTimeout(_tapPrimaryRepeatTime, &(GestureHandler::s_getInstance()));
               _timer.start();
#endif
            }

            // Set internal flags
            _isInitialActionsExecuted = true;
            detected = true;
            _isTapDetected = true;
            _previousPoint = GestureHandler::s_getInstance().getLastTouchPoint1();
         }

         //send tap move
         if (_isTapDetected && isPoint1Press && !isPoint2Press && (_previousPoint != GestureHandler::s_getInstance().getLastTouchPoint1()))
         {
            setEventType(GestureEvent::ET_MOVE);

            setPoint1Velocity(GestureHandler::s_getInstance().getLastVelocity1());
            setPoint1VelocityValid(true);

            ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

            receiver->processGestureEvent(getGesture());
            _previousPoint = GestureHandler::s_getInstance().getLastTouchPoint1();
         }

         if (_isTapDetected && (!isPoint1Press || isPoint2Press))
         {
            //need to send release events only if there is no velocity(user is intended to fling not tap-release).
            if (!isPoint2Press)
            {
               sendTapStartEvent();
               // When the release was detected, finish the gesture
               // Set the event type and parameters
               setEventType(GestureEvent::ET_END);
               setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
               setPoint1Valid(true);

               ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                   ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                   ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

               // Inform receiver about the gesture
               receiver->processGestureEvent(getGesture());

               if (false == _isDoubleTapDetected)
               {
                  //update only on ET_END of ET_START; to avoid detection of multiple double taps
#ifdef GESTURE_SIMULATION
                  _LastTapEndedTime = TimerManager::clockGetElapsedTime();
#else
                  _LastTapEndedTime = hmibase::util::Ticker::getTickCountMsec();
#endif
               }

               _isTapDetected = false;

               // Remove itself
               setAlive(false); //GestureHandler::s_getInstance().removeDetector(this);
               return false;
            }
            else
            {
               //if the release event is with velocity or other finger is touched then remove tap gesture
               setAlive(false); //GestureHandler::s_getInstance().removeDetector(this);
               //                     _isTapDetected = false;
               //                     _isDoubleTapDetected = false;
               return false;
            }
         }
      }
   }

   return detected;
}


int TapGestureDetector::getPriority()
{
   //return priority
   return _gesturePriority;
}


void TapGestureDetector::processTimerEvent(int userId)
{
   IGestureListener* receiver = getReceiver();
   if (receiver != 0)
   {
#ifdef GESTURE_SIMULATION
      switch (userId)
#else
      (void)userId;
      switch (_activeTimerId)
#endif
      {
         case TIMERID_TOUCH_LONG:
         {
            sendTapStartEvent();
            // Timer for touch long has expired, send the touch hold gesture to the receiver and start repeat timer
            // (repeat can start only when hold timer expired)
            setEventType(GestureEvent::ET_HOLD);

            ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

            receiver->processGestureEvent(getGesture());
#ifdef GESTURE_SIMULATION
            TimerManager::s_getInstance().startTimer(this, static_cast<int>(TIMERID_TOUCH_REPEAT), _tapPrimaryRepeatTime);
#else
            if (_tapPrimaryRepeatTime > 0)
            {
               _timer.setName("TapGestureDetector->RepeatTimer", "");
               _timer.setTimerId(GestureEvent::GT_PressHoldRepeatTap);
               _activeTimerId = TIMERID_TOUCH_REPEAT;
               _timer.setTimeout(_tapPrimaryRepeatTime, &(GestureHandler::s_getInstance()));
               _timer.start();
            }
#endif

            // Log the touch-long information (only once for touch-long and touch-repeat)
            if (!_isTapHoldRepeatLogged)
            {
               _isTapHoldRepeatLogged = true;
               //   GUI_UTIL_InteractionLogger::s_logTouchInformation(GUI_EVENT_GESTURE_HOLD_REPEAT, receiver,
               //                                                       getGesture().point1, Vector2D (-1, -1));
            }
            break;
         }

         case TIMERID_TOUCH_REPEAT:
         {
            sendTapStartEvent();
            // Timer for touch repeat has expired, send the touch repeat gesture to the receiver and retrigger the
            // repeat timer
            setEventType(GestureEvent::ET_REPEAT);

            ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

            receiver->processGestureEvent(getGesture());
#ifdef GESTURE_SIMULATION
            TimerManager::s_getInstance().startTimer(this, static_cast<int>(TIMERID_TOUCH_REPEAT), _tapPrimaryRepeatTime);
#else
            _timer.setName("TapGestureDetector", "");
            _timer.setTimerId(GestureEvent::GT_PressHoldRepeatTap);
            _timer.setTimeout(_tapPrimaryRepeatTime, &(GestureHandler::s_getInstance()));
            _timer.start();
#endif

            // Log the touch-repeat information (only once for touch-long and touch-repeat)
            if (!_isTapHoldRepeatLogged)
            {
               _isTapHoldRepeatLogged = true;
               // for Logging Gestures
               //   GUI_UTIL_InteractionLogger::s_logTouchInformation(GUI_EVENT_GESTURE_HOLD_REPEAT, receiver,
               //                                                   getGesture().point1, Vector2D (-1, -1));
            }
            break;
         }
         case TIMERID_TOUCH_STARTEVDELAY:
         {
            sendTapStartEvent();
         }
         break;

         default:
            break;
      }
   }
}


void TapGestureDetector::sendTapStartEvent()
{
   IGestureListener* receiver = getReceiver();
   if ((getReceiver() != 0) && _isTapDetected &&
         (false == _isStartEvSent) && (false == _isDoubleTapDetected)) //if Tap Detected Send ET_START
   {
      GestureEvent tempGesture;
      tempGesture._gestureType = GestureEvent::GT_PressHoldRepeatTap;
      tempGesture._pt1 = GestureHandler::s_getInstance().getStartPoint1();
      tempGesture._isPoint1Valid = true;
      tempGesture._gestureState = GestureEvent::ET_START;
      // stop the timeout
#ifdef GESTURE_SIMULATION
      TimerManager::s_getInstance().stopTimer(this, static_cast<int>(TIMERID_TOUCH_STARTEVDELAY), false);
#else
      _timer.stop();
#endif
      _isStartEvSent = true;

      ETG_TRACE_USR4_THR(("TapGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                          ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                          ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

      receiver->processGestureEvent(tempGesture); //Sending Tap start Event
   }
}


}
}


}
