/* ***************************************************************************************
* FILE:          SwipeGestureDetector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SwipeGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SwipeGestureDetector.cpp.trc.h"
#endif


#include "SwipeGestureDetector.h"
#include "IGestureListener.h"
#include "GestureHandler.h"

namespace hmibase {
namespace input {
namespace gesture {
SwipeGestureDetector::SwipeGestureDetector(IGestureListener* receiver, int gesturePriority, GestureEvent::Direction direction, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, int minSpeed, int minDistance) :
   GestureDetector(receiver, parallelGestureRecognitionEnabled, parallelRecognizableGestureType),
   _swipeMinSpeed(minSpeed),
   _swipeMinDistance(minDistance),
   _swipeDirection(direction),
   _isSwipeMinDistanceExceed(minDistance <= 0),
   _gesturePriority(gesturePriority)
{
   //set gesture
   setGestureType(GestureEvent::GT_SwipeFling);
   setDirection(direction);

   setGestureDetectorType(GestureEvent::GT_SwipeFling);
}


SwipeGestureDetector::~SwipeGestureDetector()
{
   //empty
}


bool SwipeGestureDetector::registerGesture(IGestureListener* receiver, int gesturePriority, GestureEvent::Direction direction, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, int minSpeed, int minDistance)
{
   // GUI_ASSERT(receiver != 0);
   //init result flag
   bool result = false;

   if (GestureHandler::s_getInstance().getRegistrationEnabled())
   {
      //get detector
      GestureDetector* detector = new SwipeGestureDetector(receiver, gesturePriority, direction, parallelGestureRecognitionEnabled, parallelRecognizableGestureType, minSpeed, minDistance);
      //   GUI_ASSERT(detector != 0);
      //register detector
      result = GestureHandler::s_getInstance().insertDetector(detector);
      if (!result)
      {
         //free detector if registration failed
         delete detector;
      }
   }

   return result;
}


void SwipeGestureDetector::cancel()
{
   //empty
}


bool SwipeGestureDetector::detect()
{
   bool isPoint1Press = GestureHandler::s_getInstance().getPosition1Valid();
   bool detected = false;

   IGestureListener* receiver = getReceiver();
   if (receiver != 0)
   {
      if (isPoint1Press)
      {
         // Check if the distance between the initial point and the current touch position has reached the minimum required distance
         _isSwipeMinDistanceExceed = _isSwipeMinDistanceExceed || (compareDistance(GestureHandler::s_getInstance().getStartPoint1(), GestureHandler::s_getInstance().getLastTouchPoint1(), _swipeMinDistance) >= 0);
      }
      else if (_isSwipeMinDistanceExceed)
      {
         // If the minimum distance has been reached, calculate the velocity of current gesture depending on registered direction
         int velocitySquared = 0;
         Vector2D ptLastVelocity = GestureHandler::s_getInstance().getLastVelocity1();
         switch (_swipeDirection)
         {
            case GestureEvent::DIR_HORIZONTAL:
               velocitySquared = ptLastVelocity.x * ptLastVelocity.x;
               break;

            case GestureEvent::DIR_VERTICAL:
               velocitySquared = ptLastVelocity.y * ptLastVelocity.y;
               break;

            case GestureEvent::DIR_2D:
               velocitySquared = ptLastVelocity.x * ptLastVelocity.x + ptLastVelocity.y * ptLastVelocity.y;
               break;

            default:
               break;
         }
         // If the current velocity crossed the declared minimum, swipe gesture is detected
         // We work on squared values to speed up calculation (skip root extraction)
         detected = (velocitySquared >= (_swipeMinSpeed * _swipeMinSpeed));

         if (detected)
         {
            // If gesture is detected, inform receiver about it
            // (gesture velocity is stored in point1 of the gesture as the point position is not needed by the receiver)
            //setPoint1(ptLastVelocity);
            setPoint1Velocity(ptLastVelocity);
            setPoint1VelocityValid(true);
            //setPoint1Valid(true);
            setEventType(GestureEvent::ET_END);
            setDirection((GestureEvent::Direction) _swipeDirection);

            ETG_TRACE_USR4_THR(("SwipeGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

            receiver->processGestureEvent(getGesture());
            //  GUI_UTIL_InteractionLogger::s_logTouchInformation(GUI_EVENT_GESTURE_SWIPE, receiver,
            //                                                     GestureHandler::s_getInstance().getLastTouchPoint1(), Vector2D (-1, -1));
         }
         else
         {
            ETG_TRACE_USR4_THR(("SwipeGestureDetector.detect: velocitySquared %d < swipeMinSpeedSquared %d\n", velocitySquared, _swipeMinSpeed * _swipeMinSpeed));
         }
      }
   }

   return detected;
}


int SwipeGestureDetector::getPriority()
{
   //return priority
   return _gesturePriority;
}


GestureEvent::Direction SwipeGestureDetector::getDirection()
{
   //return direction
   return _swipeDirection;
}


}
}


}
