/* ***************************************************************************************
* FILE:          SpreadGestureDetector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SpreadGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SpreadGestureDetector.cpp.trc.h"
#endif


#include "SpreadGestureDetector.h"
#include "GestureHandler.h"
#ifdef GESTURE_SIMULATION
#include "TimerManager.h"
#else
#include "AppUtils/Timer.h"
#endif
#include "IGestureListener.h"
#include "GestureBasetypes.h"

namespace hmibase {
namespace input {
namespace gesture {

SpreadGestureDetector::SpreadGestureDetector(IGestureListener* receiver, int gesturePriority, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, unsigned int doubleTouchMaxTime, float spreadMinFactor, float pinchMaxFactor) :
   GestureDetector(receiver, parallelGestureRecognitionEnabled, parallelRecognizableGestureType),
   _doubleTouchMaxTime(doubleTouchMaxTime),
   _initialDistanceSquared(0),
   _spreadMinFactor(spreadMinFactor),
   _pinchMaxFactor(pinchMaxFactor),
   _gesturePriority(gesturePriority),
   _isDoubleTouchDetected(false),
   _isSpreadPinchDetected(false),
   _isInitialActionsExecuted(false),
   _isGestureCancelled(false),
   _isInitialTouchDetected(false)
{
   //set gesture
   setGestureType(GestureEvent::GT_PinchSpread);
   setDirection(GestureEvent::DIR_NONE);

   setGestureDetectorType(GestureEvent::GT_PinchSpread);
}


SpreadGestureDetector::~SpreadGestureDetector()
{
   //empty
}


bool SpreadGestureDetector::registerGesture(IGestureListener* receiver, int gesturePriority, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, unsigned int doubleTouchMaxTime, float spreadMinFactor, float pinchMaxFactor)
{
   GESTURE_ASSERT(receiver != 0);
   //init result flag
   bool result = false;

   if (GestureHandler::s_getInstance().getRegistrationEnabled())
   {
      //get detector
      GestureDetector* detector = new SpreadGestureDetector(receiver, gesturePriority, parallelGestureRecognitionEnabled, parallelRecognizableGestureType, doubleTouchMaxTime, spreadMinFactor, pinchMaxFactor);
      GESTURE_ASSERT(detector != 0);
      //register detector
      result = GestureHandler::s_getInstance().insertDetector(detector);
      if (!result)
      {
         //free detector if registration failed
         delete detector;
      }
   }
   return result;
}


void SpreadGestureDetector::cancel()
{
   if ((getReceiver() != 0))
   {
      if (_isDoubleTouchDetected)
      {
         //set parameter
         setEventType(GestureEvent::ET_ABORT);
         setPoint1Valid(false);
         setPoint2Valid(false);
         ETG_TRACE_USR4_THR(("SpreadGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                             ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                             ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
         getReceiver()->processGestureEvent(getGesture());
      }
   }
}


bool SpreadGestureDetector::detect()
{
   bool isPoint1Press = GestureHandler::s_getInstance().getPosition1Valid();
   bool isPoint2Press = GestureHandler::s_getInstance().getPosition2Valid();;
   bool detected = false;
   int  currentDistanceSquared = 0;

   IGestureListener* receiver = getReceiver();
   if (receiver != 0)
   {
      // Set current touch position and press information
      setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
      setPoint1Valid(isPoint1Press);

      setPoint2(GestureHandler::s_getInstance().getLastTouchPoint2());
      setPoint2Valid(isPoint2Press);

      if (_isGestureCancelled)
      {
         // Cancel the detection in case the time between the first and second touch is greater than configured
         // using _doubleTouchMaxTime member variable
         setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
         return false;
      }

      if (isPoint1Press && isPoint2Press)
      {
         //stop the timer on double touch detection.
#ifdef GESTURE_SIMULATION
         TimerManager::s_getInstance().stopTimer(this, static_cast<int>(TIMERID_DOUBLE_TOUCH_TIME), false);
#else
         _timer.stop();
#endif
         // Two points are currently pressed, so get current distance between them
         currentDistanceSquared = (GestureHandler::s_getInstance().getLastTouchPoint2().x - GestureHandler::s_getInstance().getLastTouchPoint1().x) * (GestureHandler::s_getInstance().getLastTouchPoint2().x - GestureHandler::s_getInstance().getLastTouchPoint1().x) +
                                  (GestureHandler::s_getInstance().getLastTouchPoint2().y - GestureHandler::s_getInstance().getLastTouchPoint1().y) * (GestureHandler::s_getInstance().getLastTouchPoint2().y - GestureHandler::s_getInstance().getLastTouchPoint1().y);

         if (!_isInitialActionsExecuted)
         {
            // Perform the initial actions
            // Initial actions are executed only once, when the gesture is detected for the first time

            // Get the receiver widget rectangle and check if both touch positions are placed
            // inside this rectangle
            GUI_Rect rcReceiver = receiver->getReceiverAbsoluteRect();
            if (!rcReceiver.contains(GestureHandler::s_getInstance().getLastTouchPoint1()) || !rcReceiver.contains(GestureHandler::s_getInstance().getLastTouchPoint2()))
            {
               // Cancel the gesture because the widget rectangle doesn't contain both touch points
               setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
               return false;
            }
            else
            {
               _isDoubleTouchDetected = true;
               // Save the current distance as an initial distance, which is later used for the factor calculation
               _initialDistanceSquared = currentDistanceSquared;

               setEventType(GestureEvent::ET_HOLD);
               ETG_TRACE_USR4_THR(("SpreadGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                   ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                   ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

               // Send the gesture hold event to the receiver
               receiver->processGestureEvent(getGesture());
            }

            _isInitialActionsExecuted = true;
         }

         // To remove lint warning
         if (_initialDistanceSquared == 0)
         {
            _initialDistanceSquared = 1;
         }

         float fFactorSquared = static_cast<float>(currentDistanceSquared) / static_cast<float>(_initialDistanceSquared);

         bool bDistanceReached = (fFactorSquared >= _spreadMinFactor * _spreadMinFactor) ||
                                 (fFactorSquared <= _pinchMaxFactor * _pinchMaxFactor);

         if (!_isSpreadPinchDetected && bDistanceReached)
         {
            ETG_TRACE_USR4_THR(("SpreadGestureDetector.detect:  sending start"));
#ifdef GESTURE_SIMULATION
            TimerManager::s_getInstance().stopTimer(this, static_cast<int>(TIMERID_DOUBLE_TOUCH_TIME), false);
#else
            _timer.stop();
#endif
            setEventType(GestureEvent::ET_START);
            ETG_TRACE_USR4_THR(("SpreadGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

            // Send the gesture START event to the receiver
            receiver->processGestureEvent(getGesture());
            _isSpreadPinchDetected = true;
            detected = true;
            return detected;
         }
      }

      if (_isDoubleTouchDetected)
      {
         // This block is executed in case the double touch was detected before
         // and a touch event with new coordinates is received now

         if (!isPoint1Press || !isPoint2Press)
         {
            // In case at least one touch release is detected, send the gesture END event and remove current
            // gesture detector
            setEventType(GestureEvent::ET_END);
            setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
            setPoint1Valid(true);
            setPoint2(GestureHandler::s_getInstance().getLastTouchPoint2());
            setPoint2Valid(true);

            ETG_TRACE_USR4_THR(("SpreadGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

            receiver->processGestureEvent(getGesture());
            _isDoubleTouchDetected = false;
            setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
            //send event to interaction logger only once after complete pinch/spread action
            //GUI_UTIL_InteractionLogger::s_logTouchInformation(GUI_EVENT_GESTURE_SPREAD, receiver,
            //                                                        getGesture().point1, getGesture().point2);
            return false;
         }
         else
         {
            if (_isSpreadPinchDetected)
            {
               //GUI_UTIL_InteractionLogger::s_logTouchInformation(GUI_EVENT_GESTURE_SPREAD, receiver,
               //                                                        getGesture().point1, getGesture().point2);
               setEventType(GestureEvent::ET_MOVE);

               ETG_TRACE_USR4_THR(("SpreadGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                   ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                   ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));

               // If spread or pinch gesture was detected send the gesture MOVE event
               receiver->processGestureEvent(getGesture());
            }
         }
      }
      else
      {
         // In case the double touch was not detected
         if (isPoint1Press || isPoint2Press)
         {
            if (!_isInitialTouchDetected)
            {
               // In case only one touch point is valid, start timer to check if the next touch point will be received
               // in the defined time
#ifdef GESTURE_SIMULATION
               TimerManager::s_getInstance().startTimer(this, static_cast<int>(TIMERID_DOUBLE_TOUCH_TIME), _doubleTouchMaxTime);
#else
               _timer.setName("SpreadGestureDetector", "");
               _timer.setTimerId(GestureEvent::GT_PinchSpread);
               _timer.setTimeout(TIMERID_DOUBLE_TOUCH_TIME, _doubleTouchMaxTime, &(GestureHandler::s_getInstance()));
               _timer.start();
#endif
               _isInitialTouchDetected = true;
            }
         }
      }
   }

   return detected;
}


int SpreadGestureDetector::getPriority()
{
   //get priority
   return _gesturePriority;
}


void SpreadGestureDetector::processTimerEvent(int userId)
{
   //case handling
   switch (userId)
   {
      case TIMERID_DOUBLE_TOUCH_TIME:
         //set cancel flag
         _isGestureCancelled = true;
         break;

      default:
         break;
   }
}


}
}


}
