/* ***************************************************************************************
* FILE:          TapGestureDetector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TapGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "lint_deactivation.h"
#include "RawGestureDetector.h"
#include "IGestureListener.h"
#include "GestureHandler.h"

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RawGestureDetector.cpp.trc.h"
#endif

namespace hmibase {
namespace input {
namespace gesture {

RawGestureDetector::RawGestureDetector(IGestureListener* receiver, int gesturePriority) :
   GestureDetector(receiver, false, GestureEvent::GT_None),
   _gesturePriority(gesturePriority)
{
   //set gesture
   m_bGestureDetected = false;
   setGestureType(GestureEvent::GT_RawTouchData);
}


RawGestureDetector::~RawGestureDetector()
{
   //empty
}


bool RawGestureDetector::registerGesture(IGestureListener* receiver, int gesturePriority)
{
   // GUI_ASSERT(receiver != 0);

   // Create the gesture detector and add it to the list of detectors handled by the GestureHandler
   GestureDetector* detector = new RawGestureDetector(receiver, gesturePriority);
   //    GUI_ASSERT(detector != 0);
   bool result = GestureHandler::s_getInstance().insertDetector(detector);

   if (!result)
   {
      delete detector;
   }

   return result;
}


void RawGestureDetector::cancel()
{
   // Just send cancel to registered component
   if ((getReceiver() != 0) && (m_bGestureDetected == true))
   {
      setEventType(GestureEvent::ET_ABORT);
      setPoint1Valid(false);
      ETG_TRACE_USR4_THR(("RawGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                          ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                          ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
      getReceiver()->processGestureEvent(getGesture());
      m_bGestureDetected = false;
   }
}


bool RawGestureDetector::detectGestureStart()
{
   bool result = false;
   IGestureListener* receiver = getReceiver();
   if ((receiver != 0) && (m_bGestureDetected == false))
   {
      if (GestureHandler::s_getInstance().getPosition1Valid())
      {
         setEventType(GestureEvent::ET_START);
         setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
         setPoint1Valid(true);
         ETG_TRACE_USR4_THR(("RawGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                             ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                             ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
         receiver->processGestureEvent(getGesture());
         m_bGestureDetected = true;
         result = true;
      }
   }

   return result;
}


bool RawGestureDetector::detect()
{
   bool detected = false;

   if (!m_bGestureDetected)
   {
      detected = detectGestureStart();
   }
   else
   {
      IGestureListener* receiver = getReceiver();
      if (receiver != 0)
      {
         setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
         setPoint1Valid(true);
         setEventType(GestureHandler::s_getInstance().getPosition1Valid() ? GestureEvent::ET_RAW_DATA : GestureEvent::ET_END);

         ETG_TRACE_USR4_THR(("RawGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                             ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                             ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
         receiver->processGestureEvent(getGesture());
         detected = true;
      }
   }

   return detected;
}


int RawGestureDetector::getPriority()
{
   return _gesturePriority;
}


}
}


}
