/* ***************************************************************************************
* FILE:          GestureDetector.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#ifndef _GestureDetector_H
#define _GestureDetector_H

#include "GestureBasetypes.h"

#ifndef GESTURE_SIMULATION
#include "BaseContract/generated/BaseGesturesMsgs.h"
#include "hmibase/util/TimerThreaded.h"
#endif

namespace hmibase {
namespace input {
namespace gesture {

class IGestureListener;

/**
*  @brief Interface for gesture detector
*/
class GestureDetector
{
   public:
      /**
      *  @brief Priorities for gesture
      */
      enum GesturePriority
      {
         PRIORITY_INVALID = -1,
         PRIORITY_TAP,    ///< Priority of PressHoldRepeatTap gesture
         PRIORITY_DRAG,   ///< Priority of DragNudge gesture
         PRIORITY_SWIPE,  ///< Priority of SwipeFling gesture
         PRIORITY_SPREAD, ///< Priority of PinchSpread gesture
         PRIORITY_ROTATE, ///< Priority of Rotate gesture
         PRIORITY_RAW_DATA,
         PRIORITY_LEGACY_TOUCH
      };

      GestureDetector* _nextGestureDetector; //< pointer to the next gesture detector

      /**
      *  @brief Un-Initialize object.
      *
      *  Frees all allocated memory.
      *
      */
      virtual ~GestureDetector();

      /**
      *  @brief Get priority of the gesture
      *
      *  @return  Gesture priority (refers to GesturePriority enumeration)
      *
      */
      virtual int  getPriority() = 0;

      /**
      *  @brief Get the direction of the gesture
      *
      *  Direction is valid only for DragNudge and SwipeFling gestures
      *
      *  @return  Gesture direction (refers to Direction enumeration)
      *
      */
      virtual GestureEvent::Direction getDirection();

      /**
      *  @brief Cancel the gesture
      *
      *  Cancelling the gesture stops all internal timers (if any)
      *  and sends the abort command to gesture receiver (if needed).
      *
      */
      virtual void cancel() = 0;

      /**
      *  @brief Detects the gesture
      *
      *  Function process the input touchMsg and checks if all conditions
      *  needed to detect a gesture are fulfilled. If yes - function will
      *  inform registered receiver about detecting the gesture.
      *
      *  @param[in] touchMsg Touch event data
      *
      *  @return True if the gesture was detected, false otherwise
      *
      */
      virtual bool detect() = 0;

      /**
      *  @brief Checks if the gesture can be combined with the gesture
      *          passed as a parameter.
      *
      *  Gestures can always be combined when they are of different type
      *  (have different priorities).
      *  When the type is the same gestures can be combined only when
      *  the gesture direction is different and one of the directions is
      *  not a 2D gesture.
      *
      *  @return  True if the gesture can be combined with other gestures.
      *
      */
      virtual bool canBeCombined(GestureDetector* detector);

      /**
      *  @brief Get next gesture detector
      *
      *  @return gesture detector
      *
      */
      GestureDetector*& getNextDetector()
      {
         return _nextGestureDetector;
      }

      /**
      *  @brief Get the touch receiver
      *
      *  @return touch receiver
      *
      */
      IGestureListener* getReceiver()
      {
         return _receiver;
      }

      /**
      *  @brief Set the gesture timestamp
      *
      *  @param[in] timestamp Current gesture timestamp
      *
      */
      void setTimestamp(int timestamp)
      {
         _currentGesture._nTimestamp = timestamp;
      }

      bool isAlive()
      {
         return _isAlive;
      }
      void setAlive(bool flag)
      {
         _isAlive = flag;
      }
      GestureEvent getGesture()
      {
         return _currentGesture;
      }

      /**
      *  @brief return Flag indicating Simultanious Gesture recognition
      *
      *  @return true if Simultanious Gesture recognition Enabled else False
      *
      */
      bool getParallelGestureRecognitionEnabled()
      {
         return _isParallelGestureRecognitionEnabled;
      }

      /**
      *  @brief get  Gesture Which Can be Recognized Simultaniously
      *
      *  @return Gesture Type
      *
      */
      GestureEvent::GestureType getSimultaneouslyRecognizableGestureType()
      {
         return _parallelRecognizableGestureType;
      }

      /**
      *  @brief  Process a timer event.
      *
      *  @param[in]  userId  Identifier for this timer
      *
      */
      virtual void processTimerEvent(int userId);

      /************************************************************************/
      /* gesture timer interface                                             */
      /************************************************************************/
      virtual bool OnGestureTimerExpired(const unsigned int type, unsigned int timerId)
      {
         if (type == _type)
         {
            processTimerEvent(static_cast<int>(timerId));
         }
         return false;
      }

   protected:
      /**
      *  @brief Initialize object.
      *
      *  Initialaze member variables with defauld values.
      *
      *  @param[in] receiver Receiver of the gesture
      *
      */
      GestureDetector(IGestureListener* receiver, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType);

      /**
      *  @brief Compare two distance values.
      *
      *  Compares distance between two points(_point1, _point2) with distance value
      *  specified as a parameter(distance).
      *
      *  Function takes also the gesture direction into account e.g.
      *  if the direction is set to horizontal, function will take only the
      *  horizontal distance between two points.
      *
      *  @return
      *          In case distance between points _point1 and _point2 is:
      *          1) equal to distance - functin returs 0.
      *          2) greater than distance - functin returs positive value.
      *          3) smaller than distance - functin returs negative value.
      *
      */
      int compareDistance(const Vector2D& point1, const Vector2D& point2, int distance);

      /**
      *  @brief Compare two distance values.
      *
      *  Compares distance between two points(point1, point2) with distance value
      *  specified as a parameter(distance).
      *
      *  Function doesn't take the gesture direction into account (always 2D distance)
      *
      *  @return True if the distance between point1 and point2 is greater then distance.
      *
      */
      bool distanceExceeded(const Vector2D& point1, const Vector2D& point2, int distance);

      /**
      *  @brief Set the gesture type
      *
      *  @param gestureType Type of the gesture e.g. GT_DragNudge
      *
      */
      void setGestureType(GestureEvent::GestureType gestureType)
      {
         _currentGesture._gestureType = gestureType;
      }

      /**
      *  @brief Set the gesture event type
      *
      *  @param[in] gestureState Type of the event e.g. ET_START, ET_MOVE, ET_END
      *
      */
      void setEventType(GestureEvent::GestureState gestureState)
      {
         _currentGesture._gestureState = gestureState;
      }

      /**
      *  @brief Set the gesture direction
      *
      *  @param[in] direction Direction of the gesture e.g. DIR_VERTICAL
      *
      */
      void setDirection(GestureEvent::Direction direction)
      {
         _currentGesture._gestureDirection = direction;
      }

      /**
      *  @brief Set the point1 coordinates of the gesture
      *
      *  @param[in] point Point coordinates
      *
      */
      void setPoint1(const Vector2D& point)
      {
         _currentGesture._pt1 = point;
      }

      /**
      *  @brief Set the point1 validity flag
      *
      *  @param[in] bool
      *
      */
      void setPoint1Valid(bool valid)
      {
         _currentGesture._isPoint1Valid = valid;
      }

      /*@brief Set the point1 coordinates of the gesture
         *
         *  @param[in] point Point coordinates
         *
         */
      void setPoint1Velocity(const Vector2D& velocity)
      {
         _currentGesture._velocity1 = velocity;
      }

      /**
      *  @brief Set the point1 velocity validity flag
      *
      *  @param[in] bool
      *
      */
      void setPoint1VelocityValid(bool valid)
      {
         _currentGesture._isPoint1VelocityValid = valid;
      }

      /**
      *  @brief Set the point1 coordinates of the gesture
      *
      *  @param[in] point Point coordinates
      *
      */
      void setPoint2(const Vector2D& point)
      {
         _currentGesture._pt2 = point;
      }

      /**
      *  @brief Set the point2 validity flag
      *
      *  @param[in] bool
      *
      */
      void setPoint2Valid(bool valid)
      {
         _currentGesture._isPoint2Valid = valid;
      }

      /*@brief Set the point2 coordinates of the gesture
      *
      *  @param[in] point Point coordinates
      *
      */
      void setPoint2Velocity(const Vector2D& velocity)
      {
         _currentGesture._velocity2 = velocity;
      }

      /**
      *  @brief Set the point2 velocity validity flag
      *
      *  @param[in] bool
      *
      */
      void setPoint2VelocityValid(bool valid)
      {
         _currentGesture._isPoint2VelocityValid = valid;
      }

      /**
      *  @brief Set the point1 coordinates of the gesture
      *
      *  @param[in] point Point coordinates
      *
      */
      void setPoint3(const Vector2D& point)
      {
         _currentGesture._pt3 = point;
      }

      /**
      *  @brief Set the point2 validity flag
      *
      *  @param[in] bool
      *
      */
      void setPoint3Valid(bool valid)
      {
         _currentGesture._isPoint3Valid = valid;
      }

      /*@brief Set the point3 coordinates of the gesture
      *
      *  @param[in] point Point coordinates
      *
      */
      void setPoint3Velocity(const Vector2D& velocity)
      {
         _currentGesture._velocity3 = velocity;
      }

      /**
      *  @brief Set the point3 velocity validity flag
      *
      *  @param[in] bool
      *
      */
      void setPoint3VelocityValid(bool valid)
      {
         _currentGesture._isPoint3VelocityValid = valid;
      }

      /**
      *  @brief Set the rotate angle(in radians) when the rotate gesture is performed
      *
      *  @param[in] rotate angle (in radians)
      *
      */
      void setRotateAngle(int16_t rotateAngle)
      {
         _currentGesture._rotateAngle = rotateAngle;
      }

   private:
      IGestureListener* _receiver; //< Receiver of the gestures
      GestureEvent _currentGesture;  //< Current gesture
      bool _isAlive;   //< Flag indicating detection is alive
      bool _isParallelGestureRecognitionEnabled; //< Flag indicating if Simultaneous Gesture Recognition Enabled or not
      GestureEvent::GestureType  _parallelRecognizableGestureType; // Gesture which Can be Recognized woth Current Gesture

   protected:
#ifdef GESTURE_SIMULATION
#else
      hmibase::util::TimerThreaded _timer;
#endif
      unsigned int _type;

      void setGestureDetectorType(GestureEvent::GestureType type)
      {
         _type = static_cast<unsigned int>(type);
      }
};


}
}


}

#endif // #ifndef _GestureDetector_H
