/* ***************************************************************************************
* FILE:          RootDaemonClient.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RootDaemonClient.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include <RootDaemonClient/RootDaemon.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <syslog.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sstream>
#include <vector>
#include <string>
#include <unistd.h>

#include <hmibase/util/Permissions.h>

#ifdef __cplusplus
extern "C"
{
#endif

bool setProcessNiceLevel(id_t pid, int prio)
{
   int errCode = setpriority(PRIO_PGRP, pid, prio);
   return errCode == 0 ? true : false;
}


bool exists(const std::string& name)
{
   struct stat buffer;
   return (stat(name.c_str(), &buffer) == 0);
}


bool setThreadNiceLevel(id_t tid, int nice)
{
   int errCode = setpriority(PRIO_PROCESS, tid, nice);
   return errCode == 0 ? true : false;
}


CmdData HMIBASE_ROOTDAEMON_CLIENT_PerformRootOpAsRoot(const int cmdNum, std::string args)
{
   CmdData result;
   syslog(LOG_INFO, "HMIBase_rootdaemon_client:command(),executing cmdNum %d [%s]", cmdNum, args.c_str());

   std::string buf;                 // Have a buffer string
   std::stringstream ss(args);       // Insert the string into a stream

   std::vector<std::string> arguments; // Create vector to hold our strings

   while (ss >> buf)
   {
      arguments.push_back(buf);
   }

   switch (cmdNum)
   {
      case E_HMIBASE_ROOTDAEMON_CMD_CODE_NICE_LVL_FG:
      {
         result.errorNo = ERR_CANNOT_EXEC_CMD;
         if (arguments.size() == 1)
         {
            long int pid = strtol(arguments[0].c_str(), NULL, 10);
            if (errno == 0)
            {
               result.errorNo = (setProcessNiceLevel((id_t)pid, -10) ? ERR_NONE : ERR_CANNOT_EXEC_CMD);
            }
         }
      }
      break;
      case E_HMIBASE_ROOTDAEMON_CMD_CODE_NICE_LVL_BG:
      {
         result.errorNo = ERR_CANNOT_EXEC_CMD;
         if (arguments.size() == 1)
         {
            long int pid = strtol(arguments[0].c_str(), NULL, 10);
            if (errno == 0)
            {
               result.errorNo = (setProcessNiceLevel((id_t)pid, 0) ? ERR_NONE : ERR_CANNOT_EXEC_CMD);
            }
         }
      }
      break;

      case E_HMIBASE_ROOTDAEMON_CMD_CODE_THREAD_NICENESS_5:
      {
         result.errorNo = ERR_CANNOT_EXEC_CMD;
         if (arguments.size() == 1)
         {
            id_t tid = static_cast<id_t>(strtol(arguments[0].c_str(), NULL, 10));
            if (errno == 0)
            {
               result.errorNo = (setThreadNiceLevel(tid, 5) ? ERR_NONE : ERR_CANNOT_EXEC_CMD);
            }
         }
      }
      break;

      case E_HMIBASE_ROOTDAEMON_CMD_CODE_CHOWN_DYN_PARTITION:
      {
         result.errorNo = ERR_NONE;

         if (arguments.size() == 1)
         {
            std::string path = arguments[0];
            if (path.rfind("/var/opt/bosch/dynamic/hmi", 0) == 0)
            {
               if (exists(path))
               {
                  int status = chown(path.c_str(), ::hmibase::util::user::aid_hmibase_core, ::hmibase::util::group::aid_hmibase);

                  if (status != 0)
                  {
                     result.errorNo = ERR_CANNOT_EXEC_CMD;
                     syslog(LOG_ERR, "HMIBase_rootdaemon_client:command(),E_HMIBASE_ROOTDAEMON_CMD_CODE_CHOWN_DYN_PARTITION unable to chown for %s, error %d", path.c_str(), errno);
                  }
               }
               else
               {
                  result.errorNo = ERR_CANNOT_EXEC_CMD;
                  syslog(LOG_ERR, "HMIBase_rootdaemon_client:command(),E_HMIBASE_ROOTDAEMON_CMD_CODE_CHOWN_DYN_PARTITION error accessing path %s, error %d", path.c_str(), errno);
               }
            }
            else
            {
               result.errorNo = ERR_CANNOT_EXEC_CMD;
               syslog(LOG_ERR, "HMIBase_rootdaemon_client:command(),E_HMIBASE_ROOTDAEMON_CMD_CODE_CHOWN_DYN_PARTITION path %s not pointing to hmi folder in dynamic partition", path.c_str());
            }
         }
      }
      break;

      default:
         result.errorNo = ERR_UNKNOWN_CMD;
         break;
   }
   return result;
}


#ifdef VARIANT_S_FTR_ENABLE_HMIBASE_ROOTDAEMON_CLIENT_SO
//#include "IClientSharedlib.h"

const char* getClientName()
{
   return "aid_hmibase";
}


const char* getClientGroupName()
{
   return "aid_hmibase";
}


const char* getClientUserName()
{
   return "aid_hmibase";
}


CmdData command(const int cmdNum, std::string args)
{
   return HMIBASE_ROOTDAEMON_CLIENT_PerformRootOpAsRoot(cmdNum, args);
}


#endif

#ifdef __cplusplus
}


#endif
