/* ***************************************************************************************
* FILE:          ProxyHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ProxyHandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "ProxyHandler.h"
#include "AppBase/HmiAsfComponentBase.h"
#include "hmibase/util/StringUtils.h"
#include "hmibase/util/Ticker.h"
#include "hmibase/util/TraceUtils.h"
#include "bosch/cm/ai/hmi/appctrl/HMIAppCtrlConst.h"
#include "bosch/cm/ai/hmi/appctrl/trace/CommandsConst.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
//#include "AppBase/app_base_defines.h"
#include "View/IMessageSystem.h"

#include "MsgFwAbstraction/IMessagePostObject.h"

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_COMMON_APPCTRL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ProxyHandler.cpp.trc.h"
#endif


using namespace ::asf::core;
using namespace ::bosch::cm::ai::hmi::appctrl::HMIAppCtrl;
using namespace ::bosch::cm::ai::hmi::appctrl::trace::Commands;

namespace hmibase {
namespace services {
namespace hmiappctrl {


bool mapGadgetTargetState(const bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus state, hmibase::gadget::gadgetSceneStatus& mapped)
{
   switch (state)
   {
      case bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__ACTIVE:
         mapped = hmibase::gadget::ACTIVE;
         break;
      case bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__INACTIVE:
         mapped = hmibase::gadget::INACTIVE;
         break;
      case bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__HIDDEN:
         mapped = hmibase::gadget::HIDDEN;
         break;
      default:
         return false;
         break;
   }
   return true;
}


ProxyHandler::ProxyHandler() :
   _hmiAppCtrlProxy_QueryClient_RegId(0)
   , _hmiAppCtrlProxy_SetDisplay_RegId(0)
   , _hmiAppCtrlProxy_SetMultiDisplay_RegId(0)
   , _hmiAppCtrlProxy_DisplayStatusEval_RegId(0)
   , _hmiAppCtrlProxy_TraceQuery_RegId(0)
   , _hmiAppCtrlProxy_KeyInfoForwarding_RegId(0)
   , _hmiAppCtrlProxy_EncoderInfoForwarding_RegId(0)
   , _hmiAppCtrlProxy_SubSurfaceSwitchUpd_RegId(0)
   , _hmiBaseTraceCommandProxy_RegId(0)
   , _hmiBaseSimTouchProxy_RegId(0)
   , _hmiBaseSimTouchByNameProxy_RegId(0)
   , _hmiBaseSimTouchByTextProxy_RegId(0)
   , _hmiBaseSimTouchByTextIdProxy_RegId(0)
   , _hmiBaseSimTouchByIndexProxy_RegId(0)
   , _hmiBaseSetWidgetPropertyProxy_RegId(0)
   , _hmiBaseSimEnterKeyProxy_RegId(0)
   , _hmiBaseSimFocusReqProxy_RegId(0)
   , _hmiBaseSimJoystickProxy_RegId(0)
   , _hmiBaseSimRotateEncoderProxy_RegId(0)
   , _hmiAppCtrlProxy_ExternalImageProviderTouch_RegId(0)
   , _hmiAppCtrlProxy_SlaveSurfaceRepositioning_RegId(0)
   , _hmiAppCtrlProxy_SlaveSurfaceRepositioningAck_RegId(0)
   , _hmiAppCtrlProxy_DirectTextureConsumerPositionInfo_RegId(0)
   , _hmiAppCtrlProxy_DirectTextureConsumerPositionInfoResponse_RegId(0)
   , _hmiAppCtrlProxy_DrmGarbageCollectionRequest_RegId(0)
   , _hmiAppCtrlProxy_FollowUpActionRequest_regId(0)
   , _hmiAppCtrlProxy_FollowUpActionResponse_regId(0)
   , _appState(hmiAppState__HMIAPP_UNINITIALIZED)
   , _isDirectApplicationActivationEnabled(true)
   , _syncBlockId(0)
   , _baseComponent(0)
   , _hmiAppCtrlProxy(HMIAppCtrlProxy::createProxy("hmiAppCtrlPort", *this))
   , _hmiBaseTraceCommandProxy(CommandsProxy::createProxy("hmiAppCtrlTraceCmdPort", *this))
   , _myName(hmibase::app::base::HmiAsfComponentBase::getAppName())
   , _myTraceID(0)
{
   ETG_TRACE_USR4_THR(("ProxyHandler constructor"));
   _myName = hmibase::app::base::HmiAsfComponentBase::getAppName();
   HMI_APP_ASSERT(_myName.length() > 0);

   _hmiAppCtrlAdapter.setProxy(_hmiAppCtrlProxy);
   _hmiBaseTraceCmdAdapter.setProxy(_hmiBaseTraceCommandProxy);
}


ProxyHandler::~ProxyHandler()
{
}


void ProxyHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& /*stateChange*/)
{
   if (_hmiAppCtrlProxy == proxy)
   {
      ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onAvailable"));
      _hmiAppCtrlProxy_QueryClient_RegId = _hmiAppCtrlProxy->sendQueryClientRegister(*this);
      _hmiAppCtrlProxy_SetDisplay_RegId = _hmiAppCtrlProxy->sendSetDisplayRegister(*this);
      _hmiAppCtrlProxy_SetMultiDisplay_RegId = _hmiAppCtrlProxy->sendSetMultiDisplayRegister(*this);
      _hmiAppCtrlProxy_DisplayStatusEval_RegId = _hmiAppCtrlProxy->sendEvDisplayStatusEvalRegister(*this);
      _hmiAppCtrlProxy_KeyInfoForwarding_RegId = _hmiAppCtrlProxy->sendKeyInfoForwardingRegister(*this);
      _hmiAppCtrlProxy_EncoderInfoForwarding_RegId = _hmiAppCtrlProxy->sendEncoderInfoForwardingRegister(*this);
      _hmiAppCtrlProxy->sendActivateExternalImageProviderRegister(*this);
      _hmiAppCtrlProxy_ExternalImageProviderTouch_RegId = _hmiAppCtrlProxy->sendExternalImageProviderTouchRegister(*this);
      _hmiAppCtrlProxy_SubSurfaceSwitchUpd_RegId = _hmiAppCtrlProxy->sendSubSurfaceSwitchUpdateRegister(*this);
      _hmiAppCtrlProxy_SlaveSurfaceRepositioning_RegId = _hmiAppCtrlProxy->sendSlaveSurfaceRepositioningRegister(*this);
      _hmiAppCtrlProxy_SlaveSurfaceRepositioningAck_RegId = _hmiAppCtrlProxy->sendSlaveSurfaceRepositioningAckRegister(*this);
      _hmiAppCtrlProxy_DirectTextureConsumerPositionInfo_RegId = _hmiAppCtrlProxy->sendDirectTextureConsumerPositionInfoRequestRegister(*this);
      _hmiAppCtrlProxy_DirectTextureConsumerPositionInfoResponse_RegId = _hmiAppCtrlProxy->sendDirectTextureConsumerPositionInfoResponseRegister(*this);
      _hmiAppCtrlProxy_DrmGarbageCollectionRequest_RegId = _hmiAppCtrlProxy->sendDrmGarbageCollectionRequestRegister(*this);
      _hmiAppCtrlProxy_FollowUpActionRequest_regId = _hmiAppCtrlProxy->sendFollowUpActionRequestRegister(*this);
      _hmiAppCtrlProxy_FollowUpActionResponse_regId = _hmiAppCtrlProxy->sendFollowUpActionResponseRegister(*this);
//      _hmiAppCtrlProxy_TraceQuery_RegId = _hmiAppCtrlProxy->sendTraceQueryRegister(*this);

      // request to activate an external image could have been send out before proxy became available,
      // so get current status for external images and post it
      //_hmiAppCtrlProxy->sendExternalImagesStatusGet(*this);
   }

   if (proxy == _hmiBaseTraceCommandProxy)
   {
      _hmiBaseTraceCommandProxy_RegId = _hmiBaseTraceCommandProxy->sendTraceQueryRegister(*this);
      _hmiBaseSimTouchProxy_RegId = _hmiBaseTraceCommandProxy->sendSimTouchRegister(*this);
      _hmiBaseSimTouchByNameProxy_RegId = _hmiBaseTraceCommandProxy->sendSimTouchByNameRegister(*this);
      _hmiBaseSimTouchByIndexProxy_RegId = _hmiBaseTraceCommandProxy->sendSimTouchByIndexRegister(*this);
      _hmiBaseSetWidgetPropertyProxy_RegId = _hmiBaseTraceCommandProxy->sendWidgetPropertyValueSetterRegister(*this);
      _hmiBaseSimTouchByTextProxy_RegId = _hmiBaseTraceCommandProxy->sendSimTouchByTextRegister(*this);
      _hmiBaseSimTouchByTextIdProxy_RegId = _hmiBaseTraceCommandProxy->sendSimTouchByTextIdRegister(*this);
      _hmiBaseSimEnterKeyProxy_RegId = _hmiBaseTraceCommandProxy->sendSimEnterKeyRegister(*this);
      _hmiBaseSimFocusReqProxy_RegId = _hmiBaseTraceCommandProxy->sendSimFocusReqRegister(*this);
      _hmiBaseSimJoystickProxy_RegId = _hmiBaseTraceCommandProxy->sendSimJoystickRegister(*this);
      _hmiBaseSimRotateEncoderProxy_RegId = _hmiBaseTraceCommandProxy->sendSimRotateEncoderRegister(*this);

      // check if surfaces are already known
      if ((_surfaceIds.mainSurfaces.size() != 0) || (_surfaceIds.popupSurfaces.size() != 0) || (_surfaceIds.permanentSurfaces.size() != 0))
      {
         std::vector<unsigned int>v;

         //add main view surface
         for (unsigned int i = 0; i < _surfaceIds.mainSurfaces.size(); ++i)
         {
            v.push_back(_surfaceIds.mainSurfaces[i]);
         }

         //add popup surfaces
         for (unsigned int i = 0; i < _surfaceIds.popupSurfaces.size(); ++i)
         {
            v.push_back(_surfaceIds.popupSurfaces[i]);
         }

         //add permanent surfaces
         for (unsigned int i = 0; i < _surfaceIds.permanentSurfaces.size(); ++i)
         {
            v.push_back(_surfaceIds.permanentSurfaces[i]);
         }

         _hmiBaseTraceCommandProxy->sendUpdateClientStatusRequest(static_cast<uint32>(getpid()), _myName, v);
      }
   }
}


void ProxyHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& /*stateChange*/)
{
   if (_hmiAppCtrlProxy == proxy)
   {
      ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onUnavailable"));
      _hmiAppCtrlProxy->sendQueryClientDeregister(_hmiAppCtrlProxy_QueryClient_RegId);
      _hmiAppCtrlProxy->sendSetDisplayDeregister(_hmiAppCtrlProxy_SetDisplay_RegId);
      _hmiAppCtrlProxy->sendSetMultiDisplayDeregister(_hmiAppCtrlProxy_SetMultiDisplay_RegId);
      _hmiAppCtrlProxy->sendEvDisplayStatusEvalDeregister(_hmiAppCtrlProxy_SetDisplay_RegId);
      _hmiAppCtrlProxy->sendKeyInfoForwardingDeregister(_hmiAppCtrlProxy_KeyInfoForwarding_RegId);
      _hmiAppCtrlProxy->sendEncoderInfoForwardingDeregister(_hmiAppCtrlProxy_EncoderInfoForwarding_RegId);
      _hmiAppCtrlProxy->sendExternalImageProviderTouchDeregister(_hmiAppCtrlProxy_ExternalImageProviderTouch_RegId);
      _hmiAppCtrlProxy->sendSlaveSurfaceRepositioningDeregister(_hmiAppCtrlProxy_SlaveSurfaceRepositioning_RegId);
      _hmiAppCtrlProxy->sendSlaveSurfaceRepositioningAckDeregister(_hmiAppCtrlProxy_SlaveSurfaceRepositioningAck_RegId);
      _hmiAppCtrlProxy->sendDirectTextureConsumerPositionInfoRequestDeregister(_hmiAppCtrlProxy_DirectTextureConsumerPositionInfo_RegId);
      _hmiAppCtrlProxy->sendDirectTextureConsumerPositionInfoResponseDeregister(_hmiAppCtrlProxy_DirectTextureConsumerPositionInfoResponse_RegId);
      _hmiAppCtrlProxy->sendDrmGarbageCollectionRequestDeregister(_hmiAppCtrlProxy_DrmGarbageCollectionRequest_RegId);
      _hmiAppCtrlProxy->sendFollowUpActionRequestDeregister(_hmiAppCtrlProxy_FollowUpActionRequest_regId);
      _hmiAppCtrlProxy->sendFollowUpActionResponseDeregister(_hmiAppCtrlProxy_FollowUpActionResponse_regId);
//      _hmiAppCtrlProxy->sendTraceQueryDeregister(_hmiAppCtrlProxy_TraceQuery_RegId);
   }
   if (proxy == _hmiBaseTraceCommandProxy)
   {
      _hmiBaseTraceCommandProxy->sendTraceQueryDeregister(_hmiBaseTraceCommandProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimTouchDeregister(_hmiBaseSimTouchProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimTouchByNameDeregister(_hmiBaseSimTouchByNameProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimTouchByIndexDeregister(_hmiBaseSimTouchByIndexProxy_RegId);
      _hmiBaseTraceCommandProxy->sendWidgetPropertyValueSetterDeregister(_hmiBaseSetWidgetPropertyProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimTouchByTextDeregister(_hmiBaseSimTouchByTextProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimTouchByTextIdDeregister(_hmiBaseSimTouchByTextIdProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimEnterKeyDeregister(_hmiBaseSimEnterKeyProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimFocusReqDeregister(_hmiBaseSimFocusReqProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimJoystickDeregister(_hmiBaseSimJoystickProxy_RegId);
      _hmiBaseTraceCommandProxy->sendSimRotateEncoderDeregister(_hmiBaseSimRotateEncoderProxy_RegId);
   }
}


void ProxyHandler::setBaseComponent(::asf::core::BaseComponent* baseComponent)
{
   _baseComponent = baseComponent;
}


void ProxyHandler::setAppState(::bosch::cm::ai::hmi::appctrl::HMIAppCtrl::hmiAppState state)
{
   if (_appState != state)
   {
      _appState = state;
      ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler appState changed to %d pid: %d appState: %d name: '%s'", _appState, getpid(), _appState, _myName.c_str()));
      hmibase::services::hmiappctrl::UpdateClientStatusPayload payload(static_cast<uint32>(getpid()), _myName, _appState);
      sendLocalMessageHMIAppCtrl(payload);
   }
}


void ProxyHandler::onQueryClientSignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< QueryClientSignal >& /*sig*/)
{
   ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onQueryClientEvent pid: %d appState: %d name: '%s'", getpid(), _appState, _myName.c_str()));
   _hmiAppCtrlProxy->sendUpdateClientStatusRequest(static_cast<uint32>(getpid()), _myName.c_str(), _appState);
}


void ProxyHandler::onQueryClientError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< QueryClientError >& /*error*/)
{
}


void ProxyHandler::onEvDisplayStatusEvalSignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< EvDisplayStatusEvalSignal >& sig)
{
   // evaluationof statusline should be done only in one proxy, therefore it is not implemented here
   if (getpid() == sig->getData().getPidReceiver())
   {
      ETG_TRACE_FATAL_THR_DCL((_myTraceID, "[[CLIENT %20s]] missing proxy implementation for onEvDisplayStatusEvalEvent", _myName.c_str()));
   }
}


void ProxyHandler::onEvDisplayStatusEvalError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< EvDisplayStatusEvalError >& /*error*/)
{
}


void ProxyHandler::onSetDisplaySignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< SetDisplaySignal >& sig)
{
   // since no provision for default parameter in Franca decleration, currently implemented with 2 different interfaces [onSetDisplaySignal, onSetMultiDisplaySignal]
   // todo: old interface need to be scraped once project adapt to new interface
   // caution: until then incase fixing anything in this function needs to be checked against onSetMultiDisplaySignal if valid

   ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] app id %d ProxyHandler::onSetDisplaySignal enter", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId()));

   if (getpid() == sig->getData().getPidReceiver())
   {
      if (sig->getData().getDisplayRegion() == static_cast<int32>(displayRegion__MAIN_VIEW))
      {
         if (sig->getData().getDisplayRegionStatus() == static_cast<int8>(hmibase::Show))
         {
            ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] 'app id %d display id %d", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId(), ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId())));
            SYSTEMD_LOG("GUISTARTUP(n): hmibase::IN_FOREGROUND called\n");
            hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<ApplicationStateUpdMsg>(hmibase::IN_FOREGROUND, ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId())));
         }
         else if (sig->getData().getDisplayRegionStatus() == static_cast<int8>(hmibase::Hide))
         {
            ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] app id %d ProxyHandler::onSetDisplaySignal Hide", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId()));
            SYSTEMD_LOG("(n)GUISTARTUP: hmibase::IN_FOREGROUND called\n");
            hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<ApplicationStateUpdMsg>(hmibase::IN_BACKGROUND, ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId())));
         }
         else if (sig->getData().getDisplayRegionStatus() == static_cast<int8>(hmibase::PrepareHide))
         {
            ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] app id %d ProxyHandler::onSetDisplaySignal PrepareHide", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId()));
            hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<ApplicationStateUpdMsg>(hmibase::TO_BACKGROUND, ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId())));
         }
      }
      else
      {
         uint32 surfaceId = getSubSurface(ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId()), sig->getData().getDisplayRegion());

         if (surfaceId != SURFACEID_PERMANENTSURFACEBASE_MAX)
         {
            ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] 'app id %d onSetDisplaySignal call for SubSurface", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId()));

            if (sig->getData().getDisplayRegionStatus() == 1)
            {
               //show status line
               ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(), surfaceId);
               SYSTEMD_LOG("GUISTARTUP: hmibase::IN_FOREGROUND -- Show called\n");
               hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SubSurfaceUpdMsg>(
                        (hmibase::displayRegion)sig->getData().getDisplayRegion(),
                        sig->getData().getRegionType(),
                        hmibase::IN_FOREGROUND,
                        ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId())));
            }
            else if (sig->getData().getDisplayRegionStatus() == 0)
            {
               //hide status line
               ScreenBrokerClient::GetInstance().Hide(::hmibase::view::ViewIdentifier(), surfaceId);
               SYSTEMD_LOG("GUISTARTUP: hmibase::IN_BACKGROUND -- Hide called\n");
               hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SubSurfaceUpdMsg>(
                        (hmibase::displayRegion)sig->getData().getDisplayRegion(),
                        sig->getData().getRegionType(),
                        hmibase::IN_BACKGROUND,
                        ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId())));
            }
         }
      }
   }
}


void ProxyHandler::onSetDisplayError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< SetDisplayError >& /*error*/)
{
}


void ProxyHandler::onSetMultiDisplaySignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< SetMultiDisplaySignal >& sig)
{
   // since no provision for default parameter in Franca decleration, currently implemented with 2 different interfaces [onSetDisplaySignal, onSetMultiDisplaySignal]
   // todo: old interface need to be scraped once project adapt to new interface
   // caution: until then incase fixing anything in this function needs to be checked against onSetDisplaySignal if valid

   ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] app id %d ProxyHandler::onSetMultiDisplaySignal enter", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId()));

   if (getpid() == sig->getData().getPidReceiver())
   {
      if (sig->getData().getDisplayRegion() == static_cast<int32>(displayRegion__MAIN_VIEW))
      {
         if (sig->getData().getDisplayRegionStatus() == static_cast<int8>(hmibase::Show))
         {
            ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] 'app id %d display id %d", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId(), sig->getData().getDisplayid()));
            hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<ApplicationStateUpdMsg>(hmibase::IN_FOREGROUND, sig->getData().getDisplayid()));
         }
         else if (sig->getData().getDisplayRegionStatus() == static_cast<int8>(hmibase::Hide))
         {
            hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<ApplicationStateUpdMsg>(hmibase::IN_BACKGROUND, sig->getData().getDisplayid()));
         }
         else if (sig->getData().getDisplayRegionStatus() == static_cast<int8>(hmibase::PrepareHide))
         {
            hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<ApplicationStateUpdMsg>(hmibase::TO_BACKGROUND, sig->getData().getDisplayid()));
         }
      }
      else
      {
         uint32 surfaceId = getSubSurface(sig->getData().getDisplayid(), sig->getData().getDisplayRegion());
         if (surfaceId != SURFACEID_PERMANENTSURFACEBASE_MAX)
         {
            ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] 'app id %d onSetDisplaySignal call for SubSurface", _myName.c_str(), ScreenBrokerClient::GetInstance().AppId()));

            if (sig->getData().getDisplayRegionStatus() == 1)
            {
               //show status line
               ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(), surfaceId);
               hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SubSurfaceUpdMsg>(
                        (hmibase::displayRegion)sig->getData().getDisplayRegion(),
                        sig->getData().getRegionType(),
                        hmibase::IN_FOREGROUND,
                        sig->getData().getDisplayid()));
            }
            else if (sig->getData().getDisplayRegionStatus() == 0)
            {
               //hide status line
               ScreenBrokerClient::GetInstance().Hide(::hmibase::view::ViewIdentifier(), surfaceId);
               hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SubSurfaceUpdMsg>(
                        (hmibase::displayRegion)sig->getData().getDisplayRegion(),
                        sig->getData().getRegionType(),
                        hmibase::IN_BACKGROUND,
                        sig->getData().getDisplayid()));
            }
         }
      }
   }
}


void ProxyHandler::onSetMultiDisplayError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< SetMultiDisplayError >& /*error*/)
{
}


void ProxyHandler::onMultiDisplayCmdError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< MultiDisplayCmdError >& /*error*/)
{
}


void ProxyHandler::onMultiDisplayCmdResponse(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< MultiDisplayCmdResponse >& /*response*/)
{
}


void ProxyHandler::evaluateKeyForwardingRequest(int HKCode, int HKState, int displayId, int userData)
{
   KeyForwardingPayload payload(
      keyInfo(
         static_cast<uint32>(HKCode),
         static_cast<uint8>(HKState),
         static_cast<uint32>(getpid()),
         0,
         static_cast<uint32>(displayId),
         static_cast<uint32>(userData)
      )
   );
   sendLocalMessageHMIAppCtrl(payload);
}


void ProxyHandler::onKeyInfoForwardingSignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< KeyInfoForwardingSignal >& sig)
{
   if ((static_cast<uint32>(getpid()) != sig->getData().getPidSender()) && (static_cast<uint32>(getpid()) == sig->getData().getPidReceiver()))
   {
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<HKStatusChangedUpdMsg>(sig->getData().getKeyCode(), (hmibase::HardKeyStateEnum)sig->getData().getKeyState(), sig->getData().getDisplayId(), sig->getData().getUserData()));
   }
}


void ProxyHandler::onKeyInfoForwardingError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< KeyInfoForwardingError >& /*error*/)
{
}


void ProxyHandler::evaluateEncoderForwardingRequest(int EncCode, int steps, int surfaceId, int userData)
{
   EncoderForwardingPayload payload(encoderInfo(
                                       static_cast<int32>(EncCode),
                                       static_cast<int32>(steps),
                                       static_cast<uint32>(getpid()),
                                       0,
                                       static_cast<uint32>(surfaceId),
                                       static_cast<uint32>(userData)));
   sendLocalMessageHMIAppCtrl(payload);
}


void ProxyHandler::onEncoderInfoForwardingSignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< EncoderInfoForwardingSignal >& signal)
{
   if ((static_cast<uint32>(getpid()) != signal->getData().getPidSender()) && (static_cast<uint32>(getpid()) == signal->getData().getPidReceiver()))
   {
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<EncoderStatusChangedUpdMsg>(signal->getData().getEncCode(), signal->getData().getEncValue(), signal->getData().getSurfaceId(), signal->getData().getUserData()));
   }
}


void ProxyHandler::onEncoderInfoForwardingError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< EncoderInfoForwardingError >& /*error*/)
{
}


void ProxyHandler::onSubSurfaceSwitchUpdateError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::SubSurfaceSwitchUpdateError >& /*error*/)
{
}


void ProxyHandler::onSubSurfaceSwitchUpdateSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::SubSurfaceSwitchUpdateSignal >& signal)
{
   hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SubSurfaceSwitchCompleteUpdMsg>(signal->getSurfaceid(), signal->getSurfaceState()));
}


void ProxyHandler::onActivateExternalImageProviderError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< ActivateExternalImageProviderError >& /*error*/)
{
}


void ProxyHandler::onActivateExternalImageProviderSignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< ActivateExternalImageProviderSignal >& sig)
{
   size_t idx = 0;
   hmibase::gadget::gadgetSceneStatus targetState;

   while (idx < sig->getInstanceIds().size())
   {
      if (mapGadgetTargetState(sig->getTargetState(), targetState))
      {
         hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<hmibase::gadget::ActivateImageProviderReqMsg>(sig->getInstanceIds()[idx], (targetState == ::hmibase::gadget::ACTIVE) ? true : false, targetState));
      }
      ++idx;
   }
}


void ProxyHandler::onExternalImageProviderTouchError(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< ExternalImageProviderTouchError >& /*error*/)
{
}


void ProxyHandler::onExternalImageProviderTouchSignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< ExternalImageProviderTouchSignal >& sig)
{
   if (static_cast<uint32>(getpid()) != sig->getPidSender())
   {
      ETG_TRACE_USR2_THR_DCL((_myTraceID, "[[CLIENT %20s]] ProxyHandler::onExternalImageProviderTouchSignal : state %d, instance %d", _myName.c_str(), sig->getData().getTouchState(), sig->getInstanceId()));
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      Courier::Message* msg = 0;
      if (sig->getData().getTouchState() == 0xff)
      {
         msg = COURIER_MESSAGE_NEW(hmibase::input::TouchAbort)(static_cast<int32>(SURFACEID_VIRTUAL_SURFACE_OFFSET) + sig->getInstanceId());
      }
      else
      {
         msg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(static_cast<Courier::TouchMsgState::Enum>(sig->getData().getTouchState()),
               sig->getData().getXPos(),
               sig->getData().getYPos(),
               sig->getData().getTimeStamp(),
               0,
               static_cast<int32>(SURFACEID_VIRTUAL_SURFACE_OFFSET) + sig->getInstanceId());
      }
      POST_MSG_NOTRACE(msg)
#else
      // Courier Touch not supported
#endif
   }
}


void ProxyHandler::onSlaveSurfaceRepositioningError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::SlaveSurfaceRepositioningError >& /*error*/)
{
}


void ProxyHandler::onSlaveSurfaceRepositioningSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::SlaveSurfaceRepositioningSignal >& signal)
{
   if (static_cast<uint32>(getpid()) != signal->getPidSender())
   {
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<hmibase::gadget::SlaveSurfaceRepositionReqMsg>(signal->getActivate(), signal->getSurfaceId(), false));
   }
}


void ProxyHandler::onSlaveSurfaceRepositioningAckError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::SlaveSurfaceRepositioningAckError >& /*error*/)
{
}


void ProxyHandler::onSlaveSurfaceRepositioningAckSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::SlaveSurfaceRepositioningAckSignal >& signal)
{
   if (static_cast<uint32>(getpid()) != signal->getPidSender())
   {
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<hmibase::gadget::SlaveSurfaceRepositionResMsg>(signal->getActivate(), signal->getSurfaceId(), false, signal->getStatus()));
   }
}


void ProxyHandler::onDirectTextureConsumerPositionInfoRequestError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::DirectTextureConsumerPositionInfoRequestError >& /*error*/)
{
}


void ProxyHandler::onDirectTextureConsumerPositionInfoRequestSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::DirectTextureConsumerPositionInfoRequestSignal >& signal)
{
   if (static_cast<uint32>(getpid()) != signal->getPidSender())
   {
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<hmibase::gadget::PositionInfoReqMsg>(signal->getInstanceId(), false));
   }
}


void ProxyHandler::onDirectTextureConsumerPositionInfoResponseError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::DirectTextureConsumerPositionInfoResponseError >& /*error*/)
{
}


void ProxyHandler::onDirectTextureConsumerPositionInfoResponseSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::DirectTextureConsumerPositionInfoResponseSignal >& signal)
{
   if (static_cast<uint32>(getpid()) != signal->getPidSender())
   {
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      Candera::Vector2 pos(signal->getXPos(), signal->getYPos());
#else
      hmibase::view::Vector2 pos(signal->getXPos(), signal->getYPos());
#endif
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<hmibase::gadget::PositionInfoResMsg>(signal->getInstanceId(), false, pos));
   }
}


void ProxyHandler::onTraceQuerySignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQuerySignal >& sig)
{
   //ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onTraceQueryEvent '%s'", getAppName.c_str()));
   int pid = sig->getData().getPid();
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
   hmibase::TraceQueryIdEnum queryId;

   Courier::UInt32 userdata_1 = 0;
   Courier::UInt32 userdata_2 = 0;
   Candera::String userdata_3;
   Candera::String userdata_4;
   std::string tempInputData;
   std::vector<std::string> inputData = sig->getData().getQueryInput();

   if ((pid == 0) || (pid == getpid()))
   {
      switch (sig->getData().getQueryId())
      {
         case traceQueryId__GET_CURRENT_MAIN_VIEW:
            queryId = hmibase::GET_CURRENT_MAIN_VIEW;
            break;
         case traceQueryId__GET_CURRENT_POPUP_VIEW:
            queryId = hmibase::GET_CURRENT_POPUP_VIEW;
            break;
         case traceQueryId__GET_CURRENT_VISIBLE_POPUP_VIEW:
            queryId = hmibase::GET_CURRENT_VISIBLE_POPUP_VIEW;
            break;
         case traceQueryId__GET_POPUP_STACK:
            queryId = hmibase::GET_POPUP_STACK;
            break;
         case traceQueryId__GET_TOP_ACTIVE_SURFACE:
            queryId = hmibase::GET_TOP_ACTIVE_SURFACE;
            break;
         case traceQueryId__GET_TOP_ACTIVE_SURFACE_NON_PERMANENT:
            queryId = hmibase::GET_TOP_ACTIVE_SURFACE_NON_PERMANENT;
            break;
         case traceQueryId__GET_STATEMACHINE_INFO:
            queryId = hmibase::GET_STATEMACHINE_INFO;
            break;
         case traceQueryId__COLLECT_TEXTMETADATA:
            queryId = hmibase::GET_TEXTMETADATA;
            break;
         case traceQueryId__GET_HMI_STARTUP_INVESTIGATION:
            queryId = hmibase::GET_STARTUP_INVESTIGATION;
            break;
         case traceQueryId__GET_FEAT_MEMORY_STATISTICS:
            queryId = hmibase::GET_FEAT_MEM_STAT;
            break;
         case traceQueryId__GET_COURIER_MESSAGE_STATISTICS:
            queryId = hmibase::GET_COURIER_MSG_STAT;
            break;
         case traceQueryId__ABORT_SB_ANIMATIONS:
            queryId = hmibase::ABORT_SB_ANIMS;
            break;
         case traceQueryId__DUMP_CANDERA_PERF_MEASURES:
            queryId = hmibase::DUMP_CANDERA_PERF_MEASURES;
            break;
         case traceQueryId__FORCE_REDRAW:
            queryId = hmibase::FORCE_REDRAW;
            break;
         case traceQueryId__GET_TOUCHSESSION_INFO:
            queryId = hmibase::TOUCHSESSION_INFO;
            break;

         case traceQueryId__GET_SCREEN_INFO:
         {
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
            queryId = hmibase::GET_SCREEN_INFO;
         }
         break;

         case traceQueryId__GET_WIDGET_INSTANCE_INFO:
         {
            queryId = hmibase::GET_WIDGET_INSTANCE_INFO;

            if (std::find(inputData.begin(), inputData.end(), "#") == inputData.end())          //lint !e864
            {
               // No ParentWigetName specified
               userdata_3 = inputData[0].c_str();
            }
            else
            {
               //ParentWidgetName specified. Assign widgetName to userdata_3
               for (std::vector<std::string>::iterator it = inputData.begin(); it != inputData.end(); ++it)
               {
                  tempInputData.append(*it);
               }
               userdata_3 = tempInputData.c_str();
            }
         }
         break;

         case traceQueryId__GET_WIDGET_PROPERTY_INFO:
         {
            // assign the PropertyName to userdata_4
            userdata_4 = sig->getData().getQueryInput()[0].c_str();
            queryId = hmibase::GET_WIDGET_PROPERTY_INFO;

            if (std::find(inputData.begin(), inputData.end(), "#") == inputData.end())          //lint !e864
            {
               // ParentWidgetName is not specified. The TraceQueryInput vector contains PropertyName and WidgetName.
               userdata_3 = sig->getData().getQueryInput()[1].c_str();
            }
            else
            {
               // ParentWidgetName is specified. Assigning WidgetName to userdata_3
               for (std::vector<std::string>::iterator it = inputData.begin() + 1; it != inputData.end(); ++it)
               {
                  tempInputData.append(*it);
               }
               userdata_3 = tempInputData.c_str();
            }
         }
         break;
         case traceQueryId__DUMP_SCREEN:
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
            queryId = hmibase::DUMP_SCREEN;
            break;
         case traceQueryId__DUMP_SCENE:
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
            queryId = hmibase::DUMP_SCENE;
            break;
         case traceQueryId__GET_ALL_VISIBLE_TEXT_IN_GIVEN_LISTWIDGET:
            userdata_1 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[0].c_str()));
            queryId = hmibase::GET_ALL_VISIBLE_TEXT_IN_GIVEN_LISTWIDGET;
            break;

         case traceQueryId__GET_ALL_VISIBLE_TEXT_IN_VIEW:
            queryId = hmibase::GET_ALL_VISIBLE_TEXT_IN_VIEW;
            break;

         case traceQueryId__GET_ALL_VISIBLE_TEXT_IN_ALL_LISTWIDGETS:
            queryId = hmibase::GET_ALL_VISIBLE_TEXT_IN_ALL_LISTWIDGETS;
            break;

         case traceQueryId__GET_FOCUSED_WIDGET_INSTANCE_INFO:
            queryId = hmibase::GET_FOCUSED_WIDGET;
            break;
         case traceQueryId__SHOW_SCENE:
         {
            queryId = hmibase::SHOW_SCENE;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
         }
         break;
         case traceQueryId__HIDE_SCENE:
         {
            queryId = hmibase::HIDE_SCENE;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
         }
         break;
         case traceQueryId__ENABLE_ARABIC_PATCH:
         {
            queryId = hmibase::ENABLE_ARABIC_PATCH;
            userdata_1 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[0].c_str()));
         }
         break;
         case traceQueryId__TOGGLE_INVALIDATION:
         {
            queryId = hmibase::TOGGLE_INVALIDATION;
         }
         break;
         case traceQueryId__TOGGLE_RENDERSTATISTICS_OVERLAY:
         {
            queryId = hmibase::TOGGLE_RENDERSTATISTICS_OVERLAY;
         }
         break;
         case traceQueryId__TRAVERSE_VIEWS:
         {
            queryId = hmibase::TRAVERSE_VIEWS;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
            userdata_4 = sig->getData().getQueryInput()[1].c_str();
         }
         break;
         case traceQueryId__SET_LOCALE:
         {
            queryId = hmibase::SET_LOCALE;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
         }
         break;
         case traceQueryId__SET_CGITRACELEVEL:
         {
            queryId = hmibase::SET_CGITRACELEVEL;
            userdata_1 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[0].c_str()));
         }
         break;

         case traceQueryId__GET_ALL_VISIBLE_TEXTID_IN_LISTITEM:
         {
            queryId = hmibase::GET_ALL_VISIBLE_TEXTID_IN_LISTITEM;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
         }
         break;

         case traceQueryId__GET_POSSIBLE_TOUCHPOINT_ELEM_NAME:
         {
            queryId = hmibase::GET_POSSIBLE_TOUCHPOINT_ELEM_NAME;
            userdata_3 = sig->getData().getQueryInput()[0].c_str(); // viewName
            if (std::find(inputData.begin(), inputData.end(), "#") == inputData.end())          //lint !e864
            {
               // No ParentWigetName specified
               userdata_4 = inputData[1].c_str();  // widgetName
            }
            else
            {
               //ParentWidgetName specified. Assign widgetName to userdata_3
               for (std::vector<std::string>::iterator it = inputData.begin() + 1; it != inputData.end(); ++it)
               {
                  tempInputData.append(*it);
               }
               userdata_4 = tempInputData.c_str();  //(parentWidgetName#childWidgetName)
            }
         }
         break;

         case traceQueryId__GET_POSSIBLE_TOUCHPOINT_ELEM_TEXT:
         {
            queryId = hmibase::GET_POSSIBLE_TOUCHPOINT_ELEM_TEXT;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
         }
         break;

         case traceQueryId__GET_POSSIBLE_TOUCHPOINT_ELEM_TEXTID:
         {
            queryId = hmibase::GET_POSSIBLE_TOUCHPOINT_ELEM_TEXTID;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
         }
         break;

         case traceQueryId__GET_POSSIBLE_TOUCHPOINT_LISTELEM:
         {
            queryId = hmibase::GET_POSSIBLE_TOUCHPOINT_LISTELEM;
            userdata_1 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[0].c_str())); // listId
            userdata_2 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[1].c_str())); // listItemIndex
         }
         break;
         case traceQueryId__SB_START_ANIMATION:
         {
            queryId = hmibase::SB_START_ANIMATION;
            userdata_1 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[0].c_str())); // surfaceId
            userdata_2 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[1].c_str())); // animationType
         }
         break;
         case traceQueryId__SET_FRAME_RATE:
         {
            queryId = hmibase::SET_FRAME_RATE;
            userdata_3 = sig->getData().getQueryInput()[0].c_str();
         }
         break;
         case traceQueryId__GET_ALL_LOADEDSCENES:
         {
            userdata_1 = static_cast<Courier::UInt32>(atoi(sig->getData().getQueryInput()[0].c_str()));
            queryId = hmibase::GET_ALL_LOADEDSCENES;
         }
         break;

         default:
            return;
      }
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<QueryTraceInfoReqMsg>(queryId, userdata_1, userdata_2, userdata_3, userdata_4));
   }
#endif
}


void ProxyHandler::onSimTouchError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimTouchSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchSignal >& sig)
{
   PARAM_UNUSED(proxy);

   if (sig->getData().getPid() == getpid())
   {
      if (sig->getData().getStatus() == bosch::cm::ai::hmi::appctrl::trace::Commands::simulatedTouchStates__COMPACT)
      {
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
         Courier::Message* lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down,
                                  sig->getData().getXPos(),
                                  sig->getData().getYPos(),
                                  0,
                                  static_cast<Courier::PointerId>(sig->getData().getPointerId()),
                                  sig->getData().getSurfaceId());
         POST_MSG_NOTRACE(lMsg);

         lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Up,
                sig->getData().getXPos(),
                sig->getData().getYPos(),
                0,
                static_cast<Courier::PointerId>(sig->getData().getPointerId()),
                sig->getData().getSurfaceId());
         POST_MSG_NOTRACE(lMsg);
#endif
      }
      else
      {
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
         Courier::TouchMsgState::Enum touchState = Courier::TouchMsgState::Up;
         switch (sig->getData().getStatus())
         {
            case bosch::cm::ai::hmi::appctrl::trace::Commands::simulatedTouchStates__DOWN:
               touchState = Courier::TouchMsgState::Down;
               break;
            case bosch::cm::ai::hmi::appctrl::trace::Commands::simulatedTouchStates__MOVE:
               touchState = Courier::TouchMsgState::Move;
               break;
            default:
               break;
         }

         /// hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<Courier::TouchMsg>(sig->getData().getXPos(),sig->getData().getYPos(),0,
         ///      static_cast<Courier::PointerId>(sig->getData().getPointerId()),sig->getData().getSurfaceId()));

         Courier:: Message* lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(touchState,
                                   sig->getData().getXPos(),
                                   sig->getData().getYPos(),
                                   0,
                                   static_cast<Courier::PointerId>(sig->getData().getPointerId()),
                                   sig->getData().getSurfaceId());
         POST_MSG_NOTRACE(lMsg);
#endif
      }
   }
}


void ProxyHandler::onSimTouchByNameError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByNameError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimTouchByNameSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByNameSignal >& sig)
{
   PARAM_UNUSED(proxy);
   if (sig->getData().getPid() == getpid())
   {
      std::vector<bosch::cm::ai::hmi::appctrl::trace::Commands::SurfaceInfo>::const_iterator it = sig->getData().getTouchableSurfaces().begin();

      std::vector< ::hmibase::trace::surfaceInfoItem > data;
      for (; it != sig->getData().getTouchableSurfaces().end(); ++it)
      {
         data.push_back(::hmibase::trace::surfaceInfoItem(it->getSurfaceId(),
                        hmibase::util::geometry::Rect(it->getRect().getXPos(),
                              it->getRect().getYPos(),
                              it->getRect().getWidth(),
                              it->getRect().getHeight()
                                                     )
                                                         )
                       );
      }

      // post courier message to process touch request
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      hmibase::trace::SurfaceInfoProvider a(data);
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SimTouchReqMsg>(hmibase::SIM_TOUCH_BY_NAME, a, sig->getData().getWidgetName().c_str(), Courier::ViewId(sig->getData().getViewName().c_str())));
#endif
   }
}


void ProxyHandler::onSimTouchByTextError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByTextError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimTouchByTextIdError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByTextIdError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimTouchByTextSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByTextSignal >& sig)
{
   PARAM_UNUSED(proxy);
   if (sig->getData().getPid() == getpid())
   {
      std::vector<bosch::cm::ai::hmi::appctrl::trace::Commands::SurfaceInfo>::const_iterator it = sig->getData().getTouchableSurfaces().begin();
      std::vector< ::hmibase::trace::surfaceInfoItem > data;
      for (; it != sig->getData().getTouchableSurfaces().end(); ++it)
      {
         data.push_back(::hmibase::trace::surfaceInfoItem(it->getSurfaceId(),
                        hmibase::util::geometry::Rect(it->getRect().getXPos(),
                              it->getRect().getYPos(),
                              it->getRect().getWidth(),
                              it->getRect().getHeight()
                                                     )
                                                         )
                       );
      }
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      ::hmibase::trace::SurfaceInfoProvider a(data);
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SimTouchReqMsg>(hmibase::SIM_TOUCH_BY_TEXT, a, sig->getData().getWidgetText().c_str(), Courier::ViewId()));
#endif
   }
}


void ProxyHandler::onSimTouchByTextIdSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByTextIdSignal >& sig)
{
   PARAM_UNUSED(proxy);
   if (sig->getData().getPid() == getpid())
   {
      std::vector<bosch::cm::ai::hmi::appctrl::trace::Commands::SurfaceInfo>::const_iterator it = sig->getData().getTouchableSurfaces().begin();
      std::vector< ::hmibase::trace::surfaceInfoItem > data;
      for (; it != sig->getData().getTouchableSurfaces().end(); ++it)
      {
         data.push_back(::hmibase::trace::surfaceInfoItem(it->getSurfaceId(),
                        hmibase::util::geometry::Rect(it->getRect().getXPos(),
                              it->getRect().getYPos(),
                              it->getRect().getWidth(),
                              it->getRect().getHeight()
                                                     )
                                                         )
                       );
      }
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      ::hmibase::trace::SurfaceInfoProvider a(data);
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SimTouchReqMsg>(hmibase::SIM_TOUCH_BY_TEXT_ID, a, sig->getData().getWidgetTextId().c_str(), Courier::ViewId()));
#endif
   }
}


void ProxyHandler::onSimTouchByIndexError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByNameError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimTouchByIndexSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimTouchByIndexSignal >& sig)
{
   PARAM_UNUSED(proxy);
   if (sig->getData().getPid() == getpid())
   {
      std::vector<bosch::cm::ai::hmi::appctrl::trace::Commands::SurfaceInfo>::const_iterator it = sig->getData().getTouchableSurfaces().begin();

      std::vector< ::hmibase::trace::surfaceInfoItem > data;
      for (; it != sig->getData().getTouchableSurfaces().end(); ++it)
      {
         data.push_back(::hmibase::trace::surfaceInfoItem(it->getSurfaceId(),
                        hmibase::util::geometry::Rect(it->getRect().getXPos(),
                              it->getRect().getYPos(),
                              it->getRect().getWidth(),
                              it->getRect().getHeight()
                                                     )
                                                         )
                       );
      }

      // post courier message to process touch request
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      ::hmibase::trace::SurfaceInfoProvider a(data);
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SimTouchReqMsg>(hmibase::SIM_TOUCH_BY_INDEX, a, "", Courier::ViewId(),  sig->getData().getListID(), sig->getData().getButtonWidgetIndex()));
#endif
   }
}


void ProxyHandler::onTraceQueryError(const ::boost::shared_ptr< CommandsProxy >& /*proxy*/, const ::boost::shared_ptr< TraceQueryError >& /*error*/)
{
}


void ProxyHandler::setSurfaceIds(hmibase::SurfaceIds ids)
{
   _surfaceIds = ids;
   if (_hmiBaseSimTouchProxy_RegId != 0)
   {
      std::vector<unsigned int>v;

      //add main view surface
      for (unsigned int i = 0; i < _surfaceIds.mainSurfaces.size(); ++i)
      {
         v.push_back(_surfaceIds.mainSurfaces[i]);
      }

      //add popup surfaces
      for (unsigned int i = 0; i < _surfaceIds.popupSurfaces.size(); ++i)
      {
         v.push_back(_surfaceIds.popupSurfaces[i]);
      }

      //add permanent surfaces
      for (unsigned int i = 0; i < _surfaceIds.permanentSurfaces.size(); ++i)
      {
         v.push_back(_surfaceIds.permanentSurfaces[i]);
      }

      _hmiBaseTraceCommandProxy->sendUpdateClientStatusRequest(static_cast<uint32>(getpid()), _myName, v);
   }
}


void ProxyHandler::onWidgetPropertyValueSetterError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::WidgetPropertyValueSetterError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onWidgetPropertyValueSetterSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::WidgetPropertyValueSetterSignal >& sig)
{
   PARAM_UNUSED(proxy);

   int pid = sig->getData().getPid();
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
   std::vector<std::string> inputData = sig->getData().getWidgetNames();
   Candera::String widgetNames;
   std::string tempStringData;

   if (std::find(inputData.begin(), inputData.end(), "#") == inputData.end())                   //lint !e864
   {
      // No ParentWigetName specified
      widgetNames = inputData[0].c_str();
   }
   else
   {
      //ParentWidgetName specified. Assign widgetName to userdata_3
      for (std::vector<std::string>::iterator it = inputData.begin(); it != inputData.end(); ++it)
      {
         tempStringData.append(*it);
      }
      widgetNames = tempStringData.c_str();
   }

   if ((pid == 0) || (pid == getpid()))
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(WidgetPropertySetterReqMsg)(
                                 widgetNames,
                                 sig->getData().getPropertyName().c_str(),
                                 sig->getData().getPropertyValue().c_str()
                              );
      if (msg != 0)
      {
         msg->Post();
      }
   }
#endif
}


void ProxyHandler::onFollowUpActionRequestError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::FollowUpActionRequestError >& /*error*/)
{
}


void ProxyHandler::onFollowUpActionRequestSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::FollowUpActionRequestSignal >& signal)
{
   if ((signal->hasActionType()) && (signal->hasUserData()))
   {
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      Courier::Message* msg = COURIER_MESSAGE_NEW(FollowUpActionReqMsg)(signal->getUserData() , static_cast<hmibase::followUpAction>(signal->getActionType()));
      POST_MSG_NOTRACE(msg)
#else
      // hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<FollowUpActionReqMsg>(signal->getUserData(), static_cast<hmibase::followUpAction>(signal->getActionType()) ));
#endif
   }
}


void ProxyHandler::onFollowUpActionResponseError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::FollowUpActionResponseError >& /*error*/)
{
}


void ProxyHandler::onFollowUpActionResponseSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::HMIAppCtrlProxy >& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::FollowUpActionResponseSignal >& signal)
{
   if ((signal->hasActionType()) && (signal->hasUserData()))
   {
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      Courier::Message* msg = COURIER_MESSAGE_NEW(FollowUpActionReqMsg)(signal->getUserData() , static_cast<hmibase::followUpAction>(signal->getActionType()));
      POST_MSG_NOTRACE(msg)
#else
      // hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<FollowUpActionReqMsg)(signal->getUserData() , static_cast<hmibase::followUpAction>(signal->getActionType())));
#endif
   }
}


uint32 ProxyHandler::getSubSurface(uint32 displayid, uint32 displayregion)
{
   (void)displayid;
   ETG_TRACE_FATAL_THR_DCL((_myTraceID, "displayid[%d], displayregion[%d]", displayid, displayregion));
   hmibase::surfaceid surfaceId = SURFACEID_PERMANENTSURFACEBASE_MAX;
   switch (displayregion)
   {
      case displayRegion__STATUS_LINE:
         surfaceId = hmibase::SURFACEID_STATUSBAR;
         break;
      case displayRegion__FOOTER_LINE:
         surfaceId = hmibase::SURFACEID_FOOTERBAR;
         break;
      case displayRegion__FOOTER_LINE_2:
         surfaceId = hmibase::SURFACEID_FOOTERBAR_2;
         break;
      default:
         break;
   }

   return static_cast<uint32>(surfaceId);
}


void ProxyHandler::onSimEnterKeyError(const ::boost::shared_ptr<	bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr<  bosch::cm::ai::hmi::appctrl::trace::Commands::SimEnterKeyError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimEnterKeySignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimEnterKeySignal >& signal)
{
   PARAM_UNUSED(proxy);
   const int pid = signal->getData().getPid();
   if ((pid == 0) || (pid == getpid()))
   {
      const HardKeyStateEnum hardKeyState = HardKeyStateEnum(signal->getData().getKeyState());

      const int* sourceId = hmibase::trace::SimInputDeviceId::getItem(signal->getData().getSource());
      const int defaultSourceId = 0;
      if (sourceId == NULL)
      {
         sourceId = &defaultSourceId;
      }

      ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onSimEnterKeySignal pid: %d keyState: %d source: %50s(%d)",
                              pid, ETG_CENUM(HardKeyStateEnum, hardKeyState), signal->getData().getSource().c_str(), *sourceId));

      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<EnterKeyStatusChangedUpdMsg>(hardKeyState, *sourceId));
   }
}


void ProxyHandler::onSimFocusReqError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimFocusReqError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimFocusReqSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimFocusReqSignal >& signal)
{
   PARAM_UNUSED(proxy);
   const int pid = signal->getData().getPid();
   if ((pid == 0) || (pid == getpid()))
   {
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      const enFocusAction focusAction = enFocusAction(signal->getData().getAction());
      const enFocusTimerAction timerAction = enFocusTimerAction(signal->getData().getTimeAction());

      ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onSimFocusReqSignal pid: %d action: %d timeAction: %d view: %50s widget: %50s",
                              pid,
                              ETG_CENUM(enFocusAction, focusAction),
                              ETG_CENUM(enFocusTimerAction, timerAction),
                              signal->getData().getViewName().c_str(),
                              signal->getData().getWidgetName().c_str()));
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<FocusControlReqMsg>(
               focusAction,
               timerAction,
               ::Courier::ViewId(signal->getData().getViewName().c_str()),
               signal->getData().getWidgetName().c_str()));
#endif
   }
}


void ProxyHandler::onSimJoystickError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimJoystickError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimJoystickSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimJoystickSignal >& signal)
{
   PARAM_UNUSED(proxy);
   const int pid = signal->getData().getPid();
   if ((pid == 0) || (pid == getpid()))
   {
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      const FocusDirectionEnum focusDirection = FocusDirectionEnum(signal->getData().getDirection());

      const int* sourceId = hmibase::trace::SimInputDeviceId::getItem(signal->getData().getSource());
      const int defaultSourceId = 0;
      if (sourceId == NULL)
      {
         sourceId = &defaultSourceId;
      }

      ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onSimJoystickSignal pid: %d direction: %d source: %50s(%d)",
                              pid, ETG_CENUM(FocusDirectionEnum, focusDirection), signal->getData().getSource().c_str(), *sourceId));

      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<JoystickStatusChangedUpdMsg>(focusDirection, *sourceId));
#endif
   }
}


void ProxyHandler::onSimRotateEncoderError(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimRotateEncoderError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onSimRotateEncoderSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsProxy >& proxy, const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::SimRotateEncoderSignal >& signal)
{
   PARAM_UNUSED(proxy);
   const int pid = signal->getData().getPid();
   if ((pid == 0) || (pid == getpid()))
   {
      const int* encCode = hmibase::trace::SimInputDeviceId::getItem(signal->getData().getEncCode());

      if (encCode != NULL)
      {
         ETG_TRACE_USR4_THR_DCL((_myTraceID, "ProxyHandler onSimRotateEncoderSignal pid: %d encCode: %50s(%d) encSteps: %d userData: %d",
                                 pid, signal->getData().getEncCode().c_str(), *encCode, signal->getData().getEncSteps(), signal->getData().getUserData()));
         hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<EncoderStatusChangedUpdMsg>(*encCode, signal->getData().getEncSteps(), signal->getData().getUserData(), signal->getData().getUserData()));
      }
      else
      {
         ETG_TRACE_FATAL_THR(("ProxyHandler EncoderStatusChangedUpdMsg encCode(%50s) is not found!!", signal->getData().getEncCode().c_str()));
      }
   }
}


void ProxyHandler::onExternalImagesStatusError(const ::boost::shared_ptr< HMIAppCtrlProxy >& proxy, const ::boost::shared_ptr< ExternalImagesStatusError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onExternalImagesStatusUpdate(const ::boost::shared_ptr< HMIAppCtrlProxy >& proxy, const ::boost::shared_ptr< ExternalImagesStatusUpdate >& update)
{
   PARAM_UNUSED(proxy);
   for (std::vector <ExternalImageData>::const_iterator citer = update->getExternalImagesStatus().begin(); citer != update->getExternalImagesStatus().end(); ++citer)
   {
      hmibase::gadget::gadgetSceneStatus targetState;
      if (mapGadgetTargetState(citer->getGadgetState(), targetState))
      {
         hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<hmibase::gadget::ActivateImageProviderReqMsg>(citer->getInstanceId(), (targetState == ::hmibase::gadget::ACTIVE) ? true : false, targetState));
      }
   }
}


void ProxyHandler::onDrmGarbageCollectionRequestError(const ::boost::shared_ptr< HMIAppCtrlProxy >& proxy, const ::boost::shared_ptr< DrmGarbageCollectionRequestError >& error)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(error);
}


void ProxyHandler::onDrmGarbageCollectionRequestSignal(const ::boost::shared_ptr< HMIAppCtrlProxy >& proxy, const ::boost::shared_ptr< DrmGarbageCollectionRequestSignal >& signal)
{
   PARAM_UNUSED(proxy);
   PARAM_UNUSED(signal);
   hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<hmibase::gadget::DrmGarbageCollectorReqMsg>());
}


} // namespace hmiappctrl
} // namespace services
}
