/* ***************************************************************************************
* FILE:          SyncBlockProducerFactory.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SyncBlockProducerFactory.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "SyncBlockProducerFactory.h"
#include "hmibase/gadget/videobuffer/VideoBufferType.h"

#include "AppUtils/Timer.h"

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SyncBlockProducerFactory.cpp.trc.h"
#endif


//#define RELEASE_DRM_BUFFER_ON_PRODUCER_TERMINATE
namespace hmibase {
namespace gadget {


SyncBlockProducerFactory* SyncBlockProducerFactory::_theInstance = 0;

SyncBlockProducerFactory::SyncBlockProducerFactory() : _garbageCollectionIntervalMs(0)
{
   //setGarbageColletionIntervalMs(5000);
   _garbageCollectionTimer.setName("DrmBufferGCTimer", "");
}


SyncBlockProducerFactory::~SyncBlockProducerFactory()
{
   std::map<BufferIdentifier, SyncBlockProducerFactory::DrmBufferContainer* >::iterator it;

   for (it = _syncBlockProvider.begin(); it != _syncBlockProvider.end(); ++it)
   {
      delete it->second;
      it->second = 0;
   }

   _syncBlockProvider.erase(_syncBlockProvider.begin(), _syncBlockProvider.end());
}


bool SyncBlockProducerFactory::Init()
{
   if (_theInstance == 0)
   {
      _theInstance = new SyncBlockProducerFactory();
   }
   return true;
}


void SyncBlockProducerFactory::Deinit()
{
   if (_theInstance != 0)
   {
      delete _theInstance;
      _theInstance = 0;
   }
}


SyncBlockProducerFactory::DrmBufferContainer* SyncBlockProducerFactory::GetSyncBlockProducer(unsigned int syncBlockId, unsigned int instanceId, Accessor* accessor)
{
   static unsigned int s_AccessorId = 0;
   BufferIdentifier key(syncBlockId, instanceId);
   std::map<BufferIdentifier, SyncBlockProducerFactory::DrmBufferContainer*>::iterator it = _syncBlockProvider.find(key);

   if (it == _syncBlockProvider.end())
   {
      _syncBlockProvider[key] = new DrmBufferContainer(syncBlockId, instanceId);
      it = _syncBlockProvider.find(key);
   }

   // check if the accessor is new
   if (accessor != 0 && accessor->GetId(key) == 0)
   {
      ETG_TRACE_USR4_THR(("SyncBlockProducer::GetSyncBlockProducer new accessor ref count %d snycblock %d instanceid %d", it->second->getRefCount(), syncBlockId, instanceId));
      // new accessor, so increase reference count and assign id to accessor
      it->second->increaseRefCount();
      accessor->SetId(key, ++s_AccessorId);
   }
   else
   {
      // invalid accessor or accessor has already accessed this buffer before
      ETG_TRACE_USR4_THR(("SyncBlockProducer::GetSyncBlockProducer invalid accessor %d %d", syncBlockId, instanceId));
   }

   return it->second;
}


void SyncBlockProducerFactory::ReleaseSyncBlockProducer(unsigned int syncBlockId, unsigned int instanceId, Accessor* accessor)
{
   BufferIdentifier key(syncBlockId, instanceId);
   std::map<BufferIdentifier, SyncBlockProducerFactory::DrmBufferContainer*>::iterator it = _syncBlockProvider.find(key);

   if ((it != _syncBlockProvider.end()) && (it->second != 0))
   {
      if (accessor != 0)
      {
         // reset accessor id
         accessor->SetId(key, 0);
      }

      // decrease reference counter
      it->second->decreaseRefCount();

      // check for outstanding references and start termination if necessary
      if (it->second->getRefCount() == 0)
      {
         ETG_TRACE_USR4_THR(("SyncBlockProducer::ReleaseProducer %d %d", syncBlockId, instanceId));
         it->second->terminateProducer();

         if (_garbageCollectionIntervalMs > 0)
         {
            hmibase::util::Ticker ticker;

            uint32_t currentTime = ticker.getTickCountMsec();
            // align to full seconds
            unsigned int timestamp = currentTime + _garbageCollectionIntervalMs;

            it->second->setOutOfCacheTimestampMs(timestamp);

            setNextGCTimestamp(timestamp);
         }
      }
   }
}


bool SyncBlockProducerFactory::setGarbageColletionIntervalMs(unsigned int millisecondsInterval)
{
   if (millisecondsInterval == 0)
   {
      _garbageCollectionTimer.stop();
   }

   if (millisecondsInterval != _garbageCollectionIntervalMs)
   {
      _garbageCollectionIntervalMs = millisecondsInterval;
   }

   return true;
}


bool SyncBlockProducerFactory::GarbageCollect(GCReason reason)
{
   bool success = false;
   uint32_t totalBytes = 0;
   std::map<BufferIdentifier, DrmBufferContainer*>::iterator it = _syncBlockProvider.begin();

   // try to delete all provider with no more references on both sides
   while (it != _syncBlockProvider.end())
   {
      DrmBufferContainer* bufferContainer = it->second;
      if (bufferContainer != 0)
      {
         if (bufferContainer->getRefCount() == 0)
         {
            // no more references on provider side, so we can try to delete
            if (bufferContainer->checkTermination())
            {
               bool toBeDeleted = false;

               if (reason == GC_FORCE)
               {
                  toBeDeleted = true;
               }
               else
               {
                  // timer triggered garbage collection
                  hmibase::util::Ticker ticker;
                  uint32_t currentTime = ticker.getTickCountMsec();

                  if (bufferContainer->getOutOfCacheTimestampMs() == 0)
                  {
                     // do nothing as this element should stay in cache until GC_FORCE
                  }
                  else if (currentTime < bufferContainer->getOutOfCacheTimestampMs())
                  {
                     // another timer run is required
                     setNextGCTimestamp(bufferContainer->getOutOfCacheTimestampMs());
                  }
                  else
                  {
                     toBeDeleted = true;
                  }
               }

               if (toBeDeleted)
               {
                  if (bufferContainer->getCurrentBuffer() != 0)
                  {
                     totalBytes += bufferContainer->getCurrentBuffer()->getDataSize() * 3;
                  }
                  delete bufferContainer;

                  success = true;

                  std::map<BufferIdentifier, DrmBufferContainer*>::iterator toErase = it;
                  ++it;
                  _syncBlockProvider.erase(toErase);
               }
            }
         }
      }

      if (!success)
      {
         ++it;
      }
      success = false;
   }

   if (totalBytes > 0)
   {
      ETG_TRACE_USR1_THR(("SyncBlockProducerFactory::GarbageCollect(%d) %u bytes freed !!!", ETG_CENUM(GCReason, reason), totalBytes));
   }
   else
   {
      ETG_TRACE_USR1_THR(("SyncBlockProducerFactory::GarbageCollect(%d) nothing to be freed, %u buffers in use", ETG_CENUM(GCReason, reason), _syncBlockProvider.size()));

      if (_syncBlockProvider.size() > 0)
      {
         for (it = _syncBlockProvider.begin(); it != _syncBlockProvider.end(); ++it)
         {
            ETG_TRACE_USR1_THR(("SyncBlockProducerFactory::GarbageCollect(%d) used buffers: InstanceId %d, size 3 * %d, RefCount %d", ETG_CENUM(GCReason, reason), it->first.second, it->second->getCurrentBuffer()->getDataSize(), it->second->getRefCount()));
         }
      }
   }

   return success;
}


bool SyncBlockProducerFactory::setNextGCTimestamp(uint32_t timeStampMs)
{
   bool timerStarted = false;

   if ((_garbageCollectionTimer.getStatus() == hmibase::util::TimerThreaded::Stopped)
         || (_garbageCollectionTimer.getStatus() == hmibase::util::TimerThreaded::Expired))
   {
      // align to full seconds
      uint32_t roundedTimeStamp = (static_cast<uint32_t>((timeStampMs + 500) / 1000) * 1000);

      hmibase::util::Ticker ticker;
      uint32_t currentTime = ticker.getTickCountMsec();

      if (currentTime < roundedTimeStamp)
      {
#if defined(VARIANT_S_FTR_ENABLE_COURIERMESSAGING)
         _garbageCollectionTimer.setTimeout(0, roundedTimeStamp - currentTime, COURIER_MESSAGE_NEW(DrmGCTimeoutMsg)());
#endif
         ETG_TRACE_USR4_THR(("SyncBlockProducerFactory::setNextGCTimestamp(%u) start timer for %u ms", timeStampMs, roundedTimeStamp - currentTime));
         _garbageCollectionTimer.start();
         timerStarted = true;
      }
      else
      {
         ETG_TRACE_USR4_THR(("SyncBlockProducerFactory::setNextGCTimestamp(%u) timestamp in past (%u ms >= %u ms)", timeStampMs, currentTime, roundedTimeStamp));
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("SyncBlockProducerFactory::setNextGCTimestamp(%u) timer already running/paused", timeStampMs));
   }

   return timerStarted;
}


bool SyncBlockProducerFactory::OnMessage(const Courier::Message& msg)
{
   bool consumed = false;
#if defined(VARIANT_S_FTR_ENABLE_COURIERMESSAGING)
   switch (msg.GetId())
   {
      case hmibase::gadget::DrmGarbageCollectorReqMsg::ID:
      {
         GarbageCollect(GC_FORCE);
      }
      break;
      case DrmGCTimeoutMsg::ID:
      {
         GarbageCollect(GC_TIMER);
      }
      break;
      default:
         break;
   }
#else
   PARAM_UNUSED(msg);
#endif
   return consumed;
}


SyncBlockProducerFactory::DrmBufferContainer::DrmBufferContainer(unsigned int syncBlockId, unsigned int instanceId): _syncBlockId(syncBlockId), _instanceId(instanceId), _attached(false), _bufHandle(-1), _refCount(0), _outOfCacheTimestampMs(0)
{
   _producer = new ProducerType(static_cast<int>(syncBlockId));

   if (_producer)
   {
      _attached = _producer->attach(static_cast<int>(instanceId));
   }
}


SyncBlockProducerFactory::DrmBufferContainer::~DrmBufferContainer()
{
   if (_producer)
   {
      if (_attached)
      {
         if (!(_producer->isTerminated()))
         {
            ETG_TRACE_FATAL_THR(("SyncBlock Producer not terminated before drmBuffer free"));
         }
      }

      delete _producer;
      _producer = 0;
   }

#if defined(VARIANT_S_FTR_ENABLE_COURIERMESSAGING)
   uint32_t totalSize = 0;
   std::map<int, hmibase::gadget::videobuffer::VideoBufferType*>::iterator it;
   for (it = _drmBuffers.begin(); it != _drmBuffers.end(); ++it)
   {
      hmibase::gadget::videobuffer::VideoBufferType* videoBufferPtr = it->second;
      if (videoBufferPtr != 0)
      {
         totalSize += videoBufferPtr->getDataSize();
         delete videoBufferPtr;
      }
   }

   _drmBuffers.clear();

   Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::DrmAllocationStatusMsg)(hmibase::gadget::DEALLOCATION_SUCCESS, totalSize);
   if (msg)
   {
      msg->Post();
   }
#endif
}


bool SyncBlockProducerFactory::DrmBufferContainer::terminateProducer()
{
   if (_producer != 0)
   {
      return _producer->startTermination();
   }
   return true;
}


bool SyncBlockProducerFactory::DrmBufferContainer::checkTermination()
{
   if (_producer != 0)
   {
      return _producer->checkTermination();
   }
   return true;
}


bool SyncBlockProducerFactory::DrmBufferContainer::createDrmBuffers(int w, int h, int depth, int bpp)
{
#ifdef WIN32
   return false;
#else
   if (_producer == 0 || !_attached)
   {
      ETG_TRACE_FATAL_THR(("SyncBlockProducerFactory::DrmBufferContainer::createDrmBuffers failed, instanceId %u producer %p, attached %d", _instanceId, _producer, _attached));
      return false;
   }

   bool haveToCreateBuffers = false;
   for (unsigned int i = 0; i < 3; i++)
   {
      int bufHandle = _producer->getBufHandle(i);
      if (bufHandle == -1)
      {
         haveToCreateBuffers = true;
      }
   }

   if (haveToCreateBuffers)
   {
      ETG_TRACE_USR1_THR(("SyncBlockProducerFactory::DrmBufferContainer::createDrmBuffers creating buffers instanceId %u width %d, height %d, depth %d, bpp %d", _instanceId , w, h, depth, bpp));

      // Size and Pitch are known after VideoBuffer is allocated
      uint32_t size  = 0;
      uint32_t pitch = 0;

      for (unsigned int i = 0; i < 3; i++)
      {
         int bufHandle = _producer->getBufHandle(i);
         if (bufHandle != -1)
         {
            // buffer was already allocated, so add again for further processing
            _producer->addBufHandle(i, bufHandle);
            ETG_TRACE_USR1_THR(("SyncBlockProducerFactory::DrmBufferContainer::createDrmBuffers recover"));
         }
         else
         {
            hmibase::gadget::videobuffer::VideoBufferType* videoBufferPtr = new hmibase::gadget::videobuffer::VideoBufferType(static_cast<uint16_t>(w), static_cast<uint16_t>(h), static_cast<uint16_t>(depth), static_cast<uint16_t>(bpp));
            int id = videoBufferPtr->create();
#if defined(VARIANT_S_FTR_ENABLE_COURIERMESSAGING)
            hmibase::gadget::DrmAllocationStatusMsg* msg = COURIER_MESSAGE_NEW(hmibase::gadget::DrmAllocationStatusMsg)();

            if (id == -1)
            {
               // seems to be that allocation failed, so collect garbage, free buffers and try again
               delete videoBufferPtr;
               videoBufferPtr = NULL;
               if (SyncBlockProducerFactory::GetInstance().GarbageCollect())
               {
                  videoBufferPtr = new hmibase::gadget::videobuffer::VideoBufferType(static_cast<uint16_t>(w),
                        static_cast<uint16_t>(h),
                        static_cast<uint16_t>(depth),
                        static_cast<uint16_t>(bpp));
                  id = videoBufferPtr->create();
               }
               else
               {
                  // no memory to be freed in this process, so post message about failed allocation to force other process to free their unused drm memory
                  ETG_TRACE_FATAL_THR(("SyncBlockProducerFactory::DrmBufferContainer::createDrmBuffers failed, instanceId %u, width %d, height %d, depth %d, bpp %d", _instanceId, w, h, depth, bpp));

                  if (msg)
                  {
                     msg->SetStatus(hmibase::gadget::ALLOCATION_FAILED);
                     msg->SetSize(0);
                     msg->Post();
                  }

                  return false;
               }
            }
#endif
            size  = videoBufferPtr->getDataSize();
            pitch = videoBufferPtr->getPitch();

            if (id != -1)
            {
               _producer->addBufHandle(i, id);
               _drmBuffers[id] = videoBufferPtr;
               ETG_TRACE_USR1_THR(("SyncBlockProducerFactory::DrmBufferContainer::createDrmBuffers success,instanceId %u width %d, height %d, depth %d, bpp %d", _instanceId , w, h, depth, bpp));
#if defined(VARIANT_S_FTR_ENABLE_COURIERMESSAGING)
               if (msg)
               {
                  msg->SetStatus(hmibase::gadget::ALLOCATION_SUCCESS);
                  msg->SetSize(size);
                  msg->Post();
               }
#endif
            }
            else
            {
               ETG_TRACE_FATAL_THR(("SyncBlockProducerFactory::DrmBufferContainer::createDrmBuffers failed, instanceId %u, width %d, height %d, depth %d, bpp %d", _instanceId, w, h, depth, bpp));
               delete videoBufferPtr;
               videoBufferPtr = NULL;
#if defined(VARIANT_S_FTR_ENABLE_COURIERMESSAGING)
               if (msg)
               {
                  msg->SetStatus(hmibase::gadget::ALLOCATION_FAILED);
                  msg->SetSize(0);
                  msg->Post();
               }
#endif
               return false;
            }
         }
      }

      BufferInfoType bi(size, static_cast<uint16_t>(w), static_cast<uint16_t>(h), static_cast<uint16_t>(depth), static_cast<uint16_t>(bpp), static_cast<uint32_t>(pitch));
      _producer->setBufferInfo(bi);
   }

   _bufHandle = _producer->exchange();
   return true;
#endif
}


hmibase::gadget::videobuffer::VideoBufferType* SyncBlockProducerFactory::DrmBufferContainer::getCurrentBuffer()
{
   if ((_producer == 0) || (!_attached) || (_drmBuffers.find(_bufHandle) == _drmBuffers.end()))
   {
      ETG_TRACE_ERR_THR(("SyncBlockProducerFactory::DrmBufferContainer::getCurrentBuffer returns NULL, instanceId %u, producer %p, attached %d, bufferHandler %d", _instanceId, _producer, _attached, _bufHandle));
      return 0;
   }

   return _drmBuffers[_bufHandle];
}


int SyncBlockProducerFactory::DrmBufferContainer::getCurrentBufferId()
{
   return _bufHandle;
}


int SyncBlockProducerFactory::DrmBufferContainer::exchange()
{
   if ((_producer == 0) || (!_attached))
   {
      return -1;
   }

   _bufHandle = _producer->exchange();
   return _bufHandle;
}


SyncBlockProducerFactory::SyncBlockIdSet SyncBlockProducerFactory::GetSyncBlockIDForInstance(unsigned int producerInstanceId)
{
   SyncBlockConnectionsMap::iterator it = _syncBlockConnections.find(producerInstanceId);

   if (it != _syncBlockConnections.end())
   {
      return it->second;
   }

   // return an empty set
   return SyncBlockIdSet();
}


void SyncBlockProducerFactory::AddIdTuple(unsigned int syncBlockId, unsigned int producerInstanceId)
{
   SyncBlockConnectionsMap::iterator it = _syncBlockConnections.find(producerInstanceId);

   if (it != _syncBlockConnections.end())
   {
      it->second.insert(syncBlockId);
   }
   else
   {
      SyncBlockIdSet s;
      s.insert(syncBlockId);
      _syncBlockConnections[producerInstanceId] = s;
   }
}


}
}
