/* ***************************************************************************************
* FILE:          FTask.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FTask.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_TASK_H__
#define __FOCUS_TASK_H__

#include <queue>

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FConfigInfo.h"

namespace Focus {
//////////////////////////////////////////////////////////////////////////////////////////////
class FManager;
class FMsgQueue;
class FSession;
class FWidgetConfig;
class FViewConfig;
class FAppConfig;

//////////////////////////////////////////////////////////////////////////////////////////////
class FTask
{
   public:
      enum Result
      {
         Completed,
         Suspended,
         Error
      };

      FTask(FManager& manager) : _manager(manager) {}
      virtual ~FTask() {}

      virtual Result execute() = 0;

   protected:
      FManager& _manager;

   private:
      FTask(const FTask&);
      FTask& operator=(const FTask&);
};


//////////////////////////////////////////////////////////////////////////////////////////////
typedef std::queue<FTask*> FTaskQueue;

//////////////////////////////////////////////////////////////////////////////////////////////
class FTaskManager
{
   public:
      virtual ~FTaskManager() {}

      virtual FTask::Result executeTasks() = 0;
      virtual FTask* getExecutingTask() = 0;
      virtual unsigned int getTaskExecutionCount() const = 0;

      virtual bool hasTasks() const = 0;
      virtual void clearTasks() = 0;
      virtual void addTask(FTask* task) = 0;
      virtual bool configureTasks(FSession& session) = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FTaskFactory
{
   public:
      virtual ~FTaskFactory() {}

      virtual bool configureTasks(FTaskManager& taskManager, FSession& session) = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FSessionUpdater : public FTask
{
      typedef FTask Base;
   public:
      FSessionUpdater(FManager& manager) : Base(manager) {}
      virtual ~FSessionUpdater() {};

      virtual FTask::Result execute();
      virtual FTask::Result update(FSession& session) = 0;

   protected:
      virtual bool updateAllApps(FSession& session);
      virtual bool updateAllViews(FSession& session, FAppConfigSharedPtr appConfig);
      virtual bool updateAllWidgets(FSession& session, FViewConfig& viewConfig);

   private:
      FSessionUpdater(const FSessionUpdater&);
      FSessionUpdater& operator=(const FSessionUpdater&);

      virtual bool updateApp(FSession&, FAppConfigSharedPtr)
      {
         return true;
      }
      virtual bool updateView(FSession&, FViewConfig&)
      {
         return true;
      }
      virtual bool updateWidget(FSession&, FWidgetConfig&)
      {
         return true;
      }
};


}

#endif
