/* ***************************************************************************************
* FILE:          FManagerConfig.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FManagerConfig.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_MANAGER_CONFIG_H__
#define __FOCUS_MANAGER_CONFIG_H__

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FController.h"
#include "FSession.h"
#include "FIpc.h"

namespace Focus {
//////////////////////////////////////////////////////////////////////////////////////////////
class FTimerListener
{
   public:
      virtual ~FTimerListener() {}

      virtual void onRestarted() = 0;
      virtual void onStopped() = 0;
      virtual void onExpired() = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FTimer
{
   public:
      virtual ~FTimer() {}

      virtual void restart() = 0;
      virtual void stop() = 0;

      virtual void setTimeout(FTimerTimeout timeout) = 0;
      virtual FTimerTimeout getTimeout() const = 0;

      virtual void addListener(FTimerListener& listener) = 0;
      virtual void removeListener(FTimerListener& listener) = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class Focusable;
class FSession;
class FGroup;


//////////////////////////////////////////////////////////////////////////////////////////////
class FListener
{
   public:
      virtual ~FListener() {}

      virtual void onInternalSetFocus(FSession& session, Focusable* newFocus) = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FAvgManager
{
   public:
      virtual ~FAvgManager() {}

      virtual bool isCurrentFocusAvailable(FSession& session, FGroup* group = NULL) = 0;
      virtual Focusable* getCurrentFocusChild(FSession& session, FGroup& group) = 0;
      virtual Focusable* getDefaultFocusChild(FSession& session, FGroup& group) = 0;

      virtual Focusable* getFocus(FSession& session, FGroup* group = NULL) = 0;
      virtual void setFocus(FSession& session, Focusable* f) = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FAnimationManager
{
   public:
      virtual ~FAnimationManager() {}

      virtual bool isAnimationRunning(FFrameworkWidget* widget) = 0;
};


////////////////////////////////////////////////////////////////////////////
class FConsistencyChecker
{
   public:
      virtual ~FConsistencyChecker() {}

      //verifies if the current focus is still valid and returns this validity info
      virtual bool checkFocus() = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FVisibilityManager
{
   public:
      virtual ~FVisibilityManager() {}

      virtual bool isFocusVisible() const = 0;
      virtual void setFocusVisible(bool visible) = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FInputMsgChecker
{
   public:
      virtual ~FInputMsgChecker() {}

      virtual bool isFocusInputMessage(const FMessage& msg) const = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FMsgQueue
{
   public:
      virtual ~FMsgQueue() {}

      virtual void clear() = 0;
      virtual size_t count() const = 0;
      virtual bool empty() const = 0;

      virtual void pushBack(const FMessage& msg) = 0;
      virtual void popFront() = 0;
      virtual const FMessage* front() const = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FIpcManager : public FIpcSender, public FIpcReceiver
{
   public:
      virtual ~FIpcManager() {}

      virtual bool isOtherAppsInfoRequestInProgress() const = 0;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FOutputMsgHandler
{
   public:
      virtual ~FOutputMsgHandler() {}

      virtual bool dispatchToView(const FViewId& viewId, const FMessage& msg) = 0;
      virtual bool dispatchToViews(const FMessage& msg) = 0;
      virtual void postMessage(FMessage* msg) = 0;

      virtual void postFocusUpdateMessage(const FViewId& viewId, const FId& widgetId) = 0;
      virtual bool focusWidget(const FViewId& viewId, const FId& widgetId) = 0;
      virtual bool unfocusWidget(const FViewId& viewId, const FId& widgetId) = 0;
};


////////////////////////////////////////////////////////////////////////////
class FSessionManager : public FMsgReceiver
{
   public:
      virtual ~FSessionManager() {}

      virtual FSession* getSession() = 0;

      virtual void updateSession() = 0;
      virtual bool beginSession(FSession::Mode mode) = 0;
      virtual void endSession() = 0;
      virtual void abortSession() = 0;

      virtual bool isSessionSuspended() = 0;
      virtual void requestSessionSuspend() = 0;
      virtual void clearSessionSuspend() = 0;
};


////////////////////////////////////////////////////////////////////////////
class FAvgManager;
class FSessionManager;
class FTaskFactory;
class FTaskManager;
class FManagerConfiguration
{
      friend class FManagerConfigurator;

   public:
      FManagerConfiguration();
      virtual ~FManagerConfiguration();

      /////////////////////////////////////////////////////////////////////////
      FAppId getCurrentAppId() const
      {
         return _currentAppId;
      }

      ///////////////////////////////////////////////////////////////////////////
      const FControllerList* getControllerSet(FControllerSetId setId) const
      {
         return _defaultControllers.getList(setId);
      }
      FControllerSetId getRootGroupControllerSetId() const
      {
         return _rootGroupControllerSetId;
      }

      /////////////////////////////////////////////////////////////////////////
      FTimer* getActivityTimer() const
      {
         return _activityTimer;
      }
      FTimer* getWatchdogTimer() const
      {
         return _watchdogTimer;
      }

      /////////////////////////////////////////////////////////////////////////
      FAvgManager* getAvgManager() const
      {
         return _avgManager;
      }
      FAnimationManager* getAnimationManager() const
      {
         return _animationManager;
      }
      FConsistencyChecker* getConsistencyChecker() const
      {
         return _consistencyChecker;
      }
      FVisibilityManager* getVisibilityManager() const
      {
         return _visibilityManager;
      }
      FInputMsgChecker* getInputMsgChecker() const
      {
         return _inputMsgChecker;
      }
      FMsgQueue* getInputMsgQueue() const
      {
         return _inputMsgQueue;
      }
      FIpcManager* getIpcManager() const
      {
         return _ipcManager;
      }
      FOutputMsgHandler* getOutputMsgHandler() const
      {
         return _outputMsgHandler;
      }
      FSessionManager* getSessionManager() const
      {
         return _sessionManager;
      }
      FTaskFactory* getTaskFactory() const
      {
         return _taskFactory;
      }
      FTaskManager* getTaskManager() const
      {
         return _taskManager;
      }

      ///////////////////////////////////////////////////////////////////////////
      //bool getDefaultWrapAround() const { return _defaultWrapAround; }
      //bool getDefaultPreserveFocus() const { return _defaultPreserveFocus; }

   private:
      FManagerConfiguration(const FManagerConfiguration&);
      FManagerConfiguration& operator=(const FManagerConfiguration&);

      /////////////////////////////////////////////////////////////////////////
      FAppId _currentAppId;

      ///////////////////////////////////////////////////////////////////////////
      //map with all the registered controller sets available to be used from widgets
      FControllerListMap _defaultControllers;

      //id of the controller set used by the root group
      FControllerSetId _rootGroupControllerSetId;

      /////////////////////////////////////////////////////////////////////////
      //detects focus inactivity
      FTimer* _activityTimer;

      //watchdog when waiting for updates
      FTimer* _watchdogTimer;

      /////////////////////////////////////////////////////////////////////////
      //manages current/default focus in AVG
      FAvgManager* _avgManager;

      //manages focus animation
      FAnimationManager* _animationManager;

      //verifies if the current focus is valid
      FConsistencyChecker* _consistencyChecker;

      //manages visibility of the focus
      FVisibilityManager* _visibilityManager;

      //accumulates focus input messages for one session
      FMsgQueue* _inputMsgQueue;

      //verifies if a msg is a focus input message
      FInputMsgChecker* _inputMsgChecker;

      //used for ipc between apps with visible surfaces
      FIpcManager* _ipcManager;

      //dispatches/post messages
      FOutputMsgHandler* _outputMsgHandler;

      //manages session
      FSessionManager* _sessionManager;

      //factory used for creating of tasks associated with each session state
      FTaskFactory* _taskFactory;

      //manages task execution
      FTaskManager* _taskManager;

      ///////////////////////////////////////////////////////////////////////////
      ////default wrap around policy for groups
      //bool _defaultWrapAround;

      ////default wrap around policy for groups
      //bool _defaultPreserveFocus;
};


////////////////////////////////////////////////////////////////////////////
class FManagerConfiguration;
class FManagerConfigurator
{
   public:
      FManagerConfigurator(FManagerConfiguration& configuration) : _configuration(configuration) {}
      virtual ~FManagerConfigurator() {}

      virtual bool initialize() = 0;
      virtual bool finalize() = 0;

   protected:
      void setCurrentAppId(FAppId value)
      {
         _configuration._currentAppId = value;
      }

      void registerController(FControllerSetId setId, FController* controller)
      {
         _configuration._defaultControllers.add(setId, controller);
      }
      void unregisterController(FControllerSetId setId, const FController* controller)
      {
         _configuration._defaultControllers.remove(setId, controller);
      }
      void setRootGroupControllerSetId(FControllerSetId value)
      {
         _configuration._rootGroupControllerSetId = value;
      }

      void setActivityTimer(FTimer* value)
      {
         _configuration._activityTimer = value;
      }
      void setWatchdogTimer(FTimer* value)
      {
         _configuration._watchdogTimer = value;
      }

      void setAvgManager(FAvgManager* value)
      {
         _configuration._avgManager = value;
      }
      void setAnimationManager(FAnimationManager* value)
      {
         _configuration._animationManager = value;
      }
      void setConsistencyChecker(FConsistencyChecker* value)
      {
         _configuration._consistencyChecker = value;
      }
      void setVisibilityManager(FVisibilityManager* value)
      {
         _configuration._visibilityManager = value;
      }
      void setInputMsgQueue(FMsgQueue* value)
      {
         _configuration._inputMsgQueue = value;
      }
      void setInputMsgChecker(FInputMsgChecker* value)
      {
         _configuration._inputMsgChecker = value;
      }
      void setIpcManager(FIpcManager* value)
      {
         _configuration._ipcManager = value;
      }
      void setOutputMsgHandler(FOutputMsgHandler* value)
      {
         _configuration._outputMsgHandler = value;
      }
      void setSessionManager(FSessionManager* value)
      {
         _configuration._sessionManager = value;
      }
      void setTaskFactory(FTaskFactory* value)
      {
         _configuration._taskFactory = value;
      }
      void setTaskManager(FTaskManager* value)
      {
         _configuration._taskManager = value;
      }

      void setWrapAround(bool value)
      {
         FConfiguration::_defaultWrapAround = value;
      }
      void setPreserveFocus(bool value)
      {
         FConfiguration::_defaultPreserveFocus = value;
      }

      FManagerConfiguration& _configuration;

   private:
      FManagerConfigurator(const FManagerConfigurator&);
      FManagerConfigurator& operator=(const FManagerConfigurator&);
};


}

#endif
