/* ***************************************************************************************
* FILE:          FManager.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FManager.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_MANAGER_H__
#define __FOCUS_MANAGER_H__

#include <set>

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FContainer.h"
#include "FController.h"
#include "FManagerConfig.h"
#include "FTask.h"
#include "FSession.h"
#include "FStateInfo.h"

namespace Focus {

//////////////////////////////////////////////////////////////////////////////////////////////
class FManager : public FManagerConfiguration, public FMsgReceiver
{
      typedef FManagerConfiguration Base;

   public:
      static FManager& getInstance();

      /***********************************************************************/
      /* INITIALIZATION                                                      */
      /***********************************************************************/
      void addMsgReceiver(FMsgReceiver& msgReceiver);
      void removeMsgReceiver(const FMsgReceiver& msgReceiver);

      /***********************************************************************/
      /* LISTENERS                                                           */
      /***********************************************************************/
      void addListener(FListener& listener);
      void removeListener(const FListener& listener);
      void notifyInternalSetFocus(FSession& session, Focusable* newFocus);

      /***********************************************************************/
      /* WIDGET CONFIGURATION                                     */
      /***********************************************************************/
      /* Returns a handle associated to a widget or group in a specific view
       * which can be used to configure the focus info for that focusable item. */
      FWidgetConfig* getHandle(const FViewId& viewId, const FId& widgetId);

      /***********************************************************************/
      /* CALLBACK                                                            */
      /***********************************************************************/
      /* A message is received. */
      bool onMessage(const FMessage& msg);
      /* Callback when a view is activated or deactivated. */
      void onViewActivate(const FViewId& viewId, bool activate);
      /* Callback from widgets which indicates that they received focus. */
      void onFocusChanged(const FViewId& viewId, const FId& widgetId);

      /***********************************************************************/
      /* VISIBILITY                                                          */
      /***********************************************************************/
      void hideFocus();
      void showFocus();
      bool isFocusVisible() const;
      void setFocusVisible(bool visible);

      /***********************************************************************/
      /* SESSION                                                             */
      /***********************************************************************/
      FSession* getSession();
      bool beginSession(FSession::Mode mode);
      void updateSession();
      void printSessionDebugInfo();

      /***********************************************************************/
      /* TASKS                                                               */
      /***********************************************************************/
      /* Executes the tasks in the queue. Completed tasks will be removed and deleted.
         If a tasks returns a non Completed result the execution will stop and the same result will be returned also by this method.  */
      FTask::Result executeTasks();
      FTask* getExecutingTask();
      void configureTasks();

      /***********************************************************************/
      /* CURRENT APP STATE                                                   */
      /***********************************************************************/
      FAppStateSharedPtr getCurrentAppState() const;
      void setCurrentAppState(FAppStateSharedPtr appState);
      void initializeCurrentAppState();

      /***********************************************************************/
      /* CURRENT FOCUS                                                       */
      /***********************************************************************/
      /* Publish current focus. */
      void publishCurrentFocus();

      /* Returns the current focus. */
      const FAppCurrentFocusInfo& getCurrentFocus() const;

      /* Explicitelly sets the focus for the specified item if it is focusable. */
      void setFocus(FSession& session, const FViewId& viewId, const FId& widgetId);

      /* Verifies if the specified message is a focus input message. */
      bool isFocusInputMessage(const FMessage& msg) const;

      /***********************************************************************/
      /* CURRENT FOCUS                                                       */
      /***********************************************************************/
      /* Returns the current sequence id. */
      FSequenceId getCurrentSequenceId() const;
      /* Modifies the current sequence id. Can be used to get value from another application. */
      void setCurrentSequenceId(const FSequenceId& sequenceId);
      /* Generates a new sequence id when the group has been changed or when the focus became visible. */
      void generateNewSequenceId();

      bool isViewActive(const FViewId& viewId) const;

   private:
      FManager();
      ~FManager();

      //state info of the current app which is persisted between sessions
      FAppStateSharedPtr _currentAppState;
      //contains the focused widget (and its view) from the current application
      FAppCurrentFocusInfo _currentFocus;
      //identifies the moment when focus became visible or when the current group was changed
      //used in connection with preserve focus to validate the current state of a group
      FSequenceId _currentSequenceId;

      //these message receiver have a chance to consume messages before the focus manager
      typedef std::vector<FMsgReceiver*> MsgReceiversType;
      MsgReceiversType _msgReceivers;

      typedef std::vector<FListener*> ListenersType;
      ListenersType _listeners;

      //indicates which views are active
      typedef std::set<FViewId> ActiveViewIdsType;
      ActiveViewIdsType _activeViewIds;
};


}

#endif
