/* ***************************************************************************************
* FILE:          FManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FManager.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FManager.h"
#include "FCommon.h"
#include "FContainer.h"
#include "FConfigInfo.h"
#include "FDataSet.h"
#include "FData.h"
#include "FActiveViewGroup.h"
#include "FController.h"
#include "FStateInfo.h"
#include "FSession.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FManager.cpp.trc.h"
#endif

namespace Focus {

FManager& FManager::getInstance()
{
   static FManager instance;
   return instance;
}


FManager::FManager() : Base()
{
}


FManager::~FManager()
{
}


void FManager::addMsgReceiver(FMsgReceiver& msgReceiver)
{
   _msgReceivers.push_back(&msgReceiver);
}


void FManager::removeMsgReceiver(const FMsgReceiver& msgReceiver)
{
   FContainerUtils::remove(_msgReceivers, &msgReceiver);
}


void FManager::addListener(FListener& listener)
{
   _listeners.push_back(&listener);
}


void FManager::removeListener(const FListener& listener)
{
   FContainerUtils::remove(_listeners, &listener);
}


void FManager::notifyInternalSetFocus(FSession& session, Focusable* newFocus)
{
   for (ListenersType::iterator it = _listeners.begin(); it != _listeners.end(); ++it)
   {
      FListener* listener = *it;
      if (listener != NULL)
      {
         listener->onInternalSetFocus(session, newFocus);
      }
   }
}


bool FManager::isFocusInputMessage(const FMessage& msg) const
{
   return (getInputMsgChecker() != NULL) && (getInputMsgChecker()->isFocusInputMessage(msg));
}


bool FManager::onMessage(const FMessage& msg)
{
   for (MsgReceiversType::iterator it = _msgReceivers.begin(); it != _msgReceivers.end(); ++it)
   {
      FMsgReceiver* msgReceiver = *it;
      if ((msgReceiver != NULL) && (msgReceiver->onMessage(msg)))
      {
         return true;
      }
   }

   FSessionManager* sessionManager = getSessionManager();
   if (sessionManager != NULL)
   {
      if (isFocusInputMessage(msg) && (getInputMsgQueue() != NULL))
      {
         getInputMsgQueue()->pushBack(msg);

         if (sessionManager->getSession() == NULL)
         {
            sessionManager->beginSession(FSession::Master);
         }
         sessionManager->updateSession();

         return true;
      }

      if (sessionManager->onMessage(msg))
      {
         return true;
      }
   }

   return false;
}


void FManager::onViewActivate(const FViewId& viewId, bool active)
{
   ETG_TRACE_USR1_THR(("FManager::onViewActivate app=%d, active=%u, view=%s",
                       getCurrentAppId(), active, FVIEWID_STR(viewId)));

   if (viewId != Constants::InvalidViewId)
   {
      if (active)
      {
         _activeViewIds.insert(viewId);
      }
      else
      {
         _activeViewIds.erase(viewId);

         ////if the view of the current focused widget gets deactivated we hide the focus
         //if ((viewId == _currentFocus.ViewId) && isFocusVisible())
         //{
         //   hideFocus();
         //}
      }
   }
}


bool FManager::isViewActive(const FViewId& viewId) const
{
   return (viewId != Constants::InvalidViewId) && (_activeViewIds.find(viewId) != _activeViewIds.end());
}


FWidgetConfig* FManager::getHandle(const FViewId& viewId, const FId& widgetId)
{
   FWidgetConfig* handle = NULL;

   FSession* session = getSession();
   if (session != NULL)
   {
      FAppConfigSharedPtr appConfig = session->getCurrentAppConfig();
      if (!appConfig.PointsToNull())
      {
         FViewConfig* view = appConfig->getOrCreateChild(viewId);
         if (view != NULL)
         {
            handle = view->getOrCreateChild(widgetId);
         }
      }
   }

   return handle;
}


void FManager::hideFocus()
{
   ETG_TRACE_USR1_THR(("FManager::hideFocus app=%d", getCurrentAppId()));
   setFocusVisible(false);
   publishCurrentFocus();
}


void FManager::showFocus()
{
   ETG_TRACE_USR1_THR(("FManager::showFocus app=%d", getCurrentAppId()));
   setFocusVisible(true);
   publishCurrentFocus();
}


bool FManager::isFocusVisible() const
{
   return (getVisibilityManager() != NULL) && (getVisibilityManager()->isFocusVisible());
}


void FManager::setFocusVisible(bool visible)
{
   if (getVisibilityManager() != NULL)
   {
      getVisibilityManager()->setFocusVisible(visible);
   }
}


const FAppCurrentFocusInfo& FManager::getCurrentFocus() const
{
   return _currentFocus;
}


void FManager::publishCurrentFocus()
{
   FViewId viewId = Constants::InvalidViewId;
   FId widgetId = Constants::InvalidId;
   if ((isFocusVisible()) && (!_currentAppState.PointsToNull()))
   {
      viewId = _currentAppState->CurrentFocusInfo.ViewId;
      widgetId = _currentAppState->CurrentFocusInfo.WidgetId;
   }
   ETG_TRACE_USR1_THR(("FManager::publishCurrentFocus app=%d, view=%50s, widget=%s",
                       getCurrentAppId(), FVIEWID_STR(viewId), FID_STR(widgetId)));

   //unfocus old focused widget if it was previously valid and now was changed
   if ((getOutputMsgHandler() != NULL)
         && (_currentFocus.ViewId != Constants::InvalidViewId)
         && (_currentFocus.WidgetId != Constants::InvalidId)
         && ((_currentFocus.WidgetId != widgetId) || (_currentFocus.ViewId != viewId)))
   {
      getOutputMsgHandler()->unfocusWidget(_currentFocus.ViewId, _currentFocus.WidgetId);
   }

   _currentFocus.ViewId = viewId;
   _currentFocus.WidgetId = widgetId;

   if (getOutputMsgHandler() != NULL)
   {
      //focus new widget
      if ((_currentFocus.ViewId != Constants::InvalidViewId)
            && (_currentFocus.WidgetId != Constants::InvalidId))
      {
         getOutputMsgHandler()->focusWidget(_currentFocus.ViewId, _currentFocus.WidgetId);
      }

      //post update message
      getOutputMsgHandler()->postFocusUpdateMessage(_currentFocus.ViewId, _currentFocus.WidgetId);
   }
}


void FManager::onFocusChanged(const FViewId& viewId, const FId& widgetId)
{
   ETG_TRACE_USR1_THR(("FManager::onFocusChanged app=%d, view=%50s, widget=%s",
                       getCurrentAppId(), FVIEWID_STR(viewId), FID_STR(widgetId)));
}


void FManager::setFocus(FSession& session, const FViewId& viewId, const FId& widgetId)
{
   ETG_TRACE_USR1_THR(("FManager::setFocus app=%d, view=%50s, widget=%s",
                       getCurrentAppId(), FVIEWID_STR(viewId), FID_STR(widgetId)));

   FWidget* f = session.ActiveViewGroup.Widgets.get(FAppViewWidgetId(getCurrentAppId(), viewId, widgetId));
   if ((f != NULL) && (getAvgManager() != NULL))
   {
      getAvgManager()->setFocus(session, f);
   }
}


FSequenceId FManager::getCurrentSequenceId() const
{
   return _currentSequenceId;
}


void FManager::setCurrentSequenceId(const FSequenceId& sequenceId)
{
   ETG_TRACE_USR1_THR(("FManager::setCurrentSequenceId app=%d, id=%d",
                       getCurrentAppId(), sequenceId.Id));
   _currentSequenceId = sequenceId;
}


void FManager::generateNewSequenceId()
{
   _currentSequenceId = FSequenceId::generate();
   ETG_TRACE_USR1_THR(("FManager::generateNewSequenceId app=%d, id=%d",
                       getCurrentAppId(), _currentSequenceId.Id));
}


FAppStateSharedPtr FManager::getCurrentAppState() const
{
   return _currentAppState;
}


void FManager::setCurrentAppState(FAppStateSharedPtr appState)
{
   _currentAppState = appState;
}


void FManager::initializeCurrentAppState()
{
   if (_currentAppState.PointsToNull())
   {
      ETG_TRACE_USR1_THR(("FManager::initializeCurrentAppState initializing current app %d state", getCurrentAppId()));
      _currentAppState = FAppStateSharedPtr(FOCUS_NEW(FAppState)(getCurrentAppId()));
   }
}


FSession* FManager::getSession()
{
   return (getSessionManager() != NULL) ? getSessionManager()->getSession() : NULL;
}


bool FManager::beginSession(FSession::Mode mode)
{
   return (getSessionManager() != NULL) && getSessionManager()->beginSession(mode);
}


void FManager::updateSession()
{
   if (getSessionManager() != NULL)
   {
      getSessionManager()->updateSession();
   }
}


FTask::Result FManager::executeTasks()
{
   if (getTaskManager() != NULL)
   {
      return getTaskManager()->executeTasks();
   }
   return FTask::Completed;
}


FTask* FManager::getExecutingTask()
{
   return getTaskManager() != NULL ? getTaskManager()->getExecutingTask() : NULL;
}


void FManager::configureTasks()
{
   FSession* session = getSession();
   if ((getTaskManager() != NULL) && (session != NULL))
   {
      getTaskManager()->configureTasks(*session);
   }
}


}
