/* ***************************************************************************************
* FILE:          FIpc.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FIpc.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_IPC_H__
#define __FOCUS_IPC_H__

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FConfigInfo.h"
#include "FStateInfo.h"

namespace Focus {
//callback interface used to receive focus requests and info from other apps.
class FIpcReceiver
{
   public:
      virtual ~FIpcReceiver() {}

      //called after another app requested focus info from this app because this app has visible views.
      //received as a result of a sendOtherAppsInfoRequest in the app which has input focus.
      virtual void onCurrentAppInfoRequestReceived() = 0;

      //called when another app updated the focus state for this app after processing focus input messages.
      virtual void onCurrentAppStateReceived(FAppStateSharedPtr appState) = 0;

      //called when another app responded to a focus info request from this app.
      //should be called for each other visible app after this app called sender.sendOtherAppsInfoRequest.
      virtual void onOtherAppInfoReceived(FAppConfigSharedPtr appConfig, FAppStateSharedPtr appState) = 0;

      //called when the info for all other apps was received.
      //should be called after a call to sender.sendOtherAppsInfoRequest was done and all the info was received so that the receiver knows not to expect more onOtherAppInfoReceived calls.
      virtual void onOtherAppsInfoReceiveCompleted() = 0;
};


//used to send focus requests and info to other app.
class FIpcSender
{
   public:
      virtual ~FIpcSender() {}

      //sends a message to all other apps which have visible surfaces.
      //the Ipc receivers in those apps should get an onCurrentAppInfoRequestReceived callback.
      virtual bool sendOtherAppsInfoRequest() = 0;

      //sends a message to other app identified by appState->getId().
      //the Ipc receivers in that app should get an onCurrentAppStateReceived callback.
      //this app has input focus and processes focus info from all visible apps.
      //after that the other apps have to be updated with their new focus info.
      virtual bool sendOtherAppState(FAppStateSharedPtr appState) = 0;

      //sends a message to the app which previously requested info from all apps with visible surfaces.
      //the Ipc receivers in that app should get an onOtherAppInfoReceived callback.
      virtual bool sendCurrentAppInfo(FAppConfigSharedPtr appConfig, FAppStateSharedPtr appState) = 0;
};


//incorporates the sender and mechanism to register receivers.
class FIpcAdapter : public FIpcSender
{
   public:
      FIpcAdapter() {}
      virtual ~FIpcAdapter() {}

      void addReceiver(FIpcReceiver& receiver);
      void removeReceiver(FIpcReceiver& receiver);

   protected:
      void notifyCurrentAppInfoRequestReceived();
      void notifyCurrentAppStateReceived(Focus::FAppStateSharedPtr appState);
      void notifyOtherAppInfoReceived(Focus::FAppConfigSharedPtr appConfig, Focus::FAppStateSharedPtr appState);
      void notifyOtherAppsInfoReceiveCompleted();

   private:
      FIpcAdapter(const FIpcAdapter&);
      FIpcAdapter& operator=(const FIpcAdapter&);

      typedef std::vector<Focus::FIpcReceiver*> ReceiversType;
      ReceiversType _receivers;
};


}

#endif
