/* ***************************************************************************************
* FILE:          FGroupTraverser.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FGroupTraverser.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_GROUP_TRAVERSER_H__
#define __FOCUS_GROUP_TRAVERSER_H__

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FActiveViewGroup.h"
#include "FData.h"

namespace Focus {
////////////////////////////////////////////////////////////////////////////
class Focusable;
class FocusableVisitor;
class FGroup;
class FSession;

////////////////////////////////////////////////////////////////////////////
class FGroupTraverser
{
   public:
      FGroupTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial);
      virtual ~FGroupTraverser();

      bool advance();
      void reset();

   protected:
      bool visitChild(const FGroup& group, size_t index);
      void visit(Focusable& f);

      virtual bool advanceImpl(FGroup& group, size_t startIndex) = 0;

      FSession& _session;
      FocusableVisitor& _visitor;
      FGroup& _rootGroup;
      Focusable* _current;

   private:
      FGroupTraverser(const FGroupTraverser&);
      FGroupTraverser& operator=(const FGroupTraverser&);
};


////////////////////////////////////////////////////////////////////////////
class FDefaultGroupTraverser : public FGroupTraverser
{
      typedef FGroupTraverser Base;

   public:
      FDefaultGroupTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial, bool ascending = true);

   private:
      FDefaultGroupTraverser(const FDefaultGroupTraverser&);
      FDefaultGroupTraverser& operator=(const FDefaultGroupTraverser&);

      virtual bool advanceImpl(FGroup& group, size_t startIndex);

      bool _ascending;
};


////////////////////////////////////////////////////////////////////////////
class FGroupAscendingTraverser : public FGroupTraverser
{
      typedef FGroupTraverser Base;

   public:
      FGroupAscendingTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial);

   private:
      FGroupAscendingTraverser(const FGroupAscendingTraverser&);
      FGroupAscendingTraverser& operator=(const FGroupAscendingTraverser&);

      virtual bool advanceImpl(FGroup& group, size_t startIndex);
};


////////////////////////////////////////////////////////////////////////////
class FGroupDescendingTraverser : public FGroupTraverser
{
      typedef FGroupTraverser Base;

   public:
      FGroupDescendingTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial);

   private:
      FGroupDescendingTraverser(const FGroupDescendingTraverser&);
      FGroupDescendingTraverser& operator=(const FGroupDescendingTraverser&);

      virtual bool advanceImpl(FGroup& group, size_t startIndex);
};


////////////////////////////////////////////////////////////////////////////
class FWidget;
class FGroup;
class FocusableVisitor
{
   public:
      virtual ~FocusableVisitor() {}
      virtual void visitWidget(FWidget& widget) = 0;
      virtual void visitGroup(FGroup& group) = 0;
};


////////////////////////////////////////////////////////////////////////////
template <typename TFocusable>
class FocusableVisitorBase : public FocusableVisitor
{
   public:
      FocusableVisitorBase() : _focusable(NULL), _count(0) {}
      virtual ~FocusableVisitorBase()
      {
         _focusable = NULL;
      }

      size_t getCount() const
      {
         return _count;
      }
      TFocusable* getFocusable() const
      {
         return _focusable;
      }

      virtual void reset()
      {
         _focusable = NULL;
         _count = 0;
      }

      virtual bool isValid(TFocusable& focusable)
      {
         //keyword template is required because focusable.getData<FWidgetData>() will not compile with gcc due to an error "expected primary-expression before '>' token"
         FWidgetData* widgetData = focusable.template getData<FWidgetData>();
         return (widgetData != NULL) && widgetData->Enabled;
      }

   protected:
      void visit(TFocusable& focusable)
      {
         if (isValid(focusable))
         {
            ++_count;
            _focusable = &focusable;
         }
      }

   private:
      TFocusable* _focusable;
      size_t _count;
};


////////////////////////////////////////////////////////////////////////////
class FWidgetVisitor : public FocusableVisitorBase<FWidget>
{
      typedef FocusableVisitorBase<FWidget> Base;

   public:
      virtual ~FWidgetVisitor() {}

      virtual void visitWidget(FWidget& widget)
      {
         Base::visit(widget);
      }
      virtual void visitGroup(FGroup&) {}
};


////////////////////////////////////////////////////////////////////////////
class FGroupVisitor : public FocusableVisitorBase<FGroup>
{
      typedef FocusableVisitorBase<FGroup> Base;

   public:
      virtual ~FGroupVisitor() {}

      virtual void visitWidget(FWidget&) {}
      virtual void visitGroup(FGroup& group)
      {
         Base::visit(group);
      }
};


}
#endif
