/* ***************************************************************************************
* FILE:          FGroupTraverser.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FGroupTraverser.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FGroupTraverser.h"
#include "FActiveViewGroup.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FGroupTraverser.cpp.trc.h"
#endif

namespace Focus {

FGroupTraverser::FGroupTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial) :
   _session(session),
   _visitor(visitor),
   _rootGroup(rootGroup),
   _current(initial)
{
}


FGroupTraverser::~FGroupTraverser()
{
   _current = NULL;
}


void FGroupTraverser::reset()
{
   _current = NULL;
}


bool FGroupTraverser::advance()
{
   FGroup* group = NULL;
   Focusable* current = NULL;

   //_current was not set => use root group as starting point
   if (_current == NULL)
   {
      group = &_rootGroup;
   }
   else
   {
      //_current is a group
      group = dynamic_cast<FGroup*>(_current);

      //_current is a widget
      if (group == NULL)
      {
         group = _current->getParent();
         current = _current;
      }
   }

   FGroup* lastRefreshedGroup = NULL;
   while (group != NULL)
   {
      if (lastRefreshedGroup != group)
      {
         group->refreshReachableChildren(_session);
         lastRefreshedGroup = group;
      }

      size_t index = Constants::InvalidIndex;

      //if current is set it must be contained into group's container
      if (current != NULL)
      {
         index = group->ReachableChildren.indexOf(current);
         if (index == Constants::InvalidIndex)
         {
            //invalid container content
            return false;
         }
      }

      if (advanceImpl(*group, index))
      {
         return true;
      }

      //root group reached=>nothing else to search
      if (group == &_rootGroup)
      {
         return false;
      }

      current = group;
      group = group->getParent();
   }

   return false;
}


bool FGroupTraverser::visitChild(const FGroup& group, size_t index)
{
   Focusable* f = group.ReachableChildren.get(index);
   if (f != NULL)
   {
      visit(*f);
      return true;
   }

   return false;
}


void FGroupTraverser::visit(Focusable& f)
{
   f.visit(_visitor);
   _current = &f;
}


FDefaultGroupTraverser::FDefaultGroupTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial, bool ascending) :
   Base(session, visitor, rootGroup, initial), _ascending(ascending)
{
}


bool FDefaultGroupTraverser::advanceImpl(FGroup& group, size_t startIndex)
{
   if (group.ReachableChildren.count() > 0)
   {
      size_t i;
      //ascending traverse
      if (_ascending)
      {
         if (startIndex == Constants::InvalidIndex)
         {
            i = 0;
         }
         else
         {
            i = startIndex + 1;
         }

         while (i < group.ReachableChildren.count())
         {
            if (visitChild(group, i))
            {
               return true;
            }
            ++i;
         }
      }
      //descending traverse
      else
      {
         if (startIndex == Constants::InvalidIndex)
         {
            i = group.ReachableChildren.count();
         }
         else
         {
            i = startIndex;
         }

         while (i > 0)
         {
            if (visitChild(group, i - 1))
            {
               return true;
            }
            --i;
         }
      }
   }
   return false;
}


FGroupAscendingTraverser::FGroupAscendingTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial) :
   Base(session, visitor, rootGroup, initial)
{
}


bool FGroupAscendingTraverser::advanceImpl(FGroup& group, size_t startIndex)
{
   if (group.ReachableChildren.count() > 0)
   {
      size_t i;
      if (startIndex == Constants::InvalidIndex)
      {
         i = 0;
      }
      else
      {
         i = startIndex + 1;
      }
      while (i < group.ReachableChildren.count())
      {
         if (visitChild(group, i))
         {
            return true;
         }
         ++i;
      }
   }
   return false;
}


FGroupDescendingTraverser::FGroupDescendingTraverser(FSession& session, FocusableVisitor& visitor, FGroup& rootGroup, Focusable* initial) :
   Base(session, visitor, rootGroup, initial)
{
}


bool FGroupDescendingTraverser::advanceImpl(FGroup& group, size_t startIndex)
{
   if (group.ReachableChildren.count() > 0)
   {
      size_t i;
      if (startIndex == Constants::InvalidIndex)
      {
         i = group.ReachableChildren.count();
      }
      else
      {
         i = startIndex;
      }

      while (i > 0)
      {
         if (visitChild(group, i - 1))
         {
            return true;
         }
         --i;
      }
   }
   return false;
}


}
