/* ***************************************************************************************
* FILE:          FCourierAdapter.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FCourierAdapter.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include <queue>
#include "Courier/Messaging/MessageReferrer.h"
#include "View/CGI/CgiExtensions/AppViewHandler.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FCommon.h"
#include "FCourierAdapter.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FCourierAdapter.cpp.trc.h"
#endif

namespace Focus {

FCourierMessageQueue::~FCourierMessageQueue()
{
   while (!_queue.empty())
   {
      Courier::MessageReferrer* ref = _queue.front();
      if (ref != NULL)
      {
         FOCUS_DELETE(ref);
         ref = NULL;
      }
      _queue.pop();
   }     //lint !e438 intended
}


void FCourierMessageQueue::clear()
{
   while (!_queue.empty())
   {
      Courier::MessageReferrer* ref = _queue.front();
      if (ref != NULL)
      {
         FOCUS_DELETE(ref);
         ref = NULL;
      }
      _queue.pop();
   }     //lint !e438 intended
}


size_t FCourierMessageQueue::count() const
{
   return _queue.size();
}


bool FCourierMessageQueue::empty() const
{
   return _queue.empty();
}


void FCourierMessageQueue::pushBack(const FMessage& msg)
{
   ETG_TRACE_USR1_THR(("FCourierMessageQueue::pushBack id=%u, name=%s", msg.GetSequenceNumber(), msg.GetName()));
   _queue.push(FOCUS_NEW(Courier::MessageReferrer)(const_cast<FMessage*>(&msg)));
}


void FCourierMessageQueue::popFront()
{
   Courier::MessageReferrer* ref = _queue.front();
   if (ref != NULL)
   {
      if (ref->GetMessage() != NULL)
      {
         const FMessage& msg = *(ref->GetMessage());
         ETG_TRACE_USR1_THR(("FCourierMessageQueue::popFront id=%u, name=%s", msg.GetSequenceNumber(), msg.GetName()));
      }
      FOCUS_DELETE(ref);
      ref = NULL;
   }
   _queue.pop();
}        //lint !e438 intended


const FMessage* FCourierMessageQueue::front() const
{
   Courier::MessageReferrer* ref = _queue.front();
   if (ref != NULL)
   {
      return ref->GetMessage();
   }
   return NULL;
}


bool FCourierOutputMsgHandler::dispatchToView(const FViewId& viewId, const FMessage& msg)
{
   Courier::View* courierView = _viewHandler.FindView(viewId);
   if (courierView != NULL)
   {
      return courierView->OnMessage(msg);
   }
   return false;
}


bool FCourierOutputMsgHandler::dispatchToViews(const FMessage& msg)
{
   return _viewHandler.OnMessageViewTreeOnly(msg);
}


void FCourierOutputMsgHandler::postMessage(FMessage* msg)
{
   if (msg != NULL)
   {
      msg->Post();
   }
}


void FCourierOutputMsgHandler::postFocusUpdateMessage(const FViewId& viewId, const FId& widgetId)
{
   postMessage(COURIER_MESSAGE_NEW(FocusChangedUpdMsg)(viewId, Courier::Identifier(FID_STR(widgetId))));
}


bool FCourierOutputMsgHandler::focusWidget(const FViewId& viewId, const FId& widgetId)
{
   Courier::FrameworkWidget* widget = _viewHandler.FindWidget(viewId,

                                      Courier::CompositePath(),

                                      Courier::ItemId(widgetId.c_str()));
   if (widget != NULL)
   {
      //const Courier::FocusEvent focusEvent;
      //widget->DispatchEvent(focusEvent);
      widget->OnFocus();
      return true;
   }

   return false;
}


bool FCourierOutputMsgHandler::unfocusWidget(const FViewId& viewId, const FId& widgetId)
{
   Courier::FrameworkWidget* widget = _viewHandler.FindWidget(viewId,

                                      Courier::CompositePath(),

                                      Courier::ItemId(widgetId.c_str()));
   if (widget != NULL)
   {
      widget->OnLostFocus();
      return true;
   }

   return false;
}


FCourierTimer::FCourierTimer(FTimerTimeout timeout) :
   _timeout(timeout),
   _listeners(false)
{
   _timer = CANDERA_NEW(Util::Timer)(_timeout);
   if (_timer != NULL)
   {
      _timer->setName("FocusTimer", "");
   }
}


FCourierTimer::~FCourierTimer()
{
   if (_timer != NULL)
   {
      _timer->stop();
      CANDERA_DELETE(_timer);
      _timer = NULL;
   }
}


void FCourierTimer::setTimeout(FTimerTimeout timeout)
{
   _timeout = timeout;
   if (_timer != NULL)
   {
      _timer->setTimeout(0, timeout);
   }
}


void FCourierTimer::restart()
{
   ETG_TRACE_USR1_THR(("FCourierTimer::restart timeout=%u", _timeout));

   if ((_timer != NULL) && (_timeout > 0))
   {
      _timer->restart();

      for (size_t i = 0; i < _listeners.count(); ++i)
      {
         FTimerListener* listener = _listeners.get(i);
         if (listener != NULL)
         {
            listener->onRestarted();
         }
      }
   }
}


void FCourierTimer::stop()
{
   ETG_TRACE_USR1_THR(("FCourierTimer::stop"));

   if (_timer != NULL)
   {
      _timer->stop();

      for (size_t i = 0; i < _listeners.count(); ++i)
      {
         FTimerListener* listener = _listeners.get(i);
         if (listener != NULL)
         {
            listener->onStopped();
         }
      }
   }
}


bool FCourierTimer::onMessage(const FMessage& msg)
{
   if (msg.GetId() == TimerExpiredMsg::ID)
   {
      const TimerExpiredMsg* timerExpiredMsg = Courier::message_cast<const TimerExpiredMsg*>(&msg);
      if ((timerExpiredMsg != NULL) && (_timer != NULL) && (timerExpiredMsg->GetTimer() == _timer))
      {
         onExpired();
         return true;
      }
   }

   return false;
}


void FCourierTimer::onExpired()
{
   ETG_TRACE_USR1_THR(("FCourierTimer::onExpired"));

   if (_listeners.count() > 0)
   {
      for (size_t i = 0; i < _listeners.count(); ++i)
      {
         FTimerListener* listener = _listeners.get(i);
         if (listener != NULL)
         {
            listener->onExpired();
         }
      }
   }
}


void FCourierTimer::addListener(FTimerListener& listener)
{
   _listeners.add(&listener);
}


void FCourierTimer::removeListener(FTimerListener& listener)
{
   _listeners.remove(&listener);
}


bool FCourierInputMsgChecker::isFocusInputMessage(const FMessage& msg) const
{
   bool result = false;
   switch (msg.GetId())
   {
      //case HKStatusChangedUpdMsg::ID:
      case EnterKeyStatusChangedUpdMsg::ID:
      case EncoderStatusChangedUpdMsg::ID:
      case JoystickStatusChangedUpdMsg::ID:
      case FocusControlReqMsg::ID:
      {
         result = true;
         break;
      }

      default:
         break;
   }

   return result;
}


}
