/* ***************************************************************************************
* FILE:          FContainer.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FContainer.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_CONTAINER_H__
#define __FOCUS_CONTAINER_H__

#include <algorithm>

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"

namespace Focus {
class FContainerUtils
{
   public:
      template <typename T>
      static bool remove(std::vector<T*>& vector, const T* item)
      {
         typename std::vector<T*>::iterator it = std::find(vector.begin(), vector.end(), item);
         if (it != vector.end())
         {
            vector.erase(it);
            return true;
         }
         return false;
      }
};


template <typename TChild>
class FVectorContainer
{
      typedef std::vector<TChild*> ChildrenType;

   public:
      FVectorContainer(bool deleteChildren) : _children(), _deleteChildren(deleteChildren) {}
      virtual ~FVectorContainer()
      {
         if (_deleteChildren)
         {
            deleteAll();
         }
      }

      void add(TChild* child)
      {
         _children.push_back(child);
      }

      void addAll(FVectorContainer<TChild>& other)
      {
         for (typename ChildrenType::iterator it = other._children.begin(); it != other._children.end(); ++it)
         {
            _children.push_back(*it);
         }
      }

      void removeAt(size_t index)
      {
         TChild* child = _children[index];
         _children.erase(_children.begin() + index);
         if ((child != NULL) && _deleteChildren)
         {
            FOCUS_DELETE(child);
            child = NULL;
         }
      }

      void remove(TChild* child)
      {
         bool found = FContainerUtils::remove(_children, child);
         if (found && (child != NULL) && _deleteChildren)
         {
            FOCUS_DELETE(child);
            child = NULL;
         }
      }  //lint !e438 intended

      void removeAll()
      {
         if (_deleteChildren)
         {
            deleteAll();
         }
         _children.clear();
      }

      TChild* get(size_t index) const
      {
         return (index < count()) ? _children.at(index) : NULL;
      }

      size_t count() const
      {
         return _children.size();
      }

      size_t indexOf(const TChild* child) const
      {
         typename ChildrenType::const_iterator it = std::find(_children.begin(), _children.end(), child);
         if (it != _children.end())
         {
            return it - _children.begin();      //lint !e732 intended
         }
         return Constants::InvalidIndex;
      }

      template <typename TVal>
      size_t indexOf(bool (*func)(const TChild*, const TVal& val), const TVal& val) const
      {
         for (typename ChildrenType::const_iterator it = _children.begin(); it != _children.end(); ++it)
         {
            if (func(*it, val))
            {
               return it - _children.begin();
            }
         }
         return Constants::InvalidIndex;
      }

      void sort(bool (*compareFunction)(TChild*, TChild*))
      {
         std::stable_sort(_children.begin(), _children.end(), compareFunction);
      }

      void execute(void (*func)(TChild*))
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(*it);
         }
      }

      template <typename TVal>
      void execute(void (*func)(TChild*, TVal* val), TVal* val)
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(*it, val);
         }
      }

   private:
      ChildrenType _children;
      bool _deleteChildren;//delete children when they are removed from the container

      void deleteAll()
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            if (*it != NULL)
            {
               FOCUS_DELETE(*it);
               *it = NULL;
            }
         }
      }
};


template <typename TKey, typename TChild>
class FMapContainer
{
      typedef std::map<TKey, TChild*> ChildrenType;

   public:
      FMapContainer(bool deleteChildren) : _children(), _deleteChildren(deleteChildren) {}
      virtual ~FMapContainer()
      {
         if (_deleteChildren)
         {
            deleteAll();
         }
      }

      size_t count() const
      {
         return _children.size();
      }

      const TKey& getKeyAt(size_t index) const
      {
         typename ChildrenType::const_iterator it = _children.begin();
         std::advance(it, index);
         return it->first;
      }

      TChild* getAt(size_t index) const
      {
         if (index < _children.size())
         {
            typename ChildrenType::const_iterator it = _children.begin();
            std::advance(it, index);
            return it->second;
         }
         return NULL;
      }

      TChild* get(const TKey& id) const
      {
         typename ChildrenType::const_iterator it = _children.find(id);
         if (it != _children.end())
         {
            return it->second;
         }
         return NULL;
      }

      void add(const TKey& id, TChild* child)
      {
         _children[id] = child;
      }

      void remove(const TKey& id, bool ignoreDeleteFlag = false)
      {
         typename ChildrenType::iterator it = _children.find(id);
         if (it != _children.end())
         {
            if (!ignoreDeleteFlag && _deleteChildren && (it->second != NULL))
            {
               FOCUS_DELETE(it->second);
               it->second = NULL;
            }
            _children.erase(it);
         }
      }

      void removeAll()
      {
         if (_deleteChildren)
         {
            deleteAll();
         }
         _children.clear();
      }

      void removeWhere(bool (*func)(const TKey&, TChild*))
      {
         typename ChildrenType::iterator it = _children.begin();
         while (it != _children.end())
         {
            if (func(it->first, it->second))
            {
               if (_deleteChildren && (it->second != NULL))
               {
                  FOCUS_DELETE(it->second);
                  it->second = NULL;
               }
               _children.erase(it++);//advance the iterator than erase using the old iterator
            }
            else
            {
               ++it;
            }
         }
      }

      void execute(void (*func)(const TKey&, TChild*))
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(it->first, it->second);
         }
      }

      template <typename TVal>
      void execute(void (*func)(const TKey&, TChild*, TVal* val), TVal* val)
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(it->first, it->second, val);
         }
      }

   private:
      ChildrenType _children;
      bool _deleteChildren;//delete children when they are removed from the container

      void deleteAll()
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            if (it->second != NULL)
            {
               FOCUS_DELETE(it->second);
               it->second = NULL;
            }
         }
      }
};


}
#endif
