/* ***************************************************************************************
* FILE:          FCommon.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FCommon.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_COMMON_H__
#define __FOCUS_COMMON_H__

#include <Courier/Visualization/ViewId.h>
#include <View/CGI/CgiExtensions/MessageHandler.h>
#include <string>

#include <FeatStd/MemoryManagement/Heap.h>
#define FOCUS_NEW(type) FEATSTD_NEW(type)
#define FOCUS_DELETE(ptr) FEATSTD_DELETE(ptr)
#define FOCUS_DELETE_T(ptr, type) FEATSTD_DELETE_T(ptr, type)
#define FOCUS_NEW_ARRAY(type, count) FEATSTD_NEW_ARRAY(type, count)
#define FOCUS_DELETE_ARRAY(ptr) FEATSTD_DELETE_ARRAY(ptr)
#define FOCUS_DELETE_ARRAY_T(ptr, type) FEATSTD_DELETE_ARRAY_T(ptr, type)

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FDataSet.h"

/*****************************************************************************/
/* FORWARD DECLARATIONS                                                      */
/*****************************************************************************/

class AppViewHandler;

namespace Candera {
class Rectangle;
class Vector2;
}


namespace Courier {
class FrameworkWidget;
class Message;
class MessageReferrer;
class ViewId;
class ViewHandler;
}


namespace Focus {
/**************************************************************************/
/* TYPE MAPPING                                                           */
/**************************************************************************/
typedef int FAppId;
typedef ::Courier::ViewId FViewId;
typedef ::std::string FId;

typedef ::hmibase::view::MessageHandler FMsgReceiver;
typedef ::Courier::ViewHandler FViewHandler;
typedef ::Courier::FrameworkWidget FFrameworkWidget;
typedef ::Courier::Message FMessage;
typedef ::Candera::Rectangle FRectangle;
typedef ::Candera::Vector2 FCoordinate;
typedef short FocusOrderType;
typedef short FocusLayerType;

typedef unsigned int FControllerSetId;
typedef unsigned int FTimerTimeout;
typedef unsigned int FAnchorId;

namespace FAnchorMode {
enum Enum { None, Container, Content };
}


/**************************************************************************/
/* CONSTANTS                                                              */
/**************************************************************************/
struct Constants
{
   public:
      static const FAppId InvalidAppId;
      static const FViewId InvalidViewId;
      static const FId InvalidId;
      static const size_t InvalidIndex = ~size_t(0);
      static const FControllerSetId InvalidControllerSetId = ~FControllerSetId(0);
};


class FSequenceId
{
   public:
      FSequenceId() : Id(-1) {}

      static FSequenceId generate();

      bool operator==(const FSequenceId& other) const
      {
         return Id == other.Id;
      }

      bool operator!=(const FSequenceId& other) const
      {
         return !operator==(other);
      }

      int Id;
};


class FViewWidgetId
{
   public:
      explicit FViewWidgetId(const FViewId& viewId = Constants::InvalidViewId, const FId& widgetId = Constants::InvalidId) : ViewId(viewId), WidgetId(widgetId) {}
      FViewWidgetId(const FViewWidgetId& other) : ViewId(other.ViewId), WidgetId(other.WidgetId) {}
      FViewWidgetId& operator=(const FViewWidgetId& other);

      bool operator<(const FViewWidgetId& other) const
      {
         return (ViewId < other.ViewId) || (!(other.ViewId < ViewId) && (WidgetId < other.WidgetId));
      }

      bool operator>(const FViewWidgetId& other) const
      {
         return (ViewId > other.ViewId) || (!(other.ViewId > ViewId) && (WidgetId > other.WidgetId));
      }

      bool operator==(const FViewWidgetId& other) const
      {
         return (ViewId == other.ViewId) && (WidgetId == other.WidgetId);
      }

      bool operator!=(const FViewWidgetId& other) const
      {
         return !operator==(other);
      }

      FViewId ViewId;
      FId WidgetId;
};


class FAppViewWidgetId
{
   public:
      explicit FAppViewWidgetId(const FAppId& appId = Constants::InvalidAppId, const FViewId& viewId = Constants::InvalidViewId, const FId& widgetId = Constants::InvalidId) : AppId(appId), ViewId(viewId), WidgetId(widgetId) {}
      FAppViewWidgetId(const FAppViewWidgetId& other) : AppId(other.AppId), ViewId(other.ViewId), WidgetId(other.WidgetId) {}
      FAppViewWidgetId& operator=(const FAppViewWidgetId& other);

      bool operator<(const FAppViewWidgetId& other) const
      {
         return (AppId < other.AppId) || (!(other.AppId < AppId) && ((ViewId < other.ViewId) || (!(other.ViewId < ViewId) && (WidgetId < other.WidgetId))));
      }

      bool operator>(const FAppViewWidgetId& other) const
      {
         return (AppId > other.AppId) || (!(other.AppId > AppId) && ((ViewId > other.ViewId) || (!(other.ViewId > ViewId) && (WidgetId > other.WidgetId))));
      }

      bool operator==(const FAppViewWidgetId& other) const
      {
         return (AppId == other.AppId) && (ViewId == other.ViewId) && (WidgetId == other.WidgetId);
      }

      bool operator!=(const FAppViewWidgetId& other) const
      {
         return !operator==(other);
      }

      FAppId AppId;
      FViewId ViewId;
      FId WidgetId;
};


/**************************************************************************/
/* String of a FViewId.  */
#define FVIEWID_STR(viewId) viewId.CStr()
/* String of an FId.  */
#define FID_STR(id) id.c_str()


/**************************************************************************/
/* CONSTANTS                                                              */
/**************************************************************************/
class FConfiguration
{
      friend class FManagerConfigurator;

   public:
      static bool getDefaultWrapAround()
      {
         return _defaultWrapAround;
      }
      static bool getDefaultPreserveFocus()
      {
         return _defaultPreserveFocus;
      }

   private:
      static bool _defaultWrapAround;
      static bool _defaultPreserveFocus;
};


template<typename TId, typename TOwner>
class FComponentBase
{
   public:
      typedef TId IdType;
      typedef TOwner OwnerType;

      FComponentBase(const TId& id) : _id(id) {}
      virtual ~FComponentBase() {}

      const TId& getId() const
      {
         return _id;
      }

      template <typename TData> void setData(const TData& data)
      {
         return Data.set(data);
      }
      template <typename TData> bool hasData()
      {
         return Data.has<TData>();
      }
      template <typename TData> TData* getData()
      {
         return Data.get<TData>();
      }
      template <typename TData> TData* getOrCreateData()
      {
         return Data.getOrCreate<TData>();
      }

      FDataSet Data;

   private:
      FComponentBase(const FComponentBase&);
      FComponentBase& operator=(const FComponentBase&);

      TId _id;
};


}
#endif //__FOCUS_COMMON_H__
