/* ***************************************************************************************
* FILE:          FDefaultSessionManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultSessionManager.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultSessionManager.h"
#include "Focus/FData.h"
#include "Focus/FManager.h"
#include "Focus/FSession.h"
#include "Focus/FController.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultSessionManager.cpp.trc.h"
#endif

namespace Focus {

FDefaultSessionManager::FDefaultSessionManager(FManager& manager)
   : _manager(manager), _session(NULL)
{
}


FDefaultSessionManager::~FDefaultSessionManager()
{
   if (_session != NULL)
   {
      FOCUS_DELETE(_session);
      _session = NULL;
   }
}


bool FDefaultSessionManager::onMessage(const FMessage& msg)
{
   FTaskManager* taskManager = _manager.getTaskManager();

   if (msg.GetId() == FocusSessionFinishedUpdMsg::ID)
   {
      const FocusSessionFinishedUpdMsg& finishedMsg = static_cast<const FocusSessionFinishedUpdMsg&>(msg);
      if ((_session != NULL) && (taskManager != NULL) && (!isSessionSuspended()))
      {
         if ((finishedMsg.GetTaskExecutionCount() == taskManager->getTaskExecutionCount())
               && (!taskManager->hasTasks())
               && (_session->getStatus() == FSession::Finished))
         {
            endSession();
         }
         else
         {
            updateSession();
         }
      }
      return true;
   }

   if (msg.GetId() == UpdateFocusSessionReqMsg::ID)
   {
      if (_session != NULL)
      {
         updateSession();
      }
      return true;
   }

   if (msg.GetId() == RefreshFocusSessionReqMsg::ID)
   {
      const RefreshFocusSessionReqMsg& reqMsg = static_cast<const RefreshFocusSessionReqMsg&>(msg);
      if ((_session != NULL) && (taskManager != NULL) && (!isSessionSuspended()))
      {
         if (reqMsg.GetRefreshCrtAppInfo())
         {
            _session->setStatus(FSession::Refreshing);
            taskManager->clearTasks();
            taskManager->configureTasks(*_session);
         }
         updateSession();
      }
      return true;
   }
   return false;
}


bool FDefaultSessionManager::changeSessionState()
{
   FTaskManager* taskManager = _manager.getTaskManager();
   //if session is NULL or if there are still updaters to execute we can not change the state
   if ((_session == NULL) || (taskManager == NULL) || (taskManager->hasTasks()))
   {
      return false;
   }

   switch (_session->getStatus())
   {
      case FSession::New:
         _session->setStatus(FSession::Collecting);
         break;

      case FSession::Collecting:
      case FSession::Refreshing:
         _session->setStatus(FSession::Processing);
         break;

      case FSession::Processing:
         _session->setStatus(FSession::Publishing);
         break;

      case FSession::Publishing:
         _session->setStatus(FSession::Finished);
         break;

      case FSession::Finished:
         //Finished is the final status
         return false;

      default:
         return false;
   }

   taskManager->configureTasks(*_session);
   return true;
}


void FDefaultSessionManager::updateSession()
{
   FTaskManager* taskManager = _manager.getTaskManager();
   if (taskManager == NULL)
   {
      return;
   }

   FTask::Result result = FTask::Completed;
   bool mustRun = true;
   while (mustRun && (_session != NULL))
   {
      result = taskManager->executeTasks();

      //execution error => abort session
      if (result == FTask::Error)
      {
         mustRun = false;
         abortSession();
      }
      //execution suspended => return
      else if (result == FTask::Suspended)
      {
         mustRun = false;
      }
      //execution completed
      else
      {
         //change state
         bool stateChanged = changeSessionState();

         //no state change occured => return
         if (!stateChanged)
         {
            mustRun = false;
         }

         _manager.printSessionDebugInfo();
      }
   }

   if ((result == FTask::Completed) && (_session != NULL) && (_session->getStatus() == FSession::Finished)
         && (!isSessionSuspended()) && (_manager.getOutputMsgHandler() != NULL))
   {
      //endSession();
      FocusSessionFinishedUpdMsg* msg = COURIER_MESSAGE_NEW(FocusSessionFinishedUpdMsg)(taskManager->getTaskExecutionCount());
      if (msg != NULL)
      {
         _manager.getOutputMsgHandler()->postMessage(msg);
      }
   }
}


bool FDefaultSessionManager::beginSession(FSession::Mode mode)
{
   ETG_TRACE_USR1_THR(("FManager::beginSession app=%d, mode=%u", _manager.getCurrentAppId(), mode));
   if (_session != NULL)
   {
      ETG_TRACE_USR1_THR(("FManager::beginSession session already exists"));
      return false;
   }

   _manager.initializeCurrentAppState();

   _session = FOCUS_NEW(FSession)(mode);
   if (_session != NULL)
   {
      if (_manager.getTaskManager() != NULL)
      {
         _manager.getTaskManager()->configureTasks(*_session);
      }
   }
   return (_session != NULL);
}


void FDefaultSessionManager::endSession()
{
   ETG_TRACE_USR1_THR(("FManager::endSession app=%d", _manager.getCurrentAppId()));

   if (_session == NULL)
   {
      ETG_TRACE_USR1_THR(("FManager::endSession no session!"));
      return;
   }

   if ((_manager.getTaskManager() != NULL) && _manager.getTaskManager()->hasTasks())
   {
      ETG_TRACE_USR1_THR(("FManager::endSession there are tasks queued!"));
      abortSession();
      return;
   }

   FOCUS_DELETE(_session);
   _session = NULL;
}


void FDefaultSessionManager::abortSession()
{
   ETG_TRACE_USR1_THR(("FManager::abortSession app=%d", _manager.getCurrentAppId()));

   if (_session == NULL)
   {
      ETG_TRACE_USR1_THR(("FManager::abortSession no session"));
      return;
   }

   if (_manager.getTaskManager() != NULL)
   {
      _manager.getTaskManager()->clearTasks();
   }

   if (_manager.getInputMsgQueue() != NULL)
   {
      _manager.getInputMsgQueue()->clear();
   }

   FOCUS_DELETE(_session);
   _session = NULL;
}


struct FSessionSuspendedMarker : public FMarkerData
{
   FSessionSuspendedMarker(bool value = false) : FMarkerData(value) {}
};


class FDefaultSessionWatchdog : public FTimerListener
{
   public:
      FDefaultSessionWatchdog(Focus::FManager& manager) : _manager(manager) {}

      virtual void onRestarted() {}
      virtual void onStopped() {}
      virtual void onExpired()
      {
         ETG_TRACE_USR1_THR(("FDefaultSessionWatchdog::onExpired app=%d", _manager.getCurrentAppId()));

         FSessionManager* sessionManager = _manager.getSessionManager();
         if ((sessionManager != NULL) && (sessionManager->getSession() != NULL) && (sessionManager->isSessionSuspended()))
         {
            sessionManager->abortSession();
         }
      }

   private:
      FDefaultSessionWatchdog(const FDefaultSessionWatchdog&);
      FDefaultSessionWatchdog& operator=(const FDefaultSessionWatchdog&);

      Focus::FManager& _manager;
};


bool FDefaultSessionManager::isSessionSuspended()
{
   if (_session != NULL)
   {
      FSessionSuspendedMarker* marker = _session->Data.get<FSessionSuspendedMarker>();
      if (marker != NULL)
      {
         return marker->Value;
      }
   }
   return false;
}


void FDefaultSessionManager::requestSessionSuspend()
{
   ETG_TRACE_USR1_THR(("FDefaultSessionManager::requestSessionSuspend app=%d", _manager.getCurrentAppId()));

   if (_session != NULL)
   {
      if (!isSessionSuspended())
      {
         _session->Data.set(FSessionSuspendedMarker(true));
         if (_manager.getWatchdogTimer() != NULL)
         {
            static FDefaultSessionWatchdog* watchdogListener = NULL;
            if (watchdogListener == NULL)
            {
               watchdogListener = FOCUS_NEW(FDefaultSessionWatchdog)(_manager);
               if (watchdogListener != NULL)
               {
                  _manager.getWatchdogTimer()->addListener(*watchdogListener);
               }
            }
            _manager.getWatchdogTimer()->restart();
         }
      }
      else
      {
         ETG_TRACE_USR1_THR(("FDefaultSessionManager::requestSessionSuspend - already suspended"));
      }
   }
   else
   {
      ETG_TRACE_USR1_THR(("FDefaultSessionManager::requestSessionSuspend - no session to suspend"));
   }
}


void FDefaultSessionManager::clearSessionSuspend()
{
   ETG_TRACE_USR1_THR(("FDefaultSessionManager::clearSessionSuspend app=%d", _manager.getCurrentAppId()));

   if (_manager.getWatchdogTimer() != NULL)
   {
      _manager.getWatchdogTimer()->stop();
   }

   if (_session != NULL)
   {
      if (isSessionSuspended())
      {
         _session->Data.set(FSessionSuspendedMarker(false));
      }
   }
   else
   {
      ETG_TRACE_USR1_THR(("FDefaultSessionManager::clearSessionSuspend - no session to resume"));
   }
}


}
