/* ***************************************************************************************
* FILE:          FDefaultJoystickController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultJoystickController.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultJoystickController.h"
#include "Focus/FCommon.h"
#include "Focus/FContainer.h"
#include "Focus/FDataSet.h"
#include "Focus/FData.h"
#include "Focus/FActiveViewGroup.h"
#include "Focus/FGroupTraverser.h"
#include "Focus/FManagerConfig.h"
#include "Focus/FManager.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultJoystickController.cpp.trc.h"
#endif

//lint -esym(1768, Focus::DirectionalWidgetFinderVisitor::visitWidget) intended
namespace Focus {

bool FDefaultJoystickController::onMessage(FSession& session, FGroup& group, const FMessage& msg)
{
   if (msg.GetId() == JoystickStatusChangedUpdMsg::ID)
   {
      const JoystickStatusChangedUpdMsg* joystickMsg = Courier::message_cast<const JoystickStatusChangedUpdMsg*>(&msg);
      if (joystickMsg != NULL)
      {
         ETG_TRACE_USR1_THR(("FDefaultJoystickController::onMessage() app=%d, direction=%d, src=%u, visible=%u",
                             _manager.getCurrentAppId(), joystickMsg->GetDirection(), joystickMsg->GetSource(), _manager.isFocusVisible()));

         if (_manager.getActivityTimer() != NULL)
         {
            _manager.getActivityTimer()->restart();
         }

         //if focus is not visible, generate a new sequence so that preserve focus can correctly reuse the current focus
         if (!_manager.isFocusVisible())
         {
            _manager.generateNewSequenceId();
         }

         bool result = moveFocus(session, group, static_cast<hmibase::FocusDirectionEnum>(joystickMsg->GetDirection()));

         if (result && !_manager.isFocusVisible())
         {
            _manager.setFocusVisible(true);
         }

         //message is consumed even if focus was not changed, this could change if necessary
         return true;
      }
   }
   return false;
}


class FJoystickSearchAreaAdjuster
{
   public:
      virtual ~FJoystickSearchAreaAdjuster() {}

      FRectangle calculateSearchArea(const FRectangle& origin, hmibase::FocusDirectionEnum direction) const
      {
         FRectangle searchArea(0.0f, 0.0f, -1.0f, -1.0f);

         switch (direction)
         {
            case hmibase::Up:
               adjustUp(origin, searchArea);
               break;

            case hmibase::UpperRight:
               adjustRight(origin, searchArea);
               adjustUp(origin, searchArea);
               break;

            case hmibase::Right:
               adjustRight(origin, searchArea);
               break;

            case hmibase::LowerRight:
               adjustRight(origin, searchArea);
               adjustDown(origin, searchArea);
               break;

            case hmibase::Down:
               adjustDown(origin, searchArea);
               break;

            case hmibase::LowerLeft:
               adjustLeft(origin, searchArea);
               adjustDown(origin, searchArea);
               break;

            case hmibase::Left:
               adjustLeft(origin, searchArea);
               break;

            case hmibase::UpperLeft:
               adjustLeft(origin, searchArea);
               adjustUp(origin, searchArea);
               break;

            default:
               break;
         }

         return searchArea;
      }

   private:
      virtual void adjustUp(const FRectangle& origin, FRectangle& searchArea) const = 0;
      virtual void adjustRight(const FRectangle& origin, FRectangle& searchArea) const = 0;
      virtual void adjustDown(const FRectangle& origin, FRectangle& searchArea) const = 0;
      virtual void adjustLeft(const FRectangle& origin, FRectangle& searchArea) const = 0;
};


class RestrictedSearchAreaAdjuster : public FJoystickSearchAreaAdjuster
{
   public:
      virtual ~RestrictedSearchAreaAdjuster() {}

   private:
      virtual void adjustUp(const FRectangle& origin, FRectangle& searchArea) const
      {
         searchArea.SetHeight(origin.GetTop());
      }

      virtual void adjustRight(const FRectangle& origin, FRectangle& searchArea) const
      {
         searchArea.SetLeft(origin.GetLeft() + origin.GetWidth());
      }

      virtual void adjustDown(const FRectangle& origin, FRectangle& searchArea) const
      {
         searchArea.SetTop(origin.GetTop() + origin.GetHeight());
      }

      virtual void adjustLeft(const FRectangle& origin, FRectangle& searchArea) const
      {
         searchArea.SetWidth(origin.GetLeft());
      }
};


/*
class ExtendedSearchAreaAdjuster : public FJoystickSearchAreaAdjuster
{
public:
   virtual ~ExtendedSearchAreaAdjuster() {}

private:
   virtual void adjustUp(const FRectangle& origin, FRectangle& searchArea) const
   {
      searchArea.SetHeight(origin.GetTop() + origin.GetHeight());
   }

   virtual void adjustRight(const FRectangle& origin, FRectangle& searchArea) const
   {
      searchArea.SetLeft(origin.GetLeft());
   }

   virtual void adjustDown(const FRectangle& origin, FRectangle& searchArea) const
   {
      searchArea.SetTop(origin.GetTop());
   }

   virtual void adjustLeft(const FRectangle& origin, FRectangle& searchArea) const
   {
      searchArea.SetWidth(origin.GetLeft() + origin.GetWidth());
   }
};


*/

class DirectionalWidgetFinderVisitor : public FWidgetVisitor
{
   public:
      DirectionalWidgetFinderVisitor(const FRectangle& searchArea, FWidget& originWidget, const FRectangle& originRectangle)
         : _searchArea(searchArea), _originWidget(originWidget), _widget(NULL), _distance(0.0f)
      {
         _originX = getMiddleX(originRectangle);
         _originY = getMiddleY(originRectangle);
      }

      FWidget* getWidget() const
      {
         return _widget;
      }
      float getDistance() const
      {
         return _distance;
      }

   private:
      DirectionalWidgetFinderVisitor(const DirectionalWidgetFinderVisitor&);
      DirectionalWidgetFinderVisitor& operator=(const DirectionalWidgetFinderVisitor&);

      virtual void visitWidget(FWidget& widget)
      {
         FRectangle* rectangle = widget.getData<FRectangle>();
         if (isValid(widget) && (&_originWidget != &widget) && (rectangle != NULL))
         {
            float widgetX = getMiddleX(*rectangle);
            float widgetY = getMiddleY(*rectangle);

            if (isInRange(_searchArea.GetLeft(), _searchArea.GetWidth(), widgetX)
                  && isInRange(_searchArea.GetTop(), _searchArea.GetHeight(), widgetY))
            {
               //remember the closest widget
               float distance = getDistance(_originX, _originY, widgetX, widgetY);
               if ((_widget == NULL) || (distance < _distance))
               {
                  _widget = &widget;
                  _distance = distance;

                  ETG_TRACE_USR1_THR(("DirectionalWidgetFinderVisitor distance=%f, widget=%s",
                                      distance, FID_STR(widget.getId())));
               }
            }
         }
      }

      float getDistance(float x1, float y1, float x2, float y2) const
      {
         return Candera::Math::SquareRoot((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
      }

      float getMiddleX(const FRectangle& r) const
      {
         return r.GetLeft() + r.GetWidth() / 2;
      }
      float getMiddleY(const FRectangle& r) const
      {
         return r.GetTop() + r.GetHeight() / 2;
      }

      bool isInRange(float min, float range, float value) const
      {
         return (min <= value) && ((range == -1.0f) || (value <= min + range));
      }

      FRectangle _searchArea;

      FWidget& _originWidget;
      float _originX;
      float _originY;

      FWidget* _widget;
      float _distance;
};


FWidget* FDefaultJoystickController::tryGetNewFocus(FSession& session, FGroup& group, hmibase::FocusDirectionEnum direction,
      FWidget& originWidget, const FJoystickSearchAreaAdjuster& searchAreaAdjuster)
{
   FRectangle* originRectangle = originWidget.getData<FRectangle>();
   if (originRectangle == NULL)
   {
      ETG_TRACE_USR1_THR(("FDefaultJoystickController::tryGetNewFocus No bounds for currently focused widget %s!",
                          FID_STR(originWidget.getId())));
      return NULL;
   }

   FRectangle searchArea = searchAreaAdjuster.calculateSearchArea(*originRectangle, direction);

   DirectionalWidgetFinderVisitor widgetVisitor(searchArea, originWidget, *originRectangle);

   //if focus is not visible check origin and use it if is valid
   if (!_manager.isFocusVisible() && widgetVisitor.isValid(originWidget))
   {
      return &originWidget;
   }

   FDefaultGroupTraverser traverser(session, widgetVisitor, group, NULL, true);

   //traverse all descendents to find the nearest neighbour
   while (traverser.advance())
   {
      //nothing to do, just traverse
   }

   return widgetVisitor.getWidget();
}


bool FDefaultJoystickController::moveFocus(FSession& session, FGroup& group, hmibase::FocusDirectionEnum direction)
{
   ETG_TRACE_USR1_THR(("FDefaultJoystickController::moveFocus app=%d, direction=%d, group=%s",
                       _manager.getCurrentAppId(), direction, FID_STR(group.getId())));

   FWidget* newFocus = NULL;
   if (_manager.getAvgManager() != NULL)
   {
      FWidget* originWidget = dynamic_cast<FWidget*>(_manager.getAvgManager()->getFocus(session));
      if (originWidget == NULL)
      {
         ETG_TRACE_USR1_THR(("FDefaultJoystickController::moveFocus No widget currently focused!"));
         return false;
      }

      //first search is with a restricted search area
      newFocus = tryGetNewFocus(session, group, direction, *originWidget, RestrictedSearchAreaAdjuster());

      //extended search is disabled for now
      //if (newFocus == NULL)
      //{
      //   //second search is with an extended search area
      //   newFocus = tryGetNewFocus(group, direction, *originWidget, ExtendedSearchAreaAdjuster());
      //}

      if (newFocus != NULL)
      {
         //todo: reset old group if necessary
         _manager.getAvgManager()->setFocus(session, newFocus);
      }
   }
   return newFocus != NULL;
}


}
