/* ***************************************************************************************
* FILE:          FDefaultInputMsgProcessor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultInputMsgProcessor.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultInputMsgProcessor.h"
#include "Focus/FData.h"
#include "Focus/FManager.h"
#include "Focus/FSession.h"
#include "Focus/FController.h"
#include "Focus/FManagerConfig.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultInputMsgProcessor.cpp.trc.h"
#endif

namespace Focus {

template <typename TFocusable>
static bool dispatchToFocusable(FSession& session, TFocusable& focusable, const FControllerList& controllerList, const FMessage& msg)
{
   ETG_TRACE_USR4_THR(("FDefaultInputMsgProcessor::dispatchToFocusable msg=%30s controllerCount=%u focusable=%s",
                       msg.GetName(), controllerList.count(), FID_STR(focusable.getId())));

   bool consumed = false;
   for (size_t j = 0; (!consumed) && (j < controllerList.count()); ++j)
   {
      FController* controller = controllerList[j];
      if (controller != NULL)
      {
         consumed = controller->onMessage(session, focusable, msg);
      }
   }

   if (consumed)
   {
      ETG_TRACE_USR1_THR(("FDefaultInputMsgProcessor::dispatchToFocusable consumed by %s", FID_STR(focusable.getId())));
   }
   return consumed;
}


static const FControllerList* getControllerList(const FManager& manager, FWidgetConfig& config)
{
   FWidgetData* widgetData = config.getData<FWidgetData>();
   if (widgetData != NULL)
   {
      return manager.getControllerSet(widgetData->ControllerSetId);
   }
   return NULL;
}


bool FDefaultInputMsgProcessor::dispatchMessage(FSession& session, const FMessage& msg)
{
   ETG_TRACE_USR1_THR(("FDefaultInputMsgProcessor::dispatchMessage app=%d, id=%u, name=%s",
                       _manager.getCurrentAppId(), msg.GetSequenceNumber(), msg.GetName()));

   bool consumed = false;
   FGroup* group = NULL;

   //get focused item (can be a widget or a group)
   Focusable* focusable = (_manager.getAvgManager() != NULL) ? (_manager.getAvgManager()->getFocus(session)) : NULL;

   //if it is a widget dispatch the message to its controllers
   FWidget* widget = dynamic_cast<FWidget*>(focusable);
   if (widget != NULL)
   {
      const FControllerList* controllerList = getControllerList(_manager, widget->Config);
      if (controllerList != NULL)
      {
         consumed = dispatchToFocusable(session, *widget, *controllerList, msg) || consumed;          //lint !e845 wanted this way
      }
      group = widget->getParent();
   }
   else
   {
      group = dynamic_cast<FGroup*>(focusable);
   }

   //dispatch the message to all ancestor groups until it is consumed
   while ((!consumed) && (group != NULL))
   {
      for (size_t i = 0; i < group->Config.count(); ++i)
      {
         FWidgetConfig* config = group->Config.get(i);
         if (config != NULL)
         {
            const FControllerList* controllerList = getControllerList(_manager, *config);
            if (controllerList != NULL)
            {
               consumed = dispatchToFocusable(session, *group, *controllerList, msg) || consumed;
            }
         }
      }

      group = group->getParent();
   }

   //finally dispatch it to the root group using default controllers
   if ((!consumed) && (_manager.getRootGroupControllerSetId() != Constants::InvalidControllerSetId))
   {
      FGroup* rootGroup = session.ActiveViewGroup.getRootGroup();
      const FControllerList* controllerList = _manager.getControllerSet(_manager.getRootGroupControllerSetId());
      if ((rootGroup != NULL) && (controllerList != NULL))
      {
         consumed = dispatchToFocusable(session, *rootGroup, *controllerList, msg) || consumed;       //lint !e845 wanted this way
      }
   }

   return consumed;
}


FTask::Result FDefaultInputMsgProcessor::update(FSession& session)
{
   if (_manager.getInputMsgQueue() != NULL)
   {
      ETG_TRACE_USR1_THR(("FDefaultInputMsgProcessor::update app=%d, msgCount=%u",
                          _manager.getCurrentAppId(), _manager.getInputMsgQueue()->count()));

      while (!_manager.getInputMsgQueue()->empty())
      {
         if ((_manager.getSessionManager() != NULL) && (_manager.getSessionManager()->isSessionSuspended()))
         {
            ETG_TRACE_USR1_THR(("FDefaultInputMsgProcessor::update - suspended"));
            return FTask::Suspended;
         }

         const FMessage* msg = _manager.getInputMsgQueue()->front();
         if (msg != NULL)
         {
            dispatchMessage(session, *msg);
         }
         _manager.getInputMsgQueue()->popFront();
      }
   }

   return FTask::Completed;
}


}
