/* ***************************************************************************************
* FILE:          FDefaultAvgBuilder.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultAvgBuilder.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultAvgBuilder.h"
#include "Focus/FData.h"
#include "Focus/FManager.h"
#include "Focus/FSession.h"
#include "Focus/FController.h"
#include "Focus/FManagerConfig.h"
#include "Focus/FStateInfo.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultAvgBuilder.cpp.trc.h"
#endif

namespace Focus {

bool FDefaultAvgSessionUpdater::updateApp(FSession& session, FAppConfigSharedPtr appConfig)
{
   if (appConfig.PointsToNull())
   {
      return false;
   }

   FAppStateSharedPtr appState = session.StateInfo.get(appConfig->getId());
   if (appState.PointsToNull())
   {
      appState = FAppStateSharedPtr(FOCUS_NEW(FAppState)(appConfig->getId()));
      if (appState.PointsToNull())
      {
         return false;
      }

      session.StateInfo.add(appConfig->getId(), appState);
   }

   return updateAppEx(session, appConfig, appState);
}


bool FDefaultAvgSessionUpdater::updateAllViewsEx(FSession& session, FAppConfigSharedPtr appConfig, FAppStateSharedPtr appState)
{
   bool result = true;
   if (!appConfig.PointsToNull() && !appState.PointsToNull())
   {
      FAppStateSharedPtr appState = session.StateInfo.get(appConfig->getId());
      for (size_t i = 0; i < appConfig->getChildCount(); ++i)
      {
         FViewConfig* viewConfig = appConfig->getChild(i);
         if (viewConfig != NULL)
         {
            FViewState* viewState = appState->getOrCreateChild(viewConfig->getId());
            if (viewState != NULL)
            {
               result = updateViewEx(session, *viewConfig, *viewState) && result;
            }
            else
            {
               result = false;
            }
         }
      }
   }
   return result;
}


bool FDefaultAvgSessionUpdater::updateAllWidgetsEx(FSession& session, const FViewConfig& viewConfig, FViewState& viewState)
{
   bool result = true;
   for (size_t i = 0; i < viewConfig.getChildCount(); ++i)
   {
      FWidgetConfig* widgetConfig = viewConfig.getChild(i);
      if (widgetConfig != NULL)
      {
         result = updateWidgetEx(session, *widgetConfig, viewState) && result;
      }
   }
   return result;
}


static bool isAnchorContent(FGroupData* groupData)
{
   return (groupData != NULL) && (groupData->AnchorMode == FAnchorMode::Content);
}


static bool isAnchorContent(FWidgetConfig* widgetConfig)
{
   return (widgetConfig != NULL) && isAnchorContent(widgetConfig->Data.get<FGroupData>());
}


void FDefaultAvgBuilder::handleAsGroup(FSession& session, FViewState& viewState, FWidgetConfig& widgetConfig, const FWidgetData* widgetData, const FGroupData* groupData)
{
   (void)widgetData;
   if (groupData == NULL)
   {
      return;
   }

   ETG_TRACE_USR4_THR(("FActiveViewGroupUpdater::handleAsGroup view=%50s widget=%s",
                       FVIEWID_STR(widgetConfig.View.getId()),
                       FID_STR(widgetConfig.getId())));

   //merged groups are handled but ignored
   //if (!groupData->getMergeWithParent())
   {
      const FId& id = widgetConfig.getId();
      FGroup* group = NULL;

      //first step process non anchor content widgets
      if (!_anchorContentMergeInProgress)
      {
         if (groupData->AnchorMode != FAnchorMode::Content)
         {
            group = session.ActiveViewGroup.Groups.get(id);
            //group does not exist yet
            if (group == NULL)
            {
               const FAppViewWidgetId appViewWidgetId(widgetConfig.View.App.getId(), widgetConfig.View.getId(), id);
               //check first if a widget with the same name exists => that would not be allowed
               if (session.ActiveViewGroup.Widgets.get(appViewWidgetId) != NULL)
               {
                  ETG_TRACE_USR1_THR(("FActiveViewGroupUpdater::processWidgetConfig Expect group %s found widget", FID_STR(id)));
               }
               else
               {
                  //create group and add it
                  group = FOCUS_NEW(FGroup)(id);
                  if (group != NULL)
                  {
                     session.ActiveViewGroup.Groups.add(id, group);
                     session.ActiveViewGroup.FocusableList.push_back(group);
                  }
               }
            }
         }
      }
      //second step process anchor content widgets
      else
      {
         if (groupData->AnchorMode == FAnchorMode::Content)
         {
            group = session.ActiveViewGroup.Anchors.get(groupData->AnchorId);
            if (group == NULL)
            {
               ETG_TRACE_USR1_THR(("FActiveViewGroupUpdater::handleAsGroup Anchor container %u not registered for %s",
                                   groupData->AnchorId, FID_STR(id)));
            }
         }
      }

      if (group != NULL)
      {
         FGroupState* groupState = viewState.getOrCreateChild(widgetConfig.getId());
         if (groupState != NULL)
         {
            if (groupState->CurrentFocusInfo.Id != Constants::InvalidId)
            {
               group->Data.set(groupState->CurrentFocusInfo);
            }
            group->Data.set(groupState->UpdateSequenceId);
         }

         //add group data only if configured or not found already
         if (groupData->Configured || !group->Data.has<FGroupData>())
         {
            group->Data.set(*groupData);
         }
         group->Config.add(&widgetConfig);

         //register anchor container
         if (!_anchorContentMergeInProgress && (groupData->AnchorMode == FAnchorMode::Container))
         {
            FGroup* existingAnchor = session.ActiveViewGroup.Anchors.get(groupData->AnchorId);
            if (existingAnchor == NULL)
            {
               session.ActiveViewGroup.Anchors.add(groupData->AnchorId, group);
            }
            else if (existingAnchor != group)
            {
               ETG_TRACE_USR1_THR(("FActiveViewGroupUpdater::handleAsGroup Failed to register as anchor with id %u because it was already registered by %s",
                                   groupData->AnchorId,
                                   FID_STR(existingAnchor->getId())));
            }
            else
            {
               //nothing to do
            }
         }
      }
   }
}


void FDefaultAvgBuilder::handleAsWidget(FSession& session, FWidgetConfig& widgetConfig, const FWidgetData* widgetData, const FGroupData* groupData)
{
   (void)widgetData;

   if (groupData != NULL)
   {
      return;
   }

   const FId& id = widgetConfig.getId();
   const FAppViewWidgetId appViewWidgetId(widgetConfig.View.App.getId(), widgetConfig.View.getId(), id);

   FWidget* widget = session.ActiveViewGroup.Widgets.get(appViewWidgetId);
   //widget does not exist yet
   if (widget == NULL)
   {
      //check first if a group with the same name exists => that would not be allowed
      if (session.ActiveViewGroup.Groups.get(id) != NULL)
      {
         ETG_TRACE_USR1_THR(("FActiveViewGroupUpdater::processWidgetConfig Expect widget %s found group", FID_STR(id)));
      }
      else
      {
         //create widget and add it
         widget = FOCUS_NEW(FWidget)(id, widgetConfig);
         if (widget != NULL)
         {
            session.ActiveViewGroup.Widgets.add(appViewWidgetId, widget);
            session.ActiveViewGroup.FocusableList.push_back(widget);
         }
      }
   }
}


FGroup* FDefaultAvgBuilder::findParentGroup(FSession& session, const FWidgetConfig& widgetConfig)
{
   if (widgetConfig.getId() == Constants::InvalidId)
   {
      return NULL;
   }

   FWidgetConfig* parentWidgetConfig = widgetConfig.getParent();
   if (parentWidgetConfig == NULL)
   {
      return session.ActiveViewGroup.getRootGroup();
   }
   const FId& parentGroupId = parentWidgetConfig->getId();

   FGroupData* parentGroupData = parentWidgetConfig->Data.get<FGroupData>();
   if (parentGroupData != NULL)
   {
      FGroup* parentGroup = NULL;
      if (parentGroupData->AnchorMode == FAnchorMode::Content)
      {
         parentGroup = session.ActiveViewGroup.Anchors.get(parentGroupData->AnchorId);
         if (parentGroup == NULL)
         {
            ETG_TRACE_USR1_THR(("FDefaultAvgBuilder::findParentGroup anchor %u not found for %s", parentGroupData->AnchorId, FID_STR(parentGroupId)));
         }
      }
      else
      {
         parentGroup = session.ActiveViewGroup.Groups.get(parentGroupId);
         if (parentGroup == NULL)
         {
            parentGroup = FOCUS_NEW(FGroup)(parentGroupId);
            if (parentGroup != NULL)
            {
               session.ActiveViewGroup.Groups.add(parentGroupId, parentGroup);
               session.ActiveViewGroup.FocusableList.push_back(parentGroup);
            }
         }
      }
      return parentGroup;
   }

   return session.ActiveViewGroup.getRootGroup();
}


bool FDefaultAvgBuilder::updateWidgetEx(FSession& session, FWidgetConfig& widgetConfig, FViewState& viewState)
{
   FGroupData* groupData = widgetConfig.Data.get<FGroupData>();

   //anchor content and non anchor content are processed in different steps
   if (_anchorContentMergeInProgress != (isAnchorContent(groupData) || isAnchorContent(widgetConfig.getParent())))
   {
      return true;
   }

   FWidgetData* widgetData = widgetConfig.Data.get<FWidgetData>();
   if ((widgetData == NULL) || (widgetData->SequenceNr < 0))
   {
      return true;
   }

   handleAsGroup(session, viewState, widgetConfig, widgetData, groupData);
   handleAsWidget(session, widgetConfig, widgetData, groupData);

   return true;
}


bool FDefaultAvgBuilder::updateViewEx(FSession& session, FViewConfig& viewConfig, FViewState& viewState)
{
   session.ActiveViewGroup.Views.push_back(viewConfig.getId());

   updateAllWidgetsEx(session, viewConfig, viewState);

   return true;
}


bool FDefaultAvgBuilder::updateAppEx(FSession& session, FAppConfigSharedPtr appConfig, FAppStateSharedPtr appState)
{
   updateAllViewsEx(session, appConfig, appState);

   return true;
}


template <typename TData>
static const TData* getFocusableData(Focusable* x)
{
   FWidget* widget = dynamic_cast<FWidget*>(x);
   if (widget != NULL)
   {
      return widget->getData<TData>();
   }

   FGroup* group = dynamic_cast<FGroup*>(x);
   if (group != NULL)
   {
      return group->getData<TData>();
   }

   return NULL;
}


static bool compareByFocusOrder(Focusable* x, Focusable* y)
{
   if ((x != NULL) && (y != NULL))
   {
      const FWidgetData* xData = getFocusableData<FWidgetData>(x);
      const FWidgetData* yData = getFocusableData<FWidgetData>(y);
      if ((xData != NULL) && (yData != NULL))
      {
         if (xData->EffectiveOrder == yData->EffectiveOrder)
         {
            return xData->SequenceNr < yData->SequenceNr;
         }
         return xData->EffectiveOrder < yData->EffectiveOrder;
      }
   }
   return true;
}


static void sortGroup(const FId& id, FGroup* group)
{
   (void)id;
   if (group != NULL)
   {
      group->Children.sort(compareByFocusOrder);
   }
}


static FocusLayerType getLayer(FGroup& group)
{
   FGroupData* groupData = group.getData<FGroupData>();
   if ((groupData != NULL) && (groupData->Configured))
   {
      return groupData->Layer;
   }

   return FocusLayerType(0);
}


static void removeInaccessibleGroups(FGroup& rootGroup)
{
   FocusLayerType maxLayer = 0;

   //iterate the group to find the max layer
   for (size_t i = rootGroup.Children.count(); i > 0; --i)
   {
      size_t index = i - 1;
      //check if the child is a group
      FGroup* group = dynamic_cast<FGroup*>(rootGroup.Children.get(index));
      if (group != NULL)
      {
         //get child group layer
         FocusLayerType layer = getLayer(*group);
         //if it is greater than the max level remember it and also the index where this new max layer was found (so that later we can remove all those after it)
         if (layer > maxLayer)
         {
            maxLayer = layer;
         }
      }
   }

   //if a group with a layer greater than 0 was found
   //have another pass to remove widgets and groups with inferior layer
   if (maxLayer > 0)
   {
      for (size_t i = rootGroup.Children.count(); i > 0; --i)
      {
         size_t index = i - 1;
         FocusLayerType layer = 0;

         //check if the child is a group
         FGroup* group = dynamic_cast<FGroup*>(rootGroup.Children.get(index));
         if (group != NULL)
         {
            //get child group layer
            layer = getLayer(*group);
         }

         //if current child is not a group or if its layer is lower than the max layer => remove it
         if (layer < maxLayer)
         {
            rootGroup.Children.removeAt(index);
         }
      }
   }
}


bool FDefaultAvgBuilder::connectToParentGroup(FSession& session)
{
   for (FActiveViewGroup::FocusableListType::const_iterator it = session.ActiveViewGroup.FocusableList.begin(); it != session.ActiveViewGroup.FocusableList.end(); ++it)
   {
      Focusable* f = *it;
      if (f != NULL)
      {
         FWidgetConfig* widgetConfig = f->getConfig();
         if (widgetConfig != NULL)
         {
            FGroup* parentGroup = findParentGroup(session, *widgetConfig);
            if ((parentGroup != NULL) && (parentGroup->Children.indexOf(f) == Constants::InvalidIndex))
            {
               parentGroup->Children.add(f);
               f->setParent(parentGroup);
            }
         }
      }
   }

   return true;
}


static FocusOrderType getEffectiveFocusOrder(Focusable* f)
{
   FocusOrderType order = 0;
   while (f != NULL)
   {
      const FWidgetData* widgetData = getFocusableData<FWidgetData>(f);
      if (widgetData != NULL)
      {
         order = static_cast<FocusOrderType>(order + widgetData->Order);
      }
      f = f->getParent();
   }
   return order;
}


bool FDefaultAvgBuilder::updateEffectiveOrderRank(FSession& session)
{
   for (FActiveViewGroup::FocusableListType::const_iterator it = session.ActiveViewGroup.FocusableList.begin(); it != session.ActiveViewGroup.FocusableList.end(); ++it)
   {
      Focusable* f = *it;
      if (f != NULL)
      {
         FWidgetConfig* widgetConfig = f->getConfig();
         if (widgetConfig != NULL)
         {
            FWidgetData* widgetData = widgetConfig->Data.get<FWidgetData>();
            if ((widgetData != NULL) && (widgetData->SequenceNr >= 0))
            {
               widgetData->EffectiveOrder = getEffectiveFocusOrder(f);
            }
         }
      }
   }

   return true;
}


FTask::Result FDefaultAvgBuilder::update(FSession& session)
{
   ETG_TRACE_USR1_THR(("FDefaultAvgBuilder::update app=%d", _manager.getCurrentAppId()));

   session.ActiveViewGroup.reset();

   //process config info, first the groups with no anchors and container anchors
   _anchorContentMergeInProgress = false;
   updateAllApps(session);

   //second merge the anchor content
   _anchorContentMergeInProgress = true;
   updateAllApps(session);

   _anchorContentMergeInProgress = false;

   connectToParentGroup(session);
   updateEffectiveOrderRank(session);

   session.ActiveViewGroup.Groups.execute(sortGroup);

   FGroup* rootGroup = session.ActiveViewGroup.getRootGroup();
   if (rootGroup != NULL)
   {
      removeInaccessibleGroups(*rootGroup);
   }

   //focus visibility info will be updated automatically at the end of the session
   //overwrite FSessionFocusVisibilityUpdateConfig with false in various controllers to prevent the update
   session.Data.set(FSessionFocusVisibilityUpdateConfig(true));

   return FTask::Completed;
}


bool FAvg2FocusInfoUpdater::updateViewEx(FSession& session, FViewConfig& viewConfig, FViewState& viewState)
{
   if (_manager.getAvgManager() == NULL)
   {
      return false;
   }

   //for all groups in avg
   for (size_t i = 0; i < session.ActiveViewGroup.Groups.count(); ++i)
   {
      FGroup* avgGroup = session.ActiveViewGroup.Groups.getAt(i);

      //for all widget configs of the current group
      for (size_t j = 0; (avgGroup != NULL) && (j < avgGroup->Config.count()); ++j)
      {
         FWidgetConfig* groupConfig = avgGroup->Config.get(j);

         //if widget config belongs to current view
         if ((groupConfig != NULL) && (&(groupConfig->View) == &viewConfig))
         {
            FGroupState* groupState = viewState.getOrCreateChild(groupConfig->getId());
            if (groupState != NULL)
            {
               FSequenceId* sequenceId = avgGroup->Data.get<FSequenceId>();
               if (sequenceId != NULL)
               {
                  groupState->UpdateSequenceId = *sequenceId;
               }
               else
               {
                  groupState->UpdateSequenceId = FSequenceId();
               }

               //if the crt focus of the avg group exists in the view, set it also in the view config
               FGroupCurrentFocusInfo* avgCrtFocusInfo = avgGroup->Data.get<FGroupCurrentFocusInfo>();
               if ((avgCrtFocusInfo != NULL) && (avgCrtFocusInfo->Id != Constants::InvalidId) && viewConfig.getChild(avgCrtFocusInfo->Id))
               {
                  groupState->CurrentFocusInfo = *avgCrtFocusInfo;
               }
               //otherwise reset crt focus in the view config
               else
               {
                  groupState->CurrentFocusInfo = FGroupCurrentFocusInfo();
               }
            }
         }
      }
   }

   FSessionFocusVisibilityUpdateConfig* visibilityUpdateConfig = session.Data.get<FSessionFocusVisibilityUpdateConfig>();
   if ((visibilityUpdateConfig != NULL) && (visibilityUpdateConfig->ShouldUpdate))
   {
      //the following code is executed close to the end of a session after app.CurrentFocusInfo was updated for the current active views.
      //based on app.CurrentFocusInfo we decide which view has the focus visible
      FAppStateSharedPtr appState = session.StateInfo.get(viewConfig.App.getId());
      if (!appState.PointsToNull() && (appState->CurrentFocusInfo.ViewId == viewState.getId()))
      {
         viewState.setData(FViewFocusVisible(true));
      }
      else
      {
         viewState.setData(FViewFocusVisible(false));
      }
   }

   return true;
}


bool FAvg2FocusInfoUpdater::updateAppEx(FSession& session, FAppConfigSharedPtr appConfig, FAppStateSharedPtr appState)
{
   if ((_manager.getAvgManager() == NULL) || appConfig.PointsToNull() || appState.PointsToNull())
   {
      return false;
   }

   Focusable* crtAVGFocus = _manager.getAvgManager()->getFocus(session);
   FWidget* crtAVGWidget = dynamic_cast<FWidget*>(crtAVGFocus);
   if (crtAVGWidget != NULL)
   {
      //if current focus belongs to the specified appConfig, update it with the current value
      if (crtAVGWidget->Config.View.App.getId() == appConfig->getId())
      {
         FWidgetConfig& widgetConfig = crtAVGWidget->Config;
         FViewConfig& viewConfig = widgetConfig.View;
         appState->CurrentFocusInfo = FAppCurrentFocusInfo(viewConfig.getId(), widgetConfig.getId());
      }
      //otherwise reset the current focus of the specified appConfig
      else
      {
         appState->CurrentFocusInfo = FAppCurrentFocusInfo();
      }

      updateAllViewsEx(session, appConfig, appState);
   }
   return true;
}


FTask::Result FAvg2FocusInfoUpdater::update(FSession& session)
{
   ETG_TRACE_USR1_THR(("FAvg2FocusInfoUpdater::update app=%d", _manager.getCurrentAppId()));

   updateAllApps(session);

   return FTask::Completed;
}


}
