/* ***************************************************************************************
* FILE:          FDefaultActiveGroupController.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultActiveGroupController.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_DEFAULT_ACTIVE_GROUP_CONTROLLER_H__
#define __FOCUS_DEFAULT_ACTIVE_GROUP_CONTROLLER_H__

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include <Focus/FCommon.h>
#include <Focus/FController.h>

namespace Focus {
/**************************************************************************/
/* FORWARD DECLARATIONS                                                   */
/**************************************************************************/
class FManager;
class FGroup;
class FSession;


/**************************************************************************/
/* A focus group is active if the focus is captured inside it and can only move from one child to another.
 * If it is not active then only a special child called placeholder will be focusable, so the focus will move from outside to placeholder and again outside.
 */
/**************************************************************************/
class FDefaultActiveGroupController : public FController
{
      typedef FController Base;

   public:
      FDefaultActiveGroupController(FManager& manager, const FControllerSetId& delegateControllerId = FControllerSetId());
      virtual ~FDefaultActiveGroupController() {}

      virtual bool onMessage(FSession& session, FGroup& group, const FMessage& msg) override;

      //if the group is not active then only the placeholder is reachable
      //else all the children are reachable except for the placeholder
      virtual bool refreshReachableChildren(FSession& session, FGroup& group) override;

      //default focus is the element which gets the focus when the group gets activated
      virtual Focusable* getDefaultFocus(FSession& session, FGroup& group) const;

      //placeholder gets the focus when the group is not active
      virtual Focusable* getPlaceholder(FSession&, FGroup& group) const;

      //a group is active if it contains the current focus and this is not the placeholder
      virtual bool isGroupActive(FSession& session, FGroup& group) const;

      //activates or deactivate the group
      virtual bool activateGroup(FSession& session, FGroup& group, bool activate);

      //delegate controllers will get the unconsumed messages if the group is active
      const FControllerSetId& getDelegateControllerId() const;
      void setDelegateControllerId(const FControllerSetId& delegateControllerId);

   protected:
      virtual bool onEnterKeyStatusChangedUpdMsg(FSession& session, FGroup& group, const EnterKeyStatusChangedUpdMsg& msg);
      virtual bool onFocusGroupReqMsg(FSession& session, FGroup& group, const FocusGroupReqMsg& msg);
      virtual bool dispatchToDelegateControllers(FSession& session, FGroup& group, const FMessage& msg);

      FManager& _manager;
      FControllerSetId _delegateControllerId;

   private:
      FDefaultActiveGroupController(const FDefaultActiveGroupController&);
      FDefaultActiveGroupController& operator=(const FDefaultActiveGroupController&);
};


}
#endif
