/* ***************************************************************************************
* FILE:          SoundController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SoundController.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

//#include "hall_std_if.h"

#include "AppHmi_MasterBase/AudioInterface/SoundController.h"
#include "AppHmi_MasterBase/AudioInterface/ConnectionController.h"
#include "AppHmi_MasterBase/AudioInterface/ResourceController.h"
#include "AppHmi_MasterBase/AudioInterface/Connection.h"
#include "AppHmi_MasterBase/AudioInterface/Sink.h"

#include "AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h"


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_AUDIO_IF
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SoundController.cpp.trc.h"
#endif

using namespace hmibase::services::audio;

namespace hmibase {
namespace apphmi_master {
namespace audio {

SoundController::SoundController() : m_pSoundPropertiesSenderClient(0), m_pHmiSoundProperties(0)
{
   m_bUserMuteStatus = 0;
   m_bAudioOffMuteStatus = 0;
   m_bIsUserMuteUpdateReceived = 0;
   m_bIsAudioOffUpdateReceived = 0;
}


SoundController::~SoundController()
{
   m_pSoundPropertiesSenderClient = 0;
   m_pHmiSoundProperties = 0;
   m_bIsUserMuteUpdateReceived = 0;
   m_bIsAudioOffUpdateReceived = 0;
}


void SoundController::processLeftEncoder(int nSteps, uint16_t nSinkId)
{
   int nMode = VOL_MODE_INC;
   // Decide the decrement value of volume based on the steps
   if (nSteps < 0)
   {
      nMode = VOL_MODE_DEC;
      nSteps = -(nSteps);
   }

   Sink* pSink = NULL;
   // Fetch the sink id mapped to index
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(nSinkId);
      pSink = pResourceController->GetSinkOnIdx();
      if (pSink != NULL)
      {
         ETG_TRACE_USR4_THR(("sendSetVolumeRequest SinkId %d, EncMode %d, Steps %d", pSink->GetID(), nMode, nSteps));
      }
   }

   if (m_pSoundPropertiesSenderClient != NULL && pSink != NULL)
   {
      m_pSoundPropertiesSenderClient->sendSetVolumeRequest(pSink->GetID(), static_cast<uint16>(nMode), static_cast<int16>(nSteps));
   }
}


// Requests for User Mute State Update
void SoundController::requestUserMuteState(bool muteState, uint16_t nSinkId)
{
   int nReqMute = (muteState) ? STATE_MUTED : STATE_UNMUTED;
   Sink* pSink = NULL;
   // Fetch the sink id mapped to index
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(nSinkId);
      pSink = pResourceController->GetSinkOnIdx();
      if (pSink == NULL)
      {
         ETG_TRACE_USR4_THR(("requestUserMuteState :pSink NULL"));
      }
   }

   if ((m_pSoundPropertiesSenderClient != NULL) /*&& (getUserMuteStatus(pSink->GetID()) != muteState)*/ && (pSink != NULL))
   {
      ETG_TRACE_USR4_THR(("requestUserMuteState Requested user Mute state %d  Sink Id : %d", nReqMute, pSink->GetID()));
      m_pSoundPropertiesSenderClient->sendSetSinkMuteStateRequest(pSink->GetID(), (uint16)nReqMute);
   }
}


// Requests for Audio Off Mute State Update
void SoundController::requestAudioOffMuteState(bool muteState, uint16_t nSinkId)
{
   if ((m_pSoundPropertiesSenderClient != NULL) && (getAudioOffMuteStatus() != muteState))
   {
      ETG_TRACE_USR4_THR(("requestAudioOffMuteState Requested Audio off Mute state %d", muteState));
      m_pSoundPropertiesSenderClient->sendSetMainSinkSoundProperty(nSinkId, AUDIO_OFF_MUTE, muteState);
   }
}


void SoundController::onUserMuteStateUpdate(bool muteState)
{
   ETG_TRACE_USR4_THR(("onUserMuteStateUpdate current entertainment MuteState %d", muteState));
   m_bUserMuteStatus = muteState;
   m_bIsUserMuteUpdateReceived = true;
   if (m_pHmiSoundProperties != NULL)
   {
      m_pHmiSoundProperties->updateAudioState(m_bUserMuteStatus, m_bAudioOffMuteStatus);
   }
   sendUserMuteStateUpdate(m_bUserMuteStatus);
}


/* This is the method to get the mute status of a prticular region - Cockpit /Cabin***/
bool SoundController::requestUserMuteStatus(uint16_t nSinkId)
{
   int mutestate = 0;
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   Sink* pSink = NULL;
   //int nSinkIdx = pResourceController->GetIndexOnSinkID(nSinkId);

   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(nSinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }
   if (pSink != NULL)
   {
      mutestate = pSink->GetMuteState();
      ETG_TRACE_USR4_THR(("requestUserMuteStatus mutestate :%d , nSinkId :%d ", mutestate, nSinkId));
      switch (mutestate)
      {
         case STATE_MUTED:
            m_bUserMuteStatus = true;
            break ;
         case STATE_UNMUTED:
         case STATE_UNKNOWN:
            m_bUserMuteStatus = false;
            break;
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("requestUserMuteStatus pSink NULL "));
      m_bUserMuteStatus = false;
   }
   ETG_TRACE_USR4_THR(("requestUserMuteStatus m_bUserMuteStatus :%d ", m_bUserMuteStatus));
   return  m_bUserMuteStatus;
}


void SoundController::onSinkMuteStateChange(int nSinkId, int nMuteState)
{
   ETG_TRACE_USR4_THR(("onSinkMuteStateChange SinkId %d, current MuteState %d", nSinkId, nMuteState));

   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   if (pResourceController != NULL)
   {
      Sink* pSink = pResourceController->GetSinkOnID(nSinkId);
      if (pSink != NULL)
      {
         pSink->SetMuteState(nMuteState);
      }
      if (m_pHmiSoundProperties != NULL)
      {
         int nSinkIdx = pResourceController->GetIndexOnSinkID(nSinkId);
         bool MuteState = (nMuteState == STATE_UNMUTED) ? FALSE : TRUE;
         m_pHmiSoundProperties->updateMuteState(MuteState, nSinkIdx);
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("onSinkMuteStateChange Sink %d not found", nSinkId));
   }
}


void SoundController::onSinkMuteStateChangeUpdateMute(int nSinkId, int nMuteState)
{
   ETG_TRACE_USR4_THR(("onSinkMuteStateChangeUpdateMute SinkId %d, current MuteState %d", nSinkId, nMuteState));
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   if (pResourceController != NULL)
   {
      Sink* pSink = pResourceController->GetSinkOnID(nSinkId);
      if (pSink != NULL)
      {
         pSink->SetMuteState(nMuteState);
      }
      if (m_pHmiSoundProperties != NULL)
      {
         int nSinkIdx = pResourceController->GetIndexOnSinkID(nSinkId);
         bool MuteState = (nMuteState == STATE_UNMUTED) ? FALSE : TRUE;
         m_pHmiSoundProperties->updateMuteState(MuteState, nSinkIdx);
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("onSinkMuteStateChangeUpdateMute Sink %d not found", nSinkId));
   }
}


bool SoundController::onVolumeValueChange(int nSinkId, int nVolume)
{
   bool ret = false;

   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   if (pResourceController != NULL)
   {
      Sink* pSink = pResourceController->GetSinkOnID(nSinkId);
      if (pSink != NULL)
      {
         ETG_TRACE_USR4_THR(("onVolumeValueChange SinkId %d, current Volume %d", nSinkId, nVolume));
         pSink->SetVolume(nVolume);
         if (m_pHmiSoundProperties != NULL)
         {
            int nSinkIdx = pResourceController->GetIndexOnSinkID(nSinkId);
            m_pHmiSoundProperties->updateVolumeValue(nVolume, nSinkIdx);
         }
         ret = true;
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("onVolumeValueChange Sink %d not found", nSinkId));
   }
   return ret;
}


void SoundController::requestBeep(int nBeepType)
{
   if ((m_pSoundPropertiesSenderClient != NULL) && isBeepEnabled(nBeepType))
   {
      m_pSoundPropertiesSenderClient->sendSetSystemProperty(SYP_BEEP, static_cast<int16>(nBeepType));
   }
}


bool SoundController::isBeepEnabled(int /*nBeepType*/)
{
//   if ((nBeepType != BeepType_CLICK) || (m_pHmiSoundProperties->getConfirmationToneEnabled()))
//   {
//      return true;
//   }
//   return false;
   return true;
}


void SoundController::setSourceSoundProperty(int nPropertyType, int nValue)
{
   ETG_TRACE_USR4_THR(("setSourceSoundProperty PropertyType %d = %d", ETG_CENUM(enSoundPropertyType, nPropertyType), nValue));
   m_mainSourceSoundProperties[nPropertyType] = nValue;
}


void SoundController::onSetSinkSoundProperty(int nPropertyType, int nValue)
{
   ETG_TRACE_USR4_THR(("onSetSinkSoundProperty PropertyType %d = %d", ETG_CENUM(enSoundPropertyType, nPropertyType), nValue));
   m_mainSinkSoundProperties[nPropertyType] = nValue;
   if (nPropertyType == AUDIO_OFF_MUTE)
   {
      ETG_TRACE_USR4_THR(("onAudioOffMuteStateUpdate current MuteState %d", nValue));
      m_bAudioOffMuteStatus = nValue;
      m_bIsAudioOffUpdateReceived = true;
      if (m_pHmiSoundProperties != NULL)
      {
         m_pHmiSoundProperties->updateAudioState(m_bUserMuteStatus, m_bAudioOffMuteStatus);
      }
      sendAudioOffMuteStateUpdate(nValue);
   }
}


void SoundController::requestMainSinkSoundProperty(uint16_t nSinkId, uint16_t nPropertyType, int nValue)
{
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   Sink* pSink = NULL;
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(nSinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }

   if (m_pSoundPropertiesSenderClient != 0 && (pSink != NULL))
   {
      ETG_TRACE_USR4_THR(("requestMainSinkSoundProperty DynamicsinkID %d Property Type %d value %d", pSink->GetID(), nPropertyType, nValue));
      m_pSoundPropertiesSenderClient->sendSetMainSinkSoundProperty(pSink->GetID(), nPropertyType, static_cast<uint16>(nValue));
   }
}


void SoundController::requestSystemProperty(uint16_t nPropertyType, int nValue)
{
   if (m_pSoundPropertiesSenderClient != 0)
   {
      m_pSoundPropertiesSenderClient->sendSetSystemProperty(nPropertyType, static_cast<int16>(nValue));
   }
}


void SoundController::requestMainSinkSoundPropertiesList(uint16_t nSinkId)
{
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   Sink* pSink = NULL;
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(nSinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }

   if (m_pSoundPropertiesSenderClient != 0 && (pSink != NULL))
   {
      ETG_TRACE_USR4_THR(("requestMainSinkSoundPropertiesList DynamicsinkID %d", pSink->GetID()));
      m_pSoundPropertiesSenderClient->sendgetListMainSinkSoundProperty(pSink->GetID());
   }
}


void SoundController::requestSystemPropertiesList()
{
   if (m_pSoundPropertiesSenderClient != 0)
   {
      m_pSoundPropertiesSenderClient->sendgetListSystemProperty();
   }
}


void SoundController::onMainSinkSoundPropertyUpdate(int nPropertyType, int nValue, uint16_t nSinkId)
{
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   uint16_t nSinkIdx = DEFAULT_MAIN_SINK_ID;
   if (pResourceController != NULL)
   {
      nSinkIdx = pResourceController->GetIndexOnSinkID(nSinkId);
   }
   if (m_pHmiSoundProperties != NULL)
   {
      m_pHmiSoundProperties->sendMainSinkSoundPropertyUpdate(nPropertyType, nValue, nSinkIdx);
   }
}


void SoundController::onSystemPropertyUpdate(int nPropertyType, int nValue)
{
   if (m_pHmiSoundProperties != NULL)
   {
      m_pHmiSoundProperties->sendSystemPropertyUpdate(nPropertyType, nValue);
   }
}


void SoundController::onSystemPropertiesListUpdate(::std::vector<bosch::cm::ai::hmi::masteraudioservice::SoundProperties::SystemProperty> systempropertylist)
{
   if (m_pHmiSoundProperties != NULL)
   {
      m_pHmiSoundProperties->updateSystemPropertyList(systempropertylist);
   }
}


void SoundController::onMainSinkSoundPropertiesListUpdate(::std::vector<bosch::cm::ai::hmi::masteraudioservice::SoundProperties::SoundProperty>soundpropertylist, int sinkId)
{
   if (m_pHmiSoundProperties != NULL)
   {
      m_pHmiSoundProperties->updateMainSinkSoundPropertyList(soundpropertylist, sinkId);
   }
}


void SoundController::onSoundPropertyListUpdate(int nSinkId)
{
   ETG_TRACE_USR4_THR(("onSoundPropertyListUpdate  sinkID %d", nSinkId));
   m_mainSinkSoundPropertiesList_sink[nSinkId] = m_mainSinkSoundProperties ;
}


int SoundController::getSoundProperty(enSoundPropertyType nPropertyType)
{
   std::map<int, int>::iterator itrSrc, itrSink;
   itrSrc = m_mainSourceSoundProperties.find(static_cast<int>(nPropertyType));
   itrSink = m_mainSinkSoundProperties.find(static_cast<int>(nPropertyType));
   if (itrSrc != m_mainSourceSoundProperties.end())
   {
      return itrSrc->second;
   }
   else if (itrSink != m_mainSinkSoundProperties.end())
   {
      return itrSink->second;
   }
   else
   {
      ETG_TRACE_USR4_THR(("getSourceSoundProperty PropertyType %d not available", ETG_CENUM(enSoundPropertyType, nPropertyType)));
      return 0;
   }
}


int SoundController::getSoundPropertySink(enSoundPropertyType nPropertyType, int nSinkID)
{
   ETG_TRACE_USR4_THR(("getSoundPropertySink "));
   std::map<int, int>::iterator itr;
   std::map< int, std::map<int, int> >::iterator itrSink;
   itrSink = m_mainSinkSoundPropertiesList_sink.find(nSinkID);
   if (itrSink != m_mainSinkSoundPropertiesList_sink.end())
   {
      itr = m_mainSinkSoundProperties.find(static_cast<int>(nPropertyType));

      if (itr != m_mainSinkSoundProperties.end())
      {
         return itr->second;
      }
      else
      {
         ETG_TRACE_USR4_THR(("getSoundPropertySink PropertyType %d not available", ETG_CENUM(enSoundPropertyType, nPropertyType)));
         return -1;
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("getSoundPropertySink SinkID %d not available", nSinkID));
      return -1;
   }
}


void SoundController::setHmiStartupMute(bool bMute)
{
   if (m_pSoundPropertiesSenderClient != 0)
   {
      m_pSoundPropertiesSenderClient->sendSetSystemProperty(SYP_HMISTARTUP_MUTE, bMute);
   }
}


void SoundController::setSystemMute(bool bMute, uint16_t nSinkId)
{
   ETG_TRACE_USR4_THR(("setSystemMute SinkId %d,  bMute %d", nSinkId, bMute));
   boost::shared_ptr<ResourceController> pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>();
   Sink* pSink = NULL;

   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(nSinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }
   if (m_pSoundPropertiesSenderClient != 0 && (pSink != NULL))
   {
      ETG_TRACE_USR4_THR(("setSystemMute Dynamic SinkId %d,  bMute %d", pSink->GetID(), bMute));
      m_pSoundPropertiesSenderClient->sendSetSystemMuteRequest(pSink->GetID(), bMute);
   }
}


void SoundController::setMainSinkSoundProperty(enSoundPropertyType nPropertyType, int nValue, uint16_t nSinkId)
{
   if (m_pSoundPropertiesSenderClient != 0)
   {
      m_pSoundPropertiesSenderClient->sendSetMainSinkSoundProperty(nSinkId, nPropertyType, static_cast<uint16>(nValue));
   }
}


void SoundController::registerMuteStateUpdate(SinkMuteStateObserver* pObserver)
{
   if (::std::find(_pObservers.begin(), _pObservers.end(), pObserver) == _pObservers.end())
   {
      _pObservers.push_back(pObserver);
   }
   if (m_bIsAudioOffUpdateReceived)
   {
      sendAudioOffMuteStateUpdate(m_bAudioOffMuteStatus);
   }
   if (m_bIsUserMuteUpdateReceived)
   {
      sendUserMuteStateUpdate(m_bUserMuteStatus);
   }
}


void SoundController::deRegisterMuteStateUpdate(SinkMuteStateObserver* pObserver)
{
   ::std::vector<SinkMuteStateObserver*>::iterator itr = ::std::find(_pObservers.begin(), _pObservers.end(), pObserver);
   if (itr != _pObservers.end())
   {
      _pObservers.erase(itr);
   }
}


void SoundController::sendAudioOffMuteStateUpdate(bool bMuteState)
{
   ::std::vector<SinkMuteStateObserver*> observerList = _pObservers;
   ::std::vector<SinkMuteStateObserver*>::iterator itr = observerList.begin();
   for (; itr != observerList.end(); itr++)
   {
      (*itr)->onAudioOffMuteStateUpdate(bMuteState);
   }
}


void SoundController::sendUserMuteStateUpdate(bool bMuteState)
{
   ::std::vector<SinkMuteStateObserver*> observerList = _pObservers;
   ::std::vector<SinkMuteStateObserver*>::iterator itr = observerList.begin();
   for (; itr != observerList.end(); itr++)
   {
      (*itr)->onUserMuteStateUpdate(bMuteState);
   }
}


}
}


}
