/* ***************************************************************************************
* FILE:          HmiGeniviAudioClient.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HmiGeniviAudioClient.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"

#include "HmiGeniviAudioClient.h"
#include "AppHmi_MasterBase/AudioInterface/ConnectionController.h"
#include "AppHmi_MasterBase/AudioInterface/ResourceController.h"
#include "AppHmi_MasterBase/AudioInterface/SoundController.h"
#include "AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_AUDIO_IF
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HmiGeniviAudioClient.cpp.trc.h"
#endif

using namespace ::org::genivi::audiomanager::CommandInterface;
using namespace ::asf::core;

namespace hmibase {
namespace apphmi_master {
namespace audio {

HmiGeniviAudioClient::HmiGeniviAudioClient()
   : _CommandIFProxy(CommandInterfaceProxy::createProxy("commandInterfacePort", *this)),
     _CommandIFProxy_NumberOfMainConnectionsChanged_RegId(0),
     _CommandIFProxy_SinkAdded_RegId(0),
     _CommandIFProxy_SinkRemoved_RegId(0),
     _CommandIFProxy_SourceAdded_RegId(0),
     _CommandIFProxy_SourceRemoved_RegId(0),
     _CommandIFProxy_MainConnectionStateChanged_RegId(0),
     _CommandIFProxy_SinkAvailabilityChanged_RegId(0),
     _CommandIFProxy_SourceAvailabilityChanged_RegId(0),
     _CommandIFProxy_NewMainConnection_RegId(0)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient constructor"));
}


HmiGeniviAudioClient::~HmiGeniviAudioClient()
{
}


void HmiGeniviAudioClient::initialize()
{
   StartupSync::getInstance().registerPropertyRegistrationIF(this, _CommandIFProxy->getPortName());
   hmibase::apphmi_master::audio::AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->RegisterHmiGeniviAudioClient(this);
}


void HmiGeniviAudioClient::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void HmiGeniviAudioClient::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


void HmiGeniviAudioClient::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_CommandIFProxy == proxy)
   {
      _CommandIFProxy_NumberOfMainConnectionsChanged_RegId = _CommandIFProxy->sendNumberOfMainConnectionsChangedRegister(*this);
      _CommandIFProxy_SinkAdded_RegId = _CommandIFProxy->sendSinkAddedRegister(*this);
      _CommandIFProxy_SinkRemoved_RegId = _CommandIFProxy->sendSinkRemovedRegister(*this);
      _CommandIFProxy_SourceAdded_RegId = _CommandIFProxy->sendSourceAddedRegister(*this);
      _CommandIFProxy_SourceRemoved_RegId = _CommandIFProxy->sendSourceRemovedRegister(*this);
      _CommandIFProxy_MainConnectionStateChanged_RegId = _CommandIFProxy->sendMainConnectionStateChangedRegister(*this);
      _CommandIFProxy_SinkAvailabilityChanged_RegId = _CommandIFProxy->sendSinkAvailabilityChangedRegister(*this);
      _CommandIFProxy_SourceAvailabilityChanged_RegId = _CommandIFProxy->sendSourceAvailabilityChangedRegister(*this);
      _CommandIFProxy_NewMainConnection_RegId = _CommandIFProxy->sendNewMainConnectionRegister(*this);
      _CommandIFProxy->sendGetListMainSinksRequest(*this);
      _CommandIFProxy->sendGetListMainSourcesRequest(*this);
      _CommandIFProxy->sendGetListMainConnectionsRequest(*this);
   }
}


void HmiGeniviAudioClient::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


// Connection Interfaces
int HmiGeniviAudioClient::sendConnectRequest(uint16 u16SourceID, uint16 u16SinkID)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient sendConnectRequest"));
   return (static_cast<int>(_CommandIFProxy->sendConnectRequest(*this, u16SourceID, u16SinkID)));
}


void HmiGeniviAudioClient::onConnectResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< ConnectResponse >& reply)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onConnectResponse"));
   if (_CommandIFProxy == proxy)
   {
      //int act = reply->getAct();
      AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->mainConnectResponse(reply->getMainConnectionID(), reply->getAct());
   }
}


void HmiGeniviAudioClient::onConnectError(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< ConnectError >& /*error*/)
{
   if (_CommandIFProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("HmiGeniviAudioClient -> onConnectError"));
      AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->mainConnectError();
   }
}


void HmiGeniviAudioClient::sendDisconnectRequest(uint16 u16ConnectionID)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient sendDisconnectRequest"));
   _CommandIFProxy->sendDisconnectRequest(*this, u16ConnectionID);
}


void HmiGeniviAudioClient::onDisconnectResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< DisconnectResponse >& /*reply*/)
{
   if (_CommandIFProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("HmiGeniviAudioClient -> onDisconnectResponse"));
   }
}


void HmiGeniviAudioClient::onDisconnectError(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< DisconnectError >& /*error*/)
{
   if (_CommandIFProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("HmiGeniviAudioClient -> onDisconnectError"));
   }
}


void HmiGeniviAudioClient::onMainConnectionStateChangedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< MainConnectionStateChangedSignal >& /*sig*/)
{
   if (_CommandIFProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onMainConnectionStateChanged"));

      /* Workaround for NCG3D-14876, The below line is commented because the connection state information received from Audio is sometimes outdated compared to the
       * data received in response for GetListMainConnectionRequest. Hence the signal data is not processed but only a trigger to get Connection list is sent. */

      //AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->updateMainConnectionState(sig->getConnectionID(), sig->getConnectionState());
   }
   _CommandIFProxy->sendGetListMainConnectionsRequest(*this);
}


void HmiGeniviAudioClient::onMainConnectionStateChangedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< MainConnectionStateChangedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient -> onMainConnectionStateChangedError"));
}


void HmiGeniviAudioClient::onNumberOfMainConnectionsChangedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< NumberOfMainConnectionsChangedSignal >& /*sig*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onNumberOfMainConnectionsChanged"));
   if (_CommandIFProxy == proxy)
   {
      _CommandIFProxy->sendGetListMainConnectionsRequest(*this);
   }
}


void HmiGeniviAudioClient::onNumberOfMainConnectionsChangedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< NumberOfMainConnectionsChangedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient -> onNumberOfMainConnectionsChangedError"));
}


void HmiGeniviAudioClient::onGetListMainConnectionsResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainConnectionsResponse >& reply)
{
   ::std::vector< GetListMainConnectionsResponseListConnectionsStruct > ConnList = reply->getListConnections();
   ::std::vector< GetListMainConnectionsResponseListConnectionsStruct >::iterator it;

   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient -> onGetListMainConnectionsResponse"));
   AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->startConnectionListUpdate();
   for (it = ConnList.begin(); it != ConnList.end(); ++it)
   {
      int nConnID = it->getElem1();
      int nSrcID = it->getElem2();
      int nSinkID = it->getElem3();
      int nDelay = it->getElem4();
      int nState = it->getElem5();
      AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->addMainConnection(nConnID, nSrcID, nSinkID, nDelay, nState);
      ETG_TRACE_USR4_THR(("Connection %d, SourceID %d, SinkID %d", nConnID, nSrcID, nSinkID));
      ETG_TRACE_USR4_THR(("Delay %d, State %d", nDelay, nState));
   }
   AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->endConnectionListUpdate();
}


void HmiGeniviAudioClient::onGetListMainConnectionsError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainConnectionsError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient -> onGetListMainConnectionsError"));
}


// Resource Interfaces
void HmiGeniviAudioClient::onGetListMainSinksResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainSinksResponse >& reply)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onGetListMainSinksResponse"));
   ::std::vector< GetListMainSinksResponseListMainSinksStruct > SinkList = reply->getListMainSinks();
   ::std::vector< GetListMainSinksResponseListMainSinksStruct >::iterator it;
   ::std::string sSinkName = "";
   int nSinkID = 0;
   int nSinkAvailability = 0;
   int nVolume = 0;
   int nMuteState = 0;
   int nClassID = 0;
   for (it = SinkList.begin(); it != SinkList.end(); ++it)
   {
      nSinkID = it->getElem1();
      sSinkName.assign(it->getElem2());
      nSinkAvailability = it->getElem3().getElem1();
      nVolume = it->getElem4();
      nMuteState = it->getElem5();
      nClassID = it->getElem6();
      AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->AddSink(nSinkID, sSinkName, nSinkAvailability, nVolume, nMuteState, nClassID);
   }
}


void HmiGeniviAudioClient::onGetListMainSinksError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainSinksError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onGetListMainSinksError"));
}


void HmiGeniviAudioClient::onSinkAddedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SinkAddedSignal >& sig)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSinkAdded"));
   SinkAddedSignalNewSinkStruct stSink = sig->getNewSink();
   int nSinkID = stSink.getElem1();
   int nSinkAvailability = stSink.getElem3().getElem1();
   int nVolume = stSink.getElem4();
   int nMuteState = stSink.getElem5();
   int nClassID = stSink.getElem6();
   ::std::string sSinkName = stSink.getElem2();
   AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->AddSink(nSinkID, sSinkName, nSinkAvailability, nVolume, nMuteState, nClassID);
}


void HmiGeniviAudioClient::onSinkAddedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SinkAddedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSinkAddedError"));
}


void HmiGeniviAudioClient::onSinkAvailabilityChangedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< SinkAvailabilityChangedSignal >& sig)
{
   if (_CommandIFProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSinkAvailabilityChanged"));
      AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->UpdateSinkAvailability(sig->getSinkID(), sig->getAvailability().getElem1());
      _CommandIFProxy->sendGetListMainSinksRequest(*this);
   }
}


void HmiGeniviAudioClient::onSinkAvailabilityChangedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SinkAvailabilityChangedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSinkAvailabilityChangedError"));
}


void HmiGeniviAudioClient::onSinkRemovedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SinkRemovedSignal >& sig)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSinkRemoved"));
   AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->RemoveSink(sig->getRemovedSinkID());
}


void HmiGeniviAudioClient::onSinkRemovedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SinkRemovedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSinkRemovedError"));
}


void HmiGeniviAudioClient::onGetListMainSourcesResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainSourcesResponse >& reply)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onGetListMainSourcesResponse"));
   ::std::vector< GetListMainSourcesResponseListMainSourcesStruct > SrcList = reply->getListMainSources();
   ::std::vector< GetListMainSourcesResponseListMainSourcesStruct >::iterator it;
   ::std::string sSrcName = "";
   int nSrcID = 0;
   int nSrcAvailability = 0;
   int nSrcAvailabilityReason = SRC_AVAIL_UNKNOWN;
   int nClassID = 0;
   for (it = SrcList.begin(); it != SrcList.end(); ++it)
   {
      nSrcID = it->getElem1();
      sSrcName.assign(it->getElem2());
      nSrcAvailability = it->getElem3().getElem1();
      nSrcAvailabilityReason = it->getElem3().getElem2();
      nClassID = it->getElem4();
      AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->AddSource(nSrcID, sSrcName, nSrcAvailability, nSrcAvailabilityReason, nClassID);
   }
}


void HmiGeniviAudioClient::onGetListMainSourcesError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainSourcesError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onGetListMainSourcesError"));
}


void HmiGeniviAudioClient::onSourceAddedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SourceAddedSignal >& sig)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSourceAdded"));
   SourceAddedSignalNewSourceStruct stSrc = sig->getNewSource();
   int nSrcID = stSrc.getElem1();
   int nSrcAvailability = stSrc.getElem3().getElem1();
   int nSrcAvailabilityReason = stSrc.getElem3().getElem2();
   int nClassID = stSrc.getElem4();
   ::std::string sSrcName = stSrc.getElem2();
   AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->AddSource(nSrcID, sSrcName, nSrcAvailability, nSrcAvailabilityReason, nClassID);
}


void HmiGeniviAudioClient::onSourceAddedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SourceAddedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSourceAddedError"));
}


void HmiGeniviAudioClient::onSourceAvailabilityChangedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, const ::boost::shared_ptr< SourceAvailabilityChangedSignal >& sig)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSourceAvailabilityChanged"));
   if (_CommandIFProxy == proxy)
   {
      AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->UpdateSourceAvailability(sig->getSourceID(), sig->getAvailability().getElem1(), sig->getAvailability().getElem2());
      _CommandIFProxy->sendGetListMainSourcesRequest(*this);
   }
}


void HmiGeniviAudioClient::onSourceAvailabilityChangedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SourceAvailabilityChangedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSourceAvailabilityChangedError"));
}


void HmiGeniviAudioClient::onSourceRemovedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SourceRemovedSignal >& sig)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSourceRemoved"));
   AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->RemoveSource(sig->getRemovedSourceID());
}


void HmiGeniviAudioClient::onSourceRemovedError(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< SourceRemovedError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onSourceRemovedError"));
}


void HmiGeniviAudioClient::onNewMainConnectionSignal(const ::boost::shared_ptr< CommandInterfaceProxy  >& proxy, const ::boost::shared_ptr< NewMainConnectionSignal >& /*signal*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onNewMainConnectionSignal"));
   if (_CommandIFProxy == proxy)
   {
      _CommandIFProxy->sendGetListMainConnectionsRequest(*this);
   }
}


void HmiGeniviAudioClient::onNewMainConnectionError(const ::boost::shared_ptr< CommandInterfaceProxy  >& /*proxy*/, const ::boost::shared_ptr< NewMainConnectionError >& /*error*/)
{
   ETG_TRACE_USR4_THR(("HmiGeniviAudioClient onNewMainConnectionError"));
}


}
}


}
