/* ***************************************************************************************
* FILE:          HmiAudioSourceChange.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HmiAudioSourceChange.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "HmiAudioSourceChange.h"
#include "AudioControllerObjectManager.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_AUDIO_IF
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HmiAudioSourceChange.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;
using namespace ::org::genivi::audiomanager::CommandInterface;
using namespace ::hmibase::apphmi_master::audio;

namespace hmibase {
namespace services {
namespace audio {
HmiAudioSourceChange::HmiAudioSourceChange() :
   AudioSourceChangeStub("audioSourceChangePort")
{
}


HmiAudioSourceChange::~HmiAudioSourceChange()
{
}


void HmiAudioSourceChange::initialize(boost::shared_ptr<ConnectionController> connectionController, boost::shared_ptr<SoundController> soundController, boost::shared_ptr<ResourceController> resourceController)
{
   AudioControllerObjectManager::initialize(connectionController, soundController, resourceController);
   AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->RegisterCallbackObject(this);
   AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->RegisterCallbackObject(this);
   _soundPropsRecvClient.initialize();
   _soundPropsSenderClient.initialize();
}


void HmiAudioSourceChange::onActivateSourceRequest(const ::boost::shared_ptr< ActivateSourceRequest >& request)
{
   uint16_t nSinkId = request->getSourceInfo().getSinkId();
   if (nSinkId == 0)
   {
      nSinkId = DEFAULT_MAIN_SINK_ID;
   }
   AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->sourceRequest(request->getSourceInfo().getSrcId(), request->getSourceInfo().getSubSrcId(), nSinkId);//added sinkid
   AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->setUserSourceAction(request->getForceViewChange());
   AudioSourceChangeStub::sendActivateSourceResponse();
}


void HmiAudioSourceChange::onDeactivateSourceRequest(const ::boost::shared_ptr< DeactivateSourceRequest >& request)
{
   uint16_t nSinkId = request->getSourceInfo().getSinkId();
   if (nSinkId == 0)
   {
      nSinkId = DEFAULT_MAIN_SINK_ID;
   }
   AudioControllerObjectManager::getInstance().getConnectionController<ConnectionController>()->sourceDisconnect(request->getSourceInfo().getSrcId(), request->getSourceInfo().getSubSrcId(), nSinkId);//added sinkid
   AudioSourceChangeStub::sendDeactivateSourceResponse();
}


void HmiAudioSourceChange::onGetSourceListRequest(const ::boost::shared_ptr< GetSourceListRequest >& request)
{
   ::std::vector< sourceDetails > srcList;
   AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetGroupSrcList(request->getSrcGroupId(), &srcList);
   AudioSourceChangeStub::sendGetSourceListResponse(srcList);
}


void HmiAudioSourceChange::sendSourceActivatedRequest(int32 nSrcIndex, int32 nSinkId)
{
   ETG_TRACE_USR4_THR(("HmiAudioSourceChange sendSourceActivatedRequest"));
   AudioSourceChangeStub::sendSourceActivatedSignal(nSrcIndex, nSinkId);
}


void HmiAudioSourceChange::sendSourceListUpdateRequest(int32 nGrpIndex)
{
   ETG_TRACE_USR4_THR(("HmiAudioSourceChange sendSourceListUpdateRequest"));
   AudioSourceChangeStub::sendSourceListChangedSignal(nGrpIndex);
}


void HmiAudioSourceChange::sendSourceAvailabilityUpdateRequest(int32 nSrcIndex, int32 nSubSrcId, int32 nAvailability, int32 nAvailabilityReason)
{
   ETG_TRACE_USR4_THR(("HmiAudioSourceChange sendSourceAvailabilityUpdateRequest"));
   sourceDetails srcAvailability(nSrcIndex, nSubSrcId, nAvailability, nAvailabilityReason);
   AudioSourceChangeStub::sendSourceAvailabilityChangedSignal(srcAvailability);
}


void HmiAudioSourceChange::sendSetActiveSource(int32 nSourceId, int32 nSubSourceId,  int32 nSinkId, uint32 connectionState)
{
   sourceData srcIdx(nSourceId, nSubSourceId, nSinkId, connectionState);
   if (AudioSourceChangeStub::getActiveSource() != srcIdx)
   {
      AudioSourceChangeStub::setActiveSource(srcIdx);
      sendSourceActivatedRequest(nSourceId, nSinkId);
   }
}


void HmiAudioSourceChange::sendSetActiveSources(::std::vector<stSourceInfo>& activeSources)
{
   ::std::vector<sourceData> activeSourceList;
   for (uint32 index = 0; index < activeSources.size(); index++)
   {
      sourceData oData(static_cast<int32>(activeSources[index].srcId), static_cast<int32>(activeSources[index].deviceId), activeSources[index].sinkId, activeSources[index].connectionState);
      activeSourceList.push_back(oData);
   }
   bool bEntriesAdded = addNewEntries(activeSourceList);
   bool bEntriesRemoved = removeOldEntries(activeSourceList);
   if (bEntriesAdded || bEntriesRemoved)
   {
      AudioSourceChangeStub::setActiveSourceList(_activeSourceList);
   }
}


bool HmiAudioSourceChange::removeOldEntries(::std::vector<sourceData>& activeSourceList)
{
   ::std::vector<sourceData>::iterator iter = _activeSourceList.begin();
   bool bListChanged = false;
   for (; iter != _activeSourceList.end();)
   {
      if (std::find(activeSourceList.begin(), activeSourceList.end(), *iter) == activeSourceList.end())  //lint !e864
      {
         iter = _activeSourceList.erase(iter);
         bListChanged = true;
      }
      else
      {
         ++iter;
      }
   }
   return bListChanged;
}


bool HmiAudioSourceChange::addNewEntries(::std::vector<sourceData>& activeSourceList)
{
   ::std::vector<sourceData>::iterator iter = activeSourceList.begin();
   bool bListChanged = false;
   for (; iter != activeSourceList.end(); ++iter)
   {
      if (std::find(_activeSourceList.begin(), _activeSourceList.end(), *iter) == _activeSourceList.end())
      {
         _activeSourceList.push_back(*iter);
         bListChanged = true;
      }
   }
   return bListChanged;
}


}
}


}
