/* ***************************************************************************************
* FILE:          ConnectionController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ConnectionController.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

//#include "hall_std_if.h"

#include "AppHmi_MasterBase/AudioInterface/Connection.h"
#include "AppHmi_MasterBase/AudioInterface/Sink.h"
#include "AppHmi_MasterBase/AudioInterface/Source.h"
#include "AppHmi_MasterBase/AudioInterface/IAudioManagerHandler.h"
#include "AppHmi_MasterBase/AudioInterface/IHmiAudioServiceStubHandler.h"
#include "AppHmi_MasterBase/AudioInterface/ConnectionController.h"
#include "AppHmi_MasterBase/AudioInterface/ResourceController.h"

#include "AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h"
#include "hmibase/util/Macros.h"


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_AUDIO_IF
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ConnectionController.cpp.trc.h"
#endif

namespace hmibase {
namespace apphmi_master {
namespace audio {

ConnectionController::ConnectionController()
   : m_bStartupAvailable(false)
   , m_bGroupRequestActive(false)
   , _userSourceAction(false)
   , m_pIAudioManagerHandler(NULL)
   , m_pHmiAudioServiceStubHandler(NULL)
   , m_ConnectACT(0)
   , m_pMainConnection(NULL)
   , m_pSourceConnect(NULL)
   , m_pSinkConnect(NULL)
{
}


ConnectionController::~ConnectionController()
{
   m_pHmiAudioServiceStubHandler = NULL;
   m_pIAudioManagerHandler = NULL;
   m_pMainConnection = NULL;
   m_pSourceConnect = NULL;
   m_pSinkConnect = NULL;
}


void ConnectionController::sendConnectRequest(Source* pSource, Sink* pSink)
{
   if (m_ConnectACT != 0)
   {
      ETG_TRACE_USR4_THR(("sendConnectRequest called while response pending for previous connection"));
   }
   if ((m_pIAudioManagerHandler != NULL) && (pSource != NULL) && (pSink != NULL))
   {
      ETG_TRACE_USR4_THR(("sendConnectRequest called sourceID = %d , sinkId = %d ", pSource->GetID(), pSink->GetID()));
      m_ConnectACT = m_pIAudioManagerHandler->sendConnectRequest(static_cast<uint16>(pSource->GetID()), static_cast<uint16>(pSink->GetID()));
      m_pSourceConnect = pSource;
      m_pSinkConnect = pSink;
      //System demute should be sent to Audio
   }
}


void ConnectionController::sendDisconnectRequest(const Connection* pConnection)
{
   if (m_pIAudioManagerHandler != NULL && pConnection != NULL)
   {
      m_pIAudioManagerHandler->sendDisconnectRequest(static_cast<uint16>(pConnection->getConnectionID()));
   }
}


void ConnectionController::mainConnectResponse(int nConnectionID, int nAct)
{
   ETG_TRACE_USR4_THR(("mainConnectResponse stored act %d received act %d", m_ConnectACT,  nAct));

   if (nAct == m_ConnectACT)
   {
      m_ConnectACT = 0;
      createNewConnection(nConnectionID);
   }
   else
   {
      ETG_TRACE_USR4_THR(("mainConnectResponse, stored act and received act are different"));
   }
}


void ConnectionController::updateMainConnectionState(int nConnectionID, int nConnectionState)
{
   Connection* pConnection = NULL;
   std::map<int, Connection*>::iterator itr;
   itr = m_ConnectionMap.find(nConnectionID);
   if (itr != m_ConnectionMap.end())
   {
      pConnection = itr->second;
   }
   ETG_TRACE_USR4_THR(("Connection Update received for connection ID %d with state %d", nConnectionID, nConnectionState));
   if (pConnection != NULL)
   {
      pConnection->setConnectionState(nConnectionState);
      updateConnectionState(pConnection);
   }
   else
   {
      ETG_TRACE_USR4_THR(("Connection Update received but connection ID do not match with the list!!!"));
   }
}


void ConnectionController::updateConnectionState(Connection* pConnection)
{
   ETG_TRACE_USR4_THR(("updateConnectionState: connection state %d for connection id %d with source Id %d", pConnection->getState(), pConnection->getConnectionID(), pConnection->getSource()->GetID()));
   if (pConnection != NULL && m_pHmiAudioServiceStubHandler != NULL && pConnection->IsConnectionStateAttained() == true)
   {
      ETG_TRACE_USR4_THR(("Required connection state %d attained for connection id %d, activate signal", pConnection->getState(), pConnection->getConnectionID()));
      sendSourceActivationRequest(pConnection->getSource(), pConnection->getSink(), pConnection->getState());
   }
}


void ConnectionController::mainConnectError()
{
   ETG_TRACE_USR4_THR(("Connection error, fall back to be implemented"));
}


void ConnectionController::createNewConnection(int nConnectionID)
{
   if (m_pSourceConnect == NULL || m_pSinkConnect == NULL)
   {
      return;
   }
   Connection* pConnection = new Connection(nConnectionID, m_pSourceConnect, m_pSinkConnect);

   ETG_TRACE_USR4_THR(("createNewConnection: New connection created for Connection Id %d and Source Id %d", nConnectionID, m_pSourceConnect->GetID()));

   std::map<int, Connection*>::iterator itr;
   itr = m_ConnectionMap.find(nConnectionID);
   if (itr != m_ConnectionMap.end())
   {
      ETG_TRACE_USR4_THR(("createNewConnection: Existing connection deleted for Connection Id %d and Source Id %d with connection state %d", nConnectionID, itr->second->getSource()->GetID(), itr->second->getState()));
      delete itr->second;
      m_ConnectionMap.erase(itr->first);
   }
   m_ConnectionMap[nConnectionID] = pConnection;
}


bool ConnectionController::sourceRequest(int nSrcIndex, int nDevId, uint16_t nSinkId)
{
   bool bRet = false;
   if (nSrcIndex < static_cast<int>(SRC_INVALID))
   {
      int nGrpIndex = nSrcIndex / SRC_GROUP_SIZE;
      Source* pSource = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetSourceOnIdx(nSrcIndex, nDevId);
      ETG_TRACE_USR4_THR(("sourceRequest: Request for source index %d with device Id %d and sink ID %d", nSrcIndex, nDevId, nSinkId));
      AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->mapSinkIdx(nSinkId);
      m_pSinkConnect = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetSinkOnIdx();
      if (m_pSinkConnect != NULL)
      {
         Connection* pConnection = getActiveConnection(m_pSinkConnect->GetID());
         if (pConnection != NULL)
         {
            m_pSourceConnect = pConnection->getSource();
         }
         else
         {
            m_pSourceConnect = NULL;
         }
         if (pSource != NULL && m_pSinkConnect != NULL && pSource->GetAvailability() == RES_AVAILABLE)
         {
            if (pSource != m_pSourceConnect)
            {
               if (m_pSourceConnect != NULL)
               {
                  ETG_TRACE_USR4_THR(("sourceRequest: Requested source Id %d active source Id %d", pSource->GetID(), m_pSourceConnect->GetID()));
               }
               else
               {
                  ETG_TRACE_USR4_THR(("sourceRequest: Requested source Id %d with no active source", pSource->GetID()));
               }
               sendConnectRequest(pSource, m_pSinkConnect);
            }
            else
            {
               pConnection = getMainConnection(pSource);
               if (pConnection != NULL)
               {
                  ETG_TRACE_USR4_THR(("sourceRequest: Request for source %d already active with connection state %d and connection Id %d", nSrcIndex, pConnection->getState(), pConnection->getConnectionID()));
                  if (pConnection->getState() != STATE_CONNECTED)
                  {
                     sendConnectRequest(pSource, m_pSinkConnect);
                  }
                  updateConnectionState(pConnection);
               }
               else
               {
                  ETG_TRACE_USR4_THR(("sourceRequest: Error!!! Not able to find the right connection for source %d", nSrcIndex));
               }
            }
            bRet = true;
         }
         else //Implementation to be clarified !!!
         {
            ETG_TRACE_USR4_THR(("sourceRequest: Requested source %d not available in GroupIdx %d", nSrcIndex, nGrpIndex));
         }
      }
      else
      {
         ETG_TRACE_ERR_THR(("sourceRequest: Invalid sink id %d", nSinkId));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("sourceRequest: Invalid source index %d", nSrcIndex));
   }
   return bRet;
}


void ConnectionController::sourceDisconnect(int nSrcIndex, int nDevId, uint16_t nSinkId)
{
   ETG_TRACE_USR4_THR(("sourceDisconnect: Request for source index %d with device Id %d and sink ID %d", nSrcIndex, nDevId, nSinkId));

   //Check for sink id before disconnect
   Source* pSource = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetSourceOnIdx(nSrcIndex, nDevId);
   AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->mapSinkIdx(nSinkId);
   Sink* pSinkConnect = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetSinkOnIdx();

   Connection* pConnection = NULL;
   if (pSinkConnect != NULL)
   {
      if (pSource != NULL)
      {
         pConnection = getMainConnection(pSource, pSinkConnect->GetID());
      }
      else
      {
         ETG_TRACE_USR4_THR(("sourceDisconnect: Requested Source is NULL"));
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("sourceDisconnect: Requested Sink is NULL"));
   }

   if (pConnection != NULL)
   {
      sendDisconnectRequest(pConnection);
   }
   else
   {
      ETG_TRACE_USR4_THR(("sourceDisconnect: No Active Connection for requested Source %d and Sink %d ", nSrcIndex, nSinkId));
   }
}


bool ConnectionController::groupRequest(int nGrpIndex)
{
   bool ret = false;
   Source* pSource = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetGroupLastSrc(nGrpIndex);
   if ((pSource == NULL) || (pSource->GetAvailability() != RES_AVAILABLE))
   {
      ETG_TRACE_USR4_THR(("groupRequest: Last source not available in GroupIdx %d", nGrpIndex));
      pSource = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetAvailableSrc(nGrpIndex);
   }

   if (pSource == NULL)
   {
      ETG_TRACE_USR4_THR(("groupRequest: No source available in GroupIdx %d", nGrpIndex));
      actionOnNoSrc(nGrpIndex);
   }
   else
   {
      ETG_TRACE_USR4_THR(("groupRequest: Last source index %d with source Id %d available in GroupIdx %d", pSource->GetIndex(), pSource->GetID(), nGrpIndex));
      m_bGroupRequestActive = true;
      if (false == sourceRequest(pSource->GetIndex(), pSource->GetDeviceId()))
      {
         m_bGroupRequestActive = false;
         ret = true;
      }
   }

   return ret;
}


void ConnectionController::startConnectionListUpdate()
{
   std::map<int, Connection*>::iterator itr;
   for (itr = m_ConnectionMap.begin(); itr != m_ConnectionMap.end(); ++itr)
   {
      itr->second->setPurge(true);
   }
}


void ConnectionController::addMainConnection(int& nConnID, int& nSrcID, int& nSinkID, int& nDelay, int& nState)
{
   Source* pSource = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetSourceOnID(nSrcID);
   if (pSource == NULL)
   {
      ETG_TRACE_USR4_THR(("No source found for source ID %d nSrcID", nSrcID));
      return;
   }
   std::map<int, Connection*>::iterator itr;
   itr = m_ConnectionMap.find(nConnID);
   if (itr != m_ConnectionMap.end())
   {
      int ConnState = itr->second->getState();
      itr->second->updateConnectionData(nSrcID, nSinkID, nDelay, nState);
      ETG_TRACE_USR4_THR(("Update connection %d with SrcID %d, SinkID %d and state %d", nConnID, nSrcID, nSinkID, nState));
      if (ConnState != nState)
      {
         updateConnectionState(itr->second);
      }
      itr->second->setPurge(false);
   }
   else
   {
      ETG_TRACE_USR4_THR(("New connection %d with SrcID %d, SinkID %d and state %d", nConnID, nSrcID, nSinkID, nState));
      Sink* pSink = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetSinkOnID(nSinkID);
      if (pSink != NULL)
      {
         Connection* pMainConnection = new Connection(nConnID, pSource, pSink, nDelay, nState);
         m_ConnectionMap[nConnID] = pMainConnection;
         updateConnectionState(pMainConnection);
      }
   }
}


void ConnectionController::updateActiveConnections()
{
   ::std::vector<stSourceInfo> activeSources;
   bool sourceChangeInProgress = false;
   for (std::map<int, Connection*>::iterator itr = m_ConnectionMap.begin(); itr != m_ConnectionMap.end(); ++itr)
   {
      if (itr->second->IsConnectionStateAttained())
      {
         stSourceInfo sourceInfo;
         sourceInfo.srcId = static_cast<uint32>(itr->second->getSource()->GetIndex());
         sourceInfo.deviceId = static_cast<uint32>(itr->second->getSource()->GetDeviceId());
         sourceInfo.connectionState = static_cast<uint32>(itr->second->getState());

         //Get sink index on received id
         sourceInfo.sinkId = static_cast<uint16_t>(AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetIndexOnSinkID(itr->second->getSink()->GetID()));
         activeSources.push_back(sourceInfo);
      }
      if (itr->second->getState() == STATE_CONNECTING || itr->second->getState() == STATE_DISCONNECTING)
      {
         sourceChangeInProgress = true;
      }
   }
   if (m_pHmiAudioServiceStubHandler != 0 && sourceChangeInProgress == false)
   {
      m_pHmiAudioServiceStubHandler->sendSetActiveSources(activeSources);
   }
   onNewActiveConnections();
}


void ConnectionController::endConnectionListUpdate()
{
   std::vector<int> purgedList;
   for (std::map<int, Connection*>::iterator itr = m_ConnectionMap.begin(); itr != m_ConnectionMap.end(); ++itr)
   {
      if (itr->second->getPurge())
      {
         purgedList.push_back(itr->first);
      }
   }
   for (std::vector<int>::iterator itr = purgedList.begin(); itr != purgedList.end(); ++itr)
   {
      for (std::map<int, Connection*>::iterator itr_map = m_ConnectionMap.begin(); itr_map != m_ConnectionMap.end(); ++itr_map)
      {
         ETG_TRACE_USR4_THR(("Connection id %d (purge list) to be removed from list, %d (connection map)", *itr, itr_map->first));
         // Update the active source if Active source is disconnected physically or disconnected through system error
         if (*itr == itr_map->first)
         {
            ETG_TRACE_USR4_THR(("ConnectionController sendSetActiveSource sourceid %d SubsourceId %d sinkID %d connectionstate %d ", itr_map->second->getSource()->GetIndex(), itr_map->second->getSource()->GetDeviceId(), itr_map->second->getSink()->GetID(), itr_map->second->getState()));
            m_pHmiAudioServiceStubHandler->sendSetActiveSource(itr_map->second->getSource()->GetIndex(), itr_map->second->getSource()->GetDeviceId(), itr_map->second->getSink()->GetID(), STATE_DISCONNECTED);
         }
      }

      ETG_TRACE_USR4_THR(("Connection id %d removed from list", *itr));
      m_ConnectionMap.erase(*itr);
   }
   updateActiveConnections();
}


void ConnectionController::sendLastApplicationRequest()
{
// to be implemented in project specific derivation
}


void ConnectionController::sendSourceActivationRequest(Source* pSource, Sink* pSink, int nState)
{
   int nSrcIdx = static_cast<int>(SRC_INVALID);
   if (pSource != NULL)
   {
      nSrcIdx = pSource->GetIndex();
   }
   else
   {
      return;
   }

   int nGrpIdx = nSrcIdx / SRC_GROUP_SIZE;

   int nSinkIdx = static_cast<int>(SINK_INVALID);
   if (pSink != NULL)
   {
      nSinkIdx = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetIndexOnSinkID(pSink->GetID());
   }
   else
   {
      return;
   }

   if (m_pHmiAudioServiceStubHandler != 0)
   {
      m_pHmiAudioServiceStubHandler->sendSetActiveSource(pSource->GetIndex(), pSource->GetDeviceId(), nSinkIdx, static_cast<uint32>(nState));
   }
   else
   {
      HMI_APP_ASSERT_ALWAYS();
      return;
   }
   sendSourceActivationMsgRequest(nSrcIdx);

   evalApplicationStart(nGrpIdx, pSource);

   if (pSource->GetType() != static_cast<int>(SOURCE_ANNOUNCEMENT) && pSource->GetType() != static_cast<int>(SOURCE_MIX))
   {
      AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->SetGroupLastSrc(pSource);
   }
}


void ConnectionController::evalApplicationStart(int /*grpIndex*/, Source* /*src*/)
{
   // to be implemented in project specific derivation if needed
}


void ConnectionController::sendSourceActivationMsgRequest(int /*nSrcIndex*/)
{
   // to be implemented in project specific derivation
}


void ConnectionController::actionOnNoSrc(int /*nGrpIndex*/)
{
   // to be implemented in project specific derivation
}


Connection* ConnectionController::getMainConnection(const Source* pSource, uint16_t nSinkId)
{
   std::map<int, Connection*>::reverse_iterator itr;
   for (itr = m_ConnectionMap.rbegin(); itr != m_ConnectionMap.rend(); ++itr)
   {
      if ((itr->second->getSource() == pSource) && (itr->second->getSink()->GetID() == nSinkId))
      {
         ETG_TRACE_USR4_THR(("getMainConnection: connection found with connection Id %d, source Id %d and sink Id %d", itr->second->getConnectionID(), pSource->GetID(), nSinkId));
         return itr->second;
      }
   }
   return NULL;
}


Connection* ConnectionController::getActiveConnection(uint16_t nSinkId)
{
   std::map<int, Connection*>::iterator itr;
   ETG_TRACE_USR4_THR(("getActiveConnection %d", nSinkId));
   for (itr = m_ConnectionMap.begin(); itr != m_ConnectionMap.end(); ++itr)
   {
      ETG_TRACE_USR4_THR(("getActiveConnection source in Connection map  : %d ", itr->second->getSource()->GetID()));
      if ((itr->second->IsConnectionStateAttained()) && (itr->second->getSource()->GetType() == static_cast<int>(SOURCE_MAIN)) && (itr->second->getSink()->GetID() == nSinkId))
      {
         ETG_TRACE_USR4_THR(("getActiveConnection Active source : %d on sink %d", itr->second->getSource()->GetID(), (itr->second->getSink()->GetID())));
         return itr->second;
      }
   }
   ETG_TRACE_USR4_THR(("No Active connection on the sink"));
   return NULL;
}


}
}


}
