/* ***************************************************************************************
* FILE:          StartupSync.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  StartupSync.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "StartupSync.h"


StartupSync* StartupSync::_theInstance = 0;

StartupSync& StartupSync::getInstance()
{
   // lazy instantiation
   if (_theInstance == 0)
   {
      _theInstance = new StartupSync();
   }
   return *_theInstance;
}


StartupSync::StartupSync() : _guiReady(false)
{
}


StartupSync::~StartupSync()
{
   _theInstance = 0;
}


void StartupSync::registerPropertyRegistrationIF(PropertyRegistrationIF* propRegIF, const std::string& portName)
{
   _propertyRegistration.insert(std::pair<const std::string, PropertyRegistrationIF*>(portName, propRegIF));
}


void StartupSync::onAvailable(const ::boost::shared_ptr<asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (_guiReady)
   {
      notifyRegisteredClients(proxy, stateChange);
   }
   else
   {
      _pendingNotifications.insert(std::pair<boost::shared_ptr<asf::core::Proxy>, asf::core::ServiceStateChange>(proxy, stateChange));
   }
}


void StartupSync::onUnavailable(const ::boost::shared_ptr<asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (_guiReady)
   {
      notifyRegisteredClients(proxy, stateChange);
   }
   else
   {
      _pendingNotifications.erase(proxy);
   }
}


void StartupSync::onGuiStartupFinished()
{
   notifyRegisteredClients();
}


void StartupSync::notifyRegisteredClients()

{
   _guiReady = true;

   //CGI: After gui with SM is ready,post callbacks for services which are available.
   //Qt: After SM is ready, post callbacks for services which are available.
   std::map<boost::shared_ptr<asf::core::Proxy>, asf::core::ServiceStateChange>::const_iterator itr;
   for (itr = _pendingNotifications.begin(); itr != _pendingNotifications.end(); ++itr)
   {
      notifyRegisteredClients((*itr).first, (*itr).second);
   }
   _pendingNotifications.clear(); // clear all proxies once notified.
}


void StartupSync::notifyRegisteredClients(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (proxy)
   {
      typedef std::multimap<const std::string, PropertyRegistrationIF*>::const_iterator PropertyRegistrationITR;

      PropertyRegistrationITR clientitr = _propertyRegistration.find(proxy->getPortName());
      if (clientitr != _propertyRegistration.end())
      {
         std::pair<PropertyRegistrationITR, PropertyRegistrationITR> range = _propertyRegistration.equal_range((*clientitr).first);
         PropertyRegistrationITR rangeitr;

         for (rangeitr = range.first; rangeitr != range.second; ++rangeitr)
         {
            // only if the no previous registrations exists trigger to register
            if (stateChange.getCurrentState() == asf::core::ServiceState__Available)
            {
               if (stateChange.getPreviousState() == asf::core::ServiceState__Disconnected)
               {
                  (*rangeitr).second->registerProperties(proxy, stateChange);
               }
               else if (stateChange.getPreviousState() == asf::core::ServiceState__Suspended)
               {
                  (*rangeitr).second->onServiceResumed(proxy);
               }
            }
            else if (stateChange.getCurrentState() == asf::core::ServiceState__Disconnected)
            {
               (*rangeitr).second->deregisterProperties(proxy, stateChange);
            }
            else // ServiceState__Suspended
            {
               // previous registrations are still valid
               (*rangeitr).second->onServiceSuspended(proxy);
            }
         }
      }
   }
}
