/* ***************************************************************************************
* FILE:          ScreenBrokerClient.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScreenBrokerClient.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef VARIANT_S_FTR_ENABLE_SBCLIENT_MOCK

#include "gui_std_if.h"
#include "hmi_trace_if.h"

#include "AppBase/SystemConfiguration.h"
#include "ScreenBrokerClient.h"
#include "ScreenBroker/Client/Client.h"
#include "IAppViewSettings.h"
#include "ScreenBrokerSettings/Settings.h"
#include "AppBase/HmiAsfComponentBase.h"

#include "MsgFwAbstraction/IMessagePostObject.h"

#ifndef VARIANT_S_FTR_ENABLE_QT_APPLICATION
#include "AppBase/ScreenBrokerClient/KeyMappingBase.h"
#include "View/CGI/InputHandling/WindowInputEventHook.h"
#endif

#include "Trace/StartupInvestigation.h"
#include <algorithm>

#include "AppBase/ILM_Accessor.h"
// #include "BaseContract/generated/BaseTypes_SurfaceLayerIds.h"

//////// TRACE IF ///////////////////////////////////////////////////
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_SBCLIENT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ScreenBrokerClient.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


using namespace ::hmibase;
using namespace ::hmibase::util;


// ------------------------------------------------------------------------
ScreenBrokerClient::ScreenBrokerClient()
   : _mAppViewSettings(0),
     _mInitialized(false),
     _mAppId(0),
     _mMainSurfaceId(0),
     _mStatusBarSurfaceId((ScreenBroker::UInt32)hmibase::SURFACEID_STATUSBAR),
     _mFooterBarSurfaceId((ScreenBroker::UInt32)hmibase::SURFACEID_FOOTERBAR),
     _mVideoSurfaceId(0),
     _mScaling(ScreenBroker::Scaling::None),
     _mVideoSurfaceState(ScreenBroker::SurfaceState::Unmapped),
     _mStatusBarSurfaceState(ScreenBroker::SurfaceState::Unmapped),
     _mFooterBarSurfaceState(ScreenBroker::SurfaceState::Unmapped),
     _mInputEventHook(0)
{
#if defined VARIANT_S_FTR_ENABLE_IGNORE_DISPLAYID_IN_SINGLEDISPLAY_SETUP
   std::vector<unsigned int> screenIds;
   ILM_Accessor::getScreenIDs(screenIds);
   isMultiDisplaySetup = screenIds.size() > 1 ? true : false;
#endif
}


// ------------------------------------------------------------------------
ScreenBrokerClient& ScreenBrokerClient::GetInstance()
{
   // Lazy initialization of screen broker client thread singleton
   static ScreenBrokerClient lScreenBrokerClient;
   return lScreenBrokerClient;
}


// ------------------------------------------------------------------------
ScreenBroker::UInt32 ScreenBrokerClient::AppId() const
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::AppId failed, ScreenBrokerClient not yet initialized, return AppId: %d!", _mAppId));
   }

   return _mAppId;
}


// ------------------------------------------------------------------------
bool ScreenBrokerClient::IsMainSurface(ScreenBroker::UInt32 surfaceId)
{
   bool bFound = false;
   std::map<ScreenBroker::UInt32, ScreenBroker::SurfaceState::Enum>::const_iterator itr;

   hmibase::util::ScopedCriticalSection lock(_mCs);

   itr = _mMainSurfacesMap.find(surfaceId);
   if (itr != _mMainSurfacesMap.end())
   {
      bFound = true;
   }
   return bFound;
}


ScreenBroker::UInt32 ScreenBrokerClient::SurfaceId() const
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::SurfaceId failed, ScreenBrokerClient not yet initialized, return MainSurfaceId: %d!",
                         _mMainSurfaceId));
   }

   return _mMainSurfaceId;
}


// ------------------------------------------------------------------------
ScreenBroker::UInt32 ScreenBrokerClient::StatusBarSurfaceId() const
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::StatusBarSurfaceId failed, ScreenBrokerClient not yet initialized, return SystemPopupSurfaceId: %d!",
                         _mStatusBarSurfaceId));
   }

   return _mStatusBarSurfaceId;
}


// ------------------------------------------------------------------------
ScreenBroker::UInt32 ScreenBrokerClient::FooterBarSurfaceId() const
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::FooterBarSurfaceId failed, ScreenBrokerClient not yet initialized, return FooterBarSurfaceId: %d!",
                         _mFooterBarSurfaceId));
   }
   return _mFooterBarSurfaceId;
}


// ------------------------------------------------------------------------
ScreenBroker::UInt32 ScreenBrokerClient::VideoSurfaceId() const
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::VideoSurfaceId failed, ScreenBrokerClient not yet initialized, return VideoSurfaceId: %d!",
                         _mVideoSurfaceId));
   }

   return _mVideoSurfaceId;
}


// ------------------------------------------------------------------------
ScreenBroker::Scaling::Enum ScreenBrokerClient::Scaling() const
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::Scaling failed, ScreenBrokerClient not yet initialized, return Scaling: %d!", _mScaling));
   }

   return _mScaling;
}


bool ScreenBrokerClient::CheckSurfaceIdDisplayIdMapping(ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 displayId)
{
#if defined VARIANT_S_FTR_ENABLE_IGNORE_DISPLAYID_IN_SINGLEDISPLAY_SETUP
   // there is only one display so we can ignore the display id
   if (isMultiDisplaySetup == false)
   {
      return true;
   }
#endif

   ScreenBroker::UInt32 displayIdFromScreenBroker = GetDisplayID(surfaceId);

   return (displayIdFromScreenBroker == displayId);
}


// ------------------------------------------------------------------------
bool ScreenBrokerClient::InitClient(::asf::core::BaseComponent* baseComponent)
{
   // Initialize ScreenBroker client connection

   if (baseComponent == 0)
   {
      SYSTEMD_LOG("GUISTARTUP ERROR: ScreenBrokerClient::InitClient failed, baseComponent=0\n");
      HMI_APP_ASSERT_ALWAYS();
      _mInitialized = false;
   }
   else
   {
      _mInitialized = ScreenBroker::Client::GetInstance().InitClientApi(this, HmiAsfComponentBase::getAppName(), baseComponent);
   }
   SYSTEMD_LOG("GUISTARTUP(2): ScreenBrokerClient::InitClient %s\n", _mInitialized ? "succesfull" : "failed");
   return _mInitialized;
}


// ------------------------------------------------------------------------
bool ScreenBrokerClient::Init(ScreenBroker::UInt32 appId,
                              std::vector<unsigned int> MainSurfaces,
                              std::vector<unsigned int> VideoSurfaces,
                              std::vector<unsigned int> PopupSurfaces,
                              ScreenBroker::Scaling::Enum scaling,
                              const IAppViewSettings* appViewSettings,
                              WindowInputEventHook* inputEventHook)
{
   if ((0 == _mAppViewSettings) && (0 != appViewSettings))
   {
      if (_mInitialized)
      {
         _mAppId = appId;
         _mVideoSurfaceId = 0;
         _mScaling = scaling;
         _mAppViewSettings = appViewSettings;

         _mInputEventHook = inputEventHook;

         if (MainSurfaces.size() > 0)
         {
            _mMainSurfaceId = MainSurfaces[0]; // as a temporary implementation: to support backward compatibility
         }
         for (unsigned int i = 0; i < MainSurfaces.size(); ++i)
         {
            _mMainSurfacesMap[MainSurfaces[i]] = ScreenBroker::SurfaceState::Deregistered;
         }
         for (unsigned int i = 0; i < VideoSurfaces.size(); ++i)
         {
            _mVideoSurfacesMap[VideoSurfaces[i]] = ScreenBroker::SurfaceState::Deregistered;
         }
         for (unsigned int i = 0; i < PopupSurfaces.size(); ++i)
         {
            _mPopupSurfaceStatesMap[PopupSurfaces[i]] = ScreenBroker::SurfaceState::Deregistered;
         }

         ETG_TRACE_USR4_THR(("Initialize ScreenBroker Application's screenbroker config: AppId=%u, NrOfMainSurfaces=%u, NrOfPopupSurfaces=%d Scaling=%d",
                             (unsigned int)_mAppId,
                             MainSurfaces.size(),
                             PopupSurfaces.size(),
                             _mScaling));
      }
      else
      {
         ETG_TRACE_FATAL_THR(("ScreenBrokerClient failed to initialize"));
         HMI_APP_ASSERT_ALWAYS();
         return false;
      }
   }
   return true;
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::Terminate() const
{
   ScreenBroker::Client::GetInstance().Destroy();
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::Synchronize() const
{
   ScreenBroker::Client::GetInstance().Synchronize();
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::Show(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, bool bIsPopup, bool bForce, ScreenBroker::UInt32 displayId,
                              ScreenBroker::UInt32 entryCustomAnimationType,//AnimationType::None
                              ScreenBroker::UInt32 exitCustomAnimationType)//AnimationType::None)
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::Show(surface: %d) failed, ScreenBrokerClient not yet initialized!", surfaceId));
      return;
   }

   ScreenBroker::UInt32 lUserData = 0;

   if (bIsPopup)
   {
      surfaceId = _mAppViewSettings->GetPopupSurfaceId(viewId, displayId);
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::Show popup surfaceId: %d displayId: %d, viewId: %s, ", surfaceId, displayId, viewId.CStr()));

      std::string viewName(viewId.CStr());
      viewName.append(HmiAsfComponentBase::getAppName());
      ::hmibase::view::ViewIdentifier newviewId(viewName.c_str());

      lUserData = ScreenBroker::UInt32(newviewId.GetHashId());
      SetReverseViewHashMap(viewId, lUserData);

      ETG_TRACE_USR4_THR(("ScreenBrokerClient::Show popup lUserData__ %d ", lUserData));

      if (bForce)
      {
         ScreenBroker::PopupPresentationArg \
         args(1,
              ScreenBroker::Modality::System,
              0,
              0,
              0,
              ScreenBroker::HorizontalAlignment::Center,
              ScreenBroker::VerticalAlignment::Center,
              ScreenBroker::Scaling::None,
              1);

         ShowPopupPrivate(surfaceId,
                          lUserData,
                          _mAppViewSettings->GetShowPopupRequestId(viewId, displayId),
                          args);
      }
      else
      {
         ShowPopupPrivate(surfaceId,
                          lUserData,
                          _mAppViewSettings->GetShowPopupRequestId(viewId, displayId),
                          _mAppViewSettings->GetPopupPresentationArg(viewId, displayId));
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::Show surfaceId: %d", surfaceId));
      ScreenBrokerClient::GetInstance().ShowSurface(surfaceId,
            lUserData,
            _mAppViewSettings->GetShowSurfaceRequestId(viewId, displayId),
            entryCustomAnimationType,
            exitCustomAnimationType);
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::Hide(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, bool bIsPopup, ScreenBroker::UInt32 displayId,
                              ScreenBroker::UInt32 entryCustomAnimationType,//AnimationType::None
                              ScreenBroker::UInt32 exitCustomAnimationType)//AnimationType::None))
{
   if (0 == _mAppViewSettings)
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::Hide(surface: %d) failed, ScreenBrokerClient not yet initialized!", surfaceId));
      return;
   }

   ScreenBroker::UInt32 lUserData = 0;

   // Forward hide request
   if (bIsPopup)
   {
      surfaceId = _mAppViewSettings->GetPopupSurfaceId(viewId, displayId);
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::Hide popup surfaceId: %d viewId: %s", surfaceId, viewId.CStr()));

      std::string viewName(viewId.CStr());
      viewName.append(HmiAsfComponentBase::getAppName());
      ::hmibase::view::ViewIdentifier newviewId(viewName.c_str());

      lUserData = ScreenBroker::UInt32(newviewId.GetHashId());
      SetReverseViewHashMap(viewId, lUserData);

      ETG_TRACE_USR4_THR(("ScreenBrokerClient::Hide popup lUserData__ %d ", lUserData));

      HidePopupPrivate(surfaceId,
                       lUserData,
                       _mAppViewSettings->GetHidePopupRequestId(viewId, displayId));
   }
   else
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::Hide surfaceId: %d", surfaceId));

      ScreenBrokerClient::GetInstance().HideSurface(surfaceId,
            lUserData,
            _mAppViewSettings->GetHideSurfaceRequestId(viewId, displayId),
            entryCustomAnimationType,
            exitCustomAnimationType);
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::ShowSurface(ScreenBroker::UInt32 surfaceId,
                                     ScreenBroker::UInt32 userData,
                                     ScreenBroker::UInt32 requestId,
                                     ScreenBroker::UInt32 entryCustomAnimationType,//AnimationType::None
                                     ScreenBroker::UInt32 exitCustomAnimationType) const //AnimationType::None)
{
   if (0 != surfaceId)
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::ShowSurface surfaceId: %d userData: %u requestId: %d", surfaceId, userData, requestId));
      // Overwrite the reserved value to set the option in request ID of interface ActivateApplication
      // in order to show the surface (for more details see ScreenBrokerProtocol).
      ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ShowSurfaceMode::Enum> lRequestId(requestId);
      lRequestId.Set(ScreenBrokerProtocol::ShowSurfaceMode::Enum(0));
      ScreenBroker::RequestArg lRequestArg(lRequestId.Get(),
                                           _mAppId,
                                           userData);
      ActivateApplication(lRequestArg, surfaceId, entryCustomAnimationType, exitCustomAnimationType);
   }
}


void ScreenBrokerClient::StartAnimation(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 animationType)
{
   ETG_TRACE_USR4_THR(("ScreenBrokerClient::StartAnimation surfaceId: %d, AnimationType %u", surfaceId, animationType));
   ScreenBroker::UInt32 lUserData = 0;
   ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ShowSurfaceMode::Enum> lRequestId(_mAppViewSettings->GetShowSurfaceRequestId(viewId, static_cast<ScreenBroker::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY)));
   ScreenBroker::RequestArg lRequestArg(lRequestId.Get(),
                                        _mAppId,
                                        lUserData);
   ScreenBroker::ClientApi::StartAnimation(lRequestArg, surfaceId, animationType);
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::StartLayerAnimation(ScreenBroker::UInt32 layerId, std::vector<ScreenBroker::UInt32> surfaceList , ScreenBroker::UInt32 animationType)
{
   ETG_TRACE_USR4_THR(("ScreenBrokerClient::StartLayerAnimation Layer: %d, AnimationType %u", layerId, animationType));
   ScreenBroker::UInt32 lUserData = 0;
   ScreenBroker::UInt32 lRequestId = 0;
   ScreenBroker::RequestArg lRequestArg(lRequestId ,
                                        _mAppId,
                                        lUserData);
   ScreenBroker::ClientApi::StartLayerAnimation(lRequestArg, layerId, surfaceList, animationType);
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::HideSurface(ScreenBroker::UInt32 surfaceId,
                                     ScreenBroker::UInt32 userData,
                                     ScreenBroker::UInt32 requestId,
                                     ScreenBroker::UInt32 entryCustomAnimationType,//AnimationType::None
                                     ScreenBroker::UInt32 exitCustomAnimationType) const //AnimationType::None)
{
   if (0 != surfaceId)
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::HideSurface surfaceId: %d userData: %u requestId: %d", surfaceId, userData, requestId));
      // Overwrite the reserved value to clear the option in request ID of interface ActivateApplication
      // in order to hide the surface (for more details see ScreenBrokerProtocol).
      ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::HideSurfaceMode::Enum> lRequestId(requestId);
      lRequestId.Clear(ScreenBrokerProtocol::HideSurfaceMode::Enum(0));
      ScreenBroker::RequestArg lRequestArg(lRequestId.Get(),
                                           _mAppId,
                                           userData);
      ActivateApplication(lRequestArg, surfaceId, entryCustomAnimationType, exitCustomAnimationType);
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::ShowPopupPrivate(ScreenBroker::UInt32 surfaceId,
      ScreenBroker::UInt32 userData,
      ScreenBroker::UInt32 requestId,
      const ScreenBroker::PopupPresentationArg& popupPresentationArg) const

{
   if (0 != surfaceId)
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::ShowPopup surfaceId: %d userData: %u requestId: %d", surfaceId, userData, requestId));
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::ShowPopup popupPresentationArgs: prio: %d, modality: %d, presentTime: %d, minPresentTime: %d, validityPeriod: %d, hAlign: %d, vAlign: %d, scaling: %d ",
                          popupPresentationArg.Priority(), popupPresentationArg.Modality(), popupPresentationArg.PresentationTime(), popupPresentationArg.MinimumPresentationTime(),
                          popupPresentationArg.ValidityPeriod(), popupPresentationArg.HorizontalAlignment(), popupPresentationArg.VerticalAlignment(), popupPresentationArg.Scaling()));

      ScreenBroker::RequestArg lRequestArg(requestId,
                                           _mAppId,
                                           userData);

      ScreenBroker::ClientApi::ShowPopup(lRequestArg, surfaceId, popupPresentationArg);
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::HidePopupPrivate(ScreenBroker::UInt32 surfaceId,
      ScreenBroker::UInt32 userData,
      ScreenBroker::UInt32 requestId) const
{
   if (0 != surfaceId)
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::HidePopup surfaceId: %d userData: %u requestId: %d", surfaceId, userData, requestId));

      ScreenBroker::RequestArg lRequestArg(requestId,
                                           _mAppId,
                                           userData);
      ScreenBroker::ClientApi::HidePopup(lRequestArg, surfaceId);
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::OnSurfaceStateChanged(ScreenBroker::UInt32 userData,
      ScreenBroker::UInt32 surfaceId,
      ScreenBroker::UInt32 displayId,
      ScreenBroker::UInt32 displayAlias,
      ScreenBroker::SurfaceState::Enum surfaceState)
{
   int nState = (int)SURFACESTATE_INVISIBLE;
   ::hmibase::view::ViewIdentifier lViewId;

   //######################### find view ID ########################################
   if (ProjectSettings::IsPopup(surfaceId))
   {
      lViewId = GetReverseViewHashMap(userData);
      if ((lViewId.CStr() != 0) && (strcmp(lViewId.CStr(), "") != 0))
      {
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged hash value %u for surface %d maps to view '%s'", userData, surfaceId, lViewId.CStr()));
      }
      else
      {
         ETG_TRACE_ERR_THR(("ScreenBrokerClient::OnSurfaceStateChanged hash value %u for surface %d not found in view ID mapping table!", userData, surfaceId));
      }
   }
   else
   {
      // use empty viewID for main views, since it is not used by main views (default constructor is already called above)
      // reason is that the HMI-App does not call Show() for every new main view it wants to show, only once the application
      // gets activated/ gets in foreground.
   }

   switch (surfaceState)
   {
      case ScreenBroker::SurfaceState::AnimationStarted:
      {
         nState = (int)SURFACESTATE_ANIM_STARTED;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged Animation of surface %d started, surface state: %d (SurfaceState_AnimationStarted), userData: %u, viewId: %s \n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
         //while an animation is running, this surface/view shell not be touchable
         DisableTouch(lViewId, surfaceId, true);
         break;
      }
      case ScreenBroker::SurfaceState::AnimationStopped:
      {
         nState = (int)SURFACESTATE_ANIM_STOPPED;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged Animation of surface %d stopped, surface state: %d (SurfaceState_AnimationStopped), userData: %u, viewId: %s \n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
         //after animation is finished, enable touch again for that view/surface
         DisableTouch(lViewId, surfaceId, false);
         break;
      }
      case ScreenBroker::SurfaceState::Registered:
      {
         nState = (int)SURFACESTATE_REGISTERED;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged surface ID: %d, surface state: %d (SurfaceState_Registered), userData: %u, viewId: %s\n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
         SetDisplayID(surfaceId, displayId, displayAlias);
         break;
      }
      case ScreenBroker::SurfaceState::Deregistered:
      {
         nState = (int)SURFACESTATE_DEREGISTERED;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged surface ID: %d, surface state: %d (SurfaceState_Deregistered), userData: %u, viewId: %s \n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
         break;
      }
      case ScreenBroker::SurfaceState::Unqueued:
      {
         nState = (int)SURFACESTATE_UNQUEUED;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged surface ID: %d, surface state: %d (SurfaceState_Unqueued), userData: %u, viewId: %s \n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
         break;
      }
      case ScreenBroker::SurfaceState::Queued:
      {
         nState = (int)SURFACESTATE_QUEUED;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged surface ID: %d, surface state: %d (SurfaceState_Queued), userData: %u, viewId: %s \n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
         break;
      }
      case ScreenBroker::SurfaceState::Mapped:
      {
         static bool firstTimeMapped = false;
         if (!firstTimeMapped && (surfaceId > (ScreenBroker::UInt32)SURFACEID_APPLICATIONMAINBASE) && (surfaceId > (ScreenBroker::UInt32)SURFACEID_APPLICATIONMAINBASE_MAX))
         {
            firstTimeMapped = true;
            HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_MAIN_SURFACE_FIRST_TIME_VISIBLE);
         }
         hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<Courier::RenderReqMsg>());

         nState = (int)SURFACESTATE_VISIBLE;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged surface ID: %d, surface state: %d (SurfaceState_Visible), userData: %u, viewId: %s \n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
         break;
      }
      case ScreenBroker::SurfaceState::Unmapped:
      {
         nState = (int)SURFACESTATE_INVISIBLE;
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnSurfaceStateChanged surface ID: %d, surface state: %d (SurfaceState_Invisible), userData: %u, viewId: %s \n",
                             surfaceId, surfaceState, userData, lViewId.CStr()));
#ifndef VARIANT_S_FTR_ENABLE_QT_APPLICATION
         if (NULL != getWindowInputEventHook())
         {
            getWindowInputEventHook()->ResetInputKeyEvent(surfaceId);
         }
#endif
         break;
      }
      default:
         // nothing to do
         break;
   }

   //################### store surface state and send msg ##########################
   // Registered/Deregistered status cannot have valid user data, since viewID is not valid. Since the register/deregister request
   // comes only once per surface per registration/deregistration and is not connected to a view ID (only to a surface),
   //it is triggered by the GeniviDevicePackage (RegisterSurface()/DeregisterSurface()).

   if (surfaceState < ScreenBroker::SurfaceState::AnimationStarted)
   {
      // Custom surface states are not stored, since they are only valid for a very short time (while an animation is running)
      // furthermore after an animation the state would stay to "AnimationStopped", since the "Mapped" is send before
      //the animations starts
      if (IsMainSurface(surfaceId))
      {
         hmibase::util::ScopedCriticalSection lock(_mCs);
         _mMainSurfacesMap[surfaceId] = surfaceState;
      }
      if (surfaceId == _mStatusBarSurfaceId)
      {
         _mStatusBarSurfaceState = surfaceState;
      }

      if (surfaceId == _mFooterBarSurfaceId)
      {
         _mFooterBarSurfaceState = surfaceState;
      }

      if (surfaceId == _mVideoSurfaceId)
      {
         _mVideoSurfaceState = surfaceState;
      }
      if (ProjectSettings::IsPopup(surfaceId))
      {
         hmibase::util::ScopedCriticalSection lock(_mCs);
         if (IsInPopupSurfaceMap(surfaceId))
         {
            _mPopupSurfaceStatesMap[surfaceId] = surfaceState;
         }
      }
   }

   // for legacy check if display id or display alias to be send
   hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SurfaceStateChangedUpdMsg>(surfaceId, nState, GETVIEWID_FROM_STR(lViewId.CStr()), (getenv("SB_DISPLAY_ID_LEGACY") == 0) ? displayAlias : displayId));
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::OnCurrentStatus(ScreenBroker::UInt32 requestId,
      ScreenBroker::UInt32 surfaceId,
      ScreenBroker::UInt32 status,
      ScreenBroker::UInt32 userData)
{
   ::hmibase::view::ViewIdentifier lViewId;
   switch (requestId)
   {
      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::ActivePopups:
      {
         lViewId = GetReverseViewHashMap(userData);
         std::bitset<sizeof(status)> lStatus(status);
         ETG_TRACE_FATAL_CLS_THR((TR_CLASS_HMI_FW, "ActivePopups[%16s]: displayId %u, surfaceId %u, visible(%d) '%s'",
                                  hmibase::trace::getAppName().c_str(),
                                  GetDisplayID(surfaceId),
                                  surfaceId,
                                  lStatus.test(0),
                                  lViewId.CStr()));
         break;
      }

      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActivePopup:
      {
         lViewId = GetReverseViewHashMap(userData);
         std::bitset<sizeof(status)> lStatus(status);

         ETG_TRACE_FATAL_CLS_THR((TR_CLASS_HMI_FW, "TopActivePopup[%16s]: displayId %u, surfaceId %u, visible(%d) '%s'",
                                  hmibase::trace::getAppName().c_str(),
                                  GetDisplayID(surfaceId),
                                  surfaceId,
                                  lStatus.test(0),
                                  lViewId.CStr()));
         break;
      }
      case ScreenBrokerProtocol::CurrentStatusRequestId::TopActiveVisiblePopup:
      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveVisiblePopup:
      {
         lViewId = GetReverseViewHashMap(userData);
         std::bitset<sizeof(status)> lStatus(status);

         if (surfaceId != 0)
         {
            if (requestId == ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveVisiblePopup)
            {
               ETG_TRACE_FATAL_CLS_THR((TR_CLASS_HMI_FW, "TopActiveVisiblePopup[%16s]: displayId %u, surfaceId %u, visible(%d) '%s'",
                                        hmibase::trace::getAppName().c_str(),
                                        GetDisplayID(surfaceId),
                                        surfaceId,
                                        lStatus.test(0),
                                        lViewId.CStr()));
            }
            else
            {
               ETG_TRACE_USR4_CLS_THR((TR_CLASS_HMI_FW, "TopActiveVisiblePopup[%16s]: displayId %u, surfaceId %u, visible(%d) '%s'",
                                       hmibase::trace::getAppName().c_str(),
                                       GetDisplayID(surfaceId),
                                       surfaceId,
                                       lStatus.test(0),
                                       lViewId.CStr()));
            }
         }

         // send a response message back, so that the initial requester can collect responses from all processes and show only one trace at the end

         hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SbCurrentStatusUpdMsg>(hmibase::GET_CURRENT_VISIBLE_POPUP_VIEW, GetDisplayID(surfaceId), surfaceId, lStatus.test(0),
               GETVIEWID_FROM_STR(lViewId.CStr())));
         break;
      }

      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveSurfaceNonPermanent:
      {
         lViewId = GetReverseViewHashMap(userData);
         ETG_TRACE_FATAL_CLS_THR((TR_CLASS_HMI_FW, "TopActiveSurfaceNonPermanent[%16s]: displayId %u, surfaceId %u, '%s'",
                                  hmibase::trace::getAppName().c_str(),
                                  GetDisplayID(surfaceId),
                                  surfaceId,
                                  lViewId.CStr()));
         break;
      }
      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveSurface:
      {
         lViewId = GetReverseViewHashMap(userData);
         ETG_TRACE_FATAL_CLS_THR((TR_CLASS_HMI_FW, "TopActiveSurface[%16s]: displayId %u, surfaceId %u, '%s'",
                                  hmibase::trace::getAppName().c_str(),
                                  GetDisplayID(surfaceId),
                                  surfaceId,
                                  lViewId.CStr()));
         break;
      }

      case ScreenBrokerProtocol::CurrentStatusRequestId::PopupInForeground:
      {
         lViewId = GetReverseViewHashMap(userData);
         ETG_TRACE_USR4_CLS_THR((TR_CLASS_HMI_FW, "ForegroundPopup[%16s]: displayId %u, surfaceId %u, LayerId %u view '%s' ",
                                 hmibase::trace::getAppName().c_str(),
                                 GetDisplayID(surfaceId),
                                 surfaceId,
                                 status,
                                 lViewId.CStr()));

         hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<PopupDisplayStatusUpdMsg>(hmibase::Show, GetDisplayID(surfaceId), surfaceId, status, userData, GETVIEWID_FROM_STR(lViewId.CStr())));
         break;
      }

      case ScreenBrokerProtocol::CurrentStatusRequestId::PopupToBackground:
      {
         lViewId = GetReverseViewHashMap(userData);
         ETG_TRACE_USR4_CLS_THR((TR_CLASS_HMI_FW, "BackgroundPopup[%16s]: displayId %u, surfaceId %u, LayerId %u view '%s' ",
                                 hmibase::trace::getAppName().c_str(),
                                 GetDisplayID(surfaceId),
                                 surfaceId,
                                 status,
                                 lViewId.CStr()));

         hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<PopupDisplayStatusUpdMsg>(hmibase::Hide, GetDisplayID(surfaceId), surfaceId, status, userData, GETVIEWID_FROM_STR(lViewId.CStr())));
         break;
      }

      default:
         break;
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::OnLayerStateChanged(ScreenBroker::UInt32 requestId,
      ScreenBroker::UInt32 layerId,
      ScreenBroker::LayerState::Enum layerState)
{
   PARAM_UNUSED(requestId);
   ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnLayerStateChanged LayerId[%d], State: %d.", layerId, layerState));
   hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<LayerStateChangedUpdMsg>(layerId, layerState));
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::SetPopupFilterAll(ScreenBroker::UInt32 displayID)
{
   ScreenBroker::PopupPresentationArg lFilter;

   ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetPopupFilterAll called, appID: %d.", _mAppId));
   SetPopupFilterPrivate(true, displayID, lFilter);
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::SetPopupFilterPrio(ScreenBroker::UInt32 prio, ScreenBroker::UInt32 displayID)
{
   ScreenBroker::PopupPresentationArg lFilter;
   lFilter.SetPriority(prio);

   ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetPopupFilterPrio with prio = %d called, appID: %d.", prio, _mAppId));
   SetPopupFilterPrivate(false, displayID, lFilter);
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::SetPopupFilterPrivate(bool disableAll,
      ScreenBroker::UInt32 displayID,
      const ScreenBroker::PopupPresentationArg& filter)
{
   ScreenBroker::UInt32 lRequestId = 0;
   if (displayID == hmibase::DEFAULT_UNKNOWN_DISPLAY)
   {
      displayID = GetDisplayID(SurfaceId());
   }
   ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetPopupFilterPrivate request processed, appID: %d. displayID: %d", _mAppId, displayID));
   ScreenBroker::RequestArg lRequestArg(lRequestId,
                                        _mAppId,
                                        displayID);
   ScreenBroker::ClientApi::SetPopupFilter(lRequestArg, disableAll, filter);
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::ClearPopupFilterPublic(ScreenBroker::UInt32 displayID)
{
   if (displayID == hmibase::DEFAULT_UNKNOWN_DISPLAY)
   {
      displayID = GetDisplayID(SurfaceId());
   }
   ScreenBroker::UInt32 lRequestId = 0;
   ScreenBroker::RequestArg lRequestArg(lRequestId ,
                                        _mAppId,
                                        displayID);

   ScreenBroker::ClientApi::ClearPopupFilter(lRequestArg);
   ETG_TRACE_USR4_THR(("ScreenBrokerClient::ClearPopupFilterPublic request processed, appID: %d. displayID: %d", _mAppId, displayID));
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::AbortAnimations(void) const
{
   ScreenBroker::UInt32 actionData = 0;
   ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ActionId::Enum> actionId(static_cast<ScreenBroker::UInt32>(ScreenBrokerProtocol::ActionId::StopAnimations));

   //Abort all running/queued animations
   ScreenBroker::ClientApi::Action(actionId.Get(), actionData);

   ETG_TRACE_USR4_THR(("ScreenBrokerClient::AbortAnimations request processed, actionId: %d, actionData: %d.", actionId.Get(), actionData));
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::ExplicitSBCommit(void) const
{
   ScreenBroker::UInt32 actionData = 0;
   ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ActionId::Enum> actionId(static_cast<ScreenBroker::UInt32>(ScreenBrokerProtocol::ActionId::Commit));

   //Abort all running/queued animations
   ScreenBroker::ClientApi::Action(actionId.Get(), actionData);

   ETG_TRACE_USR4_THR(("ScreenBrokerClient::ExplicitSBCommit request processed, actionId: %d, actionData: %d.", actionId.Get(), actionData));
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::SetInputFocusPublic(ScreenBroker::UInt32 surfaceId,
      bool keyboardFocus,
      bool pointerFocus,
      ScreenBroker::Int32 priority,
      ScreenBroker::UInt32 focusData) const
{
   if (0 != surfaceId)
   {
      ScreenBroker::ClientApi::SetInputFocus(surfaceId, keyboardFocus, pointerFocus, priority, focusData);
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetInputFocusPublic request processed, surfaceId: %d, keyboardFocus: %s.",
                          surfaceId, keyboardFocus ? "true" : "false"));
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetInputFocusPublic request processed, surfaceId: %d, pointerFocus: %s.",
                          surfaceId, pointerFocus ? "true" : "false"));
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::SetInputFocusPublic(ScreenBroker::UInt32 surfaceId,
      bool keyboardFocus,
      bool pointerFocus) const
{
   if (0 != surfaceId)
   {
      ScreenBroker::ClientApi::SetInputFocus(surfaceId, keyboardFocus, pointerFocus, 0, 0);
   }
}


// ------------------------------------------------------------------------
bool ScreenBrokerClient::DisableTouch(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, bool animRunning)
{
   bool bReturn = false;

   if (ProjectSettings::IsPopup(surfaceId))
   {
      //send popup animation status to Controller, it will then disable the popup view
      //this can be done, since we have access to the view ID for popups
      //if we would disable the touch acceptance for the popup surface at ILM, the touches would
      //be send to the underlying main view surface, this behaviour is not wanted
      bReturn = SendPopupAnimStatus(viewId, animRunning);
   }
   else
   {
      //enable/disable touch input acceptance for main view surface
      //this is done, since we have no access to the view IDs of main views here
      //since not for every main view switch a Show() is called by the application
      //only when the application comes active, a Show() is called.
      bReturn = SetTouchAcceptance(surfaceId, !animRunning);
   }

   return bReturn;
}


// ------------------------------------------------------------------------
bool ScreenBrokerClient::SetTouchAcceptance(ScreenBroker::UInt32 surfaceId, bool enable)
{
   bool bChangeNeeded = true;
   struct ilmSurfaceProperties surfaceProperties;

   bool bReturn = hmibase::ILM_Accessor::getSurfaceProperties(surfaceId, &surfaceProperties);

   if (!enable)
   {
      //Animation is started
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
      if (!(surfaceProperties.focus & ILM_INPUT_DEVICE_TOUCH))
#else
      if (!(surfaceProperties.inputDevicesAcceptance & ILM_INPUT_DEVICE_TOUCH))
#endif
      {
         //The touch input acceptance shell be disabled, but it is already disabled
         //seems to be that this surface does not accepts touch at all, so do nothing
         //and remember this for the call with enable=true for this surface.
         bChangeNeeded = false;
         _mNoTouchSurfaceVec.push_back(surfaceId);
      }
   }
   else
   {
      //Animation is stopped
      std::vector<ScreenBroker::UInt32>::iterator itr;
      itr =  std::find(_mNoTouchSurfaceVec.begin(), _mNoTouchSurfaceVec.end(), surfaceId);

      if (itr != _mNoTouchSurfaceVec.end())
      {
         //found surface, so touch acceptance was disabled for this surface before animation was started
         //so do not enable it now
         bChangeNeeded = false;
         _mNoTouchSurfaceVec.erase(itr);
      }
   }

   if (bChangeNeeded)
   {
      bReturn = bReturn && hmibase::ILM_Accessor::updateInputAcceptance(surfaceId, ILM_INPUT_DEVICE_TOUCH, enable);

      if (bReturn)
      {
         if (enable)
         {
            ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetTouchAcceptance for surface %d to true, app=%d.",
                                surfaceId, _mAppId));
         }
         else
         {
            ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetTouchAcceptance for surface %d to false, app=%d.",
                                surfaceId, _mAppId));
         }
      }
      else
      {
         if (enable)
         {
            ETG_TRACE_ERR_THR(("ScreenBrokerClient::SetTouchAcceptance for surface %d to true failed, app=%d!",
                               surfaceId, _mAppId));
         }
         else
         {
            ETG_TRACE_ERR_THR(("ScreenBrokerClient::SetTouchAcceptance for surface %d to false failed, app=%d!",
                               surfaceId, _mAppId));
         }
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::SetTouchAcceptance for surface %d ignored, since it does not accept touch input at all, app=%d.",
                          surfaceId, _mAppId));
   }

   return bReturn;
}


// ------------------------------------------------------------------------

bool ScreenBrokerClient::SendPopupAnimStatus(const ::hmibase::view::ViewIdentifier& viewId, bool animRunning)
{
   bool bReturn = false;

   if ((viewId.CStr() != 0) && (strcmp(viewId.CStr(), "") != 0))
   {
      //send popup anim status to AppControllerBase which will then send the corresponding ActivateReqMsg.
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SbAnimationStateChangedUpdMsg>(animRunning, GETVIEWID_FROM_STR(viewId.CStr())));
      if (animRunning)
      {
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::SendPopupAnimStatus with animRunning=true in app=%d for view: '%s'.",
                             _mAppId, viewId.CStr()));
      }
      else
      {
         ETG_TRACE_USR4_THR(("ScreenBrokerClient::SendPopupAnimStatus with animRunning=false in app=%d for view: '%s'.",
                             _mAppId, viewId.CStr()));
      }
      bReturn = true;
   }
   else
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::SendPopupAnimStatus view ID is empty (app=%d)! So touch for this popup view cannot be disabled during animation!",
                         _mAppId));
   }
   return bReturn;
}


// ------------------------------------------------------------------------------------------------
bool ScreenBrokerClient::IsInPopupSurfaceMap(ScreenBroker::UInt32 surfaceId)
{
   bool bFound = false;
   std::map<ScreenBroker::UInt32, ScreenBroker::SurfaceState::Enum>::const_iterator itr;
   itr = _mPopupSurfaceStatesMap.find(surfaceId);
   if (itr != _mPopupSurfaceStatesMap.end())
   {
      bFound = true;
   }

   if (bFound)
   {
      ETG_TRACE_USR4_THR(("ScreenBrokerClient::IsInPopupSurfaceMap surface ID found in popup surface map: %d.", surfaceId));
   }
   else
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::IsInPopupSurfaceMap surface ID NOT found in popup surface map: %d!", surfaceId));
   }

   return bFound;
}


// ------------------------------------------------------------------------------------------------
ScreenBroker::SurfaceState::Enum ScreenBrokerClient::GetMainSurfaceState(ScreenBroker::UInt32 surfaceId)
{
   ScreenBroker::SurfaceState::Enum bReturnState = ScreenBroker::SurfaceState::Deregistered;
   if (IsMainSurface(surfaceId))
   {
      hmibase::util::ScopedCriticalSection lock(_mCs);
      bReturnState = _mMainSurfacesMap[surfaceId];
   }
   else
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::GetMainSurfaceState failed since this surface is not a Main surface: %d! app=%d",
                         surfaceId, _mAppId));
   }
   return bReturnState;
}


ScreenBroker::SurfaceState::Enum ScreenBrokerClient::GetPopupSurfaceState(ScreenBroker::UInt32 surfaceId)
{
   ScreenBroker::SurfaceState::Enum bReturnState = ScreenBroker::SurfaceState::Unmapped;
   hmibase::util::ScopedCriticalSection lock(_mCs);

   if (IsInPopupSurfaceMap(surfaceId))
   {
      bReturnState = _mPopupSurfaceStatesMap[surfaceId];
   }
   else
   {
      ETG_TRACE_ERR_THR(("ScreenBrokerClient::GetPopupSurfaceState failed since this surface is no popup surface: %d! app=%d",
                         surfaceId, _mAppId));
   }

   return bReturnState;
}


void ScreenBrokerClient::SendClosePopupOnExternalTouch(ScreenBroker::UInt32 surfaceId)
{
   ScreenBroker::ClientApi::CloseOnExternalTouch(ScreenBroker::RequestArg(0, _mAppId, surfaceId));
   ETG_TRACE_USR4_THR(("ScreenBrokerClient::SendClosePopupOnExternalTouch request processed, appID: %d, SurfaceId:%u.", _mAppId, surfaceId));
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::OnCloseOnExternalTouch(bool status)
{
   hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SetClosePopupExternalTouchMsg>(status));
   ETG_TRACE_USR4_THR(("ScreenBrokerClient::OnCloseOnExternalTouch status %d.", status));
}


// ------------------------------------------------------------------------
ScreenBroker::UInt32 ScreenBrokerClient::GetDisplayID(ScreenBroker::UInt32 surfaceId)
{
   hmibase::util::ScopedCriticalSection lock(_mCs);
   const ScreenBroker::UInt32* displayid = SurfaceDisplayMap::getItem(surfaceId);
   if (displayid)
   {
      return *(SurfaceDisplayMap::getItem(surfaceId));
   }
   return ::hmibase::DEFAULT_UNKNOWN_DISPLAY;
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::SetDisplayID(ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 displayId, ScreenBroker::UInt32 displayAlias)
{
   hmibase::util::ScopedCriticalSection lock(_mCs);
   const char* env = getenv("SB_DISPLAY_ID_LEGACY");
   if (env)
   {
      SurfaceDisplayMap::registerItem(surfaceId, displayId);
   }
   else
   {
      SurfaceDisplayMap::registerItem(surfaceId, displayAlias);
   }
}


// ------------------------------------------------------------------------
::hmibase::view::ViewIdentifier& ScreenBrokerClient::GetReverseViewHashMap(ScreenBroker::UInt32 userData)
{
   hmibase::util::ScopedCriticalSection lock(_mCs);
   return _mReverseViewHashMap[userData];
}


// ------------------------------------------------------------------------

void ScreenBrokerClient::SetReverseViewHashMap(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 lUserData)
{
   hmibase::util::ScopedCriticalSection lock(_mCs);
   _mReverseViewHashMap[lUserData] = viewId;
}


// ------------------------------------------------------------------------
void ScreenBrokerClient::GetSurfaces(const ScreenBroker::UInt32& displayId, std::list<ScreenBroker::UInt32>* surfaceList)
{
   SurfaceDisplayMap::getIds(displayId, surfaceList);
}


#else // VARIANT_S_FTR_ENABLE_SBCLIENT_MOCK

/*************************************************************************************************************/
/*   ScreenBrokerClient_MOCK                                                                                 */
/*************************************************************************************************************/

#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"

#include "MsgFwAbstraction/IMessagePostObject.h"

#ifndef VARIANT_S_FTR_ENABLE_QT_APPLICATION
#include "View/CGI/CgiExtensions/PopupConfig.h"
#else
#include "View/Qt/QtExtensions/PopupConfig.h"
#endif

#ifdef WIN32
#include "AppBase/Platform/Win32/WindowInputEventHook.h"
#endif

ScreenBrokerClient::ScreenBrokerClient()
   : _mAppViewSettings(0),
     _mInitialized(false),
     _mAppId(0),
     _mMainSurfaceId(0),
     _mStatusBarSurfaceId((ScreenBroker::UInt32)hmibase::SURFACEID_STATUSBAR),
     _mFooterBarSurfaceId((ScreenBroker::UInt32)hmibase::SURFACEID_FOOTERBAR),
     _mVideoSurfaceId(0),
     _mScaling(ScreenBroker::Scaling::None),
     _mVideoSurfaceState(ScreenBroker::SurfaceState::Mapped),
     _mStatusBarSurfaceState(ScreenBroker::SurfaceState::Mapped),
     _mFooterBarSurfaceState(ScreenBroker::SurfaceState::Mapped),
     _mInputEventHook(0)
{
}


ScreenBrokerClient& ScreenBrokerClient::GetInstance()
{
   // ensure correct initialization / cleanup order of singletons
   // static_cast<void>(::Util::Timer::TimerThreaded::createTimerThread());
   // Lazy initialization of screen broker client thread singleton
   static ScreenBrokerClient lScreenBrokerClient;
   return lScreenBrokerClient;
}


///
ScreenBroker::UInt32 ScreenBrokerClient::SurfaceId() const
{
   return _mMainSurfaceId;
}


bool ScreenBrokerClient::IsMainSurface(ScreenBroker::UInt32 surfaceId)
{
   return (_mMainSurfaceId == surfaceId);
}


ScreenBroker::UInt32  ScreenBrokerClient::GetDisplayID(ScreenBroker::UInt32 surfaceId)
{
   PARAM_UNUSED(surfaceId);
   return 0;
}


ScreenBroker::UInt32 ScreenBrokerClient::StatusBarSurfaceId() const
{
   return _mStatusBarSurfaceId;
}


ScreenBroker::UInt32 ScreenBrokerClient::FooterBarSurfaceId() const
{
   return _mFooterBarSurfaceId;
}


bool ScreenBrokerClient::CheckSurfaceIdDisplayIdMapping(ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 displayId)
{
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(displayId);

   return true;
}


bool ScreenBrokerClient::Init(ScreenBroker::UInt32 appId,
                              std::vector<unsigned int> MainSurfaces,
                              std::vector<unsigned int> VideoSurfaces,
                              std::vector<unsigned int> PopupSurfaces,
                              ScreenBroker::Scaling::Enum scaling,
                              const IAppViewSettings* appViewSettings,
                              WindowInputEventHook* inputEventHook)
{
   PARAM_UNUSED(appId);
   PARAM_UNUSED(VideoSurfaces);
   PARAM_UNUSED(PopupSurfaces);
   PARAM_UNUSED(scaling);
   PARAM_UNUSED(appViewSettings);
   PARAM_UNUSED(inputEventHook);
   if (MainSurfaces.size() > 0)
   {
      _mMainSurfaceId = MainSurfaces[0]; // as a temporary implementation: to support backward compatibility
   }

   _mStatusBarSurfaceId = 0;  // todo,
   _mFooterBarSurfaceId = (uint32_t)hmibase::SURFACEID_FOOTERBAR;
   return true;
}


///
void ScreenBrokerClient::Show(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, bool bIsPopup, bool bForce, ScreenBroker::UInt32 displayId,
                              ScreenBroker::UInt32 /*entryCustomAnimationType*/,
                              ScreenBroker::UInt32 /*exitCustomAnimationType*/)
{
   PARAM_UNUSED(bForce);

   if (bIsPopup)
   {
      //ST_POPUP_CONFIG* PopupConfigLoc = FindPopupConfig(ScreenBroker::ViewId(viewId.CStr()), displayId);
      hmibase::view::ST_POPUP_CONFIG* PopupConfigLoc = hmibase::view::PopUpConfig::FindPopupConfig(viewId, displayId);
      if (PopupConfigLoc)
      {
         surfaceId = PopupConfigLoc->surfaceID;
      }
      // update popup table
      std::map<unsigned int, std::vector<::hmibase::view::ViewIdentifier>>::iterator it = popupsOnSurface.find(surfaceId);
      if (it != popupsOnSurface.end())
      {
         bool alreadyIn = false;
         for (std::vector<::hmibase::view::ViewIdentifier>::iterator it2 = it->second.begin(); it2 != it->second.end(); it2++)
         {
            if (*it2 == viewId)
            {
               alreadyIn = true;;
               break;
            }
         }
         if (alreadyIn == false)
         {
            it->second.push_back(viewId);
         }
      }
      else
      {
         std::vector<::hmibase::view::ViewIdentifier> v;
         v.push_back(viewId);
         popupsOnSurface[surfaceId] = v;
      }

      if (PopupConfigLoc && PopupConfigLoc->presTime > 0)
      {
         // time based popup, so create a timer to close the popup after presentation time elapsed
         _popupTimer.setName("ScreenBrokerClient", "PopupTimer");
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
         _popupTimer.setTimeout(0, PopupConfigLoc->presTime, COURIER_MESSAGE_NEW(SBPopupTimerExpiryUpdMsg)(surfaceId, (int)hmibase::SURFACESTATE_INVISIBLE, Courier::ViewId(viewId.CStr())));
         _popupTimer.start();
#else
         // todo : Qt
#endif
      }
   }
   updateSurfaceState(surfaceId, (int)hmibase::SURFACESTATE_VISIBLE, viewId);
}


///
void ScreenBrokerClient::Hide(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, bool bIsPopup, ScreenBroker::UInt32 displayId,
                              ScreenBroker::UInt32 /*entryCustomAnimationType*/,
                              ScreenBroker::UInt32 /*exitCustomAnimationType*/)
{
   if (bIsPopup)
   {
      //ST_POPUP_CONFIG* PopupConfigLoc = FindPopupConfig(ScreenBroker::ViewId(viewId.CStr()), displayId);
      hmibase::view::ST_POPUP_CONFIG* PopupConfigLoc = hmibase::view::PopUpConfig::FindPopupConfig(viewId, displayId);
      if (PopupConfigLoc)
      {
         surfaceId = PopupConfigLoc->surfaceID;
      }
      std::map<unsigned int, std::vector<::hmibase::view::ViewIdentifier>>::iterator it = popupsOnSurface.find(surfaceId);
      if (it != popupsOnSurface.end())
      {
         for (std::vector<::hmibase::view::ViewIdentifier>::iterator it2 = it->second.begin(); it2 != it->second.end(); it2++)
         {
            if (*it2 == viewId)
            {
               it->second.erase(it2);
               break;
            }
         }
      }
   }

   updateSurfaceState(surfaceId, (int)hmibase::SURFACESTATE_INVISIBLE, viewId);
}


/// Set input focus for a certain surface of this HMI-App with focus priority
void ScreenBrokerClient::SetInputFocusPublic(ScreenBroker::UInt32 /*surfaceId*/,
      bool /*keyboardFocus*/,
      bool /*pointerFocus*/,
      ScreenBroker::Int32 /*priority*/,
      ScreenBroker::UInt32 /*focusData*/) const
{
   // todo
}


/// Set input focus for a certain surface of this HMI-App without focus priority
void ScreenBrokerClient::SetInputFocusPublic(ScreenBroker::UInt32 /*surfaceId*/,
      bool /*keyboardFocus*/,
      bool /*pointerFocus*/) const
{
   // todo
}


void ScreenBrokerClient::GetSurfaces(const ScreenBroker::UInt32& displayId, std::list<ScreenBroker::UInt32>* surfaceList)
{
   SurfaceDisplayMap::getIds(displayId, surfaceList);
}


bool ScreenBrokerClient::hasPopup(unsigned int surfaceId)
{
   std::map<unsigned int, std::vector<::hmibase::view::ViewIdentifier>>::iterator it = popupsOnSurface.find(surfaceId);
   if (it != popupsOnSurface.end())
   {
      if (it->second.empty())
      {
         return false;
      }
      else
      {
         return true;
      }
   }
   return false;
}


void ScreenBrokerClient::updateSurfaceState(ScreenBroker::UInt32 surfaceId, int state, const ::hmibase::view::ViewIdentifier& viewId)
{
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
   bool isVisible;
   if (surfaceId < hmibase::SURFACEID_APPLICATIONMAINBASE_MAX)
   {
      isVisible = state == hmibase::SURFACESTATE_VISIBLE;
   }
   else
   {
      isVisible = hasPopup(surfaceId);
   }
   hmibase::app::base::WindowInputEventHook::updateSurfaceStatus(surfaceId, isVisible);
#endif

   if (state == hmibase::SURFACESTATE_VISIBLE)
   {
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<Courier::RenderReqMsg>());
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SurfaceStateChangedUpdMsg>(surfaceId, (int)hmibase::SURFACESTATE_VISIBLE, GETVIEWID_FROM_STR(viewId.CStr()), 0));
   }
   if (state == hmibase::SURFACESTATE_INVISIBLE)
   {
      hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<SurfaceStateChangedUpdMsg>(surfaceId, (int)hmibase::SURFACESTATE_INVISIBLE, GETVIEWID_FROM_STR(viewId.CStr()), 0));
   }
}


ScreenBroker::SurfaceState::Enum ScreenBrokerClient::GetMainSurfaceState(ScreenBroker::UInt32 surfaceId)
{
   PARAM_UNUSED(surfaceId);
   return ScreenBroker::SurfaceState::Mapped;;
}


void ScreenBrokerClient::AbortAnimations() const
{
}


void ScreenBroker::ClientApi::OnGetDisplayId(const ScreenBroker::RequestArg& requestArg, ScreenBroker::UInt32 displayId)
{
   PARAM_UNUSED(requestArg);
   PARAM_UNUSED(displayId);
}


void ScreenBroker::ClientApi::OnGetPreferredDimensions(const ScreenBroker::RequestArg& requestArg, const ScreenBroker::DimensionArg& dimensionArg)
{
   PARAM_UNUSED(requestArg);
   PARAM_UNUSED(dimensionArg);
}


void ScreenBroker::ClientApi::OnSurfaceStateChanged(ScreenBroker::UInt32 userData, ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 displayId, ScreenBroker::UInt32 displayAlias, ScreenBroker::SurfaceState::Enum surfaceState)
{
   PARAM_UNUSED(userData);
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(displayId);
   PARAM_UNUSED(displayAlias);
   PARAM_UNUSED(surfaceState);
}


void ScreenBroker::ClientApi::OnError(const ScreenBroker::RequestArg& requestArg, ScreenBroker::Int32 systemErrno, ScreenBroker::Int32 presentationErrno)
{
   PARAM_UNUSED(requestArg);
   PARAM_UNUSED(systemErrno);
   PARAM_UNUSED(presentationErrno);
}


void ScreenBroker::ClientApi::OnKeyEvent(ScreenBroker::UInt32 userData, ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 keyCode, KeyState::Enum keyState)
{
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(userData);
   PARAM_UNUSED(keyCode);
   PARAM_UNUSED(keyState);
}


void ScreenBroker::ClientApi::OnCurrentStatus(ScreenBroker::UInt32 requestId, ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 status, ScreenBroker::UInt32 userData)
{
   PARAM_UNUSED(requestId);
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(status);
   PARAM_UNUSED(userData);
}


void ScreenBroker::ClientApi::OnLayerStateChanged(ScreenBroker::UInt32 requestId, ScreenBroker::UInt32 layerId, ScreenBroker::LayerState::Enum layerState)
{
   PARAM_UNUSED(requestId);
   PARAM_UNUSED(layerId);
   PARAM_UNUSED(layerState);
}


void ScreenBroker::ClientApi::OnCloseOnExternalTouch(bool status)
{
   PARAM_UNUSED(status);
}


void ScreenBroker::ClientApi::OnBuildInLayerInfo(const std::vector<BuildInLayerInfo>& info)
{
   PARAM_UNUSED(info);
}


void ScreenBroker::ClientApi::OnServiceStatusChanged(bool available)
{
   PARAM_UNUSED(available);
}


void ScreenBrokerClient::SetPopupFilterAll(ScreenBroker::UInt32 displayID)
{
   PARAM_UNUSED(displayID);
}


void ScreenBrokerClient::SetPopupFilterPrio(ScreenBroker::UInt32 prio, ScreenBroker::UInt32 displayID)
{
   PARAM_UNUSED(prio);
   PARAM_UNUSED(displayID);
}


void ScreenBrokerClient::ClearPopupFilterPublic(ScreenBroker::UInt32 displayID)
{
   PARAM_UNUSED(displayID);
}


::ScreenBroker::UInt32 ScreenBrokerClient::AppId() const
{
   return 0;
}


bool ScreenBrokerClient::InitClient(::asf::core::BaseComponent* baseComponent)
{
   PARAM_UNUSED(baseComponent);

   _mInitialized = true;
   return _mInitialized;
}


#endif
