/* ***************************************************************************************
* FILE:          ILM_Accessor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ILM_Accessor.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include <hmibase/sys_std_if.h>
#include "AppBase/ILM_Accessor.h"
#include "AppUtils/HmiMacros.h"

#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
// _TODO intermediate solution for Qt TGW
typedef bool ilmErrorTypes;
#undef VARIANT_S_FTR_ENABLE_IVI_SHELL
#else
#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include <ilm/ilm_input.h>

#ifdef VARIANT_S_FTR_ENABLE_IVI_OUTPUT_CONTROL
#include "ivi-output-control-client-protocol.h"
#endif

#endif


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ILM_Accessor.cpp.trc.h"
#endif

namespace hmibase {

//****** ivi-output-control helper
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL

#ifdef VARIANT_S_FTR_ENABLE_IVI_OUTPUT_CONTROL
class IviOutputControlHelper
{
   public:

      typedef struct t_outputcontrol_context
      {
         struct wl_display* display;
         struct wl_registry* registry;
         struct wl_list list_screen;
         struct ivi_output_control* controller;

         t_outputcontrol_context() : display(0), registry(0), controller(0)
         {
         }
      } outputcontrol_context;

      struct screen_context
      {
         struct wl_list link;
         struct wl_output* output;
         unsigned int name;
         unsigned int caps;
         char connectorName[256];
         struct ivi_output_control_screen* screen_controller;

         screen_context() : output(0), name(0), caps(0), screen_controller(0)
         {
         }
      };

      static IviOutputControlHelper& GetInstance()
      {
         static IviOutputControlHelper instance;
         return instance;
      }
      bool Init(void* nativeHandle);
      void Deinit();
      bool IsInitialized()
      {
         return s_isInitialized;
      }

      outputcontrol_context& GetOutputControlContext()
      {
         return s_ctx;
      }

   private:

      IviOutputControlHelper()
      {
      }

      static const struct wl_registry_listener* GetRegistryEventHandler();
      static struct ivi_output_control_screen_listener* GetScreenListener();

      static void RegistryEventHandler(void* data,
                                       struct wl_registry* registry,
                                       uint32_t name,
                                       const char* interface,
                                       uint32_t version);
      static void RegistryEventHandlerRemove(void* data,
                                             struct wl_registry* registry,
                                             uint32_t name);

      static void ScreenCapabilities(void* data,
                                     struct ivi_output_control_screen* ivi_outputcontrol_screen,
                                     uint32_t capabilities);
      static void ScreenName(void* data,
                             struct ivi_output_control_screen* ivi_outputcontrol_screen,
                             const char* name);

      static outputcontrol_context s_ctx;
      static bool s_isInitialized;
};


bool IviOutputControlHelper::s_isInitialized = false;
IviOutputControlHelper::outputcontrol_context IviOutputControlHelper::s_ctx;


bool IviOutputControlHelper::Init(void* nativeHandle)
{
   if (s_isInitialized)
   {
      return true;
   }

   if (nativeHandle == 0)
   {
      return false;
   }

   s_ctx.display = static_cast<struct wl_display*>(nativeHandle);

   wl_list_init(&s_ctx.list_screen);

   s_ctx.registry = wl_display_get_registry(s_ctx.display);
   if (s_ctx.registry == NULL)
   {
      ETG_TRACE_ERR((" Failed to get registry"));
      return false;
   }

   if (wl_registry_add_listener(s_ctx.registry,
                                IviOutputControlHelper::GetRegistryEventHandler(), &s_ctx))
   {
      ETG_TRACE_ERR(("Failed to add registry listener"));
      return false;
   }

   // get globals
   if (wl_display_roundtrip(s_ctx.display) == -1)
   {
      ETG_TRACE_ERR(("Failed to initialize wayland connection"));
      return false;
   }

   if (!s_ctx.controller)
   {
      ETG_TRACE_ERR((" ivi_output-controller not available"));
      return false;
   }

   struct screen_context* ctx_scrn;

   /* add listener if it is not added */
   wl_list_for_each(ctx_scrn, &s_ctx.list_screen, link)
   {
      if (!ctx_scrn->screen_controller)
      {
         ctx_scrn->screen_controller =
            (struct ivi_output_control_screen*) ivi_output_control_create_screen(
               s_ctx.controller, ctx_scrn->output);
         ivi_output_control_screen_add_listener(ctx_scrn->screen_controller,
                                                IviOutputControlHelper::GetScreenListener(), ctx_scrn);
      }
   }

   // get ivi-output-controller callback done
   if (wl_display_roundtrip(s_ctx.display) == -1)
   {
      ETG_TRACE_ERR((" Failed to do roundtrip queue:"));
      return FALSE;
   }

   s_isInitialized = true;

   return true;
}


void IviOutputControlHelper::Deinit()
{
   struct screen_context* ctx_scrn, *next;

   /*remove wl_output and corresponding screen context*/
   wl_list_for_each_safe(ctx_scrn, next, &s_ctx.list_screen, link)
   {
      if (ctx_scrn->screen_controller != NULL)
      {
         ivi_output_control_screen_destroy(ctx_scrn->screen_controller);
      }

      if (ctx_scrn->output != NULL)
      {
         wl_output_destroy(ctx_scrn->output);
      }

      wl_list_remove(&ctx_scrn->link);
      free(ctx_scrn);
   }

   if (s_ctx.display)
   {
      wl_display_flush(s_ctx.display);
      wl_display_roundtrip(s_ctx.display);
   }

   if (s_ctx.registry)
   {
      wl_registry_destroy(s_ctx.registry);
   }

   s_isInitialized = false;
}


void IviOutputControlHelper::RegistryEventHandler(void* data,
      struct wl_registry* registry,
      uint32_t name,
      const char* interface,
      uint32_t version)
{
   outputcontrol_context* ctx = static_cast<outputcontrol_context*>(data);

   if (strcmp(interface, "ivi_output_control") == 0)
   {
      ctx->controller = (struct ivi_output_control*) wl_registry_bind(
                           registry, name, &ivi_output_control_interface, version);
   }
   else if (strcmp(interface, "wl_output") == 0)
   {
      struct screen_context* ctx_scrn = (struct screen_context*) calloc(1,
                                        sizeof * ctx_scrn);

      if (ctx_scrn == NULL)
      {
         ETG_TRACE_ERR(("Failed to allocate memory for screen_context"));
         return;
      }
      ctx_scrn->output = (struct wl_output*) wl_registry_bind(registry, name,
                         &wl_output_interface, 1);
      if (ctx_scrn->output == NULL)
      {
         free(ctx_scrn);
         ETG_TRACE_ERR((" Failed to registry bind wl_output"));
         return;
      }

      if (ctx->controller)
      {
         ctx_scrn->screen_controller =
            (struct ivi_output_control_screen*) ivi_output_control_create_screen(
               ctx->controller, ctx_scrn->output);
         ivi_output_control_screen_add_listener(ctx_scrn->screen_controller,
                                                GetScreenListener(), ctx_scrn);
      }

      ctx_scrn->name = name;
      wl_list_insert(&ctx->list_screen, &ctx_scrn->link);
   }
}


void IviOutputControlHelper::RegistryEventHandlerRemove(void* data,
      struct wl_registry* registry,
      uint32_t name)
{
   outputcontrol_context* ctx = (outputcontrol_context*)data;
   struct screen_context* ctx_scrn, *next;
   PARAM_UNUSED(registry);

   /*remove wl_output and corresponding screen context*/
   wl_list_for_each_safe(ctx_scrn, next, &ctx->list_screen, link)
   {
      if (ctx_scrn->name == name)
      {
         if (ctx_scrn->screen_controller != NULL)
         {
            ivi_output_control_screen_destroy(ctx_scrn->screen_controller);
         }

         if (ctx_scrn->output != NULL)
         {
            wl_output_destroy(ctx_scrn->output);
         }

         wl_list_remove(&ctx_scrn->link);
         free(ctx_scrn);
      }
   }
}


void IviOutputControlHelper::ScreenCapabilities(void* data,
      struct ivi_output_control_screen* ivi_outputcontrol_screen,
      uint32_t capabilities)
{
   struct screen_context* ctx_screen = (struct screen_context*)data;
   (void) ivi_outputcontrol_screen;

   ctx_screen->caps |= capabilities;
}


void IviOutputControlHelper::ScreenName(void* data,
                                        struct ivi_output_control_screen* ivi_outputcontrol_screen,
                                        const char* name)
{
   struct screen_context* ctx_screen = (struct screen_context*)data;
   (void) ivi_outputcontrol_screen;

   strcpy((ctx_screen->connectorName), name);
}


struct ivi_output_control_screen_listener* IviOutputControlHelper::GetScreenListener()
{
   static struct ivi_output_control_screen_listener screen_listener =
   {
      ScreenCapabilities,
      ScreenName
   };

   return &screen_listener;
}


const struct wl_registry_listener* IviOutputControlHelper::GetRegistryEventHandler()
{
   static const struct wl_registry_listener registry_control_listener =
   {
      RegistryEventHandler,
      RegistryEventHandlerRemove
   };

   return &registry_control_listener;
}


#endif // VARIANT_S_FTR_ENABLE_IVI_OUTPUT_CONTROL

#endif


//***********************************************

bool ILM_Accessor::setSurfacePosition(unsigned int surfaceId, int posX, int posY)
{
#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(posX);
   PARAM_UNUSED(posY);
   return false;
#else
   t_ilm_int sourceX;
   t_ilm_int sourceY;
   t_ilm_int sourceWidth;
   t_ilm_int sourceHeight;
   t_ilm_int destX;
   t_ilm_int destY;
   t_ilm_int destWidth;
   t_ilm_int destHeight;
   SYSTEMD_LOG("ILM_Accessor::setSurfacePosition start\n");
   ilmSurfaceProperties properties;
   ilmErrorTypes err = ilm_getPropertiesOfSurface(surfaceId, &properties);

   if (posX < 0)
   {
      sourceX = -posX;
      destX = 0;
      sourceWidth = properties.sourceWidth;
      destWidth = properties.sourceWidth - sourceX;
   }
   else
   {
      sourceX = 0;
      destX = posX;
      sourceWidth = properties.sourceWidth;
      destWidth = properties.sourceWidth;
   }

   if (posY < 0)
   {
      sourceY = -posY;
      destY = 0;
      sourceHeight = properties.sourceHeight;
      destHeight = properties.sourceHeight - sourceY;
   }
   else
   {
      sourceY = 0;
      destY = posY;
      sourceHeight = properties.sourceHeight;
      destHeight = properties.sourceHeight;
   }
   ETG_TRACE_USR1_THR(("ILM_Accessor::setSurfacePosition(%d, %d, %d)", surfaceId, posX, posY));
   ETG_TRACE_USR4_THR(("ILM_Accessor::setSurfacePosition: ilm_getPropertiesOfSurface(%d), source(%d, %d, %d, %d), dest(%d, %d, %d, %d)",
                       surfaceId,
                       properties.sourceX,
                       properties.sourceY,
                       properties.sourceWidth,
                       properties.sourceHeight,
                       properties.destX,
                       properties.destY,
                       properties.destWidth,
                       properties.destHeight));

   ETG_TRACE_USR4_THR(("ILM_Accessor::setSurfacePosition: ilm_surfaceSetSourceRectangle(%d, %d, %d, %d, %d)", surfaceId, sourceX, sourceY, sourceWidth, sourceHeight));
   err = ilm_surfaceSetSourceRectangle(surfaceId, sourceX, sourceY, sourceWidth, sourceHeight);
   if (ILM_SUCCESS == err)
   {
      ETG_TRACE_USR4_THR(("ILM_Accessor::setSurfacePosition: ilm_surfaceSetDestinationRectangle(%d, %d, %d, %d, %d)", surfaceId, destX, destY, destWidth, destHeight));
      err = ilm_surfaceSetDestinationRectangle(surfaceId, destX, destY, destWidth, destHeight);
   }

   if (ILM_SUCCESS == err)
   {
      err = ilm_commitChanges();
   }
   SYSTEMD_LOG("ILM_Accessor::setSurfacePosition end rc=%d, expected %d \n", err, ILM_SUCCESS);
   return (ILM_SUCCESS == err);
#endif
}


bool ILM_Accessor::setSurfaceDestinationRectangle(unsigned int surfaceId, unsigned int x, unsigned int y, unsigned int w, unsigned int h)
{
#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(x);
   PARAM_UNUSED(y);
   PARAM_UNUSED(w);
   PARAM_UNUSED(h);
   return false;
#else
   ETG_TRACE_USR1_THR(("ILM_Accessor::setSurfaceDestinationRectangle: Surface %d, Position (%u, %u), Dimension (%u, %u)", surfaceId, x, y, w, h));
   ilmErrorTypes err = ilm_surfaceSetDestinationRectangle(surfaceId, x, y, w, h);
   if (ILM_SUCCESS == err)
   {
      err = ilm_commitChanges();
   }
   return (ILM_SUCCESS == err);
#endif
}


bool ILM_Accessor::updateInputAcceptance(unsigned int surfaceId, unsigned int devices, bool acceptance)
{
#if defined (VARIANT_S_FTR_ENABLE_ILM_MOCK)
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(devices);
   PARAM_UNUSED(acceptance);
   return false;
#elif defined (VARIANT_S_FTR_ENABLE_IVI_SHELL)

   t_ilm_uint numSeats = 0;
   t_ilm_string* availableSeats;
   ilmInputDevice bitmask = 0;
   ilmErrorTypes err = ilm_getInputDevices(ILM_INPUT_DEVICE_ALL, &numSeats, &availableSeats);

   if ((err == ILM_SUCCESS) && (numSeats > 0))
   {
      std::vector<t_ilm_string> x(numSeats);
      for (unsigned int i = 0; i < numSeats; ++i)
      {
         err = ilm_getInputDeviceCapabilities(availableSeats[i], &bitmask);

         if (err == ILM_SUCCESS)
         {
            if (bitmask & devices)
            {
               if (acceptance)
               {
                  x.push_back(availableSeats[i]);
               }
            }
         }
      }
      err = ilm_setInputAcceptanceOn(surfaceId, static_cast<t_ilm_uint>(x.size()), &x[0]);
   }
   return ((err == ILM_SUCCESS) && (numSeats > 0));
#else
   ilmErrorTypes ilmError = ilm_UpdateInputEventAcceptanceOn(surfaceId, devices, acceptance);

   if (ilmError == ILM_SUCCESS)
   {
      // Commit changes to ILM
      ilmError = ilm_commitChanges();
   }

   return (ilmError == ILM_SUCCESS);
#endif
}


bool ILM_Accessor::setSynchronizedSurface(unsigned int* surfaces, unsigned int count)
{
#if defined (VARIANT_S_FTR_ENABLE_ILM_MOCK)
   PARAM_UNUSED(surfaces);
   PARAM_UNUSED(count);
#elif defined (VARIANT_S_FTR_ENABLE_IVI_SHELL)
   PARAM_UNUSED(surfaces);
   PARAM_UNUSED(count);
   // use LockScreen instead
#else
   ilmErrorTypes err = ilm_setSynchronizedSurfaces(surfaces, count);
   if (err == ILM_SUCCESS)
   {
      err = ilm_commitChanges();
      if (err == ILM_SUCCESS)
      {
         ETG_TRACE_USR1_THR((" Set Synchronized Surfaces called..."));
         for (unsigned int i = 0 ; i < count ; i++)
         {
            ETG_TRACE_USR1_THR((" for Surfaces[%u]", surfaces[i]));
         }
         return true;
      }
   }
   ETG_TRACE_USR1_THR((" Set Synchronized Surfaces Failed ..."));
#endif
   return false;
}


bool ILM_Accessor::removeSynchronizedSurface(unsigned int* surfaces, unsigned int count)
{
#if(defined (VARIANT_S_FTR_ENABLE_IVI_SHELL) || defined (VARIANT_S_FTR_ENABLE_ILM_MOCK))

   PARAM_UNUSED(surfaces);
   PARAM_UNUSED(count);

   ETG_TRACE_FATAL_THR(("ToDo Gen4: Use alternative for ilm_removeSynchronizedSurfaces"));
#else
   ilmErrorTypes err = ilm_removeSynchronizedSurfaces(surfaces, count);
   if (err == ILM_SUCCESS)
   {
      err = ilm_commitChanges();
      if (err == ILM_SUCCESS)
      {
         ETG_TRACE_USR1_THR((" Remove Synchronized Surfaces called..."));
         for (unsigned int i = 0 ; i < count ; i++)
         {
            ETG_TRACE_USR1_THR((" for Surfaces[%u]", surfaces[i]));
         }
         return true;
      }
   }
   ETG_TRACE_USR1_THR((" Remove Synchronized Surfaces Failed ..."));
#endif
   return false;
}


bool ILM_Accessor::getSurfaceProperties(unsigned int surfaceId, struct ilmSurfaceProperties* properties)
{
#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
   PARAM_UNUSED(surfaceId);
   PARAM_UNUSED(properties);
   return false;
#else
   ilmErrorTypes ilmError = ilm_getPropertiesOfSurface(surfaceId, properties);
   return (ilmError == ILM_SUCCESS);
#endif
}


bool ILM_Accessor::takeSurfaceSnapshot(const char* name, unsigned int surface)
{
#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
   PARAM_UNUSED(name);
   PARAM_UNUSED(surface);
   return false;
#else
   ilmErrorTypes ilmError = ILM_FAILED;
   if (name != 0)
   {
      char cmd[512];
      memset(cmd, 0, sizeof(cmd));

      strcpy(cmd, "/tmp/");
      char buffer[20];
      SNPRINTF(buffer, sizeof(buffer), "SurfaceId_%d_", surface);
      strncat(cmd, buffer, 20);
      strncat(cmd, name, 480);
      for (int mv = 5; cmd[mv] != 0; ++mv)
      {
         if (cmd[mv] == '/' || cmd[mv] == '\\' || cmd[mv] == '#')
         {
            cmd[mv] = '_';
         }
      }
      strcat(cmd, ".bmp");
      ilmError = ilm_takeSurfaceScreenshot(cmd, surface);

      if (ilmError != ILM_SUCCESS)
      {
         ETG_TRACE_FATAL_THR(("ilm_takeSurfaceScreenshot failed with error code %d", ilmError));
      }
      else
      {
         ETG_TRACE_FATAL_THR(("Screenshot of surface %d to %s", surface, cmd));
      }
   }
   return (ilmError == ILM_SUCCESS);
#endif
}


bool ILM_Accessor::takeSnapshot(const char* name)
{
#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
   PARAM_UNUSED(name);
   return false;
#else
   t_ilm_uint numberOfIds = 0;
   t_ilm_uint* pIDs = 0;
   ilmErrorTypes ret = ILM_FAILED;

   if (numberOfIds == 0)
   {
      ret = ilm_getScreenIDs(&numberOfIds, &pIDs);
      if (ILM_SUCCESS != ret)
      {
         return ret;
      }
   }

   if (name != NULL)
   {
      for (unsigned int i = 0; i < numberOfIds; ++i)
      {
         char cmd[512];
         memset(cmd, 0, sizeof(cmd));

         strcpy(cmd, "/tmp/");
         strncat(cmd, name, 480);
         for (int mv = 5; cmd[mv] != 0; ++mv)
         {
            if (cmd[mv] == '/' || cmd[mv] == '\\' || cmd[mv] == '#')
            {
               cmd[mv] = '_';
            }
         }
         strcat(cmd, ".bmp");
         ret = ilm_takeScreenshot(pIDs[i], cmd);
         if (ret != ILM_SUCCESS)
         {
            ETG_TRACE_FATAL_THR(("ilm_TakeSnapshot failed with errorcode %d", ret));
         }
         else
         {
            ETG_TRACE_FATAL_THR(("Screenshot of screen %d with scene %s to /tmp", pIDs[i], name));
         }
      }
   }

   if (pIDs)
   {
      free(pIDs);
   }
   return (ret == ILM_SUCCESS);;
#endif
}


bool ILM_Accessor::getScreenIDs(std::vector<unsigned int>& screenIds)
{
#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
   PARAM_UNUSED(screenIds);
   return false;
#else
   t_ilm_uint length = 0;
   t_ilm_uint* pArray;
   SYSTEMD_LOG("GUISTARTUP(1): ilm_init()\n");
   ilmErrorTypes errCode = ilm_init();

   if (errCode == ILM_SUCCESS)
   {
      errCode = ilm_getScreenIDs(&length, &pArray);
      if (errCode == ILM_SUCCESS)
      {
         for (unsigned int i = 0; i < length; ++i)
         {
            screenIds.push_back(pArray[i]);
         }
         free(pArray);
      }
   }
   return (errCode == ILM_SUCCESS);
#endif
}


bool ILM_Accessor::getSurfaceIds(std::vector<unsigned int>& surfaceIds)
{
#ifdef VARIANT_S_FTR_ENABLE_ILM_MOCK
   PARAM_UNUSED(surfaceIds);
   return false;
#else
   t_ilm_int length = 0;
   t_ilm_uint* pArray;
   SYSTEMD_LOG("GUISTARTUP(1): ilm_init()\n");
   ilmErrorTypes errCode = ilm_init();

   if (errCode == ILM_SUCCESS)
   {
      errCode = ilm_getSurfaceIDs(&length, &pArray);
      if (errCode == ILM_SUCCESS)
      {
         for (int i = 0; i < length; ++i)
         {
            surfaceIds.push_back(pArray[i]);
         }

         free(pArray);
      }
   }
   return (errCode == ILM_SUCCESS);
#endif
}


bool ILM_Accessor::init(void* nativeHandle)
{
#if defined VARIANT_S_FTR_ENABLE_IVI_SHELL && defined VARIANT_S_FTR_ENABLE_IVI_OUTPUT_CONTROL
   return IviOutputControlHelper::GetInstance().Init(nativeHandle);
#else
   PARAM_UNUSED(nativeHandle);
   return true;
#endif
}


void ILM_Accessor::deInit()
{
#if defined VARIANT_S_FTR_ENABLE_IVI_SHELL && defined VARIANT_S_FTR_ENABLE_IVI_OUTPUT_CONTROL
   IviOutputControlHelper::GetInstance().Deinit();
#endif
}


bool ILM_Accessor::lockUnlockScreen(bool lock, const char* connectorName)
{
#if defined VARIANT_S_FTR_ENABLE_IVI_SHELL && defined VARIANT_S_FTR_ENABLE_IVI_OUTPUT_CONTROL
   if (!IviOutputControlHelper::GetInstance().IsInitialized())
   {
      return false;
   }

   struct IviOutputControlHelper::screen_context* ctx_scrn, *next;
   IviOutputControlHelper::outputcontrol_context& ctx = IviOutputControlHelper::GetInstance().GetOutputControlContext();
   wl_list_for_each_safe(ctx_scrn, next, &ctx.list_screen, link)
   {
      if ((connectorName == 0) || (!strcmp(ctx_scrn->connectorName, connectorName)))
      {
         if (ctx_scrn->caps & IVI_OUTPUT_CONTROL_SCREEN_CAPABILITY_DPMS)
         {
            ivi_output_control_screen_set_dpms(ctx_scrn->screen_controller,
                                               lock ? IVI_OUTPUT_CONTROL_SCREEN_DPMS_MODE_FREEZE : IVI_OUTPUT_CONTROL_SCREEN_DPMS_MODE_ON);
         }
         ETG_TRACE_USR1(("Set DPMS:for %25s connector, lock %d", connectorName, lock));
      }
      wl_display_roundtrip(ctx.display);
   }
#else
   PARAM_UNUSED(lock);
   PARAM_UNUSED(connectorName);
#endif

   return true;
}


}
