/* ***************************************************************************************
* FILE:          HallComponentBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HallComponentBase.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "HallComponentBase.h"

//#ifndef _LINUXX86_64_
//#include <systemd/sd-daemon.h>
//#endif

#include "MsgFwAbstraction/IMessagePostObject.h"

#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HallComponentBase.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace app {
namespace base {


HallComponentBase::HallComponentBase(const char* appName, const std::string& busName, const std::string& objName):
   HmiAsfComponentBase(appName),
   CourierMessageReceiverStub("courierMessageReceiverPort"),
   _lcmHandler(busName, objName, this)
{
}


HallComponentBase::~HallComponentBase()
{
}


void HallComponentBase::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


void HallComponentBase::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


// CGI: Response from CGI, init ok and Message system is running ==> former Courier::StartupMsg received,
//  signal LCM, that HMI is now ready
// Qt: Gui startup is not required, because SM is on HALL side, GUI can connect to HALL and gets a refresh
//  signal LCM, that HMI is now ready.
bool HallComponentBase::onGuiStartupFinished()
{
   ETG_TRACE_USR1(("HallComponentBase --- onGuiStartupFinished -> sd_notify(0, READY=1);"));
   _lcmHandler.setAppStartReady();
   SYSTEMD_LOG("GUISTARTUP(4): lcmHandler.setAppStartReady() called\n");
   SYSTEMD_LOG("GUISTARTUP(5): sd_notify(0, READY=1) called\n");
   return true;
}


// Response from SM, ready. ==> former GuiStartupFinishedUpdMsg received
bool HallComponentBase::onStatemachineStartupFinished()
{
   ETG_TRACE_USR1(("HallComponentBase --- GuiStartupFinishedUpdMsg received from SM, ready to start!"));
   SYSTEMD_LOG("GUISTARTUP(x): hallComponentReady called, notifyRegisteredClients\n");
   // After app is ready, post callbacks for services which are available.
   StartupSync::getInstance().notifyRegisteredClients(); // former onGuiStartupFinished();
   // Finaly the app has to inform the master with appStartupReadyForMaster(), that the
   //  app is ready for screenbroker actions like reister surface, show ... etc
   return true;
}


//  Qt: HMI application informs LCM/systemd that startup is finished
//  CGI: not used
void HallComponentBase::hallComponentReady()
{
   onGuiStartupFinished();
   onStatemachineStartupFinished();
}


// Req to get the HMI SUB state
bool HallComponentBase::onChangeHmiSubStateReq()
{
   ETG_TRACE_USR1(("HallComponentBase --- ChangeHmiSubStateReqMsg -> lcmHandler.sendHmiStateMessage"));
   _lcmHandler.sendHmiStateMessage();
   return true;
}


bool HallComponentBase::OnAppModeChange(hmibase::app::base::lcm::ShutdownType /*newState*/, hmibase::app::base::lcm::ShutdownType /*oldState*/)
{
   ETG_TRACE_USR1_THR(("onApplicationPowerStateChangeRequest is not overwritten by HALL component of application %s", hmibase::trace::getAppName().c_str()));
   return true;
}


void HallComponentBase::onLoadPersistency()
{
   // callback from LCM-Handler
   ETG_TRACE_USR1_THR(("onLoadPersistency is not overwritten by HALL component of application %s", hmibase::trace::getAppName().c_str()));
}


void HallComponentBase::onSavePersistency()
{
   // callback from LCM-Handler
   ETG_TRACE_USR1_THR(("onSavePersistency is not overwritten by HALL component of application %s", hmibase::trace::getAppName().c_str()));
}


void HallComponentBase::onAppSubStateReady(hmibase::HMIStateEnum val)
{
   // callback from LCM-Handler
   ETG_TRACE_USR1(("HallComponentBase::onAppSubStateReady HMIStateEnum=%d", val));
   hmibase::IMessagePostObject::s_getInstance().postMsg(*newMessage<HMISubStateChangedUpdMsg>(val));
}


void HallComponentBase::SendLifeCycleRequestCompleteRequest()
{
   ETG_TRACE_USR1(("HallComponentBase::SendLifeCycleRequestCompleteRequest -> lcmHandler"));
   _lcmHandler.SendLifeCycleRequestCompleteRequest();
}


void HallComponentBase::onReceiveCourierMessageRequest(const boost::shared_ptr<CourierTunnelService::CourierMessageReceiver::ReceiveCourierMessageRequest>& /*msg*/)
{
   // incomming ping from tunnelservice, GUIMain has something added to the message queue
   processViewEvents();
}


} // namespace
} // namespace
} // namespace
