/* ***************************************************************************************
* FILE:          AppControllerBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppControllerBase.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include <hmibase/sys_std_if.h>
#include "AppControllerBase.h"
#include "HMIAppCtrl/Proxy/ProxyHandler.h"
#include "Trace/StartupInvestigation.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#ifndef VARIANT_S_FTR_DISABLE_FOCUSHANDLING
#include "Focus/FManager.h"
#endif
#include "app_base_defines.h"
#include "AppBase/ILM_Accessor.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_COMMON_APPCTRL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AppControllerBase.cpp.trc.h"
#endif


using namespace ::hmibase::services::hmiappctrl;

namespace hmibase {
namespace app {
namespace base {

AppControllerBase* AppControllerBase::_controllerbase = NULL;

#ifndef WIN32
bool mapGadgetTargetState(const hmibase::gadget::gadgetSceneStatus state, bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus& mapped)
{
   switch (state)
   {
      case hmibase::gadget::ACTIVE:
         mapped = bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__ACTIVE;
         break;
      case hmibase::gadget::INACTIVE:
         mapped = bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__INACTIVE;
         break;
      case hmibase::gadget::HIDDEN:
         mapped = bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__HIDDEN;
         break;
      default:
         return false;
         break;
   }
   return true;
}


bool mapGadgetTargetState(const bool activate, bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus& mapped)
{
   mapped = activate ? bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__ACTIVE : bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus__INACTIVE;
   return true;
}


#endif

// ------------------------------------------------------------------------
AppControllerBase::AppControllerBase(hmibase::services::hmiappctrl::ProxyHandler& proxyHandler) :
   _hmiAppCtrlProxyHandler(proxyHandler)
{
   _controllerbase = this;
   // set the environment variable HMIAPP_STANDALONE to any value to start the application in
   // standalone mode, means without master application
#ifdef WIN32
   _standalone = true;
#else
   _standalone = ::getenv("HMIAPP_STANDALONE") != 0 ? true : false;
#endif
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const Courier::StartupMsg& /*msg*/)
{
   HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_COURIER_STARTUP_MESSAGE_RECEIVED);
   if (_standalone)
   {
      //activate main surface
      //todo: to activate more than one main surface at startup for standalone app
      unsigned int surfaceID = ScreenBrokerClient::GetInstance().SurfaceId();
      unsigned int displayID = ScreenBrokerClient::GetInstance().GetDisplayID(surfaceID);
      SYSTEMD_LOG("GUISTARTUP(n): onApplicationStateForeground() called from standalone\n");
      POST_MSG(COURIER_MESSAGE_NEW(ApplicationStateUpdMsg)(hmibase::IN_FOREGROUND, displayID));
   }
#ifndef WIN32
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "StartupMsg received, send status 'Application_Initialized'"));
   SYSTEMD_LOG("GUISTARTUP(3): HMIAPP_INITIALIZED() send to master\n");
   _hmiAppCtrlProxyHandler.setAppState(bosch::cm::ai::hmi::appctrl::HMIAppCtrl::hmiAppState__HMIAPP_INITIALIZED);
#endif
   return false;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const HKStatusChangedUpdMsg& msg)
{
   //don't consume and forward to master app the focus related messages
#ifndef VARIANT_S_FTR_DISABLE_FOCUSHANDLING
   if (Focus::FManager::getInstance().isFocusInputMessage(msg))
   {
      return false;
   }
#endif
#ifndef WIN32
   // message has not been consumed by the components state machine, so post it to the master
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Send hardkey message to AppHmi_Master"));
   _hmiAppCtrlProxyHandler.evaluateKeyForwardingRequest((int)msg.GetHKCode(), (int)msg.GetHKState(), (int)msg.GetDisplayID(), (int)msg.GetUserData());

   return true;
#else
   return false;
#endif
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const EncoderStatusChangedUpdMsg& msg)
{
   //don't consume and forward to master app the focus related messages
#ifndef VARIANT_S_FTR_DISABLE_FOCUSHANDLING
   if (Focus::FManager::getInstance().isFocusInputMessage(msg))
   {
      return false;
   }
#endif

#ifndef WIN32
   // message has not been consumed by the components state machine, so post it to the master
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Send encoder message to AppHmi_Master"));
   _hmiAppCtrlProxyHandler.evaluateEncoderForwardingRequest((int)msg.GetEncCode(), (int)msg.GetEncSteps(), (int)msg.GetSurfaceId(), (int)msg.GetUserData());
#endif
   return true;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const PlayBeepReqMsg& msg)
{
#ifndef WIN32
   // message has not been consumed by the components state machine, so post it to the master
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Send Beep message to AppHmi_Master"));

   BeepForwardingPayload payload(msg.GetBeepType());
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const SetApplicationAsMandatoryReqMsg& msg)
{
#ifndef WIN32
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Send SetApplicationAsMandatoryReqMsg message "));

   ApplicationMandatoryPayload payload(_hmiAppCtrlProxyHandler.getName(), msg.GetValue());
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const ShowHMISubSurfaceReqMsg& msg)
{
#ifndef WIN32
   // send status line message to HMI-Master
   //sender name, receiver name
   DisplayStatusEvalPayload payload(::bosch::cm::ai::hmi::appctrl::HMIAppCtrl::MultidisplayData(0,
                                    (msg.GetDisplayID() == hmibase::DEFAULT_UNKNOWN_DISPLAY) ?
                                    ScreenBrokerClient::GetInstance().GetDisplayID(ScreenBrokerClient::GetInstance().SurfaceId()) :
                                    msg.GetDisplayID(),
                                    _hmiAppCtrlProxyHandler.getName(),
                                    hmibase::app::base::appHmi_Master,
                                    (int)msg.GetDisplayRegion(),
                                    (int)msg.GetDisplayRegionType(),
                                    (int)msg.GetShowSubSurface()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);

#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const PopupFilterPrioEnableReqMsg& msg)
{
#ifndef WIN32
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Enable popup filter prio for all popups with prio > %d.", (int)msg.GetPrio()));

   //call screen broker to set the popup filter
   ScreenBrokerClient::GetInstance().SetPopupFilterPrio(msg.GetPrio());
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const PopupFilterAllEnableReqMsg& /*msg*/)
{
#ifndef WIN32
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Enable popup filter all."));

   //call screen broker to set the popup filter
   ScreenBrokerClient::GetInstance().SetPopupFilterAll();
#endif
   return true;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const PopupFilterDisableReqMsg& /*msg*/)
{
#ifndef WIN32
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Disable popup filter."));

   //call screen broker to set the popup filter
   ScreenBrokerClient::GetInstance().ClearPopupFilterPublic();
#endif
   return true;
}


// -------------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const MultiDisplayPopupFilterReqMsg& msg)
{
#ifndef WIN32

   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Filter the pop up <type> : %d <displayId>: %d", static_cast<int>(msg.GetPopupFilterType()), msg.GetDisplayID()));

   //call screen broker to set the popup filter
   if (msg.GetPopupFilterType() == hmibase::ENABLE_PRIO)
   {
      ScreenBrokerClient::GetInstance().SetPopupFilterPrio(msg.GetPrio(), msg.GetDisplayID());
   }
   else if (msg.GetPopupFilterType() == hmibase::ENABLE_ALL)
   {
      ScreenBrokerClient::GetInstance().SetPopupFilterAll(msg.GetDisplayID());
   }
   else if (msg.GetPopupFilterType() == hmibase::DISABLE_ALL)
   {
      ScreenBrokerClient::GetInstance().ClearPopupFilterPublic(msg.GetDisplayID());
   }
   else
   {
      // do nothing
   }
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::ExternalImageProviderReqMsg& msg)
{
#ifndef WIN32
   std::vector<int32_t>v;
   v.push_back(msg.GetInstanceId());

   bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus targetState;
   if (mapGadgetTargetState(msg.GetTargetState(), targetState) == false)
   {
      mapGadgetTargetState(msg.GetActivate(), targetState);
   }

   RequestExternalImageProviderActivationPayload payload(targetState, v, static_cast<int32>(_hmiAppCtrlProxyHandler.getSyncBlockId()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::ExternalImageProviderGroupReqMsg& msg)
{
#ifndef WIN32
   bosch::cm::ai::hmi::appctrl::HMIAppCtrl::gadgetStatus targetState;
   if (mapGadgetTargetState(msg.GetTargetState(), targetState) == false)
   {
      mapGadgetTargetState(msg.GetActivate(), targetState);
   }

   RequestExternalImageProviderActivationPayload payload(targetState, msg.GetInstanceIds(), static_cast<int32>(_hmiAppCtrlProxyHandler.getSyncBlockId()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::TouchForwardReqMsg& msg)
{
#ifndef WIN32
   bosch::cm::ai::hmi::appctrl::HMIAppCtrl::touchInfo data(static_cast<uint8>(msg.GetState()), msg.GetXPos(), msg.GetYPos(), static_cast<uint32>(msg.GetTimeStamp()));

   RequestExternalImageProviderTouchPayload payload(static_cast<uint32>(getpid()), data, static_cast<int32_t>(msg.GetInstanceId()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::TouchAbortReqMsg& msg)
{
#ifndef WIN32
   bosch::cm::ai::hmi::appctrl::HMIAppCtrl::touchInfo data(0xff, 0, 0, 0);

   RequestExternalImageProviderTouchPayload payload(static_cast<uint32>(getpid()), data, static_cast<int32_t>(msg.GetInstanceId()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::SlaveSurfaceRepositionReqMsg& msg)
{
   if (msg.GetIPCForwarding() == false)
   {
      return false;
   }
#ifndef WIN32
   RequestSlaveSurfaceRepositioningPayload payload(static_cast<uint32>(getpid()), msg.GetActivate(), static_cast<uint32_t>(msg.GetSlaveSurfaceId()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::SlaveSurfaceRepositionResMsg& msg)
{
   if (msg.GetIPCForwarding() == false)
   {
      return false;
   }
#ifndef WIN32
   ResponseSlaveSurfaceRepositioningPayload payload(static_cast<uint32>(getpid()), msg.GetActivate(), static_cast<uint32_t>(msg.GetSlaveSurfaceId()), msg.GetStatus());
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::PositionInfoReqMsg& msg)
{
   if (msg.GetIPCForwarding() == false)
   {
      return false;
   }
#ifndef WIN32
   RequestDirectTextureConsumerPositionInfoPayload payload(static_cast<uint32>(getpid()), static_cast<uint32_t>(msg.GetInstanceId()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::PositionInfoResMsg& msg)
{
   if (msg.GetIPCForwarding() == false)
   {
      return false;
   }
#ifndef WIN32
   ResponseDirectTextureConsumerPositionInfoPayload payload(
      static_cast<uint32>(getpid()),
      static_cast<uint32_t>(msg.GetInstanceId()),
      msg.GetPosition().GetX(),
      msg.GetPosition().GetY());
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const hmibase::gadget::DrmAllocationStatusMsg& msg)
{
#ifndef WIN32
   DrmBufferAllocationStatusPayload payload(getpid(), bosch::cm::ai::hmi::appctrl::HMIAppCtrl::drmBufferAllocationStatus__DRM_BUFFER_DEALLOCATION_FAILED, 0);

   switch (msg.GetStatus())
   {
      case hmibase::gadget::ALLOCATION_SUCCESS:
      {
         payload._status = bosch::cm::ai::hmi::appctrl::HMIAppCtrl::drmBufferAllocationStatus__DRM_BUFFER_ALLOCATION_SUCCESS;
         payload._bufferSize = msg.GetSize();
      }
      break;
      case hmibase::gadget::ALLOCATION_FAILED:
      {
         payload._status = bosch::cm::ai::hmi::appctrl::HMIAppCtrl::drmBufferAllocationStatus__DRM_BUFFER_ALLOCATION_FAILED;
         payload._bufferSize = 0;
      }
      break;
      case hmibase::gadget::DEALLOCATION_SUCCESS:
      {
         payload._status = bosch::cm::ai::hmi::appctrl::HMIAppCtrl::drmBufferAllocationStatus__DRM_BUFFER_DEALLOCATION_SUCCESS;
         payload._bufferSize = msg.GetSize();
      }
      break;
      case hmibase::gadget::DEALLOCATION_FAILED:
      {
         payload._status = bosch::cm::ai::hmi::appctrl::HMIAppCtrl::drmBufferAllocationStatus__DRM_BUFFER_DEALLOCATION_FAILED;
         payload._bufferSize = 0;
      }
      break;
      default:
         return true;
         break;
   }
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const RenderingCompleteMsg& msg)
{
#ifndef WIN32
   ApplicationRenderedUpdatePayload payload((int)msg.GetSurfaceId());
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


bool AppControllerBase::onCourierMessage(const SurfaceStateChangedUpdMsg& msg)
{
   if (_hmiAppCtrlProxyHandler.isDirectApplicationActivationEnabled() == false)
   {
      if ((msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_DEREGISTERED))
            && (ScreenBrokerClient::GetInstance().IsMainSurface(msg.GetSurfaceId())))
      {
#ifndef WIN32
         ApplicationSwitchCompletePayload payload((int) msg.GetSurfaceId(), (int) msg.GetState());
         _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#endif
      }
   }
   else
   {
      if ((msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_VISIBLE) || msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_INVISIBLE))
            && (ScreenBrokerClient::GetInstance().IsMainSurface(msg.GetSurfaceId())))
      {
#ifndef WIN32
         ApplicationSwitchCompletePayload payload((int) msg.GetSurfaceId(), (int) msg.GetState());
         _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
#endif
      }
   }
   return false;
}


bool AppControllerBase::onCourierMessage(const ClosePopupOnExternalTouchReqMsg& msg)
{
   PARAM_UNUSED(msg);
#ifndef WIN32
#ifndef VARIANT_S_FTR_DISABLE_FOCUSHANDLING
   if (msg.GetTouchState() == Courier::TouchMsgState::Down)
   {
      //call ScreenBrokerClient
      ScreenBrokerClient::GetInstance().SendClosePopupOnExternalTouch(msg.GetSurfaceId());
   }
#endif
#endif
   return true;
}


bool AppControllerBase::setSynchronizedSurfaces(unsigned int* surface, unsigned int count)
{
   return hmibase::ILM_Accessor::setSynchronizedSurface(surface, count);
}


bool AppControllerBase::removeSynchronizedSurfaces(unsigned int* surface, unsigned int count)
{
   return hmibase::ILM_Accessor::removeSynchronizedSurface(surface, count);
}


bool AppControllerBase::onCourierMessage(const ApplicationStateUpdMsg& msg)
{
   PARAM_UNUSED(msg);
   return false;
}


// ------------------------------------------------------------------------
bool AppControllerBase::onCourierMessage(const StartLayerAnimationReqMsg& msg)
{
#ifndef WIN32
   ETG_TRACE_USR1_THR_DCL((_hmiAppCtrlProxyHandler.getTraceId(), "Start Layer Animation %d.", (int)msg.GetLayerId()));

   //call screen broker to start Layer animation with the requested surfaces
   std::vector<ScreenBroker::UInt32> lSurfaceIdList(msg.GetSurfaceIdList().begin(), msg.GetSurfaceIdList().end());
   if (lSurfaceIdList.empty())
   {
      lSurfaceIdList.push_back(0);
   }
   ScreenBrokerClient::GetInstance().StartLayerAnimation(static_cast<ScreenBroker::UInt32>(msg.GetLayerId()),
         lSurfaceIdList,
         static_cast<ScreenBroker::UInt32>(msg.GetAnimationType()));
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


}
}


}
