/* ***************************************************************************************
* FILE:          CCASocketGateway.cpp
* SW-COMPONENT:  LCM-Generic
*  DESCRIPTION:  CCASocketGateway.cpp is part of Bosch LCM framework Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include<stdio.h>
#include<string.h>
#include<stdlib.h>
#include<sys/socket.h>
#include<arpa/inet.h>
#include<unistd.h>
#include<pthread.h>
#include<errno.h>
#include <sstream>

#include "CCASocketGateway.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

static bool g_remoteConnected = false;
static int  g_remoteSockId = -1;
static char* g_destinationAddress = NULL;
static bool g_debug = false;

#define SOCKETGW_BYTE_0_HEADER_START                0x5A
#define SOCKETGW_BYTE_1_HEADER_START                0xA5
#define SOCKETGW_BYTE_2_HEADER_START                0x0F
#define SOCKETGW_BYTE_3_CMD_START_CCA_RECEIVER      0xE1
#define SOCKETGW_BYTE_3_CMD_SEND_CCA_MSG            0xE2


#define TRACE_MSG(args)  { if (g_debug) { printf args; printf(" [line %d]\n",__LINE__); } }
#define TRACE_INFO(args)   { printf args; }
#define TRACE_ERR(args) { fprintf(stderr, "\n ===> CCASocketGateway :: ERROR occured in line %d\n",__LINE__); printf args; printf("\n"); }

static bool socketSend(int sockId, const unsigned char* buf, ssize_t total_bytes_towrite);

#define HMIBASE_TGW_DEFAULT_PORT  2348

HostCcaSocketGateway::HostCcaSocketGateway() : _port(0)
{
   g_destinationAddress = ::getenv("TARGET_GATEWAY_ADDRESS");
   if (g_destinationAddress == NULL)
   {
      TRACE_ERR(("environment TARGET_GATEWAY_ADDRESS not set\n"));
      exit(1);
   }

   _port = HMIBASE_TGW_DEFAULT_PORT;  // default port
}


// add services which should be mirrored on client side (HOST)
void HostCcaSocketGateway::addServicesToMirror(std::string csvMboxNumbers)
{
   // for example "0,42,43"
   std::string token;
   std::istringstream tokenStream(csvMboxNumbers);
   while (std::getline(tokenStream, token, ','))
   {
      int val = atoi(token.c_str());
      if (val >= 0)
      {
         addServiceIdToMirror(val);
      }
   }
}


// add service which should be mirrored on client side (HOST)
void HostCcaSocketGateway::addServiceIdToMirror(uint32_t serviceId)
{
   _vecServiceID.push_back(serviceId);
}


// add applictions services which should be mirrored on server side (TARGET)
void HostCcaSocketGateway::addAppIdsToMirror(std::string csvMboxNumbers)
{
   // for example "160"
   std::string token;
   std::istringstream tokenStream(csvMboxNumbers);
   while (std::getline(tokenStream, token, ','))
   {
      int val = atoi(token.c_str());
      if (val >= 0)
      {
         addAppIdToMirror(val);
      }
   }
}


// add appliction services which should be mirrored on server side (TARGET)
void HostCcaSocketGateway::addAppIdToMirror(uint32_t appId)
{
   _vecAppId.push_back(appId);
}


/* static */
void* HostCcaSocketGateway::clientMainThreadProc(void* classpointer)
{
   HostCcaSocketGateway* threadImpl = static_cast<HostCcaSocketGateway*>(classpointer);
   if (threadImpl)
   {
      threadImpl->clientMainThread();
   }
   return 0;
}


// main thread on HOST side, which connects to TARGET and creates a
//  couple of CCA receiver threads ( setup of MBX mirrors for services ).
//  Dispatches incomming socket data.
bool HostCcaSocketGateway::clientMainThread()
{
   amt_bInit();
   scd_init();

   //Create socket
   g_remoteSockId = socket(AF_INET, SOCK_STREAM, 0);
   if (g_remoteSockId == -1)
   {
      TRACE_ERR(("HostCcaSocketGateway::clientMainThread() Could not create socket err=%s", strerror(errno)));
      return false;
   }

   TRACE_MSG(("HostCcaSocketGateway::clientMainThread() Socket created -> %d\n", g_remoteSockId));

   struct sockaddr_in server;
   memset(&server, 0, sizeof(server));
   server.sin_addr.s_addr = inet_addr(g_destinationAddress);
   server.sin_family = AF_INET;
   server.sin_port = htons(_port);

   while (1)
   {
      //Connect to remote server
      while (connect(g_remoteSockId, (struct sockaddr*)&server, sizeof(server)) < 0)
      {
         TRACE_INFO(("HostCcaSocketGateway::clientMainThread() server connection was not successful ... \n"));
         OSAL_s32ThreadWait(1500);
      }

      TRACE_INFO(("HostCcaSocketGateway::clientMainThread() server connection established!"));

      // start client threads
      g_remoteConnected = true;

      std::vector<uint32_t>::iterator it;
      for (it = _vecServiceID.begin(); it != _vecServiceID.end(); it++)
      {
         uint32_t id = *it;
         TRACE_MSG(("HostCcaSocketGateway::clientMainThread() start client thread for -> AppID: %d", id));
         // setup MBX mirrors for services on HOST side
         MessageReceiverMirror::startCcaMessageReceiver(id);
      }

      for (it = _vecAppId.begin(); it != _vecAppId.end(); it++)
      {
         uint8_t msg2Send[10];

         msg2Send[0] = SOCKETGW_BYTE_0_HEADER_START;
         msg2Send[1] = SOCKETGW_BYTE_1_HEADER_START;
         msg2Send[2] = SOCKETGW_BYTE_2_HEADER_START;
         msg2Send[3] = SOCKETGW_BYTE_3_CMD_START_CCA_RECEIVER;
         // length of payload data
         msg2Send[4] = 0;
         msg2Send[5] = 0;
         msg2Send[6] = 0;
         msg2Send[7] = 2;
         // target id as payload for this command
         msg2Send[8] = (*it & 0xff00) >>  8;
         msg2Send[9] = (*it & 0x00ff) >>  0;

         TRACE_MSG(("HostCcaSocketGateway::clientMainThread() setup remote -> AppID: %d (0x%04x) (sock: %d)",
                    *it, *it, g_remoteSockId));
         socketSend(g_remoteSockId, msg2Send, 10);
      }

      while (waitForSocketMessage(g_remoteSockId))
         ;

      //stop clients
      g_remoteConnected = false;
   }

   pthread_exit(0);
}


// Opens CCA message queue, forwards CCA Message content via socket to the counterpart.
// This code is for TARGET and HOST
void MessageReceiverMirror::ccaMessageReceiverThreadProc(uint32_t appid)
{
   // create message queue
   OSAL_tMQueueHandle _hCcaInQueue = OSAL_C_INVALID_HANDLE;

   tChar szName[32];
   snprintf(szName, sizeof(szName), "mbx_%d", appid);

   _hCcaInQueue = OSAL_C_INVALID_HANDLE;
   if (OSAL_ERROR == OSAL_s32MessageQueueOpen(szName, OSAL_EN_READWRITE, &_hCcaInQueue))
   {
      if (OSAL_ERROR == OSAL_s32MessageQueueCreate(
               szName,
               20,
               SCD_MAILBOX_MAX_MESSAGE_LENGTH,
               OSAL_EN_READWRITE,
               &_hCcaInQueue
            )
         )
      {
         TRACE_ERR(("MessageReceiverMirror::ccaMessageReceiverThreadProc() FAILED to open/create queue."));
      }
   }

   TRACE_MSG(("MessageReceiverMirror::ccaMessageReceiverThreadProc() OSAL msgQueue '%s' created --> wait for messages.", szName));
   while (g_remoteConnected && (_hCcaInQueue != OSAL_C_INVALID_HANDLE))
   {
      amt_tclBaseMessage oMsgObject;
      tS32  s32RetVal = OSAL_s32MessageQueueWait(
                           _hCcaInQueue,
                           (uint8_t*)oMsgObject.prGetOSALMsgHandle(), /*pointer of the MsgHandle field*/
                           sizeof(OSAL_trMessage),
                           OSAL_NULL,
                           1000000
                        );

      if (s32RetVal != OSAL_ERROR)
      {
         // gets the pointer to shared memory
         if (sizeof(OSAL_trMessage) == s32RetVal)
         {
            TRACE_MSG(("MessageReceiverMirror::ccaMessageReceiverThreadProc(%s) New message received", szName));
            oMsgObject.pu8SharedMemBase = OSAL_pu8MessageContentGet(* (oMsgObject.prGetOSALMsgHandle()), OSAL_EN_READWRITE);

            if (oMsgObject.pu8SharedMemBase != OSAL_NULL)
            {
               // send message to host via socket
               uint32_t messageSize = oMsgObject.u32GetSize();

               // RTC-542179/ASF-5252: don't increment the message size by u32DynMsgSize() as this is already part of u32GetSize
               //if (oMsgObject.u32GetDynMsgSize() > 0)
               //{
               //   messageSize += oMsgObject.u32GetDynMsgSize();
               //}

               TRACE_MSG(("MessageReceiverMirror::ccaMessageReceiverThreadProc(%s) sockSend payload size = %d", szName, messageSize));
               uint8_t msg2Send[10];

               msg2Send[0] = SOCKETGW_BYTE_0_HEADER_START;
               msg2Send[1] = SOCKETGW_BYTE_1_HEADER_START;
               msg2Send[2] = SOCKETGW_BYTE_2_HEADER_START;
               msg2Send[3] = SOCKETGW_BYTE_3_CMD_SEND_CCA_MSG;

               msg2Send[4] = (messageSize & 0xff000000) >> 24;
               msg2Send[5] = (messageSize & 0x00ff0000) >> 16;
               msg2Send[6] = (messageSize & 0x0000ff00) >>  8;
               msg2Send[7] = (messageSize & 0x000000ff) >>  0;

               socketSend(g_remoteSockId, msg2Send, 8);
               socketSend(g_remoteSockId, oMsgObject.pu8GetSharedMemBase(), messageSize);
            }

            // release memory of message object
            oMsgObject.bDelete();
         }
         else
         {
            if (OSAL_u32ErrorCode() != OSAL_E_TIMEOUT)
            {
               // error
            }
            else
            {
               // timeout
            }
         }
      }
      else
      {
         // ETG_TRACE_ERR_CLS((SPM_TRACE_CLASS_SPM, "bMessageQueueWait, receive error: %s", OSAL_coszErrorText(OSAL_u32ErrorCode())));
      }
   }

   OSAL_s32MessageQueueDelete(szName);
   TRACE_MSG(("MessageReceiverMirror::ccaMessageReceiverThreadProc() OSAL msgQueue '%s' deleted!", szName));
}


// Creates the CCA message receiver as a thread. This code is for TARGET and HOST.
void MessageReceiverMirror::startCcaMessageReceiver(uint32_t appId)
{
   TRACE_MSG(("MessageReceiverMirror::startCcaMessageReceiver() start client: ID=%d", appId));

   char procSpecName[32] = {0};
   snprintf(procSpecName, sizeof(procSpecName), "GwMsgRecMbx%04x", appId);

   OSAL_trThreadAttribute rAttr;
   rAttr.szName       = (tString)procSpecName;
   rAttr.s32StackSize = 10000;
   rAttr.u32Priority  = 100;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)MessageReceiverMirror::ccaMessageReceiverThreadProc;
   rAttr.pvArg        = (void*)(long)appId;

   OSAL_tThreadID hThreadId = OSAL_ThreadSpawn(&rAttr);
   if (hThreadId == OSAL_ERROR)
   {
      TRACE_ERR(("MessageReceiverMirror::startCcaMessageReceiver(): Failed to spawn thread!"));
   }
}


// Creates HOST main thread, which connects via a socket to the Target
void HostCcaSocketGateway::connectTarget()
{
   pthread_t client_thread;
   TRACE_MSG(("HostCcaSocketGateway::connectTarget() destination_ip=%s port=%d", g_destinationAddress, _port));
   if (pthread_create(&client_thread, NULL, HostCcaSocketGateway::clientMainThreadProc, this) < 0)
   {
      TRACE_ERR(("HostCcaSocketGateway::connectTarget() could not create thread, failed."));
   }
   int count = 0;
   while ((g_remoteConnected == false) && (count < 10))
   {
      usleep(500 * 1000);   // sleep 500ms
      ++count;
   }
}


// Creates HOST main thread, which connects via a socket to the Target
void HostCcaSocketGateway::connectTarget(const char* csvMailboxesServices, const char* csvMailboxesAppId)
{
   if (csvMailboxesServices)
   {
      addServicesToMirror(csvMailboxesServices);
   }
   if (csvMailboxesAppId)
   {
      addAppIdsToMirror(csvMailboxesAppId);
   }
   connectTarget();
}


// constructor
TargetCcaSocketGateway::TargetCcaSocketGateway()
{
}


/* static */
void* TargetCcaSocketGateway::socketServerThreadProc(void* classpointer)
{
   TargetCcaSocketGateway* threadImpl = static_cast<TargetCcaSocketGateway*>(classpointer);
   if (threadImpl)
   {
      threadImpl->socketServerThread();
   }
   return 0;
}


// Main thread on Target. Creates a socket server, unpacks the CCA message payload
//  and forward the CCA message stream with OSAL_s32MessageQueuePost().
bool TargetCcaSocketGateway::socketServerThread()
{
   g_remoteConnected = true;

   while (waitForSocketMessage(g_remoteSockId))
   {
      ;
   }

   g_remoteConnected = false;
   g_remoteSockId = -1;

   return true;
}


bool TargetCcaSocketGateway::startSocketServer(uint32_t port)
{
   amt_bInit();
   scd_init();

   TRACE_MSG(("argetCcaSocketGateway::startSocketServer() start cca gateway as server on port='%d'", port));

   // create socket
   int socket_desc = socket(AF_INET, SOCK_STREAM, 0);
   if (socket_desc == -1)
   {
      TRACE_ERR(("TargetCcaSocketGateway::startSocketServer() Could not create socket err=%s", strerror(errno)));
      return OSAL_ERROR;
   }
   TRACE_MSG(("TargetCcaSocketGateway::startSocketServer() Port=%d Socket created --> sock=%d", port, socket_desc));

   // prepare the sockaddr_in structure
   struct sockaddr_in server;
   server.sin_family = AF_INET;
   server.sin_addr.s_addr = INADDR_ANY;
   server.sin_port = htons(port);
   memset(&server.sin_zero, 0, sizeof(server.sin_zero));

   // bind
   if (bind(socket_desc, (struct sockaddr*)&server, sizeof(server)) < 0)
   {
      //print the error message
      TRACE_ERR(("TargetCcaSocketGateway::startSocketServer() bind failed. err=%s", strerror(errno)));
      close(socket_desc);
      return OSAL_ERROR;
   }
   TRACE_MSG(("TargetCcaSocketGateway::startSocketServer() bind done"));

   // listen
   listen(socket_desc, 3);

   TRACE_MSG(("TargetCcaSocketGateway::startSocketServer() Waiting for incoming connections..."));

   // accept and incoming connection
   int c = sizeof(struct sockaddr_in);
   int client_sock;
   std::vector<int>client_sock_fds;
   struct sockaddr_in client;
   while ((client_sock = accept(socket_desc, (struct sockaddr*)&client, (socklen_t*)&c)))
   {
      TRACE_MSG(("TargetCcaSocketGateway::startSocketServer() Connection accepted: sock=%d", client_sock));

      pthread_t sniffer_thread;
      if (g_remoteSockId == -1)
      {
         g_remoteSockId = client_sock;

         if (pthread_create(&sniffer_thread, NULL,  TargetCcaSocketGateway::socketServerThreadProc, this) < 0)
         {
            TRACE_ERR(("TargetCcaSocketGateway::startSocketServer() Could not create thread err=%s", strerror(errno)));
            close(client_sock);
            close(socket_desc);
            return OSAL_ERROR;
         }

         TRACE_MSG(("TargetCcaSocketGateway::startSocketServer() Handler assigned"));
      }
      // remember open fdds for later closure
      client_sock_fds.push_back(client_sock);
   }

   while (client_sock_fds.size() != 0)
   {
      close(client_sock_fds.back());
      client_sock_fds.pop_back();
   }

   if (client_sock < 0)
   {
      TRACE_ERR(("TargetCcaSocketGateway::startSocketServer(): accept failed err=%s", strerror(errno)));
      close(socket_desc);
      return OSAL_ERROR;
   }

   close(socket_desc);
   return OSAL_OK;
}


class MyBinaryStreamMessage : public amt_tclBaseMessage
{
   public:
      MyBinaryStreamMessage(uint8_t* pStreamData, uint32_t u32MessageSize)
      {
         vSetDynMsgSize(u32MessageSize);
         bAllocateMessage();
         vSetStreamU8(0, pStreamData, u32MessageSize, u32MessageSize);
         //uint32_t res1 = u32GetDynMsgSize();
         //uint32_t res2 = u32GetSize();
         // printf("xxx %d %d", res1, res2);
      }
};


// Header is 8 Bytes long
uint32_t SocketReceiver::waitForSocketHeader(int sockId, uint32_t& cmdId)
{
   uint8_t inMessage[10];
   if (recv(sockId, &inMessage[0], 1, 0) == 1 && inMessage[0] == SOCKETGW_BYTE_0_HEADER_START)
   {
      if (recv(sockId, &inMessage[1], 1, 0) == 1 && inMessage[1] == SOCKETGW_BYTE_1_HEADER_START)
      {
         if (recv(sockId, &inMessage[2], 1, 0) == 1 && inMessage[2] == SOCKETGW_BYTE_2_HEADER_START)
         {
            if (recv(sockId, &inMessage[3], 1, 0) == 1 &&
                  (inMessage[3] == SOCKETGW_BYTE_3_CMD_START_CCA_RECEIVER || inMessage[3] == SOCKETGW_BYTE_3_CMD_SEND_CCA_MSG))
            {
               cmdId = inMessage[3];
               for (int ii = 4; ii < 8; ++ii)
               {
                  ssize_t readSize = recv(sockId, &inMessage[ii], 1, 0);
                  if (readSize != 1)
                  {
                     return 0;
                  }
               }
               // header complete
               uint32_t ccaMsgContentLength = (uint32_t)(inMessage[4] << 24) + (inMessage[5] << 16) + (inMessage[6] << 8) + inMessage[7];
               return ccaMsgContentLength;
            }
         }
      }
   }
   return 0;
}


/*
 * socket message receiver for client and server
 */
bool SocketReceiver::waitForSocketMessage(int sockId)
{
   bool bConnectionUp = true;

   // receive a message from socket, this code is for server an client

   while (bConnectionUp)
   {
      uint32_t cmdId = 0;
      uint32_t ccaMsgContentLength = waitForSocketHeader(sockId, cmdId);
      if (ccaMsgContentLength == 0)
      {
         TRACE_ERR(("SocketReceiver::waitForSocketMessage() ERROR: no content in package !!!"));
         bConnectionUp = false;
         return bConnectionUp;
      }
      //
      // now socket message header is available, read the cca payload
      //
      uint32_t remainingBytes = ccaMsgContentLength;

      TRACE_MSG(("SocketReceiver::waitForSocketMessage() remainingBytes %d", ccaMsgContentLength));

      uint8_t* pMsgData = new uint8_t[ccaMsgContentLength + 30];
      ssize_t index = 0;

      while ((remainingBytes > 0) && bConnectionUp)
      {
         // here is interesting, if recv comes back. This should allways happened
         ssize_t readSize = recv(sockId, pMsgData + index, remainingBytes, 0);
         if (readSize <= 0)
         {
            bConnectionUp = false;
            TRACE_MSG(("SocketReceiver::waitForSocketMessage() read of cca payload failed"));
         }
         else
         {
            index += readSize;
            if (index > ccaMsgContentLength)
            {
               TRACE_ERR(("SocketReceiver::waitForSocketMessage() ERROR: index overflow !!!"));
            }
            remainingBytes -= readSize;
            TRACE_MSG(("SocketReceiver::waitForSocketMessage() successfull read payload size=%ld, remainingBytes=%u", readSize, remainingBytes));
         }
      }

      TRACE_MSG(("SocketReceiver::waitForSocketMessage() sockId %d --> CmdId: %d on : len(%d)", sockId, cmdId, ccaMsgContentLength));

      if (cmdId == SOCKETGW_BYTE_3_CMD_SEND_CCA_MSG)
      {
         // stream to cca container and deliver the received ip packet to CCA mbox
         MyBinaryStreamMessage oMyBaseMsg((uint8_t*)(pMsgData), ccaMsgContentLength);

         uint16_t targetID = oMyBaseMsg.u16GetTargetAppID();
         TRACE_MSG(("cSocketReceiver::postMessage() Msg to send for mbx='%d', length='%d'", targetID, ccaMsgContentLength));
         OSAL_tMQueueHandle hAppMsgQueue = scd_OpenQueue(targetID);
         if (hAppMsgQueue != OSAL_C_INVALID_HANDLE)
         {
            tS32 rc = OSAL_s32MessageQueuePost(hAppMsgQueue, (tCU8*)oMyBaseMsg.prGetOSALMsgHandle(),
                                               sizeof(OSAL_trMessage),
                                               OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);
            if (rc != OSAL_OK)
            {
               TRACE_ERR(("SocketReceiver::postMessage() Post to app '%d' failed.", targetID));
               oMyBaseMsg.bDelete();
            }
         }
         else
         {
            TRACE_ERR(("SocketReceiver::postMessage() Cannot open queue for app '%d'", targetID));
         }
      }

      else if (cmdId == SOCKETGW_BYTE_3_CMD_START_CCA_RECEIVER)
      {
         // start CCA message handler
         uint32_t appId = 0;
         TRACE_MSG(("ccaSocketGw::array size %d\n", (int)(sizeof(*pMsgData) / sizeof(uint8_t))));
         if (ccaMsgContentLength == 2)
         {
            appId += (pMsgData[0] << 8) & 0xff00;
            appId += (pMsgData[1]) & 0x00ff;
            MessageReceiverMirror::startCcaMessageReceiver(appId);
         }
      }
      else
      {
         TRACE_ERR(("SocketReceiver::waitForSocketMessage() ERROR: unknown message type !!!"));
      }

      if (pMsgData)
      {
         delete []pMsgData;
         pMsgData = NULL;
      }
   }
   return bConnectionUp;
}


static bool socketSend(int g_remoteSockId, const unsigned char* buf, ssize_t total_bytes_towrite)
{
   ssize_t total_bytes_written = 0;

   while (total_bytes_written != total_bytes_towrite)
   {
      assert(total_bytes_written < total_bytes_towrite);
      ssize_t bytes_written = write(g_remoteSockId,
                                    &buf[total_bytes_written],
                                    total_bytes_towrite - total_bytes_written);
      if (bytes_written == -1)
      {
         TRACE_ERR(("MessageReceiverMirror::ccaMessageReceiverThreadProc() Write message via socket failed"));
         return false;
      }
      total_bytes_written += bytes_written;
   }
   return true;
}


#if !defined(_LINUXX86_64_) && !defined(GEN3HOST)

#ifdef __cplusplus
extern "C" {
#endif

void vStartApp(int argc, char** argv)
{
   int c;
   uint32_t port = HMIBASE_TGW_DEFAULT_PORT;

   while ((c = getopt(argc, argv, "dp:")) != -1)
   {
      switch (c)
      {
         case 'p':
            port = atoi(optarg);
            break;
         case 'd':
            g_debug = true;
            break;
         default:
            break;
      }
   }

   TargetCcaSocketGateway gateway;
   if (gateway.startSocketServer(port) == true)
   {
      _exit(1);
   }
}


#ifdef __cplusplus
}


#endif

#endif // gen3host
