/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
***************************************************************************/

/*!
*\file     ethernetbus.cpp
*\brief
*
*\author   CM-CI3/ESW-Losch
*
*\par Copyright:
*(c) 2017 Robert Bosch Car Multimedia GmbH
*
*\par History:
* See history of revision control system
***************************************************************************/

/****************************************************************************
| includes
|--------------------------------------------------------------------------*/
#include <ifaddrs.h>
#include <time.h>
#include <arpa/inet.h>
#include <linux/if_packet.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "DoIP_ServerCfg.h"
#include "DoIP_Protocoll.h"
#include "DoIP_Server.h"
#include "DoIP_Factory.h"
#include "DoIP_Channel.h"


#ifdef DOIP_USE_FACTORY
   /** this method is not part of the doip_lib and must also be implemented by the user process in case 
    DOIP_USE_FACTORY is not set*/
DoIP_Channel* DoIP_Channel::poCreate(DoIP_Connection* /* poConn */, DoIP_ChannelType /* type */, tU8 /* busNumber */, tU16 /* testerSA */, tU16 /* targetSA */, tU32 /* baseaddr */, tU8 /* addrext */) {
   return nullptr;
}
#endif

class DoIP_DefaultFactory :
public DoIP_Factory {
public:
   virtual DoIP_Channel *createChannel(DoIP_Connection* poConn, DoIP_ChannelType type, tU8 busNumber, tU16 testerSA, tU16 targetSA, tU32 baseaddr, tU8 addrext) {
      return DoIP_Channel::poCreate(poConn, type, busNumber, testerSA, targetSA, baseaddr, addrext);
   };
};


tS32 tclEthernetBus::s32CurrentTime = 0;

tclEthernetBus::tclEthernetBus(tU8 u8Number, const char* pszDevice)
 : cu8BusNumber(u8Number),
   pszDeviceName(NULL),
   bValidMacAddress(false),
   bIPv4allocated(false),
   bIPv6allocated(false),
   poDoIP_Protocoll(NULL),
   u32IPCheckTimer(cu32IPCheckIntervall)
{
   pszDeviceName = new char[strlen(pszDevice)+1];
   strcpy(pszDeviceName, pszDevice);

   memset(&IPv4,0,sizeof(IPv4));
   memset(&IPv4_bcast,0,sizeof(IPv4_bcast));
   memset(&IPv6,0,sizeof(sockaddr_in6));
}

tclEthernetBus::tclEthernetBus(tU8 u8Number, sockaddr_in rAddr)
 : cu8BusNumber(u8Number),
   pszDeviceName(NULL),
   bValidMacAddress(false),
   bIPv4allocated(false),
   bIPv6allocated(false),
   poDoIP_Protocoll(NULL),
   u32IPCheckTimer(cu32IPCheckIntervall)
{
   IPv4 = rAddr;
   memset(&IPv4_bcast,0,sizeof(IPv4_bcast));
   memset(&IPv6,0,sizeof(IPv6));
}

tclEthernetBus::~tclEthernetBus()
{
   if(poDoIP_Protocoll != NULL)
   {
      delete poDoIP_Protocoll;
      poDoIP_Protocoll = NULL;
   }
   if(pszDeviceName != NULL)
   {
      delete[] pszDeviceName;
   }
}

void tclEthernetBus::vTraceStackDump(void)
{
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &IPv4.sin_addr, addressBuffer, INET_ADDRSTRLEN);

   DOIP_TRACE_NOTICE("\t#%d: %s MAC=%02X:%02X:%02X:%02X:%02X:%02X IP=%s",
                     cu8BusNumber, ((pszDeviceName != NULL) ? pszDeviceName : ""),
                     au8MacAddress[0], au8MacAddress[1], au8MacAddress[2], au8MacAddress[3],
                     au8MacAddress[4], au8MacAddress[5], addressBuffer);

   if(poDoIP_Protocoll != NULL)
   {
      poDoIP_Protocoll->vTraceStackDump();
   }
}

bool tclEthernetBus::bStart()
{
   bCheckIPAddress();
   if(poDoIP_Protocoll != NULL)
   {
      if(bIPv4allocated)
      {
         poDoIP_Protocoll->vLocalIpAddrAssignmentChg(true, IPv4, IPv4_bcast, IPv6);
      }
   }
   return true;
}

bool tclEthernetBus::bStop()
{
   if(poDoIP_Protocoll != NULL)
   {
      poDoIP_Protocoll->vLocalIpAddrAssignmentChg(false, IPv4, IPv4_bcast, IPv6);
   }
   return true;
}

void tclEthernetBus::vCommonTimerTick(void)
{
   if(poDoIP_Protocoll != NULL)
   {
      poDoIP_Protocoll->vCommonTimerTick();
   }

   tU32 u32NextTime = u32IPCheckTimer > cu32CommonTimerStep ?
                        u32IPCheckTimer - cu32CommonTimerStep : 0;

   if(u32IPCheckTimer > 0 && u32NextTime == 0)
   {
      u32IPCheckTimer = cu32IPCheckIntervall;

      // check for address changes and notify DoIP server in case
      if(bCheckIPAddress())
      {
         // address changed!
         if(poDoIP_Protocoll != NULL)
         {
            poDoIP_Protocoll->vLocalIpAddrAssignmentChg(bIPv4allocated || bIPv6allocated, IPv4, IPv4_bcast, IPv6);
         }
      }
   }
   else
   {
      u32IPCheckTimer = u32NextTime;
   }
}


static inline tU8 hexToBin(char hexNibble)
{
   if (hexNibble>='0' && hexNibble<='9') {
      return (tU8)(hexNibble - '0');
   }
   if (hexNibble>='A' && hexNibble<='F') {
      return (tU8)(hexNibble - 'A' + 10);
   }
   if (hexNibble>='a' && hexNibble<='f') {
      return (tU8)(hexNibble - 'a' + 10);
   }
   return 0;
}

static inline tU8 hexToBin(char hexNibbleHigh, char hexNibbleLow) {
   return (tU8)(hexToBin(hexNibbleHigh) << 4) | hexToBin(hexNibbleLow);
}

tU8 tclEthernetBus::bcd2byte(char highNibble, char lowNibble)
{
    tU8 byteValue = 0;
    if (highNibble < '0' || highNibble > '9' || lowNibble < '0' || lowNibble > '9')
    {
        DOIP_TRACE_ERR("tclEthernetBus::bcd2Byte: BCD value out of range High: %c Low: %c . Return 0 instead", highNibble, lowNibble);
    }
    else
    {
        byteValue = (tU8)((highNibble - '0') << 4);
        byteValue |= (tU8)(lowNibble - '0');
    }
    return byteValue;
}


DoIP_Protocoll* tclEthernetBus::poSetDoIPConfig(tU16 u16Port, tU16 u16SourceAddr, const char* pszEID, const char* pszGID) {
   return poSetDoIPConfig(u16Port, u16SourceAddr, pszEID, pszGID, new DoIP_DefaultFactory());
}

DoIP_Protocoll* tclEthernetBus::poSetDoIPConfig(tU16 u16Port, tU16 u16SourceAddr, const char* pszEID, const char* pszGID, DoIP_Factory *poFactory)
{
   DOIP_TRACE_NOTICE("tclEthernetBus::poSetDoIPConfig: port=%u srcaddr=0x%02X EID=%s GID=%s",
                     u16Port, u16SourceAddr, ((pszEID != NULL) ? pszEID : ""), ((pszGID != NULL) ? pszGID : ""));

   // convert the EID and GID from BCD to Binary format or pass them on as NULL pointers
   tU8 au8EID[6];
   tU8 au8GID[6];


   if(pszEID != NULL)
   {
      if(strlen(pszEID) == 12)
      {
         au8EID[0] = hexToBin(pszEID[0], pszEID[1]);
         au8EID[1] = hexToBin(pszEID[2], pszEID[3]);
         au8EID[2] = hexToBin(pszEID[4], pszEID[5]);
         au8EID[3] = hexToBin(pszEID[6], pszEID[7]);
         au8EID[4] = hexToBin(pszEID[8], pszEID[9]);
         au8EID[5] = hexToBin(pszEID[10], pszEID[11]);
      }
      else
      {
         DOIP_TRACE_ERR("tclEthernetBus::poSetDoIPConfig: EID=%s must define exactly 6 bytes", pszEID);
      }
   }

   if(pszGID != NULL)
   {
      if(strlen(pszGID) == 12)
      {
         au8GID[0] = bcd2byte(pszGID[0], pszGID[1]);
         au8GID[1] = bcd2byte(pszGID[2], pszGID[3]);
         au8GID[2] = bcd2byte(pszGID[4], pszGID[5]);
         au8GID[3] = bcd2byte(pszGID[6], pszGID[7]);
         au8GID[4] = bcd2byte(pszGID[8], pszGID[9]);
         au8GID[5] = bcd2byte(pszGID[10], pszGID[11]);         
      }
      else
      {
         DOIP_TRACE_ERR("tclEthernetBus::poSetDoIPConfig: GID=%s must define exactly 6 bytes", pszGID);
      }
   }

   if(poDoIP_Protocoll != NULL)
   {
      delete poDoIP_Protocoll;
      poDoIP_Protocoll = NULL;
   }

   poDoIP_Protocoll = poFactory->createProtocoll(this, u16Port, u16SourceAddr, (pszEID == NULL) ? NULL : au8EID, (pszGID == NULL) ? NULL : au8GID);
   return poDoIP_Protocoll;
}

bool tclEthernetBus::bCheckIPAddress()
{
   struct ifaddrs * ifAddrStruct=NULL;
   struct ifaddrs * ifa=NULL;
   void * tmpAddrPtr=NULL;
   bool   bIPchanged = false;
   bool   bIPfound = false;

   getifaddrs(&ifAddrStruct);

   for (ifa = ifAddrStruct; ifa != NULL; ifa = ifa->ifa_next)
   {
      if (ifa->ifa_addr)
      {
         char addressBuffer[INET6_ADDRSTRLEN] = "???";
         char addressBufferBcast[INET6_ADDRSTRLEN] = "!!!";

         if(ifa->ifa_addr != NULL)
         {
            if (ifa->ifa_addr->sa_family == AF_INET)
            {
               // is a valid IP4 Address
               tmpAddrPtr = &((struct sockaddr_in *)ifa->ifa_addr)->sin_addr;

               if(pszDeviceName != NULL && strcmp(ifa->ifa_name, pszDeviceName) == 0)
               {
                  // XML configuration defines an ethernet device - extract the IP and broadcast IP from it
                  bIPfound = true;
                  IPv4 = *((struct sockaddr_in *)ifa->ifa_addr);

                  // calculate broadcast address from IP and netmask
                  IPv4_bcast = IPv4;
                  IPv4_bcast.sin_addr.s_addr |= ((struct sockaddr_in *)ifa->ifa_netmask)->sin_addr.s_addr;
                  IPv4_bcast.sin_addr.s_addr |= ~(((struct sockaddr_in *)ifa->ifa_netmask)->sin_addr.s_addr);

                  if(!bIPv4allocated) bIPchanged = true;
                  bIPv4allocated = true;
               }
               else if(pszDeviceName == NULL && memcmp(tmpAddrPtr, &IPv4.sin_addr, sizeof(IPv4.sin_addr)) == 0)
               {
                  // XML configuration defines an IP address - address is allocated
                  bIPfound = true;
                  // fill the name
                  pszDeviceName = new char[strlen(ifa->ifa_name)+1];
                  strcpy(pszDeviceName, ifa->ifa_name);

                  // calculate broadcast address from IP and netmask
                  IPv4_bcast = IPv4;
                  IPv4_bcast.sin_addr.s_addr |= ((struct sockaddr_in *)ifa->ifa_netmask)->sin_addr.s_addr;
                  IPv4_bcast.sin_addr.s_addr |= ~(((struct sockaddr_in *)ifa->ifa_netmask)->sin_addr.s_addr);

                  if(!bIPv4allocated) bIPchanged = true;
                  bIPv4allocated = true;
               }

               if(bIPchanged)
               {
                  inet_ntop(AF_INET, &IPv4.sin_addr, addressBuffer, INET_ADDRSTRLEN);
                  inet_ntop(AF_INET, &IPv4_bcast.sin_addr, addressBufferBcast, INET_ADDRSTRLEN);
                  DOIP_TRACE_DEBUG("tclEthernetBus::bCheckIPAddress: changed %s (%s/%s)", ifa->ifa_name, addressBuffer, addressBufferBcast);
               }
            }
            else if (ifa->ifa_addr->sa_family == AF_INET6)
            {
               // is a valid IP6 Address
               tmpAddrPtr=&((struct sockaddr_in6 *)ifa->ifa_addr)->sin6_addr;

               if(pszDeviceName != NULL && strcmp(ifa->ifa_name, pszDeviceName) == 0)
               {
                  // XML configuration defines an ethernet device - extract the IPv6 from it
                  bIPfound = true;
                  IPv6 = *((struct sockaddr_in6 *)ifa->ifa_addr);
                  if(!bIPv6allocated) bIPchanged = true;
                  bIPv6allocated = true;
               }
               else if(pszDeviceName == NULL && memcmp(tmpAddrPtr, &IPv6.sin6_addr, sizeof(IPv6.sin6_addr)) == 0)
               {
                  // XML configuration defines an IPv6 address - address is allocated
                  bIPfound = true;
                  if(!bIPv6allocated) bIPchanged = true;
                  bIPv6allocated = true;
               }
            }
            else if (ifa->ifa_addr->sa_family == AF_PACKET)
            {
               if(pszDeviceName != NULL && strcmp(ifa->ifa_name, pszDeviceName) == 0)
               {
                  // extract the MAC for defined ethernet device
                  struct sockaddr_ll *s = (struct sockaddr_ll*)ifa->ifa_addr;
                  memcpy(au8MacAddress, s->sll_addr, (s->sll_halen > 6) ? 6 : s->sll_halen);
                  bValidMacAddress = true;
               }
            }
         }
      }
   }
   if (ifAddrStruct!=NULL) freeifaddrs(ifAddrStruct);

   if(bIPv4allocated && !bIPfound)
   {
      bIPv4allocated = false;
      bIPchanged = true;
   }
   if(bIPv6allocated && !bIPfound)
   {
      bIPv6allocated = false;
      bIPchanged = true;
   }

   return bIPchanged;
}


bool tclEthernetBus::bRegister(DoIP_Server* poServer, char const* acVIN, tU8 u8VINlen)
{
   if(poDoIP_Protocoll != NULL)
   {
      poDoIP_Protocoll->vSetVIN((tU8*)acVIN, u8VINlen);
      return poDoIP_Protocoll->bRegister(poServer);
   }
   return false;
}

bool tclEthernetBus::bRemove(DoIP_Server* poServer)
{
   if(poDoIP_Protocoll != NULL)
   {
      return poDoIP_Protocoll->bRemove(poServer);
   }
   return false;
}


tS32 tclEthernetBus::s32GetTimestamp()
{
   return s32CurrentTime;
}

void tclEthernetBus::vUpdateTimestamp()
{
   struct timespec rElapsedTime;
   clock_gettime(CLOCK_MONOTONIC, &rElapsedTime);
   s32CurrentTime = ((rElapsedTime.tv_sec*1000)+(rElapsedTime.tv_nsec/1000000)) & 0x7FFFFFFF; // suppress negative values!
}


