/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DoIP_Connection.h
 *\brief    A DoIP connection reflects and manages the connection of this
 *          host and the DoIP tester. It contains the routing tables and
 *          instantiates channels on activation request.
 *
 *\author   CM-CI1/ENP1 - Losch
 *
 *\par Copyright:
 *(c) 2018 Robert Bosch Car Multimedia GmbH
 *
 *\par History:
 * See history of revision control system
 ***************************************************************************/

#ifndef DOIP_CONNECTION_H
#define DOIP_CONNECTION_H


#include "DoIP_ServerCfg.h"
#include "DoIP_Protocoll.h"


class DoIP_Channel;
class DoIP_Server;
class DoIP_Socket;

struct DoIP_AliveCheck
{
   tU32 aliveCheckTimer;
   tU8  aliveCheckBuffer[DOIP_ALIVECHECK_BUFFERSIZE];
   tU8  aliveCheckState;
};




struct DoIP_TXConnectionStatemachine: public DoIP_TCPConnectionStatemachine
{

   DoIP_TXConnectionStatemachine():
      txState(DOIP_CHANNEL_TX_IDLE)
   {}

   tU8  txBuffer[(DOIP_HEADER_SIZE + 2u + 2u + DOIP_CFG_cu32TCPMaxRequestSize)];
   tU8  txState;

};


struct DoIP_RXConnectionStatemachine: public DoIP_TCPConnectionStatemachine
{
   DoIP_RXConnectionStatemachine():
      nextReadLen(0),
      payloadCounter(0),
      recvBufferUsed(0),
      diagMsgTargetAddress(0),
      diagMsgSourceAddress(0),
      cnt(0)
  {}

   tU8     ackBuffer[DOIP_CFG_cu32TCPBufferSize]; // for receiving TCP frames
   tU8     curDoipHeader[DOIP_HEADER_SIZE]; // for receiving TCP frames
   uint32_t nextReadLen;

   tU32 payloadCounter; /* number of payload bytes in payloadBuffer */
   tU8  payloadBuffer[DOIP_CFG_cu32TCPMaxRequestSize + DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN]; // for completely assembled payload (out of multiple TCP frames)

   uint8_t recvBuffer[DOIP_CFG_cu32TCPBufferSize];
   uint32_t recvBufferUsed; // number of bytes in recvBuffer

   tU16 diagMsgTargetAddress;
   tU16 diagMsgSourceAddress;

   tU32 cnt;

   void reset() {
      state=DOIP_TCP_IDLE;
      payloadLength=0;
      payloadType=DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK;
      nextReadLen=0;
      payloadCounter=0;
      recvBufferUsed=0;
      diagMsgTargetAddress=0;
      diagMsgSourceAddress=0;
   }

};

class DoIP_Connection
{
   // Must not be copied or assigned to prevent multiple close calls
   // for socketFd member in destructor.
    DoIP_Connection ( const DoIP_Connection& obj ) = delete;                                                           \
    DoIP_Connection& operator = ( const DoIP_Connection& obj ) = delete;
public:
   DoIP_Connection(DoIP_Factory *poFactory, DoIP_Protocoll* poServer, DoIP_Socket *psocket);
   virtual ~DoIP_Connection();

   MOCK_VIRTUAL void vTraceStackDump(void);

   static const tU16 cu16TesterAddrInit = 0xFFFF;
   static const tU32 TCPSOCKET_LISTENER_STACKSIZE  = (96*1024);

   static const bool cbDiagMsgFastProcessing = true; // processes DiagMsgs directly from RxThread instead of CommonTimer (increases performance)

   MOCK_VIRTUAL DoIP_Server* poGetDoIPServer();
   MOCK_VIRTUAL bool bRemove(DoIP_Channel* poChan);

   MOCK_VIRTUAL void vCommonTimerTick(void);

   MOCK_VIRTUAL void vConnectionOpen(); // immediate open for usage, called DoIP_Protocol on opened socket
   MOCK_VIRTUAL bool bConnectionIsActive();
   MOCK_VIRTUAL void vConnectionClose(bool isSoConModeChg);

   MOCK_VIRTUAL bool bChannel_Transmit(DoIP_Channel* poChannel, tU8 const au8Data[], tU32 u32Len);

   MOCK_VIRTUAL tU16 u16GetTesterAddress() { return rActiveTesterCfg->first; }
   MOCK_VIRTUAL void vRoutingActivationAuthenticationResp(bool Authentified, tU8* AuthenticationResData);
   MOCK_VIRTUAL bool bCheckRoutingActivationRegistrationStatus(tU8 registrationStatus);
   MOCK_VIRTUAL int iGetSocket();

protected:
   // check received doip-header (start of each doip-message)
   void vHandleRxDoipHeader();
   // check received payload
   void vHandleRxPayloadIndication();
   // dispatch complete doip-message accoding to payload-type
   void vHandleRxMsgComplete();

   // handle doip-meesage DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_REQ
   void vHandleRxRoutingActivationRequest();

   // handle doip-meesage DOIP_PAYLOAD_TYPE_DIAG_MSG
   void vHandleRxAliveCheckResp();

   // handle doip-meesage DOIP_PAYLOAD_TYPE_DIAG_MSG
   void vHandleRxDiagMsg();


   // helper to assemble doip-header
   void setDoipHeader(uint16_t payloadType, uint32_t payloadLength, uint8_t *doipHeader);



   bool bIsRoutingActivationTypeSupported(tU8 activationType);
   bool bIsConnectionAlreadyAssginedToOtherSourceAddress(tU16 sourceAddress);
   bool bIsActivationTypeActive(tU8 activationType);
   bool bCheckRoutingActivationAuthenticationStatus(tU8 authenticationStatus);



protected:
   DoIP_Channel* poFindChannel(tU16 targetAddr);
   void vChannelCloseAll();
   void vOpenTCP();
   void vCloseTCP();
   void vSetGeneralInactivity();
   void vSetInitialInactivity();
   void vHandleTimer();
   void vHandleMessages();
   bool bSendMessage(tU8* data, bool isDiagMessage);

   tU16 u16GetAckNackResponseLength();
   tU8  u8CheckDiagMsgHeader();
   bool bCheckTaConfiguration(); // check if target address is contained in activated route
   bool bCheckTaAtConfiguration(); // check if route to target is activated
   void vPrepareDiagnosticAcknowledgeMessage(tU8 responseCode);
   void vDiagnosticMessageAcknowledgeTxConfirmation();
   void vDiagnosticMessageTxConfirmation();
   void vGenericAcknowledgeTxConfirmation();


   bool bRoutingActivationIsValid(tU16 sourceAddr);
   void vRoutingActivationMainFunction();
   void vRoutingActivationRxHandleRequest();
   void vRoutingActivationTxConfirmation();
   void vRoutingActivationPollResults();
   void vRoutingActivationPrepareResponse(tU8 responseCode, const tU8 *oem);
   bool bIsSourceAddressRegistered(tU16 sourceAddr);
   bool bIsSourceAddressRegisteredToConnectionWithAnotherActivationType(tU16 sourceAddr, tU8 activationType);
   void vCheckRoutingAuthentication(tU16 sourceAddr, tU8 activationType);
   void vCheckRoutingConfirmation(tU16 sourceAddr, tU8 activationType);
   void vSetRoutingActivationAuthenticationStatus(tU8 authenticationStatus);
   void vSetRoutingActivationConfirmationStatus(tU8 confirmationStatus);
   void vSetRoutingActivationStatus(bool active, bool registered, tU16 sourceAddr,tU8 activationType);
   void vUpdateRoutingActivationStatus(bool active, bool registered);

   void vAliveCheckMainFunction();
   void vAliveCheckTxPrepareRequest();
   void vAliveCheckTxConfirmation();
   void vAliveCheckRxHandle();
   bool bAliveCheckIsActive();
   void vAliveCheckRequest();

protected:
   std::map<tU16, DoIP_TesterCfg>::iterator           rActiveTesterCfg;
   std::map<tU8, DoIP_RoutingActivationCfg>::iterator rActiveRouteCfg;

   std::list<DoIP_Channel*> oActiveChannels; // targets that are activated by current route and connection is established
   DoIP_Channel*  poCurrentChannel;
   sockaddr_in remoteAddr;
   sockaddr_in6 remoteAddrIpv6;

   // TCP
   tU8 state;

   tU32 timerInactivity;
   DoIP_AliveCheck aliveCheck;
   DoIP_RoutingActivation routingActivation;
   DoIP_TXConnectionStatemachine TX;
   DoIP_RXConnectionStatemachine RX;

   pthread_t rTCPReadThread;

   bool bReadNextFrameTCP(void);
   static void* pvTCPReadThread(void* pvParent);
protected:
   DoIP_Protocoll* poDoIPProtocol;
   DoIP_Factory *mpoFactory;
   DoIP_Socket *poSocket;
private:
   
   ssize_t readNBytes(uint8_t *buffer, uint32_t numBytesRequested);

   
   tU32 getU32(void *src) {
      tU32 res;
      poDoIPProtocol->vMemCopyWithEndianessConversion(&res, src, 4);
      return res;
   }
   tU16 getU16(void *src) {
      tU16 res;
      poDoIPProtocol->vMemCopyWithEndianessConversion(&res, src, 2);
      return res;
   }
   tU8 getU8(void *src) {
      tU8 res;
      poDoIPProtocol->vMemCopyWithEndianessConversion(&res, src, 1);
      return res;
   }

   void setU32(void *dst, tU32 src) {
      poDoIPProtocol->vMemCopyWithEndianessConversion(dst, &src, 4);
   }
   void setU16(void *dst, tU16 src) {
      poDoIPProtocol->vMemCopyWithEndianessConversion(dst, &src, 2);
   }
   void setU8(void *dst, tU8 src) {
      poDoIPProtocol->vMemCopyWithEndianessConversion(dst, &src, 1);
   }


   void copyU32(void *dst, void *src) {
      poDoIPProtocol->vMemCopyWithEndianessConversion(dst, src, 4);
   }
   void copyU16(void *dst, void *src) {
      poDoIPProtocol->vMemCopyWithEndianessConversion(dst, src, 2);
   }
   void copyU8(void *dst, void *src) {
      poDoIPProtocol->vMemCopyWithEndianessConversion(dst, src, 1);
   }
};


#endif /*DOIP_CONNECTION_H*/
