/**********************************************************************************************************************
 *
 * COPYRIGHT RESERVED, Robert Bosch GmbH, 2013. All rights reserved.
 * The reproduction, distribution and utilization of this document as well as the communication of its contents to
 * others without explicit authorization is prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 **********************************************************************************************************************/

#ifndef DOIP_TYPES_H
#define DOIP_TYPES_H



#include "DoIP_BaseTypes.h"

#ifndef DOIP_USE_EXTERNAL_TRACE
#include "DoIP_TraceTypes.h"
#endif


#ifndef MOCK_VIRTUAL
#define MOCK_VIRTUAL
#ifdef UNIT_TEST
#error "MOCK_VIRTUAL has to be defined if UNIT_TEST is defined"
#endif
#endif



#include <map>
#include <list>


//msg header structure
#define DOIP_MSG_POS_PROTOCOL_VERSION           0u
#define DOIP_MSG_SIZE_PROTOCOL_VERSION          1u
#define DOIP_MSG_POS_INV_PROTOCOL_VERSION       1u//(DOIP_MSG_POS_PROTOCOL_VERSION+DOIP_MSG_SIZE_PROTOCOL_VERSION)
#define DOIP_MSG_SIZE_INV_PROTOCOL_VERSION      1u
#define DOIP_MSG_POS_PAYLOAD_TYPE               2u//(DOIP_MSG_POS_INV_PROTOCOL_VERSION+DOIP_MSG_SIZE_INV_PROTOCOL_VERSION)
#define DOIP_MSG_SIZE_PAYLOAD_TYPE              2u
#define DOIP_MSG_POS_PAYLOAD_LENGTH             4u//(DOIP_MSG_POS_PAYLOAD_TYPE+DOIP_MSG_SIZE_PAYLOAD_TYPE)
#define DOIP_MSG_SIZE_PAYLOAD_LENGTH            4u
#define DOIP_MSG_POS_CONTENT                    8u//(DOIP_MSG_POS_PAYLOAD_LENGTH+DOIP_MSG_SIZE_PAYLOAD_LENGTH)

#define DOIP_HEADER_SIZE                        DOIP_MSG_POS_CONTENT

#define DOIP_ALIVECHECK_BUFFERSIZE              (DOIP_HEADER_SIZE + DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_RESP)

//payload types
#define DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK       0x0000u
#define DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ            0x0001u
#define DOIP_PAYLOAD_TYPE_VEHICLE_ID_EID_REQ        0x0002u
#define DOIP_PAYLOAD_TYPE_VEHICLE_ID_VIN_REQ        0x0003u
#define DOIP_PAYLOAD_TYPE_VEHICLE_ID_RESP           0x0004u
#define DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_REQ    0x0005u
#define DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_RESP   0x0006u
#define DOIP_PAYLOAD_TYPE_ALIVE_CHECK_REQ           0x0007u
#define DOIP_PAYLOAD_TYPE_ALIVE_CHECK_RESP          0x0008u
#define DOIP_PAYLOAD_TYPE_ENTITY_STATUS_REQ         0x4001u
#define DOIP_PAYLOAD_TYPE_ENTITY_STATUS_RESP        0x4002u
#define DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_REQ        0x4003u
#define DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_RESP       0x4004u
#define DOIP_PAYLOAD_TYPE_DIAG_MSG                  0x8001u
#define DOIP_PAYLOAD_TYPE_DIAG_MSG_ACK              0x8002u
#define DOIP_PAYLOAD_TYPE_DIAG_MSG_NACK             0x8003u

//payload length
#define DOIP_PAYLOAD_LENGTH_VEHICLE_ID_REQ              0x00000000u
#define DOIP_PAYLOAD_LENGTH_VEHICLE_ID_EID_REQ          0x00000006u
#define DOIP_PAYLOAD_LENGTH_VEHICLE_ID_VIN_REQ          0x00000011u
#define DOIP_PAYLOAD_LENGTH_VEHICLE_ID_RESP             0x00000021u
#define DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_7    0x00000007u
#define DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_11   0x0000000Bu
#define DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_RESP_9   0x00000009u
#define DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_RESP_13  0x0000000Du
#define DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_RESP            0x00000002u
#define DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_REQ             0x00000000u
#define DOIP_PAYLOAD_LENGTH_ENTITY_STATUS_REQ           0x00000000u

#define DOIP_PAYLOAD_LENGTH_ENTITY_STATUS_RESP_LONG     0x00000007u
#define DOIP_PAYLOAD_LENGTH_ENTITY_STATUS_RESP_SHORT    0x00000003u

#define DOIP_PAYLOAD_LENGTH_DIAG_POWERMODE_REQ          0x00000000u
#define DOIP_PAYLOAD_LENGTH_DIAG_POWERMODE_RESP         0x00000001u
#define DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN                0x00000004u
#define DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_ACK_MIN      0x00000005u
#define DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_NACK_MIN     0x00000005u
#define DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_NACK         0x00000001u

//negative acknowledge codes
#define DOIP_NACK_PROTOCOL_INFO_INCORRECT   0x00u
#define DOIP_NACK_UNKNOWN_PAYLOAD_TYPE      0x01u
#define DOIP_NACK_MSG_TOO_LARGE             0x02u
#define DOIP_NACK_OUT_OF_MEMORY             0x03u
#define DOIP_NACK_INVALID_PAYLOAD_LENGTH    0x04u

//protocol version
#define DOIP_PROTOCOL_VERSION                0x02u
#define DOIP_PROTOCOL_VERSION_INV            0xFDu//(~DOIP_PROTOCOL_VERSION)
#define DOIP_PROTOCOL_VERSION_DEFAULT        0xFFu
#define DOIP_PROTOCOL_VERSION_DEFAULT_INV    0x00u//(~DOIP_PROTOCOL_VERSION_DEFAULT)

//further action
#define DOIP_FURTHER_ACTION_NOT_REQUIRED                    0x00u
#define DOIP_FURTHER_ACTION_ROUTING_ACTIVATION_REQUIRED     0x10u

/* VIN & GID synchronized */
#define DOIP_VIN_GID_SYNCHRONIZED        0x00u
#define DOIP_VIN_GID_NOT_SYNCHRONIZED    0x10u

/* VIN & GID sync state */
#define DOIP_GIDSYNC_IDLE               0x00u
#define DOIP_GIDSYNC_PENDING            0x01u
#define DOIP_GIDSYNC_SUCCESS            0x02u

//routing activation types
#define DOIP_ROUTING_ACTIVATION_TYPE_DEFAULT            0x00u
#define DOIP_ROUTING_ACTIVATION_TYPE_WWH_OBD            0x01u
#define DOIP_ROUTING_ACTIVATION_TYPE_CENTRAL_SECURITY   0xE0u

//UDP port range acc. to ISO13400-2 [DoIP-135]
#define DOIP_UDP_PORT_MIN           0xC000u
#define DOIP_UDP_PORT_MAX           0xFFFFu

#define DOIP_UDP_PORT_DISCOVERY     13400u //0x3458
#define DOIP_TCP_PORT_DATA          13400u
#define DOIP_UNIX_PORT_DATA         33400u
#define DOIP_UDP_DISCOVERY_REPETITIONEN 03u

typedef enum
{
   DOIP_GATEWAY = 0x00u
   ,DOIP_NODE   = 0x01u
}DoIP_NodeType;

/* Socket connection state */
#define DOIP_CON_OFFLINE                    0x00u
#define DOIP_CON_OFFLINE_OPEN_REQUESTED     0x01u
#define DOIP_CON_ONLINE                     0x02u
#define DOIP_CON_ONLINE_CLOSE_NEEDED        0x03u
#define DOIP_CON_ONLINE_CLOSE_REQUESTED     0x04u

/* UDP connection state */
#define DOIP_UDP_IDLE                       0x00u
#define DOIP_UDP_DISCOVERY                  0x01u

/* UDP queue state*/
#define DOIP_UDP_DATA_EMPTY                 0x00u
#define DOIP_UDP_DATA_RECEIVED              0x01u
#define DOIP_UDP_DATA_PROCESSED             0x02u
#define DOIP_UDP_DATA_READY                 0x03u
#define DOIP_UDP_DATA_WAITING_POWER_MODE    0x04u

/* TCP connection state */
#define DOIP_TCP_IDLE                       0x00u
#define DOIP_TCP_RX_LOCK                    0x01u
#define DOIP_TCP_RX_RECEIVING               0x02u
#define DOIP_TCP_RX_READY2SEND              0x03u
#define DOIP_TCP_RX_WAIT4CONFIRMATION       0x04u
#define DOIP_TCP_TX_LOCK                    0x05u
#define DOIP_TCP_TX_WAIT2SEND               0x06u
#define DOIP_TCP_TX_SENDING                 0x07u
#define DOIP_TCP_TX_WAIT4CONFIRMATION       0x08u







/**
 **************************************************************************************************
 * macro to suppress QAC warnings for unused parameters
 **************************************************************************************************
 */

#define DOIP_GA_RESPCODE_INVALID_PROTOCOL               0x00u
#define DOIP_GA_RESPCODE_PAYLOAD_TYPE_UNSUPPORTED       0x01u
#define DOIP_GA_RESPCODE_EXCEEDS_MAX_REQ_BYTES          0x02u
#define DOIP_GA_RESPCODE_SOR_EXCEEDS_DOIP_BUFFER        0x03u
#define DOIP_GA_RESPCODE_INVALID_PAYLOAD_LENGTH_TYPE    0x04u

#define DOIP_E_UNINIT                       0x01
#define DOIP_E_PARAM_POINTER                0x02
#define DOIP_E_INVALID_PDU_SDU_ID           0x03
#define DOIP_E_INVALID_PARAMETER            0x04
#define DOIP_E_INIT_FAILED                  0X05
#define DOIP_E_PENDING                      0x0A //10
#define DOIP_E_ACTIVAITON_LINE              0x10

#define DOIP_E_NOT_SUPPORTED                0xFD
#define DOIP_E_ASSERT                       0xFE
#define DOIP_E_SHOULD_NOT_REACH             0xFF

typedef unsigned int            tDoIPResult;

#define DOIP_SUCCESS                                         ((tDoIPResult) 0x00000000)
#define DOIP_DONE                                            ((tDoIPResult) 0x00000000)
#define DOIP_E_NOERROR                                       ((tDoIPResult) 0x00000000)
#define DOIP_E_NO_ERROR                                      ((tDoIPResult) 0x00000000)

#define DOIP_FAILED                                          ((tDoIPResult) 0x00000001)
#define DOIP_NOT_DONE                                        ((tDoIPResult) 0x00000001)
#define DOIP_E_ERROR                                         ((tDoIPResult) 0x00000001)


#define DOIP_NOT_READY 0
#define DOIP_READY 1
#define DOIP_NOT_SUPPORTED 2


#define DOIP_ALIVECHECK_IDLE                0x00u
#define DOIP_ALIVECHECK_REQUESTED           0x01u
#define DOIP_ALIVECHECK_READY2SEND          0x02u
#define DOIP_ALIVECHECK_WAIT4CONFIRMATION   0x03u
#define DOIP_ALIVECHECK_WAIT4RESPONSE       0x04u
#define DOIP_ALIVECHECK_RECEIVED            0x05u

#define DOIP_DM_ACKCODE_SUCCESS             0x00u

#define DOIP_DM_NACKCODE_INVALID_SA         0x02u
#define DOIP_DM_NACKCODE_UNKNOWN_TA         0x03u
#define DOIP_DM_NACKCODE_MSG_TOO_LARGE      0x04u
#define DOIP_DM_NACKCODE_OUT_OF_MEM         0x05u
#define DOIP_DM_NACKCODE_TA_UNREACHABLE     0x06u
#define DOIP_DM_NACKCODE_UNKNOWN_NET        0x07u
#define DOIP_DM_NACKCODE_TP_ERROR           0x08u

#define DOIP_DM_NACKCODE_UNDEFINED          0xFFu

#define  DOIP_MIN(a,b)  (((a)<(b))?(a):(b))
#define  DOIP_MAX(a,b)  (((a)>(b))?(a):(b))


#define DOIP_CHANNEL_TX_IDLE                0x00u
#define DOIP_CHANNEL_TX_LOCK                0x01u
#define DOIP_CHANNEL_TX_READY2SEND          0x02u
#define DOIP_CHANNEL_TX_SENDING             0x03u
#define DOIP_CHANNEL_TX_WAIT4CONFIRMATION   0x04u



typedef enum
{
   DOIP_LOCAL = 0,

   DOIP_CAN_NORMAL,
   DOIP_CAN_EXTENDED,
   DOIP_CAN_MIXED,
   DOIP_CAN_NORMAL_FIXED,
   DOIP_CAN_REMOTE_NORMAL,
   DOIP_CAN_REMOTE_EXTENDED,
   DOIP_CAN_REMOTE_MIXED,
   DOIP_CAN_REMOTE_NORMAL_FIXED,

   DOIP_UDS_CAN_NORMALFIXED,
   DOIP_UDS_CAN_NORMAL,
   DOIP_UDS_CAN_MIXED,
   DOIP_UDS_CAN_EXTENDED,
   DOIP_KWP_CAN_NORMALFIXED,
   DOIP_KWP_CAN_NORMAL,
   DOIP_KWP_CAN_MIXED,
   DOIP_KWP_CAN_EXTENDED,
   DOIP_UDS_CAN_REMOTE_NORMALFIXED,
   DOIP_UDS_CAN_REMOTE_NORMAL,
   DOIP_UDS_CAN_REMOTE_MIXED,
   DOIP_UDS_CAN_REMOTE_EXTENDED,
   DOIP_KWP_CAN_REMOTE_NORMALFIXED,
   DOIP_KWP_CAN_REMOTE_NORMAL,
   DOIP_KWP_CAN_REMOTE_MIXED,
   DOIP_KWP_CAN_REMOTE_EXTENDED,

   DOIP_J1939,
   DOIP_J1939_DM,
} DoIP_ChannelType;

typedef struct
{
   DoIP_ChannelType  type; // type of channel used to reach the target
   tU16            targetSA; // address of the target
   tU32            targetBaseAddr; //e.g. for IsoTp NormalFixed
   tU16            targetAddrExt; //e.g. for IsoTp NormalFixed
   tU8             busNumber; // busnumber to reach the target
} DoIP_RoutingActivationTarget;

typedef struct
{
   //tU8  routingActivationNumber;
   bool bAuthRequ; // indicate that authentication is required for the route
   std::list<DoIP_RoutingActivationTarget> targetCfg; // list of all targets that can be reached by this route 
} DoIP_RoutingActivationCfg;


typedef struct
{
   //tU16 testerSA;
   tU16 numByteDiagAckNack; //according to DoIP big enough to store 0x1999.9999.9999.9999
   // define a route (routeNumber or actNum) and the channels (DoIP_RoutingActivationCfg) that belong to the route
   //map<routeNumber, DoIP_RoutingActivationCfg>
   std::map<tU8, DoIP_RoutingActivationCfg> oRouteActCfg;
} DoIP_TesterCfg;

/**
 * @ingroup DOIP_TYPES_H
 * Definition of the configuration structure that is passed to DoIP_Init().
 */
typedef struct
{
   tU16 portNumber; // UDP-Port the server listens on ???

   // identification
    tU8  acEID[6];
    bool bValidEID;
    tU8  acGID[6];
    bool bValidGID;
    char acVIN[17];
    bool bValidVIN;

   // configured testers map<tester-addr, TesterCfg>
    std::map<tU16, DoIP_TesterCfg> oTesters;
} DoIP_ConfigType;


struct DoIP_TCPConnectionStatemachine {


   DoIP_TCPConnectionStatemachine():
      state(DOIP_TCP_IDLE),
      payloadLength(0),
      payloadType(0)
   {}

   tU8  state;
   tU32 payloadLength;
   tU16 payloadType;
};

#define DOIP_ROUTINGACTIVATION_BUFFERSIZE          (DOIP_HEADER_SIZE + DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_RESP_13)


struct DoIP_RoutingActivationMessage {
   tU16 testerAddress;
   tU8  activationType;
   tU8  iso[4];
   tU8  oem[4];
   bool  hasOem;
};


struct DoIP_RoutingActivation
{
   tU16 routingActivationSA; // address of tester
   tU8  routingActivationBuffer[DOIP_ROUTINGACTIVATION_BUFFERSIZE];
   tU8  routingActivationType;
   tU8  routingActivationStatus;    //TODO rename
   tU8  routingActivationOemBufferReq[4];
   tU8  routingActivationOemBufferResp[4];
   tU8  routingActivationState;
   DoIP_RoutingActivationMessage rx;

};




#endif

