/*!
 * \file       vd_diaglog_dtcConfigurationChk.cpp
 *
 * \brief      class to check whether DTC is enabled or disabled
 *
 * \details    ...
 *
 * \component  Diaglog
 *
 * \ingroup    DTC Enable/Disable Configuration
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       26.05.2020
 *
 * \copyright  (c) 2020 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_VD_DIAGLOG_DTC_CONFIGURATION_CHECK__
#include "project/framework/dtcConfiguration/vd_diaglog_dtcConfigurationChk.h"
#endif

#ifndef __INCLUDED_VD_DIAGLOG_DTC_DATAPOOL_CONFIGURATION__
#include "project/framework/dtcConfiguration/vd_diaglog_dtcDataPoolConfig.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_itc_definition_common
#include <common/framework/vd_diaglog_itc_definition_common.h>
#endif

#ifndef __INCLUDED_VD_DIAGLOG_DTC_CONFIGURATION_DEFS__
#include "project/framework/dtcConfiguration/vd_diaglog_dtcConfig_defs.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_trace
#include <common/framework/vd_diaglog_trace.h>  
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_dtcConfigurationChk.cpp.trc.h"   
#endif

#define DTC_ENABLED		((tU8)0x01)

// Instantiate the static class object
dtcConfigurationChk* dtcConfigurationChk::_poDtcCfgChk = NULL;

dtcConfigurationChk*
getInstanceOfDTCCfgChecker ( void )
{
   return dtcConfigurationChk::getInstance();
}

void
releaseInstanceOfDTCCfgChecker ( void )
{
   dtcConfigurationChk::deleteInstance();
}

/**
 * Description     : getInstance, returns the instance of the class
 *                   dtcConfigurationChk
 * @param[in]      : none
 * @param[in]      : none
 * @return         : dtcConfigurationChk*
 */
dtcConfigurationChk* 
dtcConfigurationChk::getInstance()
{
	if (_poDtcCfgChk == NULL)
	{
		_poDtcCfgChk = new dtcConfigurationChk();
	}
	
	return _poDtcCfgChk;
}

/**
 * Description     : deleteInstance, deletes the instance of the class
 *                   dtcConfigurationChk
 * @param[in]      : none
 * @param[in]      : none
 * @return         : void
 */
void
dtcConfigurationChk::deleteInstance ( void )
{
	if ( _poDtcCfgChk )
	{  
		(void) _poDtcCfgChk->tearDown();
		delete _poDtcCfgChk; 
		_poDtcCfgChk = 0; 
	}
}

/**
 * Description     : Constructor of dtcConfigurationChk
 *
 * @param[in]      : none
 * @param[in]      : None
 * @return         : void
 */
dtcConfigurationChk::dtcConfigurationChk()
: mDtcCfgData(0)
{
   ETG_TRACE_USR1_THR(("--> dtcConfigurationChk::dtcConfigurationChk"));
   ETG_TRACE_USR1_THR(("<-- dtcConfigurationChk::dtcConfigurationChk")); 
}

/**
 * Description     : Destructor of dtcConfigurationChk
 *
 * @param[in]      : none
 * @param[in]      : None
 * @return         : void
 */
dtcConfigurationChk::~dtcConfigurationChk()
{
	if ( mDtcCfgData )
    {
       delete [] mDtcCfgData;
       mDtcCfgData = 0;
    }
   
	if (_poDtcCfgChk != NULL )
	{
		delete _poDtcCfgChk;
	}
	_poDtcCfgChk = NULL;
}

/**
 * Description     : Level 2 Constructor of dtcConfigurationChk
 *
 * @param[in]      : none
 * @param[in]      : None
 * @return         : void
 */
tDiaResult
dtcConfigurationChk::setup ( void )
{
	ETG_TRACE_USR1_THR(("---> dtcConfigurationChk::setup"));
	tDiaResult retCode = DIA_FAILED;
	
	dtcDataPoolConfig* poDpCfg = getInstanceOfDTCDataPoolConfig();
	if(poDpCfg)
	{
		mDtcCfgData = new tU8[DTC_CFG_DATAPOOL_DATA_LENGTH];
		if(DIA_SUCCESS == poDpCfg->getDTCCfg(mDtcCfgData))
		{
			ETG_TRACE_USR1_THR(("--- DTC CFG Datapool Read Succeeded"));
			prepareDatabase();
			retCode = DIA_SUCCESS;
		}
		else
		{
			ETG_TRACE_USR1_THR(("--- DTC CFG Datapool Read Failed"));
		}
	}
   
	ETG_TRACE_USR1_THR(("<--- dtcConfigurationChk::setup"));
	return retCode;
}

/**
 * Description     : Level 2 Destructor of dtcConfigurationChk
 *
 * @param[in]      : none
 * @param[in]      : None
 * @return         : void
 */
tDiaResult
dtcConfigurationChk::tearDown ( void )
{
	ETG_TRACE_USR1_THR(("---> dtcConfigurationChk::setup"));
	ETG_TRACE_USR1_THR(("<--- dtcConfigurationChk::setup"));
    return DIA_SUCCESS;
}

/**
* Description     : prepareDatabase, prepares the DTC cfg map
*
* @param[in]      : void
* @return         : void
*/
void 
dtcConfigurationChk::prepareDatabase( void )
{
	ETG_TRACE_USR1_THR(("--> dtcConfigurationChk::prepareDatabase(...)"));
	
	for(int i = 0; i < g_dtcListSize(); ++i)
	{
		_dtcConfigTable.insert({g_dtcList[i], mDtcCfgData[i]});
		//ETG_TRACE_USR1_THR(( "---  Updated database with DTC 0x%03x", (tU32)g_dtcList[i]));
		ETG_TRACE_USR1_THR(( "---  Updated database with DTC 0x%03x and it is %s", (tU32)g_dtcList[i], ((mDtcCfgData[i] == DTC_ENABLED) ? "ENABLED" : "DISABLED")));
	}
	
	ETG_TRACE_USR1_THR(("<-- dtcConfigurationChk::prepareDatabase(...)"));
}

/**
* Description     : isDTCEnabled, check if the ITC is enabled or disabled
*
* @param[in]      : tU16 ITC
* @return         : bool - yes/no
*/
bool 
dtcConfigurationChk::isDTCEnabled( tU16 ITC )
{
	ETG_TRACE_USR1_THR(("--> dtcConfigurationChk::isDTCEnabled(tU16 ITC)"));
	
	bool retVal = true;
	
	ETG_TRACE_USR1_THR(("--- Received ITC to check : 0x%02x ", ITC ));
	
	//std::map<tU16, tU32>::iterator itc_it;
	
	auto itc_it = _itc2dtcMapTable.find(ITC);
	
	if(itc_it != _itc2dtcMapTable.end())
	{
		//ITC found from the table, get the corresponding DTC
		tU32 DTC = itc_it->second;
		
		retVal = isDTCEnabled( DTC );
	}
	else
	{
		ETG_TRACE_USR1_THR(("--- ITC 0x%02x is Not Found in Database, So by default status is enabled", ITC ));
	}
	
	ETG_TRACE_USR1_THR(("<-- dtcConfigurationChk::isDTCEnabled(tU16 ITC)"));
	
	return retVal;
}

/**
* Description     : isDTCEnabled, prepares the DTC cfg map
*
* @param[in]      : tU32 DTC
* @return         : bool - yes/no
*/
bool 
dtcConfigurationChk::isDTCEnabled( tU32 DTC )
{
	ETG_TRACE_USR1_THR(("--> dtcConfigurationChk::isDTCEnabled(tU32 DTC)"));
	
	bool retVal = true;
	
	ETG_TRACE_USR1_THR(("--- Received DTC to check : 0x%03x ", DTC ));
	// Got DTC, Now check if it exists in the database
	//std::map<tU16, tU8>::iterator dtc_it;
	
	auto dtc_it = _dtcConfigTable.find(DTC);
	
	if(dtc_it != _dtcConfigTable.end())
	{
		// DTC found, check in the database whether it is enabled or disabled
		tU8 dtcCfg = dtc_it->second;
		
		if( DTC_ENABLED == dtcCfg )
		{
			ETG_TRACE_USR1_THR(("--- DTC 0x%03x is Enabled", DTC ));
		}
		else
		{
			ETG_TRACE_USR1_THR(("--- DTC 0x%03x is Disabled", DTC ));
			retVal = false;
		}
	}
	else
	{
		ETG_TRACE_USR1_THR(("--- DTC 0x%03x is Not Found in Database, So by default status is enabled", DTC ));
	}
	
	
	ETG_TRACE_USR1_THR(("<-- dtcConfigurationChk::isDTCEnabled(tU32 DTC)"));
	
	return retVal;
}

/**
* Description     : itc to dtc map table
*
* This table is used to fetch the DTC when ITC is known
* 
*/
const std::map<const tU16, const tU32>  dtcConfigurationChk::_itc2dtcMapTable =
{
/*   ITC                                                    , DTC                                                                                                                  */
	{ITC_CMC_VCAN_TIME_OUT                                  , DTC_CMC_VCAN_TIME_OUT										},/* CAN 1 Time out                                        */
	{ITC_CMC_VCAN_BUS_OFF                                   , DTC_CMC_VCAN_NO_COMMUNICATION								},/* No CAN 1 communication possible                       */
	{ITC_CMC_MCAN_TIME_OUT                                  , DTC_CMC_MCAN_TIME_OUT										},/* CAN 2 Time out                                        */
	{ITC_CMC_MCAN_BUS_OFF                                   , DTC_CMC_MCAN_NO_COMMUNICATION								},/* No CAN 2 communication possible                       */
	{ITC_CMC_ETHERNET_NO_COMMUNICATION                      , DTC_CMC_ETHERNET_NO_COMMUNICATION							},/* Ethernet communication not possible                   */
	{ITC_SPM_VOLTAGE_ABOVE_THRESHOLD                        , DTC_CMC_HIGH_VOLTAGE_ERROR								},/* High voltage error                                    */
	{ITC_SPM_VOLTAGE_BELOW_THRESHOLD                        , DTC_CMC_LOW_VOLTAGE_ERROR									},/* Low voltage error              					   */
	{ITC_HEATCONTROL_DISPLAY_CRITICAL_TEMP_HIGH             , DTC_CMC_DISPLAY_OVER_TEMPERATURE							},/* Display                                               */
    {ITC_HEATCONTROL_AMPLIFIER_CRITICAL_TEMP_HIGH           , DTC_CMC_AMPLIFIER_OVER_TEMPERATURE						},/* ECU Amplifier                                         */
	{ITC_HEATCONTROL_CMC_MAIN_PROCESSOR_CRITICAL_TEMP_HIGH  , DTC_CMC_MAIN_PROCESSOR_OVER_TEMPERATURE					},/* Main processor                                        */
	{ITC_HEATCONTROL_CMC_CO_PROCESSOR_CRITICAL_TEMP_HIGH    , DTC_CMC_CO_PROCESSOR_OVER_TEMPERATURE						},/* Co processor                                          */
	{ITC_AMFM_TUNER1_SYSTEM_FAILURE                         , DTC_CMC_TUNER_AMFM_INTITIALIZATION_ERROR					},/* Tuner AM/FM - initialisation fails                    */
	{ITC_DAB_TUNER_SYSTEM_FAILURE                           , DTC_CMC_TUNER_DAB_INITIALIZATION_ERROR					},/* Tuner DAB - Initialisation error                      */
	{ITC_CMC_TUNER_SXM_INIT_FAILED                          , DTC_CMC_TUNER_SXM_INITIALIZATION_ERROR					},/* Tuner SXM - initialisation error                      */
	{ITC_CMC_TUNER_HD_RADIO_INIT_FAILED                     , DTC_CMC_TUNER_HD_RADIO_INITIALIZATION_ERROR				},/* Tuner HD-Radio - initialisation error                 */
	{ITC_CMC_AMPLIFIER_INIT_FAILED                          , DTC_CMC_AMPLIFIER_INITIALIZATION_ERROR					},/* CMC Amplifier initialisation error                    */
	{ITC_BLUETOOTH_SYSTEM_FAILURE                           , DTC_CMC_BLUETOOTH_INITIALIZATION_ERROR					},/* Bluetooth - Initialisation error                      */	
	{ITC_MIC1_SHORT_BAT                                     , DTC_CMC_EXT_MIC_SIG_LINE1_SHORT_TO_BATTERY				},/* External Microphone Signal Line 1 - short to battery  */
	{ITC_MIC1_OPEN                                          , DTC_CMC_EXT_MIC_SIG_LINE1_OPEN_LOAD_SHORT_TO_GND			},/* External Microphone Signal Line 1 - open load         */
	{ITC_MIC1_SHORT_GND                                     , DTC_CMC_EXT_MIC_SIG_LINE1_OPEN_LOAD_SHORT_TO_GND			},/* External Microphone Signal Line 1 - short to ground   */
	{ITC_MIC2_SHORT_BAT                                     , DTC_CMC_EXT_MIC_SIG_LINE2_SHORT_TO_BATTERY				},/* External Microphone Signal Line 2 - short to battery  */
	{ITC_MIC2_OPEN                                          , DTC_CMC_EXT_MIC_SIG_LINE2_OPEN_LOAD_SHORT_TO_GND			},/* External Microphone Signal Line 2 - open load         */
	{ITC_MIC2_SHORT_GND                                     , DTC_CMC_EXT_MIC_SIG_LINE2_OPEN_LOAD_SHORT_TO_GND			},/* External Microphone Signal Line 2 - short to ground   */
	{ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_VSS                     , DTC_CMC_FRONT_RT_SPKR_OUTPUT_SHORT_TO_BAT					},/* Front Right Speaker Output - short to battery         */
	{ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_ACROSS                  , DTC_CMC_FRONT_RT_SPKR_OUTPUT_SHORT_LINES					},/* Front Right Speaker Output - short +/- lines          */
	{ITC_AUDIO_SPK_RF_CIRCUIT_INTERRUPTED                   , DTC_CMC_FRONT_RT_SPKR_OUTPUT_OPEN_LOAD					},/* Front Right Speaker Output - open load                */
	{ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_GND                     , DTC_CMC_FRONT_RT_SPKR_OUTPUT_SHORT_TO_GND					},/* Front Right Speaker Output - short to ground          */
	{ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_VSS                     , DTC_CMC_FRONT_LT_SPKR_OUTPUT_SHORT_TO_BAT					},/* Front Left Speaker Output - short to battery          */
	{ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_ACROSS                  , DTC_CMC_FRONT_LT_SPKR_OUTPUT_SHORT_LINES					},/* Front Left Speaker Output - short +/- lines           */
	{ITC_AUDIO_SPK_LF_CIRCUIT_INTERRUPTED                   , DTC_CMC_FRONT_LT_SPKR_OUTPUT_OPEN_LOAD					},/* Front Left Speaker Output - open load                 */
	{ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_GND						, DTC_CMC_FRONT_LT_SPKR_OUTPUT_SHORT_TO_GND					},/* Front Left Speaker Output - short to ground           */
	{ITC_CMC_CABINOUTPUT_SHORT_TO_BATTERY			    	, DTC_CMC_CABIN_OUTPUT_SHORT_TO_BAT							},/* Cabin Output - short to battery                       */
	{ITC_CMC_CABINOUTPUT_SHORT_TO_LINES				    	, DTC_CMC_CABIN_OUTPUT_SHORT_LINES							},/* Cabin Output - short +/- lines                        */
	{ITC_CMC_CABINOUTPUT_SHORT_TO_LOAD				    	, DTC_CMC_CABIN_OUTPUT_OPEN_LOAD							},/* Cabin Output - open load                              */
	{ITC_CMC_CABINOUTPUT_SHORT_TO_GROUND				    , DTC_CMC_CABIN_OUTPUT_SHORT_TO_GND							},/* Cabin Output - short to groun                         */
	{ITC_CMC_PRIVATESPEAKER_OUTPUT_SHORT_TO_BATTERY			, DTC_CMC_PRIVATE_SPKR_OUTPUT_SHORT_TO_BAT					},/* Private Speaker Output - short to battery             */
	{ITC_CMC_PRIVATESPEAKER_OUTPUT_SHORT_TO_LINES			, DTC_CMC_PRIVATE_SPKR_OUTPUT_SHORT_LINES					},/* Private Speaker Output - short +/- lines              */
	{ITC_CMC_PRIVATESPEAKER_OUTPUT_SHORT_TO_LOAD			, DTC_CMC_PRIVATE_SPKR_OUTPUT_OPEN_LOAD						},/* Private Speaker Output - open load                    */
	{ITC_CMC_PRIVATESPEAKER_OUTPUT_SHORT_TO_GROUND		 	, DTC_CMC_PRIVATE_SPKR_OUTPUT_SHORT_TO_GND					},/* Private Speaker Output - short to ground              */
	{ITC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SHORT_TO_BATTERY     , DTC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SW_12V_SHORT_TO_BAT		},/* Switchable Output Voltage SW_12V - short to battery   */
	{ITC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SHORT_TO_GROUND      , DTC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SW_12V_SHORT_TO_GND		},/* Switchable Output Voltage SW_12V - short to groung    */
	//{ITC_USB1_OVERCURRENT                                   , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_HUB_NOT_DETECTED                              , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_HUB_NUMBER_OF_PORTS_MISMATCH                  , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_OPEN                                          , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_LINK_DISABLED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_SPEED_REDUCED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_DATA_LINE_FAULT                               , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_POWER_SUPPLY_FAULT                            , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_SHORT_GND                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_SHORT_BAT                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB1_SYSTEM_FAILURE,                                 DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_OVERCURRENT                                   , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_HUB_NOT_DETECTED                              , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_HUB_NUMBER_OF_PORTS_MISMATCH                  , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_OPEN                                          , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_LINK_DISABLED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_SPEED_REDUCED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_DATA_LINE_FAULT                               , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_POWER_SUPPLY_FAULT                            , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_SHORT_GND                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_SHORT_BAT                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB2_SYSTEM_FAILURE,                                 DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_OVERCURRENT                                   , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_HUB_NOT_DETECTED                              , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_HUB_NUMBER_OF_PORTS_MISMATCH                  , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_OPEN                                          , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_LINK_DISABLED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_SPEED_REDUCED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_DATA_LINE_FAULT                               , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_POWER_SUPPLY_FAULT                            , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_SHORT_GND                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_SHORT_BAT                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB3_SYSTEM_FAILURE,                                 DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_OVERCURRENT                                   , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_HUB_NOT_DETECTED                              , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_HUB_NUMBER_OF_PORTS_MISMATCH                  , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_OPEN                                          , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_LINK_DISABLED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_SPEED_REDUCED                                 , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_DATA_LINE_FAULT                               , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_POWER_SUPPLY_FAULT                            , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_SHORT_GND                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_SHORT_BAT                                     , DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_USB4_SYSTEM_FAILURE,                                 DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE				},/* External USB CM-Hub - general failure                 */
	//{ITC_HEATCTRL_FAN_SYSTEM_FAILURE                        , DTC_CMC_ECU_FAN_ELECTRONIC_ERROR							},/* ECU Fan - electronic error                            */
	{ITC_CMC_MCP_HOMEBUTTON_KEY_JAM							, DTC_CMC_HOME_BUTTON_KEY_JAM								},/* HOMEBUTTON - key jam                                  */
	{ITC_CMC_MCP_HOMEBUTTON_ILLUMINATION_ERROR				, DTC_CMC_HOME_BUTTON_AMBER_LED_ILLUMINATION_ERROR			} /* AMBER LED of HOMEBUTTON - illumination error          */
};

const tU32 g_dtcList[] =
{
	/* DTC in the same order as in the $22/2E 1F 81 feature */
	DTC_CMC_VCAN_TIME_OUT									,
    DTC_CMC_VCAN_NO_COMMUNICATION							,
    DTC_CMC_MCAN_TIME_OUT									,
    DTC_CMC_MCAN_NO_COMMUNICATION							,
    DTC_CMC_ETHERNET_NO_COMMUNICATION						,
    DTC_CMC_HIGH_VOLTAGE_ERROR								,
    DTC_CMC_LOW_VOLTAGE_ERROR								,
    DTC_CMC_DISPLAY_OVER_TEMPERATURE						,
    DTC_CMC_AMPLIFIER_OVER_TEMPERATURE						,
    DTC_CMC_MAIN_PROCESSOR_OVER_TEMPERATURE					,
    DTC_CMC_CO_PROCESSOR_OVER_TEMPERATURE					,
    DTC_CMC_TUNER_AMFM_INTITIALIZATION_ERROR				,
    DTC_CMC_TUNER_DAB_INITIALIZATION_ERROR					,
    DTC_CMC_TUNER_SXM_INITIALIZATION_ERROR					,
    DTC_CMC_TUNER_HD_RADIO_INITIALIZATION_ERROR				,
    DTC_CMC_AMPLIFIER_INITIALIZATION_ERROR					,
    DTC_CMC_BLUETOOTH_INITIALIZATION_ERROR					,
    DTC_CMC_EXT_MIC_SIG_LINE1_SHORT_TO_BATTERY				,
    DTC_CMC_EXT_MIC_SIG_LINE1_OPEN_LOAD_SHORT_TO_GND		,
    DTC_CMC_EXT_MIC_SIG_LINE2_SHORT_TO_BATTERY				,
    DTC_CMC_EXT_MIC_SIG_LINE2_OPEN_LOAD_SHORT_TO_GND		,
    DTC_CMC_FRONT_RT_SPKR_OUTPUT_SHORT_TO_BAT				,
    DTC_CMC_FRONT_RT_SPKR_OUTPUT_SHORT_LINES				,
    DTC_CMC_FRONT_RT_SPKR_OUTPUT_OPEN_LOAD					,
    DTC_CMC_FRONT_RT_SPKR_OUTPUT_SHORT_TO_GND				,
    DTC_CMC_FRONT_LT_SPKR_OUTPUT_SHORT_TO_BAT				,
    DTC_CMC_FRONT_LT_SPKR_OUTPUT_SHORT_LINES				,
    DTC_CMC_FRONT_LT_SPKR_OUTPUT_OPEN_LOAD					,
    DTC_CMC_FRONT_LT_SPKR_OUTPUT_SHORT_TO_GND				,
    DTC_CMC_CABIN_OUTPUT_SHORT_TO_BAT						,
    DTC_CMC_CABIN_OUTPUT_SHORT_LINES						,
    DTC_CMC_CABIN_OUTPUT_OPEN_LOAD							,
    DTC_CMC_CABIN_OUTPUT_SHORT_TO_GND						,
    DTC_CMC_PRIVATE_SPKR_OUTPUT_SHORT_TO_BAT				,
    DTC_CMC_PRIVATE_SPKR_OUTPUT_SHORT_LINES					,
    DTC_CMC_PRIVATE_SPKR_OUTPUT_OPEN_LOAD					,
    DTC_CMC_PRIVATE_SPKR_OUTPUT_SHORT_TO_GND				,
    DTC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SW_12V_SHORT_TO_BAT	,
    DTC_CMC_SWITCHABLE_OUTPUT_VOLTAGE_SW_12V_SHORT_TO_GND	,
    //DTC_CMC_ELECTRICAL_USB_CM_HUB_GENERAL_FAILURE			,
    //DTC_CMC_ECU_FAN_ELECTRONIC_ERROR						,
    DTC_CMC_HOME_BUTTON_KEY_JAM								,
    DTC_CMC_HOME_BUTTON_AMBER_LED_ILLUMINATION_ERROR		
};

int g_dtcListSize()
{
  return (static_cast<int>(sizeof(g_dtcList)/sizeof(tU32)));
}
	