//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_inc_comm.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog INC Interface
//
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim
// HISTORY:
// Date      | Author                       | Modification
// 24.10.13  | BSOT Plischke                | Initial version
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#define BYTE_POSITION_DEM_EVENTS_NUMBER   4

#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include "inc.h"
#include "inc_ports.h"
#include "dgram_service.h"


#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_inc_comm
   #include "vd_diaglog_inc_comm.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_memory_master
   #include <common/framework/vd_diaglog_memory_master.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
#include <common/framework/vd_diaglog_system_variables.h>
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_inc_comm.cpp.trc.h"
#endif

//lint -e750 => ignore unused defines
//lint -e749 => ignore unused enums
#define DIAGLOG_INC_MAJOR_VERSION  1
#define DIAGLOG_INC_MINOR_VERSION  0

#define DIAGLOG_INC_THREAD_NAME                         "DIAGLOG_INC_THREAD"
#define DIAGLOG_INC_THREAD_PRIO                         35
#define DIAGLOG_INC_THREAD_STACKSIZE                    5000
#define DEFAULT_MSG_SIZE                                4096

#define DIAGLOG_INC_INACTIVE    0
#define DIAGLOG_INC_ACTIVE      1

#define SCC_EVENTMEMORY_C_COMPONENT_STATUS           0x20
#define SCC_EVENTMEMORY_R_COMPONENT_STATUS           0x21
#define SCC_EVENTMEMORY_R_MAIN_REJECT                0x0B
#define SCC_EVENTMEMORY_R_SCC_REJECT                 0x0C
#define SCC_EVENTMEMORY_C_TEST_RESULT_SCC2MAIN       0x30
#define SCC_EVENTMEMORY_R_TEST_RESULT_SCC2MAIN       0x31
#define SCC_EVENTMEMORY_C_TEST_RESULT_MAIN2SCC       0x32
#define SCC_EVENTMEMORY_R_TEST_RESULT_MAIN2SCC       0x33
#define SCC_EVENTMEMORY_C_COMPONENT_STATUS_SCC2MAIN  0x34
#define SCC_EVENTMEMORY_R_COMPONENT_STATUS_SCC2MAIN  0x35
#define SCC_EVENTMEMORY_C_CLEAR_SCC2MAIN             0x40
#define SCC_EVENTMEMORY_R_CLEAR_SCC2MAIN             0x41
#define SCC_EVENTMEMORY_C_CLEAR_MAIN2SCC             0x42
#define SCC_EVENTMEMORY_R_CLEAR_MAIN2SCC             0x43
#define SCC_EVENTMEMORY_C_EXT_DATA_SCC2MAIN          0x50
#define SCC_EVENTMEMORY_R_EXT_DATA_SCC2MAIN          0x51
#define SCC_EVENTMEMORY_C_EXT_DATA_MAIN2SCC          0x01
#define SCC_EVENTMEMORY_R_EXT_DATA_MAIN2SCC          0x53

#define DIAGLOG_INC_LUN    DIA_EVENTMEMORY_PORT


typedef enum
{
   VD_INC_PRE_RESULT = 0,
   VD_INC_RESULT,
   VD_INC_FINISHED
}teIncResultDataStateMachine;

typedef enum
{
   VD_INC_STATUS_ACTIVE = 1,
   VD_INC_STATUS_INACTIVE
}teIncStatusActiveInactiv;


//-----------------------------------------------------------------------------

vdl_tclIncCommunication* vdl_tclIncCommunication::mpInstance = 0;
int vdl_tclIncCommunication::m_nSocketFd = -1;
unsigned char vdl_tclIncCommunication::m_au8RecvBuffer[DEFAULT_MSG_SIZE];
unsigned char vdl_tclIncCommunication::m_au8SendBuffer[DEFAULT_MSG_SIZE];
vdDiagLog_tclApp*  vdl_tclIncCommunication::m_poGWMain = NULL;

#ifdef USE_DGRAM_SERVICE
sk_dgram* vdl_tclIncCommunication::dgram = NULL;
#endif
//-----------------------------------------------------------------------------

vdl_tclIncCommunication::vdl_tclIncCommunication (vdDiagLog_tclApp* poVdDiagLog_MainApp):
   m_oThread(0)
{
   m_poGWMain = poVdDiagLog_MainApp;
}

//-----------------------------------------------------------------------------

vdl_tclIncCommunication::~vdl_tclIncCommunication ( tVoid )
{}

//-----------------------------------------------------------------------------
bool vdl_tclIncCommunication::create ( tVoid )
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::create"));

   bool retCode = true;


  if ( isCreated() ) return retCode;

   // create socket for communication
   m_nSocketFd = socket(AF_BOSCH_INC_AUTOSAR, (int)SOCK_STREAM, 0);
   if ( m_nSocketFd == -1)
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::connect => ERROR: Cannot create socket"));
      retCode = false;
   }

   ETG_TRACE_USR3_THR(("vdl_tclIncCommunication::create: Socket with handle %d created", m_nSocketFd));

#ifdef USE_DGRAM_SERVICE
   dgram = dgram_init(m_nSocketFd, DGRAM_MAX, NULL);
   if ( dgram == NULL )
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::connect => ERROR: dgram_init failed"));
      (void) close();
      retCode = false;
   }
#endif

   return retCode;
}

//-------------------------------------------------------------------------------------------------

bool vdl_tclIncCommunication::close ( void )
{
   ETG_TRACE_USR3_THR(("vdl_tclIncCommunication::close"));
   bool retCode = true;

#ifdef USE_DGRAM_SERVICE
   dgram_exit(dgram);
#endif
   (void) ::close(m_nSocketFd);
   m_nSocketFd = 0;

   return retCode;
}
//-----------------------------------------------------------------------------

bool vdl_tclIncCommunication::connect ( tVoid )
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::connect"));

   bool retCode=true;

   struct addrinfo    hints, *infoptr=NULL, *loopptr=NULL;

   memset(&hints, 0, sizeof(hints));
   hints.ai_family=AF_INET;
#ifdef USE_DGRAM_SERVICE
   hints.ai_socktype=SOCK_DGRAM;
#else // USE_DGRAM_SERVICE
   hints.ai_socktype=SOCK_STREAM;
#endif // USE_DGRAM_SERVICE

   char portstr[16];
   snprintf(portstr, sizeof(portstr), "%d", DIAGLOG_INC_LUN);

   // get the Local Host for v850 and Bind to that Host, which is dynamically created at runtime by kernel in the file /etc/hosts
   int result=getaddrinfo("scc-local", portstr, &hints, &infoptr);
   if(result || !infoptr)
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::connect => ERROR: getaddrinfo failed for scc-local"));
      retCode=false;
   }
   else
   {
      // Bind to a local port to recieve response
      for(loopptr=infoptr; loopptr; loopptr=loopptr->ai_next)
         if(!bind(m_nSocketFd, loopptr->ai_addr, loopptr->ai_addrlen))
            break;
      if(!loopptr)
      {
         ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::connect => ERROR: Binding to local host port failed"));
         retCode=false;
      }
   }
   freeaddrinfo(infoptr);
   infoptr=NULL;

   if(!retCode)
   {
      close();
      m_nSocketFd = -1;
      return retCode;
   }



   memset(&hints, 0, sizeof(hints));
   hints.ai_family=AF_INET;
#ifdef USE_DGRAM_SERVICE
   hints.ai_socktype=SOCK_DGRAM;
#else // USE_DGRAM_SERVICE
   hints.ai_socktype=SOCK_STREAM;
#endif // USE_DGRAM_SERVICE

   snprintf(portstr, sizeof(portstr), "%d", DIAGLOG_INC_LUN);

   // get the remote hosts IP address and try to connect to its socket
   result=getaddrinfo("scc", portstr, &hints, &infoptr);
   if(result || !infoptr)
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::connect => ERROR: getaddrinfo failed for scc"));
      retCode=false;
   }
   else
   {
      // Connect to port of the v850
      for(loopptr=infoptr; loopptr; loopptr=loopptr->ai_next)
         if(!::connect(m_nSocketFd, loopptr->ai_addr, loopptr->ai_addrlen))
            break;
      if(!loopptr)
      {
         ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::connect => ERROR: Connecting to remote host port failed"));
         retCode=false;
      }
   }
   freeaddrinfo(infoptr);
   infoptr=NULL;

   if(!retCode)
   {
      close();
      m_nSocketFd = -1;
   }
   else
   {
      ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::connect => Successfully bound to local port and connected to remote port"));

      char threadName[]=DIAGLOG_INC_THREAD_NAME;

      OSAL_trThreadAttribute oThreadAttribute=
      {
         threadName,
         DIAGLOG_INC_THREAD_PRIO,
         DIAGLOG_INC_THREAD_STACKSIZE,
         (OSAL_tpfThreadEntry) ThreadFunction,
         NULL //this
      };

      m_oThread=OSAL_ThreadSpawn(&oThreadAttribute);

      if(OSAL_ERROR == m_oThread)
      {
          ETG_TRACE_FATAL_THR(( "!!! vdl_tclIncCommunication::bThreadSetup => m_threadID FAILED Size:%d Prio:%d",oThreadAttribute.s32StackSize,oThreadAttribute.u32Priority));
          retCode = false;
      }
      else
      {
          ETG_TRACE_USR3_THR(("!!! vdl_tclIncCommunication::bThreadSetup => m_threadID PASSED Size:%d Prio:%d",oThreadAttribute.s32StackSize,oThreadAttribute.u32Priority));
      }

     SendCompStatus();

   }

   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::connect"));

   return retCode;
}

//-----------------------------------------------------------------------------

bool vdl_tclIncCommunication::disconnect ( tVoid ) const
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::disconnect"));

   bool retCode = true;


   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::disconnect"));

   return retCode;
}

//------------------------------------------------------------------------------

void
vdl_tclIncCommunication::SendIncResponse (tU8 ResponseId)
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::SendIncResponse"));

   tU8 u8Channel = 0; //TODO

   size_t msgLength = 0;
   size_t iNumberOfBytes = 0;

   //copy message data to the buffer
   m_au8SendBuffer[msgLength++] = ResponseId;
   m_au8SendBuffer[msgLength++] = u8Channel;


#ifdef USE_DGRAM_SERVICE
   iNumberOfBytes = dgram_send(dgram, m_au8SendBuffer, msgLength);
#else
   iNumberOfBytes = write(m_nSocketFd, m_au8SendBuffer, msgLength);
#endif

   if (iNumberOfBytes != msgLength)
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::SendIncResponse => ERROR: Not all bytes written"));
   }
   else
   {
      ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::SendIncResponse => %d Bytes Send Data=%02x",iNumberOfBytes, ETG_LIST_LEN((tU16) iNumberOfBytes), ETG_LIST_PTR_T8(m_au8SendBuffer)));
   }

   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::SendIncResponse"));
}

//------------------------------------------------------------------------------
void
vdl_tclIncCommunication::SendClearRequest () const
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::SendClearRequest"));

   tU8 u8Channel = 0; //TODO

   tU16 msgLength = 0;
   ssize_t iNumberOfBytes = 0;

   //copy message data to the buffer
   m_au8SendBuffer[msgLength++] = SCC_EVENTMEMORY_C_CLEAR_MAIN2SCC;
   m_au8SendBuffer[msgLength++] = u8Channel;
   m_au8SendBuffer[msgLength++] = 1;
   m_au8SendBuffer[msgLength++] = 0xFF;
   m_au8SendBuffer[msgLength++] = 0xFF;
   m_au8SendBuffer[msgLength++] = 0xFF;

#ifdef USE_DGRAM_SERVICE
   iNumberOfBytes = dgram_send(dgram, m_au8SendBuffer, msgLength);
#else
   iNumberOfBytes = write(m_nSocketFd, m_au8SendBuffer, msgLength);
#endif

   if (iNumberOfBytes != msgLength)
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::SendClearRequest => ERROR: Not all bytes written"));
   }
   else
   {
      ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::SendClearRequest => %d Bytes Send Data=%02x",iNumberOfBytes, ETG_LIST_LEN((tU16) iNumberOfBytes), ETG_LIST_PTR_T8(m_au8SendBuffer)));
   }

   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::SendClearRequest"));
}

//------------------------------------------------------------------------------
void
vdl_tclIncCommunication::SendItcReport ( const tU8 au8MsgBuffer[], tU16 u16Length) const
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::SendItcReport"));

   tU8 u8Channel = 0; //TODO

   tU16 msgLength = 0;
   ssize_t iNumberOfBytes = 0;

// To Do : bsu9kor Check if Early Diagnostics is supported 
// If true,  send the DTC Event Id and its Status in the TP message for sync to SCC.
// If false, default behaviour.
// if(vdl_tclGlobalSystemVariables::bIsEarlyDiagnosticsSupported() == true){/*...*/}
// else {/*...*/} 

   //copy message data to the buffer
   m_au8SendBuffer[msgLength++] = 0x00; //TP marker
   m_au8SendBuffer[msgLength++] = SCC_EVENTMEMORY_C_TEST_RESULT_MAIN2SCC;
   m_au8SendBuffer[msgLength++] = u8Channel;
   memcpy(&m_au8SendBuffer[msgLength], au8MsgBuffer, u16Length);
   msgLength = static_cast<tU16>(msgLength + u16Length);

#ifdef USE_DGRAM_SERVICE
   iNumberOfBytes = dgram_send(dgram, m_au8SendBuffer, msgLength);
#else
   iNumberOfBytes = write(m_nSocketFd, m_au8SendBuffer, msgLength);
#endif

   if (iNumberOfBytes != msgLength)
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::SendItcReport => ERROR: Not all bytes written"));
   }
   else
   {
      ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::SendItcReport => %d Bytes Send Data=%02x",iNumberOfBytes, ETG_LIST_LEN((tU16) iNumberOfBytes), ETG_LIST_PTR_T8(m_au8SendBuffer)));
   }

   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::SendItcReport"));
}

//------------------------------------------------------------------------------

tVoid
vdl_tclIncCommunication::SendCompStatus( tVoid ) const
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::SendCompStatus"));

   unsigned char au8SendBuffer[4] = {SCC_EVENTMEMORY_C_COMPONENT_STATUS, DIAGLOG_INC_ACTIVE, DIAGLOG_INC_MAJOR_VERSION, DIAGLOG_INC_MINOR_VERSION};
   tU16 msgLength = sizeof(au8SendBuffer);
   ssize_t iNumberOfBytes = 0;

#ifdef USE_DGRAM_SERVICE
   iNumberOfBytes = dgram_send(dgram, au8SendBuffer, msgLength);
#else
   iNumberOfBytes = write(m_nSocketFd, au8SendBuffer, msgLength);
#endif

   if (iNumberOfBytes != msgLength)
   {
      ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::SendCompStatus => ERROR: Not all bytes written"));
   }
   else
   {
      ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::SendCompStatus => %d Bytes Send Data=%02x",iNumberOfBytes, ETG_LIST_LEN((tU16) iNumberOfBytes), ETG_LIST_PTR_T8(au8SendBuffer)));
   }

   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::SendCompStatus"));
}



//-----------------------------------------------------------------------------
/*static*/void vdl_tclIncCommunication::OnTestResult (tU16 SizeOfData, tU8  u8NumberOfDTCs, const tU8* pData )
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::OnTestResult"));

   ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::OnTestResult => Size=%d NumDTC=%d",SizeOfData,u8NumberOfDTCs));


   if(m_poGWMain != NULL)
   {
      // we have to run a small state maschine here to verify the Size of Data first
      teIncResultDataStateMachine eResultState = VD_INC_PRE_RESULT;

      while(eResultState != VD_INC_FINISHED)
      {
         ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::OnTestResult => eResultState:%x",ETG_CENUM(teIncResultDataStateMachine, eResultState)));
         long SizeUsed = 0;
         // check over all received reports (could be more then one)
         for(::size_t i = 0; i < u8NumberOfDTCs; ++i)
         {
            vdl_tsDiaglogTestResult    oTestResults;

            oTestResults.u32Timestamp   = 0;
            tU32 ITC = (pData[SizeUsed + 3] * 0x01000000) + (pData[SizeUsed + 2] * 0x010000) + (pData[SizeUsed + 1] * 0x0100) + (pData[SizeUsed + 0]);
            ETG_TRACE_COMP_THR(("ITC:%x",ITC));
            if ((ITC & 0xFF000000) == 0x80000000)
            {
                ETG_TRACE_COMP_THR(("OnTestResult---> Dynamic DTC Detected"));
                oTestResults.bDynamicalDTC = true;
                oTestResults.u32DTC = ITC & 0xFFFFFF;
            }
            else
            {
                ETG_TRACE_COMP_THR(("OnTestResult---> Static ITC Detected"));
                oTestResults.u32ITC = u32ConvertDemEventToItc(ITC);
            }
            SizeUsed +=4;

            bool  bUseTimestamp         = ( pData[SizeUsed] & 0x80)?true:false;
            oTestResults.eReportStatus  = (vdl_teDiaglogReportMsg)((pData[SizeUsed]) & 0x0F);
            SizeUsed++;

            if(bUseTimestamp)
            {
               oTestResults.u32Timestamp = (pData[SizeUsed + 3] * 0x01000000) + (pData[SizeUsed + 3] * 0x010000) + (pData[SizeUsed + 1] * 0x0100) + (pData[SizeUsed + 0]);
               SizeUsed +=4;
            }
            tU8  u8NumberOfExtData = pData[SizeUsed];
            SizeUsed++;

            tU8 u8ExtDataBlock = 0;
            while(u8NumberOfExtData > u8ExtDataBlock)
            {
               vdl_tsDiaglogExtData oExtDataObject;

               oExtDataObject.u8ExtId = pData[SizeUsed];
               tU8 u8ExtaDataLength   = pData[SizeUsed + 1];
               SizeUsed+=2;

               for (::size_t j = 0; j < u8ExtaDataLength; ++j)
               {
                  // fill Data
                  oExtDataObject.oData.push_back(pData[SizeUsed]);
                  SizeUsed++;
               }
               // add the Ext Data to the TestResults Object
               oTestResults.oExtData.push_back(oExtDataObject);

               // check for Next ExtData Package
               u8ExtDataBlock++;
            }// while(u8NumberOfExtData > u8ExtDataBlock)

            if(eResultState == VD_INC_RESULT)
            {
               // forward Data to Diaglog
               ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::OnTestResult => ITC=%08x Status=%02x Timestamp=%08x",oTestResults.u32ITC,oTestResults.eReportStatus,oTestResults.u32Timestamp));
               (tVoid) m_poGWMain->m_oMemoryMaster.bProcessTestReport(oTestResults);
            }// if(eResultState == VD_INC_RESULT)
         }// for(::size_t i = 0; i < u8NumberOfDTCs; ++i)

         // now switch to the next state
         switch(eResultState)
         {
            case VD_INC_PRE_RESULT:
            {
               if(SizeUsed != SizeOfData)
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::OnTestResult => SIZE MISMATCH!!! TotalSizeUsed=%x TotalSizeOfDataReceived=%x",SizeUsed,SizeOfData));
                  eResultState = VD_INC_FINISHED;
               }
               else
               {
                  eResultState = VD_INC_RESULT;
               }
               break;
            }// case VD_INC_PRE_RESULT:

            case VD_INC_RESULT:
            {
               eResultState = VD_INC_FINISHED;
               break;
            }// case VD_INC_RESULT:
            default:
               break;
         }// switch(eResultState)
      }// while(eResultState != VD_INC_FINISHED)
   }// if(m_poGWMain != NULL)
   else
   {
      ETG_TRACE_ERRMEM(("!!! vdl_tclIncCommunication::OnTestResult => m_poGWMain == NULL"));
      NORMAL_M_ASSERT_ALWAYS();
   }// else // if(m_poGWMain != NULL)

   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::OnTestResult"));
}
//-----------------------------------------------------------------------------

/*static*/ tVoid vdl_tclIncCommunication::ThreadFunction ( tPVoid /*this*/ )
{
   ETG_TRACE_USR3_THR(("--> vdl_tclIncCommunication::ThreadFunction"));

   ssize_t iNumberOfBytesReceived = 0;

   do
   {
      #ifdef USE_DGRAM_SERVICE
         iNumberOfBytesReceived = dgram_recv(dgram, m_au8RecvBuffer, DEFAULT_MSG_SIZE);
      #else
         iNumberOfBytesReceived = recv(m_nSocketFd, m_au8RecvBuffer, DEFAULT_MSG_SIZE, 0);
      #endif

      //ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => INC Rx %d bytes", iNumberOfBytesReceived));

      ETG_TRACE_COMP_THR(("--- vd_diaglog => INC Rx %d bytes Data: %02x", iNumberOfBytesReceived,
      ETG_LIST_LEN((unsigned int)iNumberOfBytesReceived), ETG_LIST_PTR_T8(const_cast<tU8*>(m_au8RecvBuffer))));

      if (iNumberOfBytesReceived > 0 && (iNumberOfBytesReceived <= DEFAULT_MSG_SIZE))
      {
         switch (m_au8RecvBuffer[0])
         {
            /***********************************/
            /*          REQUEST                */
            /***********************************/
            case SCC_EVENTMEMORY_C_TEST_RESULT_SCC2MAIN:
            {
               if (iNumberOfBytesReceived > 6)// Check for minimum length of message for any DEM events received
               {
                  tU8  u8Channel       = m_au8RecvBuffer[1];
                  ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_TEST_RESULT_SCC2MAIN[%d]",u8Channel));
                  tU16 u16Length       = static_cast<tU16>((m_au8RecvBuffer[3] * 0x0100) + m_au8RecvBuffer[2]);//Computing number of bytes received related to DEM events
                  tU8  u8NumberOfDTCs  = 0;
                  if(
                     (m_au8RecvBuffer[BYTE_POSITION_DEM_EVENTS_NUMBER] > 0)
                     &&
                     (iNumberOfBytesReceived >= (5 + u16Length))//Validation for number of bytes received
                     )
                  {
                     u8NumberOfDTCs  = m_au8RecvBuffer[4];
                     std::vector<tU8> data(m_au8RecvBuffer+5, m_au8RecvBuffer+5+u16Length);//copying DEM events data from the buffer for further processing.
                     OnTestResult(u16Length, u8NumberOfDTCs, &data[0]);
                     // send response
                     SendIncResponse(SCC_EVENTMEMORY_R_TEST_RESULT_SCC2MAIN);
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_TEST_RESULT_SCC2MAIN too short"));
                  }
               }
               else
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_TEST_RESULT_SCC2MAIN too short"));
               }
               break;
            }

            case SCC_EVENTMEMORY_C_CLEAR_SCC2MAIN:
            {
               if (iNumberOfBytesReceived > 2)
               {
                  tU8  u8Channel       = m_au8RecvBuffer[1];
                  tU8  u8MemoryId      = m_au8RecvBuffer[2];
                  tU32 u32DTCMask      = (m_au8RecvBuffer[5] * 0x010000) +(m_au8RecvBuffer[4] * 0x0100) + m_au8RecvBuffer[3];

                  ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_CLEAR_SCC2MAIN[%d] Id:%d Mask:%d",u8Channel,u8MemoryId,u32DTCMask));

                  // to do foward to DIAGLOG
                  // send response
                  SendIncResponse(SCC_EVENTMEMORY_R_CLEAR_SCC2MAIN);
               }
               else
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_CLEAR_SCC2MAIN too short"));
               }
               break;
            }

            case SCC_EVENTMEMORY_C_EXT_DATA_SCC2MAIN:
            {
               if (iNumberOfBytesReceived > 2)
               {
                  tU8  u8Channel       = m_au8RecvBuffer[1];

                  ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_EXT_DATA_SCC2MAIN[%d]",u8Channel));

                  // to do foward to DIAGLOG
                  // send response
                  SendIncResponse(SCC_EVENTMEMORY_R_EXT_DATA_SCC2MAIN);
               }
               else
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_EXT_DATA_SCC2MAIN too short"));
               }
               break;
            }

            case SCC_EVENTMEMORY_C_COMPONENT_STATUS_SCC2MAIN:
            {
               /*
                * Fill 4byte StateBitMask appropriately.
                * Format : (MsgID)0x34 (ChId)0x00 (StateBitMask)0xFFFFFFFF
                * To Do : Define meaning for each bit of statemask for appropriate actions on SCC side.
                */
               ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_C_COMPONENT_STATUS_SCC2MAIN"));
               break;
            }


            /***********************************/
            /*          RESPONSE               */
            /***********************************/

            case SCC_EVENTMEMORY_R_COMPONENT_STATUS:
            {
               // response to SCC_EVENTMEMORY_R_COMPONENT_STATUS
               if (iNumberOfBytesReceived == 4)
               {
                  tU8 u8Status = m_au8RecvBuffer[1];
                  tU8 u8VersionMajor = m_au8RecvBuffer[2];
                  tU8 u8VersionMinor = m_au8RecvBuffer[3];

                  teIncStatusActiveInactiv eStatus = (teIncStatusActiveInactiv)u8Status;
                  ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_COMPONENT_STATUS Status:%d Version%x.%x",ETG_CENUM(teIncStatusActiveInactiv,eStatus),u8VersionMajor,u8VersionMinor));
               }
               else
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_COMPONENT_STATUS too short"));
               }
               break;
            }

            case SCC_EVENTMEMORY_R_TEST_RESULT_MAIN2SCC:
            {
               // transmit response to SCC_EVENTMEMORY_C_TEST_RESULT_MAIN2SCC
               if (iNumberOfBytesReceived == 2)
               {
                  tU8 u8Channel = m_au8RecvBuffer[1];
                  ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_TEST_RESULT_MAIN2SCC[%d]",u8Channel));
               }
               else
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_TEST_RESULT_MAIN2SCC too short"));
               }
               break;
            }

            case SCC_EVENTMEMORY_R_CLEAR_MAIN2SCC:
            {
               // transmit response to SCC_EVENTMEMORY_C_CLEAR_MAIN2SCC
               if (iNumberOfBytesReceived == 2)
               {
                  tU8 u8Channel = m_au8RecvBuffer[1];
                  ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_CLEAR_MAIN2SCC[%d]",u8Channel));
               }
               else
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_CLEAR_MAIN2SCC too short"));
               }
               break;
            }

            case SCC_EVENTMEMORY_R_EXT_DATA_MAIN2SCC:
            {
               // transmit response to SCC_EVENTMEMORY_R_EXT_DATA_MAIN2SCC
               if (iNumberOfBytesReceived == 2)
               {
                  tU8 u8Channel = m_au8RecvBuffer[1];
                  ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_EXT_DATA_MAIN2SCC[%d]",u8Channel));
               }
               else
               {
                  ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_EXT_DATA_MAIN2SCC too short"));
               }
               break;
            }

            case SCC_EVENTMEMORY_R_COMPONENT_STATUS_SCC2MAIN:
            {
               // response to SCC_EVENTMEMORY_C_COMPONENT_STATUS_SCC2MAIN
               ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::ThreadFunction => SCC_EVENTMEMORY_R_COMPONENT_STATUS_SCC2MAIN"));
               break;
            }




            default:
            {
               ETG_TRACE_ERR_THR(("!!! vdl_tclIncCommunication::ThreadFunction => Unknown msgID=0x%x", m_au8RecvBuffer[0]));
               break;
            }
         } //switch (m_au8RecvBuffer[0])
      } //if (iNumberOfBytesReceived > 0)
   }
   while (1);//lint !e506

   ETG_TRACE_USR3_THR(("<-- vdl_tclIncCommunication::ThreadFunction"));
}

/*static*/tU32 vdl_tclIncCommunication::u32ConvertDemEventToItc(tU32 DemEventId)
{
   for (int i = 0; i< iEventToItcMapSize();i++)
   {
      ETG_TRACE_USR4_THR(("--- vdl_tclIncCommunication::u32ConvertDemEventToItc => DemEventId:%x DemEventIdReceived:%x",rEventToItcMap[i].DemEventId, DemEventId));

      if(DemEventId == rEventToItcMap[i].DemEventId)
      {
         ETG_TRACE_USR4_THR(("--- vdl_tclIncCommunication::u32ConvertDemEventToItc => Found DemEventId:%x => ITC:%x",rEventToItcMap[i].DemEventId, rEventToItcMap[i].ItcId));
         return rEventToItcMap[i].ItcId;
      }
   }
   ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::u32ConvertDemEventToItc => Unkown DemEventId:%x",DemEventId));
   return 0;
}

/*static*/tU32 vdl_tclIncCommunication::u32ConvertItcToDemEvent(tU32 ItcId)
{
   for (int i = 0; i< iEventToItcMapSize();i++)
   {
      ETG_TRACE_USR4_THR(("--- vdl_tclIncCommunication::u32ConvertItcToDemEvent => ItcId:%x ItcIdReceived:%x",rEventToItcMap[i].ItcId, ItcId));
      if(ItcId == rEventToItcMap[i].ItcId)
      {
         ETG_TRACE_USR4_THR(("--- vdl_tclIncCommunication::u32ConvertItcToDemEvent => Found ItcId:%x => DemEventId:%x",rEventToItcMap[i].ItcId, rEventToItcMap[i].DemEventId));
         return rEventToItcMap[i].DemEventId;
      }
   }
   ETG_TRACE_COMP_THR(("--- vdl_tclIncCommunication::u32ConvertItcToDemEvent => Unkown ItcId:%x",ItcId));
   return 0;
}
