//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_report_record.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog Report Record base class
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 27.10.05  | ESA2 Barber                  | Initial version
// 15.10.12  | BSOT Plischke                | new Diaglog 
// 22.10.15  | gpu2kor					    | fix for CMG3GB-2474
// 28.12.18  | mky6kor						| Extended data configuration(RTC-434184)
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>


#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record
   #include "vd_diaglog_report_record.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_trace
   #include <common/framework/vd_diaglog_trace.h>  
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
   #include <common/framework/vd_diaglog_system_variables.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include <common/framework/vd_diaglog_report_memory.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_store_to_emtrace
   #include <common/framework/vd_diaglog_store_to_emtrace.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_ageing
   #include <common/framework/ageing/vd_diaglog_ageing.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record_factory
   #include <project/framework/vd_diaglog_report_record_factory.h>  
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_ExtendedDataPrj
   #include <project/framework/extendedData/ExtendedDataPrj.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_snapshotdata
   #include <common/framework/snapshotData/vd_diaglog_snapshotdata.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_report_record.cpp.trc.h"   
#endif

#include <common/framework/extendedData/ExtendedData.h>


using namespace VDD;

// used for dynamic DTC as Database
const vdl_tclReportRecordConstData g_oConstDataDefault (g_oaDtcDefinitionTable_Default,1);

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord::tclReportRecord
// 
// DESCRIPTION: constructor
//
// PARAMETER:  u16TroubleCode - internal code for this error report
//             u32Group - error group for this report
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportRecord::vdl_tclReportRecord(vdl_tclReportMemory* const pMemoryObject, vdl_tclReportRecordConstData const* pReportRecordConstData, tU16 Index) :
   m_bIsIgnitionCycleActive(false),   
   m_pMemoryObject(pMemoryObject),
   m_Index(Index),
   m_u8FailOccurenceCounter(0),
   m_u8MonCycleSinceFirstFail(0),
   m_pReportRecordConstData(pReportRecordConstData),
   m_u16ITC(0),
   m_u32DTC(0),
   m_bDisabledByVariant(false),
   m_u8AgeingCounter(0)

{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vdl_tclReportRecord - ITC = %04x", u16ITC()));

   if(pReportRecordConstData == NULL)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::vdl_tclReportRecord => NULL == pReportRecordConstData"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   m_s32PassValue = u32GetPassDeltaByConf();
   m_s32FailValue = u32GetFailDeltaByConf();
   m_u8MaxAgeing  = u8GetMaxAgeByConf();
   m_uRecordStatus.u32Status = 0;
   vSetDTCCalibrationEnabled();        // default active
   vSetDTCControlEnabled();            // default active
   vSetDTCReadByCalibrationEnabled();  // default active

   // check if this ITC have Blocking dependenices
   // if yes add them to blocking Module 
   if(vdDiagLog_tclApp::m_poInstance != NULL)
   {
      // first collect BlockList
      std::vector<tU32> blockDTCList;
      const tU32* const DTCBlockListPtr = ptrGetBlockList();   
      for (int i=0; i<DIAGLOG_MAX_BLOCKDTCLIST; i++)
      {
         if(DTCBlockListPtr[i]!= 0)
         {
            blockDTCList.push_back(DTCBlockListPtr[i]);
         }
      }
      // now register at Blocking Module
      vdDiagLog_tclApp::m_poInstance->registerBlockModeCallback(u8GetMemoryId(),u16ITC(), u32GetAppID(), u32GetDTCBlockingMode(), blockDTCList);
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vdl_tclReportRecord"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord::tclReportRecord
// 
// DESCRIPTION: constructor
//
// PARAMETER:  u16TroubleCode - internal code for this error report
//             u32Group - error group for this report
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportRecord::vdl_tclReportRecord(vdl_tclReportMemory* const pMemoryObject, tU16 _u16ITC, tU32 _u32DTC) :
   m_bIsIgnitionCycleActive(false),   
   m_pMemoryObject(pMemoryObject),
   m_Index(0),
   m_u8FailOccurenceCounter(0),
   m_u8MonCycleSinceFirstFail(0),
   m_pReportRecordConstData(&g_oConstDataDefault),
   m_u16ITC(_u16ITC),
   m_u32DTC(_u32DTC),
   m_bDisabledByVariant(false),
   m_u8AgeingCounter(0)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vdl_tclReportRecord - ITC = %04x", u16ITC()));

   if(_u16ITC == 0 || _u32DTC == 0)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::vdl_tclReportRecord => INVLAID ITC:%x DTC:%x",_u16ITC, static_cast<tUInt>(_u32DTC)));
      NORMAL_M_ASSERT_ALWAYS();
   }
   m_s32PassValue = u32GetPassDeltaByConf();
   m_s32FailValue = u32GetFailDeltaByConf();
   m_u8MaxAgeing  = u8GetMaxAgeByConf();
   m_uRecordStatus.u32Status = 0;
   vSetDTCCalibrationEnabled();        // default active
   vSetDTCControlEnabled();            // default active
   vSetDTCReadByCalibrationEnabled();  // default active

   if(vdDiagLog_tclApp::m_poInstance != NULL)
   {
      vdDiagLog_tclApp::m_poInstance->unregisterBlockModeCallback(u8GetMemoryId(),u16ITC());
   }


   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vdl_tclReportRecord"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord::~vdl_tclReportRecord
// 
// DESCRIPTION: default destructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportRecord::~vdl_tclReportRecord()
{
   m_pReportRecordConstData = NULL;

   _BP_TRY_BEGIN
   {
      for(std::map<tU8, VDD::ExtendedData*>::const_iterator i = extendedData.begin(); i != extendedData.end(); ++i)
      {
         // delete now all map elements, created with new
         delete i->second;
      }
   }

   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::~vdl_tclReportRecord => FAILED"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord::vSetModified
// 
// DESCRIPTION: Send Modified to Memory Class to store the Data
//
///////////////////////////////////////////////////////////////////////////////////
//

tVoid vdl_tclReportRecord::vSetModified()
{
   if(m_pMemoryObject != NULL)
   {
      m_pMemoryObject->vSetModified();
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::vTrace
// 
// DESCRIPTION:   Trace output
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vTrace()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vTrace"));   

   if(bIsDTCCalibrationEnabled() == false)
   {
      ETG_TRACE_FATAL_THR(( "--- vdl_tclReportRecord::vTrace => DTC:%06x ITC:%04x INACTIVE",u32DTC(),u16ITC()));
   }
   else if(bIsDTCReadByCalibrationEnabled() == false)
   {
      if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_SINGLE) //lint !e774 !e506
      {
         ETG_TRACE_FATAL_THR(( "--- vdl_tclReportRecord::vTrace => DTC:%06x ITC:%04x Status:%08x OccurenceCounter:%d AgeingCounter:%d/%d => READ INACTIVE",u32DTC(),u16ITC(),m_uRecordStatus.u32Status,m_u8FailOccurenceCounter,m_u8AgeingCounter,u8GetMaxAge()));
      }
      else if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_GLOBAL) //lint !e774 !e506
      {
         ETG_TRACE_FATAL_THR(( "--- vdl_tclReportRecord::vTrace => DTC:%06x ITC:%04x Status:%08x OccurenceCounter:%d AgeingCounter:%d/%d => READ INACTIVE",u32DTC(),u16ITC(),m_uRecordStatus.u32Status,m_u8FailOccurenceCounter,vdl_tclAgeing::getGlobalAgeingCounter(),vdl_tclAgeing::getMaxGlobalAgeingCounter()));
      }
      else
      {
         ETG_TRACE_FATAL_THR(( "--- vdl_tclReportRecord::vTrace => DTC:%06x ITC:%04x Status:%08x OccurenceCounter:%d => READ INACTIVE",u32DTC(),u16ITC(),m_uRecordStatus.u32Status,m_u8FailOccurenceCounter));
      }
   }
   else
   {
      if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_SINGLE) //lint !e774 !e506
      {
         ETG_TRACE_FATAL_THR(( "--- vdl_tclReportRecord::vTrace => DTC:%06x ITC:%04x Status:%08x OccurenceCounter:%d AgeingCounter:%d/%d",u32DTC(),u16ITC(),m_uRecordStatus.u32Status,m_u8FailOccurenceCounter,m_u8AgeingCounter,u8GetMaxAge()));
      }
      else if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_GLOBAL) //lint !e774 !e506
      {
         ETG_TRACE_FATAL_THR(( "--- vdl_tclReportRecord::vTrace => DTC:%06x ITC:%04x Status:%08x OccurenceCounter:%d AgeingCounter:%d/%d",u32DTC(),u16ITC(),m_uRecordStatus.u32Status,m_u8FailOccurenceCounter,vdl_tclAgeing::getGlobalAgeingCounter(),vdl_tclAgeing::getMaxGlobalAgeingCounter()));
      }
      else
      {
         ETG_TRACE_FATAL_THR(( "--- vdl_tclReportRecord::vTrace => DTC:%06x ITC:%04x Status:%08x OccurenceCounter:%d",u32DTC(),u16ITC(),m_uRecordStatus.u32Status,m_u8FailOccurenceCounter));
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vTrace"));   
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportRecord::bEventStep
//
// DESCRIPTION:  forward the EventStep to the PlugIns
//             
// PARAMETER:    vdl_teDiaglogEventStep enEventStep: Kind of EventStep
//               vdl_tclReportRecord* pReportRecord: pointer to Data
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tBool vdl_tclReportRecord::bEventStep(vdl_teDiaglogEventStep enEventStep, vdl_tclReportRecord* pReportRecord) const
{
   tBool bReturn = false;
   if(vdDiagLog_tclApp::m_poInstance != NULL)
   {
      bReturn = vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.bEventStep(enEventStep,pReportRecord);
   }
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:bool vdl_tclCycleCountersBase:bLoad
// 
// DESCRIPTION:   override of tclDiagLogObject method - load class data from
//                a byte stream
//
// PARAMETER:  iPos - iterator to byte stream for serialisation
//
// RETURNVALUE: bool true=succesfull / false=failed
// 
///////////////////////////////////////////////////////////////////////////////////
//

bool vdl_tclReportRecord::bLoad(tStreamCIterator& iPos)
{
   // if report memory is saving itself, add class data to collection
   tU8 u8Value;
   u8Value = static_cast<tU8>(*iPos++);
   tU16 Itc = static_cast<tU16>(u8Value << 8);
   u8Value = static_cast<tU8>(*iPos++);
   Itc = static_cast<tU16>(Itc | u8Value);

   u8Value = static_cast<tU8>(*iPos++);
   tU32 Dtc = static_cast<tU32>(u8Value << 24);
   u8Value = static_cast<tU8>(*iPos++);
   Dtc |= static_cast<tU32>(u8Value << 16);
   u8Value = static_cast<tU8>(*iPos++);
   Dtc |= static_cast<tU32>(u8Value << 8);
   u8Value = static_cast<tU8>(*iPos++);
   Dtc |= static_cast<tU32>(u8Value);

   if(Itc != u16ITC())
   {
      if(u16ITC() != 0xFFFF)
      {
         // do not load the Data, matching is wrong
         ETG_TRACE_ERR_THR(( "!!! vdl_tclReportRecord::bLoad => ITC wrong %x<>%x ",Itc,u16ITC()));
         ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclReportRecord::bLoad] Index wrong %x<>%x",Itc,u16ITC()));
         iPos+=6;

         // we can stop here, Data invalid
         return false;
      }// if(u16ITC() != 0xFFFF)
      else
      {
         // this is a dynamilcal DTC, so load it
         m_u16ITC = Itc;
         m_u32DTC = Dtc;
      }
   }// if(Itc != u16ITC())

   u8Value = static_cast<tU8>(*iPos++);
   tU32 u32Status = (static_cast<tU32>(u8Value) << 24);
   u8Value = static_cast<tU8>(*iPos++);
   u32Status |= (static_cast<tU32>(u8Value) << 16);
   u8Value = static_cast<tU8>(*iPos++);
   u32Status |= (static_cast<tU32>(u8Value) << 8);
   u8Value = static_cast<tU8>(*iPos++);
   u32Status |= static_cast<tU32>(u8Value);
   m_uRecordStatus.u32Status = u32Status;

   m_u8AgeingCounter = static_cast<tU8>(*iPos++);
   m_u8FailOccurenceCounter = static_cast<tU8>(*iPos++);
   m_u8MonCycleSinceFirstFail = static_cast<tU8>(*iPos++);

   // get now freezeFrameData
   for_each(this->extendedData.begin(), this->extendedData.end(), VDD::load(&iPos));

   return true;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclCycleCountersBase::vSave
// 
// DESCRIPTION:   override of tclDiagLogObject method - save class data to 
//                a byte stream
//
// PARAMETER:  roData - byte stream for serialisation
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vSave(tStream& roData) const
{
   // if report memory is saving itself, add class data to collection
   roData.push_back(static_cast<tS8>((u16ITC() >> 8) & 0xff));
   roData.push_back(static_cast<tS8>(u16ITC()  & 0xff));

   roData.push_back(static_cast<tS8>((u32DTC() >> 24) & 0xff));
   roData.push_back(static_cast<tS8>((u32DTC() >> 16) & 0xff));
   roData.push_back(static_cast<tS8>((u32DTC() >> 8) & 0xff));
   roData.push_back(static_cast<tS8>(u32DTC() & 0xff));

   roData.push_back(static_cast<tS8>((m_uRecordStatus.u32Status >> 24) & 0xff));
   roData.push_back(static_cast<tS8>((m_uRecordStatus.u32Status >> 16) & 0xff));
   roData.push_back(static_cast<tS8>((m_uRecordStatus.u32Status >> 8) & 0xff));
   roData.push_back(static_cast<tS8>(m_uRecordStatus.u32Status & 0xff));

   roData.push_back(static_cast<tS8>(m_u8AgeingCounter));
   roData.push_back(static_cast<tS8>(m_u8FailOccurenceCounter));
   roData.push_back(static_cast<tS8>(m_u8MonCycleSinceFirstFail));

   // store now freezeFrameData
   for_each(this->extendedData.begin(), this->extendedData.end(), VDD::save(&roData));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::vClear(bool )
// 
// DESCRIPTION: reset Data
//
// PARAMETER:  bool
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vClear(bool forced)
{
   if((bIsClearAllowed()==true) || (forced==true))
   {
      m_uRecordStatus.sStatus.m_bIsFailedInPreviousIgnCycle       = false;
      m_uRecordStatus.sStatus.m_bIsPassedInCurrentIgnCycle        = false;
      m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle        = false;    
      m_uRecordStatus.sStatus.m_bIsActive                         = false;   
      m_uRecordStatus.sStatus.m_bIsHistory                        = false;
      m_uRecordStatus.sStatus.m_bIsFailedSinceDTCCleared          = false;
      m_uRecordStatus.sStatus.m_bIsNotPassedSinceDTCCleared       = true;
      m_uRecordStatus.sStatus.m_bIsPassedSinceCurrentPowerUp      = false;
      m_uRecordStatus.sStatus.m_bIsNotPassedSinceCurrentPowerUp   = true;
      m_uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle      = false;
      m_uRecordStatus.sStatus.m_bIsNotCompleteThisOperationCycle  = true;
      m_uRecordStatus.sStatus.m_bIsNotCompleteSinceLastClear      = true;
      m_uRecordStatus.sStatus.m_bIsPassedSinceDTCCleared          = false;
      
      m_u8FailOccurenceCounter = 0;
      m_u8MonCycleSinceFirstFail = 0;

      std::for_each(extendedData.begin(), extendedData.end(), VDD::clear());

      #ifdef DIAGLOG_USE_SNAPSHOT_DATA 
      if(vdDiagLog_tclApp::m_poInstance != NULL)
      {
         if(vdDiagLog_tclApp::m_poInstance->m_poSnapShotData != NULL)
         {
           vdDiagLog_tclApp::m_poInstance->m_poSnapShotData->clearSnapShotByDTC(u8GetMemoryId(), u32DTC());
         }
         else
         {
            // should never happen if DIAGLOG_USE_SNAPSHOT_DATA is defined
            ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::vClear => NULL == m_poSnapShotData"));
            NORMAL_M_ASSERT_ALWAYS();
         }
      }//if(vdDiagLog_tclApp::m_poInstance != NULL)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::vClear => NULL == m_poInstance"));
         NORMAL_M_ASSERT_ALWAYS();
      }// else // if(vdDiagLog_tclApp::m_poInstance != NULL)
      #endif


      vSetModified();
      ETG_TRACE_COMP_THR(( "--- vdl_tclReportRecord::vClear DTC:%06x ITC:%04x",u32DTC(),u16ITC()));
   }
   else
   {
      ETG_TRACE_COMP_THR(( "--- vdl_tclReportRecord::vClear DTC:%06x ITC:%04x forbidden",u32DTC(),u16ITC()));
   }
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord::processSystemEvent()
// 
// DESCRIPTION: call for Power Event change
//
// PARAMETER:  systemEvent : Power Event
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::processSystemEvent(vdl_teDiaglogSystemEvent systemEvent)
{
	ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::processSystemEvent"));

   switch(systemEvent)
   {
      case DIAGLOG_EVENT_IGNITION_CYCLE_START:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_IGNITION_CYCLE_START"));
         vIgnitionCycleStart();
         break;
      }
      case DIAGLOG_EVENT_IGNITION_CYCLE_END:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_IGNITION_CYCLE_END"));
         vIgnitionCycleEnd();
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_START:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_OPERATION_CYCLE_START"));
         vOperatingCycleStart();
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_END:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_OPERATION_CYCLE_END"));
         vOperatingCycleEnd();
         break;
      }
      case DIAGLOG_EVENT_ACC_CYCLE_START:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_ACC_CYCLE_START:"));
         // unused
         break;
      }
      case DIAGLOG_EVENT_ACC_CYCLE_END:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_ACC_CYCLE_END"));
         // unused
         break;
      }
      case DIAGLOG_EVENT_CRANK_CYCLE_START:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_CRANK_CYCLE_START"));
         // unused
         break;
      }
      case DIAGLOG_EVENT_CRANK_CYCLE_END:
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportRecord::processSystemEvent => DIAGLOG_EVENT_CRANK_CYCLE_END"));
         // unused
         break;
      }
      default:
      {
         // this one should support all, if used or not, just not to forget one
         ETG_TRACE_FATAL_THR(( "!!! vdl_tclReportRecord::processSystemEvent => systemEvent:%d not supported",systemEvent));
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }
   }// switch(systemEvent)


	ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::processSystemEvent"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportRecord::vOperatingCycleStart()
// 
// DESCRIPTION:   on OperationCycle start
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vOperatingCycleStart()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vOperatingCycleStart"));
   m_uRecordStatus.sStatus.m_bIsPassedSinceCurrentPowerUp      = false;
   m_uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle      = false;
   m_uRecordStatus.sStatus.m_bIsNotPassedSinceCurrentPowerUp   = true;
   m_uRecordStatus.sStatus.m_bIsNotCompleteThisOperationCycle  = true;
   m_uRecordStatus.sStatus.m_bIsPassedInCurrentIgnCycle        = false;
   m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle        = false;


   tU8 u8Conf = u8GetVariant();
   tU8 u8SystemConf = vdl_tclGlobalSystemVariables::u8GetSystemConf();

   // delete if variant configuration is invalid
   if((u8Conf & u8SystemConf) == 0)
   {
      ETG_TRACE_COMP_THR(( "--- vdl_tclReportRecord::vOperatingCycleStart => ITC:%04x DTC:%06x DISABLED reason:wrong Configuration(%d<>%d)",u16ITC(),u32DTC(),u8Conf,u8SystemConf));
      m_bDisabledByVariant = true;
      vClear(true);
   }

   // delete if clear on each startup is defined
   if(bIsClearOnStartup())
   {
      ETG_TRACE_USR1_THR(( "--- vdl_tclReportRecord::vOperatingCycleStart => ITC:%04x DTC:%06x Clear On First Startup ",u16ITC(),u32DTC()));
      vClear(true);
   }

   vSetModified();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vOperatingCycleStart"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportRecord::vOperatingCycleEnd()
// 
// DESCRIPTION:   on OperationCycle end
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
/*lint -e(1762) suppress 1762 for next expression*/
tVoid vdl_tclReportRecord::vOperatingCycleEnd() 
{
	// check/increment cycles since first fail if a fail occurred
	if ( true == this->bGetIsFailedInCurrentPowerCycle() ){
		// if first fail, set to one
		if ( 0 == this->m_u8MonCycleSinceFirstFail ){
			this->m_u8MonCycleSinceFirstFail = 1;
		}
		// else the dtc was active otherwise, but not for the first operation
		// cycle. and that's not a valid reason for increment any further, the
		// counter counts subsequent inactive cycles
	}
	else{
		// if a fail happened in the past increment cycles since first fail

		// if a fail happened in the past and counter not yet at max, increment
		// counter
		if ( (0 < this->m_u8MonCycleSinceFirstFail)
				&& (0xff > this->m_u8MonCycleSinceFirstFail) ){
			++(this->m_u8MonCycleSinceFirstFail);
		}
		// else inactive, but counter is 0 so no subsequent cycle. or limit is
		// reached already
	}
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportRecord::vIgnitionCycleStart()
// 
// DESCRIPTION:   on Ignition Cycle start
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vIgnitionCycleStart()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vIgnitionCycleStart"));
   m_bIsIgnitionCycleActive = true;
   if(m_uRecordStatus.sStatus.m_bIsActive == true)
   {
      m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle = true;
      vSetModified();
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vIgnitionCycleStart"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportRecord::vIgnitionCycleEnd()
// 
// DESCRIPTION:   on Ignition Cycle end
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vIgnitionCycleEnd()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vIgnitionCycleEnd"));
   if (bIsDTCEnabled())           // DTC enabled via EOL
   {
      bool bModified = false;
      m_bIsIgnitionCycleActive = false;

      // check each if realy changed, otherwise no store requiered
      if(m_uRecordStatus.sStatus.m_bIsFailedInPreviousIgnCycle != m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle)
      {
         m_uRecordStatus.sStatus.m_bIsFailedInPreviousIgnCycle = m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle;
         bModified = true;
      }
      if(m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle != false)
      {
         m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle = false;
         bModified = true;
      }
      if (m_uRecordStatus.sStatus.m_bIsPassedInCurrentIgnCycle != false)
      {
         m_uRecordStatus.sStatus.m_bIsPassedInCurrentIgnCycle = false;
         bModified = true;
      }

      if(bModified)
      {
         vSetModified();
      }
   }// if (bIsDTCEnabled()) 
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vIgnitionCycleEnd"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportRecord::vPass
// 
// DESCRIPTION:    called when a test pass has been reported 
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vPass()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vPass"));
   
   if(bIsChangeAllowedWihtoutDebounce())
   {
      if(m_bIsIgnitionCycleActive)
      {
         m_uRecordStatus.sStatus.m_bIsPassedInCurrentIgnCycle     = true;    
      }

      m_uRecordStatus.sStatus.m_bIsActive                          = false;
      m_uRecordStatus.sStatus.m_bIsNotPassedSinceDTCCleared        = false;
      m_uRecordStatus.sStatus.m_bIsPassedSinceCurrentPowerUp       = true;
      m_uRecordStatus.sStatus.m_bIsNotPassedSinceCurrentPowerUp    = false;
      m_uRecordStatus.sStatus.m_bIsNotCompleteThisOperationCycle   = false;
      m_uRecordStatus.sStatus.m_bIsNotCompleteSinceLastClear       = false;
      m_uRecordStatus.sStatus.m_bIsPassedSinceDTCCleared           = true;
      vSetModified();
   }


   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vPass"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportRecord::vPass
// 
// DESCRIPTION:    called when a test pass has been reported 
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vPassQualified()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vPassQualified"));
   
#ifdef DIAGLOG_STORE_TO_EMTRACE
   //Get the Current Bit before vPass
   const bool isActiveBeforePass = (m_uRecordStatus.sStatus.m_bIsActive)?true:false;
#endif // #ifdef DIAGLOG_STORE_TO_EMTRACE

   // check for DTC defintion, if DTC should be deleted on PASS Message
   if(bIsClearOnFirstPass())
   {
      vClear(true);
   }

   if(m_bIsIgnitionCycleActive)
   {
      m_uRecordStatus.sStatus.m_bIsPassedInCurrentIgnCycle     = true;    
   }
   m_uRecordStatus.sStatus.m_bIsActive                          = false;
   m_uRecordStatus.sStatus.m_bIsNotPassedSinceDTCCleared        = false;
   m_uRecordStatus.sStatus.m_bIsPassedSinceCurrentPowerUp       = true;
   m_uRecordStatus.sStatus.m_bIsNotPassedSinceCurrentPowerUp    = false;
   m_uRecordStatus.sStatus.m_bIsNotCompleteThisOperationCycle   = false;
   m_uRecordStatus.sStatus.m_bIsNotCompleteSinceLastClear       = false;
   m_uRecordStatus.sStatus.m_bIsPassedSinceDTCCleared           = true;
   vSetModified();

  
#ifdef DIAGLOG_STORE_TO_EMTRACE
   if (!(m_uRecordStatus.sStatus.m_bIsActive))
   {
      //Check whether the Current Bit changed - if yes update the EMTrace
      if (isActiveBeforePass == true)
      {
         tU8 MemoryId = 0;
         if(m_pMemoryObject != NULL)
         {
            MemoryId = m_pMemoryObject->u8GetMemoryId();
         }
            
         vdDiagLog_tclStoreToEmTrace::vStoreToEmTrace(MemoryId, DIAGLOG_REPORT_PASSED, u32DTC(),u16ITC());
      }
   }
#endif // #ifdef DIAGLOG_STORE_TO_EMTRACE

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vPassQualified"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::vFail
// 
// DESCRIPTION:   called when a test fail has been reported
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vFail()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vFail"));   

   //Get the Current Bit before vFailQualified
   const bool isActiveBeforeFailed  = (m_uRecordStatus.sStatus.m_bIsActive)?true:false;
   // needed for case DIAGLOG_OCCURENCE_COUNTER_ON_EACH_FAILED_ONCE_A_OPERATION_CYCLE
   const bool isFailedInCurrentPowerCycle = (m_uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle)?true:false;

   // only handle _OCCURENCE_COUNTER if not still failed
   if(isActiveBeforeFailed == false)
   {
      // increment fail detection counter, must be done before collectExtendedData
      switch(OCCURENCE_COUNTER_ALGORITHM)
      {
         case DIAGLOG_OCCURENCE_COUNTER_ON_EACH_FAILED:
         {
            if(m_u8FailOccurenceCounter < 254)
            {
               ++m_u8FailOccurenceCounter;
            }
            break;
         }
         case DIAGLOG_OCCURENCE_COUNTER_ON_EACH_FAILED_ONCE_A_OPERATION_CYCLE:
         {
            if(isFailedInCurrentPowerCycle == false)
            {
               if(m_u8FailOccurenceCounter < 254)
               {
                  ++m_u8FailOccurenceCounter;
               }
            }
            break;
         }
         default:
         {
            // maybe handled by qualified Failed
            break;
         }
      }// switch(OCCURENCE_COUNTER_ALGORITHM)
   }// if(isActiveBeforeFailed == false)

   // collect the SnapShot Data on each Fail, not on FailQualified
   collectSnapShotData(DIAGLOG_SSD_TRIGGER_BY_FAIL);
   // collect the Extended Data on each Fail, not on FailQualified
   collectExtendedData();

   if(bIsChangeAllowedWihtoutDebounce())
   {
      // change flags also if a DTC is not Qualified
      if(m_bIsIgnitionCycleActive)
      {
         m_uRecordStatus.sStatus.m_bIsFailedInCurrentIgnCycle     = true;    
      }

      m_uRecordStatus.sStatus.m_bIsActive                         = true;
      m_uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle      = true;
      m_uRecordStatus.sStatus.m_bIsNotCompleteThisOperationCycle  = false;
      m_uRecordStatus.sStatus.m_bIsNotCompleteSinceLastClear      = false;
      m_uRecordStatus.sStatus.m_bIsFailedSinceDTCCleared          = true;
      
      vSetModified();
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vFail"));   
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::vFail
// 
// DESCRIPTION:   called when a test fail has been reported
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportRecord::vFailQualified()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::vFailQualified"));   

   //Get the Current Bit before vFailQualified
   const bool isActiveBeforeFailed  = (m_uRecordStatus.sStatus.m_bIsActive)?true:false;
  
   // needed for case DIAGLOG_OCCURENCE_COUNTER_ON_EACH_FAILED_ONCE_A_OPERATION_CYCLE
   const bool isFailedInCurrentPowerCycle = (m_uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle)?true:false;

   m_uRecordStatus.sStatus.m_bIsActive                         = true;
   m_uRecordStatus.sStatus.m_bIsHistory                        = true;
   m_uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle      = true;
   m_uRecordStatus.sStatus.m_bIsNotCompleteThisOperationCycle  = false;
   m_uRecordStatus.sStatus.m_bIsNotCompleteSinceLastClear      = false;
   m_uRecordStatus.sStatus.m_bIsFailedSinceDTCCleared          = true;

   // only handle _OCCURENCE_COUNTER if not still failed
   if(isActiveBeforeFailed == false)
   {
      // increment fail detection counter
      switch(OCCURENCE_COUNTER_ALGORITHM)
      {
         case DIAGLOG_OCCURENCE_COUNTER_ON_EACH_CONFIRMED_FAILED:
         {
            if(m_u8FailOccurenceCounter < 254)
            {
               ++m_u8FailOccurenceCounter;
            }
            break;
         }
         case DIAGLOG_OCCURENCE_COUNTER_ON_EACH_CONFIRMED_FAILED_ONCE_A_OPERATION_CYCLE:
         {
            if(isFailedInCurrentPowerCycle == false)
            {
               if(m_u8FailOccurenceCounter < 254)
               {
                  ++m_u8FailOccurenceCounter;
               }
            }
            break;
         }
         default:
         {
            // maybe handled by none qualified Failed
            break;
         }
      }// switch(OCCURENCE_COUNTER_ALGORITHM)
   }// if(isActiveBeforeFailed == false)
   vSetModified();

#ifdef DIAGLOG_STORE_TO_EMTRACE
   if (m_uRecordStatus.sStatus.m_bIsActive == true)
   {
      //Check whether the Current Bit changed - if yes update the EMTrace
      if (isActiveBeforeFailed == false)
      {
         tU8 MemoryId = 0;
         if(m_pMemoryObject != NULL)
         {
            MemoryId = m_pMemoryObject->u8GetMemoryId();
         }

         vdDiagLog_tclStoreToEmTrace::vStoreToEmTrace(MemoryId, DIAGLOG_REPORT_FAILED, u32DTC(),u16ITC());
      }
   }
#endif // #ifdef DIAGLOG_STORE_TO_EMTRACE

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::vFailQualified"));   
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::bIsDTCReadByCalibrationEnabled
// 
// DESCRIPTION:   is dtc allowed calibration?
//
// PARAMETER:  void
//
// RETURNVALUE: bool true if allowed
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportRecord::bIsDTCReadByCalibrationEnabled() const                      
{
   bool bReturn = true;

   if(vdl_tclGlobalSystemVariables::bGetIgnoreCalibrationSetting() == true)
   {
      ETG_TRACE_USR1_THR(( "--- vdl_tclReportRecord::bIsDTCReadByCalibrationEnabled => ITC:%04x IGNORE SETTING by bGetIgnoreCalibrationSetting",u16ITC()));   
      // ignore following settings
      bReturn = true;
   }// if(vdl_tclGlobalSystemVariables::bGetIgnoreCalibrationSetting())
   else if (m_uRecordStatus.sStatus.m_bIsDTCReadByCalibrationEnabled == false)
   {
      ETG_TRACE_USR1_THR(( "--- vdl_tclReportRecord::bIsDTCReadByCalibrationEnabled => ITC:%04x BLOCKED by m_bIsDTCReadByCalibrationEnabled",u16ITC()));   
      bReturn = false;
   }

   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::u8GetMemoryId
// 
// DESCRIPTION:   
//
// PARAMETER:  none
//
// RETURNVALUE: Id of the Memory
// 
///////////////////////////////////////////////////////////////////////////////////
//
tU8 vdl_tclReportRecord::u8GetMemoryId() const                              
{
   tU8 MemoryId = 0xFF;
   if(m_pMemoryObject != NULL)
   {
      MemoryId = m_pMemoryObject->u8GetMemoryId();
   }
   return MemoryId;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::getExtendedData
// 
// DESCRIPTION:   
//
// PARAMETER:  tU8 aRecNo
//
// RETURNVALUE: ExtendedData
// 
///////////////////////////////////////////////////////////////////////////////////
//
ExtendedData const* vdl_tclReportRecord::getExtendedData(tU8 aRecNo) const
{
   // this is  a workaround for Ageing and Occurence Counter which are not able to collect via the ExtendedData implementation
   std::map<tU8, ExtendedData*>::const_iterator it = this->extendedData.find(DIAGLOG_XD_AGE_COUNTER);
   if(it.operator!=(this->extendedData.end()))
   {
      it->second->collect();
   }
   it = this->extendedData.find(DIAGLOG_XD_OCC_COUNTER);
   if(it.operator!=(this->extendedData.end()))
   {
      it->second->collect();
   }


   // now get the requested Data
   it = this->extendedData.find(aRecNo);
   if(it.operator==(this->extendedData.end()))
   {
      ETG_TRACE_USR1_THR(( "--- vdl_tclReportRecord::getExtendedData => DTC:%04x ExtDataId:%02x Not Found",u32DTC(),aRecNo));   
      return NULL;
   }
   ETG_TRACE_USR1_THR(( "--- vdl_tclReportRecord::getExtendedData => DTC:%04x, ITC: %04x, ExtDataId:%02x",u32DTC(),u16ITC(),aRecNo));   
   return it->second;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::getSnapShotData
// 
// DESCRIPTION: get SnapShot Data from SnapShot Module  
//
// PARAMETER:  recordNumber      : SnapShot Record ID
//             status            : referenz to status, to fill by this function
//             SnapShotDataList  : referenz to List, to fill by this function
//
// RETURNVALUE: ExtendedData
// 
///////////////////////////////////////////////////////////////////////////////////
//
//lint -e1762 // Info 1762: prioX: Member function 'vdl_tclReportRecord::getSnapShotData(unsigned char, unsigned long &, std::list<vdl_tsSnapShotDataStruct> &)' could be made const
//lint -e1764 // Info 1764: prio3: Reference parameter 'SnapShotDataList' (line 945) could be declared const ref
void vdl_tclReportRecord::getSnapShotData(tU8 recordNumber, tU32& status, tSnapShotDataList& SnapShotDataList) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::getSnapShotData"));

   // this is a special behaviour, because we have also to support Extended Data as SnapShot Data
   collectSnapShotData(DIAGLOG_SSD_TRIGGER_BY_READ);

   // first get status information
   status = u32GetStatus();
   bool isDTCSet = false;
   if(bGetIsActive() | bGetIsHistory())
   {
	   isDTCSet = true;
	   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::getSnapShotData -- DTC is set"));
   }
   else
   {
	   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::getSnapShotData -- DTC is not set"));
   }

   // now get the snapshot data self
   #ifdef DIAGLOG_USE_SNAPSHOT_DATA 
   if(vdDiagLog_tclApp::m_poInstance != NULL)
   {
      if(vdDiagLog_tclApp::m_poInstance->m_poSnapShotData != NULL)
      {
        (void) vdDiagLog_tclApp::m_poInstance->m_poSnapShotData->getSnapShotDataByDTC(u8GetMemoryId(), u32DTC(),recordNumber,SnapShotDataList,isDTCSet);
      }
      else
      {
         // should never happen if DIAGLOG_USE_SNAPSHOT_DATA is defined
         ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::getSnapShotData => NULL == m_poSnapShotData"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }//if(vdDiagLog_tclApp::m_poInstance != NULL)
   else
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::getSnapShotData => NULL == m_poInstance"));
      NORMAL_M_ASSERT_ALWAYS();
   }// else // if(vdDiagLog_tclApp::m_poInstance != NULL)
   #else
      DIA_PARAMETER_INTENTIONALLY_UNUSED( recordNumber );
      DIA_PARAMETER_INTENTIONALLY_UNUSED( SnapShotDataList );
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportRecord::getSnapShotData => no snap shot data defined for ALL DTC'S, check define DIAGLOG_USE_SNAPSHOT_DATA"));
   #endif

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::getSnapShotData"));
}
//lint +e1762
//lint +e1764

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::addExtendedData
// 
// DESCRIPTION: create the Extended Data object  
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
//lint -e593  :Custodial pointer 'pAll' (line 1010) possibly not freed or returned

void vdl_tclReportRecord::addExtendedData()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::addExtendedData"));
   if(DIAGLOG_MAX_EXTENDED_RECORDS_SUPPORTED == 0) //lint !e774 !e506
   {
      // no Extended Data defined for all DTC
      ETG_TRACE_USR2_THR(( "--- vdl_tclReportRecord::addExtendedData => NO ext Data defined for all DTC"));  
      return;
   }

   tU8 recNo = this->m_pReportRecordConstData->m_DtcDefinitionTable[m_Index].m_extendedDataRecords[0];
   if(recNo == DIAGLOG_XD_NO)
   {
      // no Extended Data defined for this DTC
      ETG_TRACE_USR2_THR(( "--- vdl_tclReportRecord::addExtendedData => NO ext Data defined for this DTC"));  
      return;
   }// if(recNo == DIAGLOG_XD_NO)

   AllRecords* pAll = new AllRecords(); 
   if(NULL == pAll)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::addExtendedData => NULL == pAll"));
      NORMAL_M_ASSERT_ALWAYS();
      return;
   }


   for(int i = 0; i < DIAGLOG_MAX_EXTENDED_RECORDS_SUPPORTED; ++i)
   {
      recNo = this->m_pReportRecordConstData->m_DtcDefinitionTable[m_Index].m_extendedDataRecords[i];
      ETG_TRACE_USR1_THR(( "--- vdl_tclReportRecord::addExtendedData => recNo=%d",recNo));  

      if(recNo == DIAGLOG_XD_NO)
      {
         // ignore
         continue;
      }

      if(recNo != DIAGLOG_XD_ALL_RECORDS)
      {
         ExtendedData* pNew = 0;
         pNew = generateExtendedData(recNo,this);
         if(pNew != NULL)
         {
            extendedData[recNo] = pNew;
            pAll->add(pNew);
         }
      }
      else
      {
         extendedData[DIAGLOG_XD_ALL_RECORDS] = pAll;
      }
   }// for(int i = 0; i < DIAGLOG_MAX_EXTENDED_RECORDS_SUPPORTED; ++i)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::addExtendedData"));  
}
//lint +e593 

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::collectExtendedData
// 
// DESCRIPTION: Collect the Data to fill all the Exended Data Objects 
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclReportRecord::collectExtendedData()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::collectExtendedData"));  
   std::for_each(extendedData.begin(), extendedData.end(), VDD::collect());
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::collectExtendedData"));  
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportRecord::collectSnapShotData
// 
// DESCRIPTION: Collect the Data to fill all the SnapShot Data Objects 
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
//lint -e1762 // Info 1762: prioX: Member function 'vdl_tclReportRecord::collectSnapShotData(void)' could be made const
void vdl_tclReportRecord::collectSnapShotData(vdl_teDiaglogSnapShotTrigger trigger) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::collectSnapShotData"));  

   #ifdef DIAGLOG_USE_SNAPSHOT_DATA
   if(vdDiagLog_tclApp::m_poInstance != NULL)
   {
      if(vdDiagLog_tclApp::m_poInstance->m_poSnapShotData != NULL)
      {
         vdDiagLog_tclApp::m_poInstance->m_poSnapShotData->updateSnapShotDataByRecord(this,trigger);
      }
      else
      {
         // should never happen
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   #else
      DIA_PARAMETER_INTENTIONALLY_UNUSED( trigger );
   #endif

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::collectSnapShotData"));  
}
//lint +e1762
ExtendedData const* vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC(tReportArray ITCList, tU8 recordNumber)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC"));
   ExtendedData const* pResult = NULL;
   vdl_tclReportRecord* ChosenITC = ITCList.at(0);
   tReportArrayCIt itITCList;
   switch(recordNumber)
   {
	   case DIAGLOG_XD_AGE_COUNTER:
		   {
			   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC -- DIAGLOG_XD_AGE_COUNTER"));
     		   std::vector<tU8> oldAgeingCount(1,0);
			   std::vector<tU8> AgeingCounter;
			      for(itITCList = ITCList.begin(); itITCList != ITCList.end(); ++itITCList)
				  {
					 AgeingCounter = (*itITCList)->getExtendedData(recordNumber)->getData();
					 if(AgeingCounter.at(0) > oldAgeingCount[0])
	                 {
	                      ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC -- ITC with higher ageing count found"));
					      oldAgeingCount[0] = AgeingCounter.at(0);
					      ChosenITC = *itITCList;
	                 }
			      }
				  if(ChosenITC != NULL)
				  {
					  pResult = ChosenITC->getExtendedData(recordNumber);
				  }
				  else
				  {
                     ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC => NULL == chosenITCRecord"));
                     NORMAL_M_ASSERT_ALWAYS();
				  }	
			   break;
           }
	   case DIAGLOG_XD_MILEAGE:
		   {
			   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC -- DIAGLOG_XD_MILEAGE"));
			   std::vector<tU8> Mileage(3,0);
			   tU32 oldMileageVal = 0;
			      for(itITCList = ITCList.begin(); itITCList != ITCList.end(); ++itITCList)
			      {
	                 Mileage = (*itITCList)->getExtendedData(recordNumber)->getData();
				     tU32 tempVal = ((Mileage.at(2)<< 16) & 0x00FF0000) | ((Mileage.at(1) << 8) & 0x0000FF00) | (Mileage.at(0) & 0x000000FF);
	                 if(tempVal > oldMileageVal)
	                 {
	                    ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC -- ITC with higher Mileage value found"));
					    oldMileageVal = tempVal;
						ChosenITC = *itITCList;
	                 }
			      }
				  if(ChosenITC != NULL)
				  {
					  pResult = ChosenITC->getExtendedData(recordNumber);
				  }
				  else
				  {
                     ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC => NULL == chosenITCRecord"));
                     NORMAL_M_ASSERT_ALWAYS();
				  }	
			   break;
           }
	   case DIAGLOG_XD_ALL_RECORDS:
		  {
			ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC -- DIAGLOG_XD_ALL_RECORDS"));
			std::vector<tU8> AgeingCounter;
    		std::vector<tU8> oldAgeingCount(1,0);
				for(itITCList = ITCList.begin(); itITCList != ITCList.end(); ++itITCList)
				  {
					 AgeingCounter = (*itITCList)->getExtendedData(DIAGLOG_XD_AGE_COUNTER)->getData();
					 if(AgeingCounter.at(0) > oldAgeingCount[0])
	                 {
					     oldAgeingCount[0] = AgeingCounter.at(0);
					     ChosenITC = *itITCList;
	                 }
			      }
				  if(ChosenITC != NULL)
				  {
					  pResult = ChosenITC->getExtendedData(recordNumber);
				  }
				  else
				  {
                     ETG_TRACE_ERRMEM(( "!!! vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC => NULL == chosenITCRecord"));
                     NORMAL_M_ASSERT_ALWAYS();
				  }		  
			break;
		  }
	   case DIAGLOG_XD_OCC_COUNTER:
	   default:
		   {
		      ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC invalid record No. %d", recordNumber));
		      break;
		   }
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC"));
   return pResult;
}
