//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_report_memory.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog Server ReportMemory base class
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 27.10.05  | ESA2 Barber                  | Initial version
// 10.08.12  | BSOT Plischke                | new Diaglog
//
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include "vd_diaglog_report_memory.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_ageing_single
   #include <common/framework/ageing/vd_diaglog_ageing_single.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_ageing_global
   #include <common/framework/ageing/vd_diaglog_ageing_global.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_service
   #include <project/interfaces/vd_diaglog_service.h>  
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
   #include <common/framework/vd_diaglog_system_variables.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_plugin_includes
   #include "project/plugin/vd_diaglog_plugin_includes.h"
#endif


/* --Includes. */
/*
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_report_memory.cpp.trc.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_typedefs
   #include <common/framework/vd_diaglog_typedefs.h>
#endif

using namespace VDD;

namespace VDD
{
struct equalsDTC
{
   equalsDTC();
   equalsDTC(tU32 aDTC) : dtc(aDTC) {}
   bool operator() (const vdl_tclReportRecord* p) const {return p->u32DTC() == this->dtc;}
private:
   tU32 dtc;
};
}// namespace VDD


///////////////////////////////////////////////////////////////////////////////////
// constructor
vdl_tclReportMemory::vdl_tclReportMemory(tU8 Id, tCString acMemoryFile, int size, const trReportRecordData* pConstDataArray):
   vdl_tclStorageUserInterface(acMemoryFile),
   m_MemoryId(Id),
   m_u8GlobalAgeingCounter(DIAGLOG_GLOBAL_AGEING_DELFAULT),
   m_u32LastReportedDTC(0),
   m_u32StatOfLastReportedDTC(0),
   m_pAgeingAlgorithm(NULL),
   m_u16NumberOfDynamicITCs(0),
   m_reportRecordConstData(pConstDataArray, size),
   m_poQualificationObject(vdDiagLog_tclApp::m_poInstance->getQualificationObject())
{  
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vdl_tclReportMemory"));
 
   // register for Qualification Handler
   if(m_poQualificationObject != NULL)
   {
      if(m_poQualificationObject->registerCallBack(this)== false)
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::vdl_tclReportMemory => registerCallBack FAILED"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::vdl_tclReportMemory => NULL == m_poQualificationObject"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vdl_tclReportMemory"));
}


// destructor
vdl_tclReportMemory::~vdl_tclReportMemory()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::~vdl_tclReportMemory"));

   _BP_TRY_BEGIN
   {
      // if the datbase has been modified do it now before power down
      vSaveNow();
      vDestroy();
   }

   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::~vdl_tclReportMemory => _BP_CATCH_ALL"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::~vdl_tclReportMemory"));
}



///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vDestroy
// 
// DESCRIPTION: destroy class contents
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vDestroy()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vDestroy"));

   // kill qualification Interface
   if(m_poQualificationObject != NULL)
   {
      if(m_poQualificationObject->unregisterCallBack(this) == false)
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::vDestroy => unregisterCallBack FAILED"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   // delete pointers in vector
   for(
      tReportArrayIt itArray = m_oReports.begin(); 
      itArray != m_oReports.end(); 
      ++itArray
      ) 
   {
      delete *itArray;
      *itArray = 0;
   }

   if(m_pAgeingAlgorithm != NULL)
   {
      delete m_pAgeingAlgorithm;
      m_pAgeingAlgorithm = NULL;
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vDestroy"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::bProcessTestReports
// 
// DESCRIPTION: process test reports in worklist
//
// PARAMETER:  midw_fi_tcl_TestResultList const& roTestReport - test report(s)
//             midw_diaglogfi_tclMsgDTCActiveStatus   roActivationStatus - DTC activation status
//
// RETURNVALUE: bool - true if data was sucessfully process
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportMemory::bProcessTestReport(tU16 u16ITC, vdl_teDiaglogReportMsg eReportMsgStatus)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::bProcessTestReport"));
   bool bReturn = false; // Default ITC invalid


   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::bProcessTestReport => Error: LOAD FAILED"));
      return false;
   }


   LockMemorySemaphore();
   {
      // for each test report
      for(tReportArrayCIt itReport =  m_oReports.begin();
          itReport != m_oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         // process this test result
         if(NULL != pReportRecord)
         {
            // serach for ITC in List
            if(pReportRecord->u16ITC() == u16ITC)
            {
               ETG_TRACE_USR2_THR(("--- vdl_tclReportMemory::bProcessTestReport => ITC:%x Type:%x",u16ITC, ETG_ENUM(DIAGLOG_REPORT,eReportMsgStatus)));
               vManageReport(pReportRecord, eReportMsgStatus);
               bReturn = true; // this ITC is valid
               break;
            }
         }
         else
         {
           ETG_TRACE_ERR_THR(("!!! vdl_tclReportMemory::bProcessTestReport => ERROR: NULL == pReportRecord"));     
         }
      }
      // if database has changed
      vCheckAndSave();
      vUpdateActiveDTCList();
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::bProcessTestReport"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::bProcessTestReports
// 
// DESCRIPTION: process test reports in worklist
//
// PARAMETER:  vdl_tsDiaglogTestResult oTestResult: Full Test Result 
//
// RETURNVALUE: bool - true if data was sucessfully process
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportMemory::bProcessTestReport(vdl_tsDiaglogTestResult oTestResult)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::bProcessTestReport"));
   bool bReturn = false; // default ITC invalid
   bool bFound = false;  // default ITC/DTC not found

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::bProcessTestReport => Error: LOAD FAILED"));
      return false;
   }


   LockMemorySemaphore();
   {
      // for each test report
      for(tReportArrayCIt itReport =  m_oReports.begin();
          itReport != m_oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         // process this test result
         if(NULL != pReportRecord)
         {
            // serach for ITC/DTC in List
            if(oTestResult.bDynamicalDTC == true)
            {
               // dnyamic DTC only support DTC's
               if(pReportRecord->u32DTC() == oTestResult.u32DTC)
               {
                  // we found the DTC
                  if(pReportRecord->isDynamicalDTC())
                  {
                     // this is a dynamical DTC, update
                     ETG_TRACE_USR2_THR(("--- vdl_tclReportMemory::bProcessTestReport => DTC:%x Type:%x",oTestResult.u32ITC, ETG_ENUM(DIAGLOG_REPORT,oTestResult.eReportStatus)));
                     vManageReport(pReportRecord, oTestResult.eReportStatus);
                     bReturn = true; // the DTC is valid
                     bFound = true;  // this DTC is still in list
                     break;
                  }
                  else
                  {
                     // we receive a dynamical DTC but it is configured as static one, reject!!!
                     ETG_TRACE_FATAL_THR(("!!! vdl_tclReportMemory::bProcessTestReport => DTC:%x Type:%x receive, but defined as static ->IGNORE",oTestResult.u32ITC, ETG_ENUM(DIAGLOG_REPORT,oTestResult.eReportStatus)));
                     bReturn = false; // reject
                     bFound = true; // this DTC is still in list
                     break;
                  }
               }
            }
            else
            {
               // statuc DTC use only ITC's
               if(pReportRecord->u16ITC() == oTestResult.u32ITC)
               {
                  ETG_TRACE_USR2_THR(("--- vdl_tclReportMemory::bProcessTestReport => ITC:%x Type:%x",oTestResult.u32ITC, ETG_ENUM(DIAGLOG_REPORT,oTestResult.eReportStatus)));
                  vManageReport(pReportRecord, oTestResult.eReportStatus);
                     bReturn = true; // the ITC is valid
                     bFound = true;  // this ITC is still in list
                  break;
               }
            }
         }
         else
         {
           ETG_TRACE_ERR_THR(("!!! vdl_tclReportMemory::bProcessTestReport => ERROR: NULL == pReportRecord"));     
         }
      }// for(tReportArrayCIt itReport =  m_oReports.begin();...

      if(bFound == false)
      {
         // DTC not found
         if(   (oTestResult.bDynamicalDTC == true) && (u8GetMemoryId() == DIAGLOG_MEMORY_CUSTOMER))
         {
            if(m_u16NumberOfDynamicITCs < this->getMaxDynamicDtcs())//lint !e568 // could be defined = 0
            {
               oTestResult.u32ITC = DYNAMICAL_ITC_START_ID + m_u16NumberOfDynamicITCs;
               // we have a dynamit DTC which we have to add to the List
               ETG_TRACE_COMP_THR(("--- vdl_tclReportMemory::bProcessTestReport => New dynamic DTC:%x ITC:%x Type:%x",oTestResult.u32DTC, oTestResult.u32ITC, ETG_ENUM(DIAGLOG_REPORT,oTestResult.eReportStatus)));     

               // first create object and add to Record List
               vdl_tclReportRecord* pReportRecord = createReportRecord((tU16)oTestResult.u32ITC,oTestResult.u32DTC);
               if(pReportRecord != NULL)
               {
                  pReportRecord->initialise();

                  // add to list
                  m_oReports.push_back(pReportRecord);

                  ++m_u16NumberOfDynamicITCs; // increase number of Dynamic DTC's
                  setModified(); // we have changes initate save

                  // now forward for handling
                  ETG_TRACE_USR2_THR(("--- vdl_tclReportMemory::bProcessTestReport => ITC:%x Type:%x",oTestResult.u32ITC, ETG_ENUM(DIAGLOG_REPORT,oTestResult.eReportStatus)));
                  vManageReport(pReportRecord, oTestResult.eReportStatus);
                  bReturn = true;
               }// if(pReportRecord != NULL)
               else
               {
                  ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::bProcessTestReport => NULL == pReportRecord"));
                  NORMAL_M_ASSERT_ALWAYS();
                  bReturn = false;
               }
            }// if(m_u16NumberOfDynamicITCs < MAX_NUMBER_OF_DYNAMIC_DATA)
            else
            {
               ETG_TRACE_ERR_THR(("--- vdl_tclReportMemory::bProcessTestReport => max Number of DynamicalDTC's received (%d) -> REJECTED ",m_u16NumberOfDynamicITCs));
            }
         }
      }// if(bFound == false)
      vCheckAndSave();
      vUpdateActiveDTCList();
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::bProcessTestReport"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vClearAll
// 
// DESCRIPTION: clear all report records
//
// PARAMETER:  void
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vClearAll()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vClearAll"));
   // clear time quali list
   if(m_poQualificationObject != NULL)
   {
      m_poQualificationObject->clearQualiListByMemoryId(m_MemoryId);
   }

   // for each report
   tReportArrayIt iPos = m_oReports.begin();
   while(iPos != m_oReports.end())
   {
      vdl_tclReportRecord* pReportRecord = (*iPos); // get report record

      if(NULL != pReportRecord)
      {        
         if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_SINGLE) //lint !e774 !e506
         {
            // if single aging send clear for each
            if(m_pAgeingAlgorithm != NULL)
            {
               m_pAgeingAlgorithm->onClearSingle(pReportRecord); // reset aging
            }
         }// if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_SINGLE)    

         if(pReportRecord->isDynamicalDTC())
         {
            // this is a dynamical DTC, this one has to be deleted now from list
            ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::vClearAll => remove Dynamcial DTC:%06x",pReportRecord->u32DTC()));
            iPos = deleteRecordByIter(iPos);
            continue; // this iterator is not longer valid -> continue with next one
         }
         else
         {
            pReportRecord->vClear(); // clear report
         }
      }// if(NULL != pReportRecord)
      ++iPos;
   }// for(tReportArrayCIt iPos = m_oReports.begin

   if(m_u16NumberOfDynamicITCs != 0)
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vClearAll => fail not all Dynamical DTC deleted %d still active",m_u16NumberOfDynamicITCs));
   }

   if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_GLOBAL) //lint !e774 !e506
   {
      // if global aging send clear only once
      if(m_pAgeingAlgorithm != NULL)
      {
         m_pAgeingAlgorithm->onClearAll(); // reset aging
      }
   }// if(AGEING_ALGORITHM == DIAGLOG_AGEING_ALGORITHM_GLOBAL)

   ETG_TRACE_ERRMEM(("DIAGLOG => MemoryId:%d ClearAll",m_MemoryId));
   setModified(); // we have changes initate save

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vClearAll"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::clearByMask
// 
// DESCRIPTION: clear requested report record
//
// PARAMETER:  tU32 u32DtcMask: DTC to delete
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tBool vdl_tclReportMemory::clearByMask(tU32 u32DtcMask, tITCVector& ITCs)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::clearByMask"));
   tBool bReturn = false;

   if(m_poQualificationObject != NULL)
   {
      m_poQualificationObject->clearQualiListByDtc(u32DtcMask);
   }

   tReportArrayIt iPos = m_oReports.begin();
   while(iPos != m_oReports.end())
   {
      vdl_tclReportRecord* pReportRecord = (*iPos); // get report record
      if(NULL != pReportRecord)
      {
         // get trouble code depending on mode
         tU32 u32DTC       = pReportRecord->u32DTC();
         if(u32DTC == u32DtcMask)
         {
            if(m_pAgeingAlgorithm != NULL)
            {
               m_pAgeingAlgorithm->onClearSingle(pReportRecord);
            }

            if(pReportRecord->isDynamicalDTC())
            {
               // this is a dynamical DTC, this one has to be deleted now from list
               ETG_TRACE_ERRMEM(("DIAGLOG => MemoryId:%d ClearByMask DTC:%06x (dynamical)",m_MemoryId, static_cast<tUInt>(u32DTC)));
               iPos = deleteRecordByIter(iPos); // delete set directly to next iterator
               setModified();
               bReturn = true;
               continue; // this iterator is not longer valid -> continue with next one
            }
            else
            {
               ETG_TRACE_ERRMEM(("DIAGLOG => MemoryId:%d ClearByMask DTC:%06x",m_MemoryId, static_cast<tUInt>(u32DTC)));
               setModified();
               bReturn = true;
               pReportRecord->vClear(); // clear report
               ITCs.push_back(pReportRecord->u16ITC());
               (void) bEventStep(EVENT_CLEAR_BY_MASK, pReportRecord); // forward info to the PlugIns
            }
            // continue, a DTC could be added multiple times
         }// if(u32DTC == u32DtcMask)
      }// if(NULL != pReportRecord)
      ++iPos;
   }// for(tReportArrayCIt itReport =  m_oReports.begin();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::clearByMask"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclReportMemory::bLoad
// 
// DESCRIPTION: load raw ReportMemeory RAM data into class members
//
// PARAMETER:  vdl_tclDiagLogObject::tStreamCIterator - array of log file data
//
// RETURNVALUE: bool true=succesfull / false=failed
// 
///////////////////////////////////////////////////////////////////////////////////
//

bool vdl_tclReportMemory::bLoad(tStream& roData)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::bLoad"));

   bool bResult = false;
   tU8 u8Value;

   tStreamCIterator iData = roData.begin();

   u8Value = static_cast<tU8>(*iData++);
   m_u32LastReportedDTC = (static_cast<tU32>(u8Value) << 24);
   u8Value = static_cast<tU8>(*iData++);
   m_u32LastReportedDTC |= (static_cast<tU32>(u8Value) << 16);
   u8Value = static_cast<tU8>(*iData++);
   m_u32LastReportedDTC |= (static_cast<tU32>(u8Value) << 8);
   u8Value = static_cast<tU8>(*iData++);
   m_u32LastReportedDTC |= static_cast<tU32>(u8Value);
   ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::bLoad => m_u32LastReportedDTC = %06x", m_u32LastReportedDTC));

   u8Value = static_cast<tU8>(*iData++);
   m_u32StatOfLastReportedDTC = (static_cast<tU32>(u8Value) << 24);
   u8Value = static_cast<tU8>(*iData++);
   m_u32StatOfLastReportedDTC |= (static_cast<tU32>(u8Value) << 16);
   u8Value = static_cast<tU8>(*iData++);
   m_u32StatOfLastReportedDTC |= (static_cast<tU32>(u8Value) << 8);
   u8Value = static_cast<tU8>(*iData++);
   m_u32StatOfLastReportedDTC |= static_cast<tU32>(u8Value);
   ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::bLoad => m_u32StatOfLastReportedDTC = %08x", m_u32StatOfLastReportedDTC));

   m_u8GlobalAgeingCounter    = static_cast<tU8>(*iData++);
   ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::bLoad => m_u8GlobalAgeingCounter = %02d", m_u8GlobalAgeingCounter));  

   u8Value = static_cast<tU8>(*iData++);
   m_u16NumberOfDynamicITCs = tU16(u8Value << 8);
   u8Value = static_cast<tU8>(*iData++);
   m_u16NumberOfDynamicITCs = tU16(m_u16NumberOfDynamicITCs | u8Value);
   ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vSave: m_u16NumberOfDynamicITCs = %04d", m_u16NumberOfDynamicITCs));

   for (int i=0; i<m_u16NumberOfDynamicITCs; ++i)
   {
      //now create the dynamic DTC Objects, before loading the Data
      vdl_tclReportRecord* pReportRecord;
      pReportRecord = createReportRecord(0xFFFF, 0xFFFFFF);
      pReportRecord->initialise();
      m_oReports.push_back(pReportRecord);
   }

   // for each report record
   for(tReportArrayCIt iPos = m_oReports.begin(); 
       iPos != m_oReports.end(); 
       ++iPos)
   {
      // set object data
      bResult = (*iPos)->bLoad(iData);
      if(bResult == false)
      {
         return false;
      }
      ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::bLoad => DTC:%06x ITC:%04x Status:%08x Index:%d",(*iPos)->u32DTC(), (*iPos)->u16ITC(), (*iPos)->u32GetStatus(), (*iPos)->u16GetIndex())); 
   }

   if(m_pAgeingAlgorithm)
   {
      m_pAgeingAlgorithm->bLoad(iData);
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::bLoad"));
   return true;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportMemory::vSave
// 
// DESCRIPTION: save raw ReportMemeory RAM data
//
// PARAMETER:  tStream - array to fill with class member data
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vSave(tStream& roData) 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vSave"));


   roData.push_back(static_cast<tS8>((m_u32LastReportedDTC >> 24) & 0xff));
   roData.push_back(static_cast<tS8>((m_u32LastReportedDTC >> 16) & 0xff));
   roData.push_back(static_cast<tS8>((m_u32LastReportedDTC >> 8) & 0xff));
   roData.push_back(static_cast<tS8>(m_u32LastReportedDTC & 0xff));
   ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vSave: m_u32LastReportedDTC = %06x", m_u32LastReportedDTC));

   roData.push_back(static_cast<tS8>((m_u32StatOfLastReportedDTC >> 24) & 0xff));
   roData.push_back(static_cast<tS8>((m_u32StatOfLastReportedDTC >> 16) & 0xff));
   roData.push_back(static_cast<tS8>((m_u32StatOfLastReportedDTC >> 8) & 0xff));
   roData.push_back(static_cast<tS8>(m_u32StatOfLastReportedDTC & 0xff));
   ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vSave: m_u32StatOfLastReportedDTC = %08x", m_u32StatOfLastReportedDTC));

   roData.push_back(static_cast<tS8>(m_u8GlobalAgeingCounter));
   ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vSave: m_u8GlobalAgeingCounter = %02d", m_u8GlobalAgeingCounter));

   roData.push_back(static_cast<tS8>((m_u16NumberOfDynamicITCs  >> 8) & 0xff));
   roData.push_back(static_cast<tS8>(m_u16NumberOfDynamicITCs  & 0xff));
   ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vSave: m_u16NumberOfDynamicITCs = %04d", m_u16NumberOfDynamicITCs));

   // for each report record
   for(
      tReportArrayCIt iPos = m_oReports.begin(); 
      iPos != m_oReports.end(); 
      ++iPos
      )
   {
      NORMAL_M_ASSERT(*iPos);
      (*iPos)->vSave(roData);
      ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vSave => DTC:%06x ITC:%04x Status:%08x Index:%d",(*iPos)->u32DTC(), (*iPos)->u16ITC(), (*iPos)->u32GetStatus(), (*iPos)->u16GetIndex())); 
   }

   if(m_pAgeingAlgorithm)
   {
      m_pAgeingAlgorithm->vSave(roData);
   }
     
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vSave"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclReportMemory::vGetSizeOfLoadBuffer
// 
// DESCRIPTION: GetSize of expected File
//
// PARAMETER:  Max, Min
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vGetSizeOfLoadBuffer(tS32& Max, tS32& Min)
{
   tStream oData;

   // collect class data to establish number of bytes to read and set array size (in this case it is not a save function)
   vSave(oData);

   // get the current size
   tS32 s32Size = static_cast<tS32>(oData.size());

   if(MAX_NUMBER_OF_DYNAMIC_DATA > 0) //lint !e774 !e506
   {
      // get the size of one Element
      tStream oDataRecored;

      // create temporary a dummy objetct
      vdl_tclReportRecord DummyDynamicRecord(this,0xFFFF,0xFFFFFF);

      // call save function to get size of Data
      DummyDynamicRecord.vSave(oDataRecored);
      tS32 s32SizeRecord = static_cast<tS32>(oDataRecored.size());

      Max = s32Size + (MAX_NUMBER_OF_DYNAMIC_DATA * s32SizeRecord); //lint !e845 // The right argument to operator '+' is certain to be 0  // MAX_NUMBER_OF_DYNAMIC_DATA could be 0
      Min = s32Size;
   }
   else
   {
      Max = s32Size;
      Min = s32Size;
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vTraceReports
// 
// DESCRIPTION: trace each report 
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vTraceReports()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vTraceReports"));
   
   // u32Timestamp yyyyyymm mmdddddh hhhhmmmm mmssssss
   tU32 u32Timestamp = u32GetDataBaseTimestamp();
 

   tU8 u8Year   = tU8((u32Timestamp >> 26) & 0x3F);
   tU8 u8Month  = tU8((u32Timestamp >> 22) & 0x0F);
   tU8 u8Day    = tU8((u32Timestamp >> 17) & 0x0F);
   tU8 u8Hour   = tU8((u32Timestamp >> 12) & 0x0F);
   tU8 u8Min    = tU8((u32Timestamp >> 6) & 0x3F);
      
   ETG_TRACE_FATAL_THR(( "################# DIAGLOG DUMP MEMORY:%x ##################", m_MemoryId)); 
   ETG_TRACE_FATAL_THR(( "--- vdl_tclReportMemory::vTraceReports => File Date %d.%d.%d %d:%d",u8Day,u8Month,u8Year,u8Hour,u8Min));

   // for each report
   for(
      tReportArrayCIt itReport = m_oReports.begin();
      itReport != m_oReports.end();
      ++itReport
      )
   {
      vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
      // null report should never happen
      if(NULL != pReportRecord)
      {
         pReportRecord->vTrace(); // output trace info
      }
      else
      {
         ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vTraceReports => Error: NULL != pReportRecord"));
      }
   }
   ETG_TRACE_FATAL_THR(( "################# DIAGLOG DUMP MEMORY:%x END ##################", m_MemoryId)); 

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vTraceReports"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vProcessOnInit
// 
// DESCRIPTION: first call to e.g. Load Data, calibrate device
//              while this, it is forbidden to get DATA from other parts
//              do this in vProcessOperatingCycleStart
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vProcessOnInit()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vProcessOnInit"));

   // check we have loaded the database at start up
   if(!bCheckLoadStatus())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessOnInit => Error: LOAD FAILED"));
      return;
   }

  
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vProcessOnInit"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::processSystemEvent
// 
// DESCRIPTION: Process SPM/LCM Events 
//
// PARAMETER:  systemEvent: Event to handle
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::processSystemEvent(vdl_teDiaglogSystemEvent systemEvent)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::processSystemEvent"));
   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessIgnitionCycleEnd => Error: LOAD FAILED"));
      return;
   }

   // send info to each DTC Record
   LockMemorySemaphore();
   {
      // first do some pre handlings
      if(systemEvent == DIAGLOG_EVENT_OPERATION_CYCLE_END)
      {
         // clear time quali list
         if(m_poQualificationObject != NULL)
         {
            m_poQualificationObject->clearQualiListByMemoryId(m_MemoryId);
         }
      }

      // for each report
      for(
         tReportArrayCIt itCode = m_oReports.begin();
         itCode != m_oReports.end();
         ++itCode
         )
      {
         (*itCode)->processSystemEvent(systemEvent);
      }//  for(tReportArrayCIt itCode = m_oReports.begin(); ...

      // inform ageing algorithm 
      if(m_pAgeingAlgorithm != NULL)
      {
         m_pAgeingAlgorithm->processSystemEvent(systemEvent);
      }

      // last do some post handlings
      switch(systemEvent)
      {
         case DIAGLOG_EVENT_IGNITION_CYCLE_START:
         case DIAGLOG_EVENT_IGNITION_CYCLE_END:
         {
            vCheckAndSave();
            break;
         }
         case DIAGLOG_EVENT_OPERATION_CYCLE_START:
         case DIAGLOG_EVENT_OPERATION_CYCLE_END:
         {
            // save Data at END of Startup
            vSaveNow();
            break;
         }
         case DIAGLOG_EVENT_ACC_CYCLE_START:
         case DIAGLOG_EVENT_ACC_CYCLE_END:
         case DIAGLOG_EVENT_CRANK_CYCLE_START:
         case DIAGLOG_EVENT_CRANK_CYCLE_END:
         {
            // unused
            break;
         }
         default:
         {
            // this one should support all, if used or not, just not to forget one
            ETG_TRACE_FATAL_THR(( "!!! vdl_tclSingleAgeing::processSystemEvent => systemEvent:%d not supported",systemEvent));
            NORMAL_M_ASSERT_ALWAYS();
            break;
         }
      }// switch(systemEvent)
   }
   FreeMemorySemaphore();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::processSystemEvent"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vOnExternalAgeingCounter
// 
// DESCRIPTION: Process Ageing Event
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vOnExternalAgeingCounter(tU8 AgeingCounter)
{

   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vOnExternalAgeingCounter"));

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vOnExternalAgeingCounter => Error: LOAD FAILED"));
      return;
   }

   // send info to each DTC Record
   LockMemorySemaphore();
   {
      if(m_pAgeingAlgorithm != NULL)
      {
         m_pAgeingAlgorithm->vOnExternalAgeingCounter(AgeingCounter);
      }
      vCheckAndSave();
   }
   FreeMemorySemaphore();



   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vOnExternalAgeingCounter"));
}



///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vManageReport
// 
// DESCRIPTION: process a single test result
//
// PARAMETER:  vdl_tclReportRecord* pReportRecord        - report affected by result
//             vdl_teDiaglogReportMsg enReportMsgStatus  - report status (PASSED, FAILED,...) 
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vManageReport (
                                             vdl_tclReportRecord*               pReportRecord, 
                                             vdl_teDiaglogReportMsg             enReportMsgStatus
                                             )
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vManageReport"));

   if(pReportRecord != NULL)
   {
      tU32 u32BlockingModeActive = 0;
      tU16 u16ITC = pReportRecord->u16ITC();
      tU32 u32DTC = pReportRecord->u32DTC();

      // process result
      switch(enReportMsgStatus)
      {
         case DIAGLOG_REPORT_PASSED:
         {
            // enabled via Calibration?
            if(pReportRecord->bIsDTCEnabled()) 
            {
               // all runtime conditions fullfilled?
               if(isBlockingModeActive(pReportRecord,u32BlockingModeActive)== false)
               {
                  teQualificationState eQualificationState = NONE_QUALIFICATION_RUNNING;
                  if(m_poQualificationObject != NULL)
                  {
                     eQualificationState = m_poQualificationObject->isQualifiying(pReportRecord);
                  }

                  if(eQualificationState == NONE_QUALIFICATION_RUNNING)
                  {
                     // continue only if changed (currenlty FAILD) do not check for interrupt of blocking mode this is not supported
                     if(bIsAlreadyPassed(pReportRecord) == false)
                     {
                        vPass(pReportRecord);

                        tS32 s32PassValue = pReportRecord->s32GetPassValue();
                        if(s32PassValue == 0)
                        {
                           // no qualiyfication necessary

                           ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x PASSED Qualified",m_MemoryId,u16ITC,u32DTC));           
                           vPassQualified(pReportRecord);

                        }// if(u32PassValue == 0)
                        else
                        {
                           // we have to qualiy it first
                           ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x PASSED Qualifing Time:%d",m_MemoryId, u16ITC,u32DTC,s32PassValue));    
                           if(m_poQualificationObject != NULL)
                           {
                              m_poQualificationObject->addToQualificationListPassed(this, pReportRecord);
                           }
                        }
                     }// if(bIsAlreadyPassed(pReportRecord) == false)
                     else 
                     {
                        // do nothing, just passed
                        ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x PASSED, but already PASSED",m_MemoryId, u16ITC,u32DTC));     
                     }
                  }// if(eQualificationState == NONE_QUALIFICATION_RUNNING)
                  else if(eQualificationState == PASS_QUALIFICATION_RUNNING)
                  {
                     // do nothing, a PASS is just quallifing
                     ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x PASSED, but already qualifing PASSED",m_MemoryId, u16ITC,u32DTC));     
                  }// else if(eQualificationState == PASS_QUALIFICATION_RUNNING)
                  else if(eQualificationState == FAIL_QUALIFICATION_RUNNING)
                  {
                     // stop this qualification now!!!   
                     ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x PASSED, breaking FAILED debouncing",m_MemoryId, u16ITC,u32DTC)); 
                     if(m_poQualificationObject != NULL)
                     {
                        m_poQualificationObject->stopQualifiying(pReportRecord);
                     }
                     vPass(pReportRecord); // set back if m_bChangeOnDebounceOnly is configured
                  }// else if(eQualificationState == FAIL_QUALIFICATION_RUNNING)
               }// if(!(u32BlockingModeActive))
               else
               {
                  // currently blocked  
                  ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x PASSED, but BlockedBy:%x",m_MemoryId, u16ITC,u32DTC, u32BlockingModeActive ));     
               }
            }// if(bIsDTCEnabled())
            else
            {
               // do nothing DTC is inactive     
               ETG_TRACE_USR1_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x PASSED, but DTC NOT ACTIVE",m_MemoryId, u16ITC,u32DTC));
            }
            break;
         }// case DIAGLOG_REPORT_PASSED:

         case DIAGLOG_REPORT_FAILED:
         {
            // enabled via Calibration?
            if(pReportRecord->bIsDTCEnabled())
            {
               // all runtime conditions fullfilled?
               if(isBlockingModeActive(pReportRecord,u32BlockingModeActive)==false)
               {        
                  teQualificationState eQualificationState = NONE_QUALIFICATION_RUNNING;
                  if(m_poQualificationObject != NULL)
                  {
                     eQualificationState = m_poQualificationObject->isQualifiying(pReportRecord);
                  }

                  if(eQualificationState == NONE_QUALIFICATION_RUNNING)
                  {
                     tS32 s32FailValue = pReportRecord->s32GetFailValue();   // has DTC a qualification time?
                     // continue only if changed (currenlty PASSED) or interrupted by blocking mode
                     if(bIsAlreadyFailed(pReportRecord) == false)
                     {
                        // this is a DTC change from Pass to failed
                        vFail(pReportRecord);
                        if(s32FailValue == 0)
                        {
                           // no qualiyfication necessary
                           ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x FAILED Qualified",m_MemoryId, u16ITC,u32DTC));           
                           vFailQualified(pReportRecord);
                        }// if(u32PassValue == 0)
                        else
                        {
                           // we have to qualiy it first
                           ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x FAILED Qualifing Time:%d",m_MemoryId, u16ITC,u32DTC,s32FailValue));  
                           if(m_poQualificationObject != NULL)
                           {
                              m_poQualificationObject->addToQualificationListFailed(this, pReportRecord);
                           }
                        }
                     }// if(bIsAlreadyFailed(pReportRecord) == false)
                     else if(  (pReportRecord->bGetIsHistory() == false)
                             &&(s32FailValue != 0))
                     {
                        // this DTC was interuppted by qualifing, so do again
                        ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x FAILED Qualifing Time:%d",m_MemoryId, u16ITC,u32DTC,s32FailValue));        
                        if(m_poQualificationObject != NULL)
                        {
                           m_poQualificationObject->addToQualificationListFailed(this, pReportRecord);
                        }
                     }
                     else
                     {
                        // do nothing, just passed
                        ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x FAILED, but already FAILED",m_MemoryId, u16ITC, u32DTC));     
                     }
                  }// if(eQualificationState == NONE_QUALIFICATION_RUNNING)
                  else if(eQualificationState == FAIL_QUALIFICATION_RUNNING)
                  {
                     // do nothing, a FAIL is just quallifing
                     ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x FAILED, but already qualifing FAILED",m_MemoryId, u16ITC, u32DTC));     
                  }// else if(eQualificationState == FAIL_QUALIFICATION_RUNNING)
                  else if(eQualificationState == PASS_QUALIFICATION_RUNNING)
                  {
                     // stop this qualification now!!!   
                     ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x FAILED, breaking PASSED debouncing",m_MemoryId, u16ITC, u32DTC)); 
                     if(m_poQualificationObject != NULL)
                     {
                        m_poQualificationObject->stopQualifiying(pReportRecord);
                     }
                     vFail(pReportRecord); // set back if m_bChangeOnDebounceOnly is configured
                  }// else if(eQualificationState == PASS_QUALIFICATION_RUNNING)
               }// if(!(u32BlockingModeActive))
               else
               {
                  ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Report received ITC:%04x DTC:%06x FAILED, but BlockedBy:%x",m_MemoryId, u16ITC, u32DTC, u32BlockingModeActive ));     
               }
            }// if(bIsDTCEnabled())
            else
            {
               // do nothing DTC is inactive     
               ETG_TRACE_USR1_CLS(( TR_CLASS_DIAGLOG_SM, ":[%d]  Report received ITC:%04x DTC:%06x FAILED, but DTC NOT ACTIVE",m_MemoryId, u16ITC, u32DTC));
            }
            break;
         }// case DIAGLOG_REPORT_FAILED:

         case DIAGLOG_REPORT_NORESULT:
         {
            // no result means to stop qualification if running
            teQualificationState eQualificationState = NONE_QUALIFICATION_RUNNING;
            if(m_poQualificationObject != NULL)
            {
               eQualificationState = m_poQualificationObject->isQualifiying(pReportRecord);
            }

            switch(eQualificationState)
            {
               case PASS_QUALIFICATION_RUNNING:
               {
                  ETG_TRACE_USR1_CLS(( TR_CLASS_DIAGLOG_SM, ":[%d]  Report received ITC:%04x DTC:%06x NO_RESULT STOP PASS qualification",m_MemoryId, u16ITC, u32DTC));
                  if(m_poQualificationObject != NULL)
                  {
                     m_poQualificationObject->stopQualifiying(pReportRecord);
                  }
                  break;
               }

               case FAIL_QUALIFICATION_RUNNING:
               {
                  ETG_TRACE_USR1_CLS(( TR_CLASS_DIAGLOG_SM, ":[%d]  Report received ITC:%04x DTC:%06x NO_RESULT STOP FAIL qualification",m_MemoryId, u16ITC, u32DTC));
                  if(m_poQualificationObject != NULL)
                  {
                     m_poQualificationObject->stopQualifiying(pReportRecord);
                  }
                  break;
               }

               case NONE_QUALIFICATION_RUNNING:
               {
                  // nothing to do
                  break;
               }
            }// switch(eQualificationState)
            break;
         }// case DIAGLOG_REPORT_NORESULT:

         case DIAGLOG_REPORT_DISABLE:
         {
            pReportRecord->vSetDTCCalibrationDisabled();
            setModified();
            ETG_TRACE_USR2_THR(( "DTC %d disabled", u16ITC));
            break;
         }// case DIAGLOG_REPORT_DISABLE:

         case DIAGLOG_REPORT_ENABLE:
         {
            pReportRecord->vSetDTCCalibrationEnabled();
            setModified();
            ETG_TRACE_USR2_THR(( "DTC %x enabled", u16ITC));
            break;
         }// case DIAGLOG_REPORT_ENABLE:

         default:
            break;
      }

       vCheckAndSave();
   }
   else
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::vManageReport => ERROR NULL != pReportRecord"));
      NORMAL_M_ASSERT_ALWAYS();
   }


   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vManageReport"));

}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vAddErrorReports
// 
// DESCRIPTION: create a report record and add it to the report memory
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vAddErrorReports() 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vAddErrorReports"));

   for(tU16 i = 0; i < m_reportRecordConstData.m_iDtcDefinitionTableSize; ++i)
   {  
      vdl_tclReportRecord* pReportRecord = NULL;
      vdl_teDiaglogReportType  eReportTyp = m_reportRecordConstData.GetReportType(i);

      switch(eReportTyp)
      {
         case DIAGLOG_REPORT_BASE:
         {
            pReportRecord = createReportRecord(&m_reportRecordConstData, i);
            if(NULL == pReportRecord)
            {
               ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::vAddErrorReports => NULL == pReportRecord"));
               NORMAL_M_ASSERT_ALWAYS();
               return;
            }
            pReportRecord->initialise();
            break;
         }

         default:
         {
            pReportRecord = poCreateOverloadRecord(eReportTyp, this, &m_reportRecordConstData, i);
            break;
         }
      }// switch(eReportTyp)

      m_oReports.push_back(pReportRecord);
   }// for(int i = 0; i < size; ++i)  



   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vAddErrorReports"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vAddAegingAlgorithm
// 
// DESCRIPTION: create a ageing allgorithm
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vAddAgeingAlgorithm() 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vAddAgeingAlgorithm"));

   switch(AGEING_ALGORITHM)
   {
      case DIAGLOG_AGEING_ALGORITHM_GLOBAL:
      {
         m_pAgeingAlgorithm = new vdl_tclGlobalAgeing(this);
         break;
      }

      case DIAGLOG_AGEING_ALGORITHM_SINGLE:
      {
         m_pAgeingAlgorithm = new vdl_tclSingleAgeing(this);
         break;
      }

      case DIAGLOG_AGEING_ALGORITHM_NONE:
      {
         m_pAgeingAlgorithm = new vdl_tclAgeing();
         break;
      }

      default:
      {
         break;
      }
   }// switch(AGEING_ALGORITHM)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vAddAgeingAlgorithm"));
}



///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vProcessProductionClearRequest
// 
// DESCRIPTION: process clear request
//
// PARAMETER: 
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tBool vdl_tclReportMemory::processClearRequest(tU32 u32DtcMask, tITCVector& ITCs)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::processClearRequest"));

   tBool bReturn = false;

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::processClearRequest => LOAD FAILED"));
      return false;
   }
  
   LockMemorySemaphore();
   {
      switch(u32DtcMask)
      {
         case DIAGLOG_CLEAR_ALL_DTC:
         {
            vClearAll();   // clear all report records
            bReturn = true;
            break;
         }
         default:
         {
            bReturn = clearByMask(u32DtcMask, ITCs);
            break;
         }
      }// switch(u32DtcMask)
      // save new status to flash
      vSaveNow();
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::processClearRequest"));
   return bReturn;
}

tS8 vdl_tclReportMemory::vProcessReadPriority(tReadReportListDM1& DM1List)
{
	if ( !isDataBaseLoaded() ){
		ETG_TRACE_COMP_THR (("!!! vdl_tclReportMemory::vProcessReadPriority => Error: LOAD FAILED"));
		return -1;
	}
	LockMemorySemaphore();
	{
		for (tReportArrayCIt itReport = m_oReports.begin();
				itReport != m_oReports.end(); ++itReport){
			vdl_tclReportRecord* pReportRecord = (*itReport);
			if( NULL != pReportRecord ){
				if( 0x08 & pReportRecord->u32GetStatus()){
					ETG_TRACE_COMP_THR (("!!! vdl_tclReportMemory::vProcessReadPriority =>DTC %x Sev %x",pReportRecord->u32DTC(),pReportRecord->u8GetSeverity()));
					struct vdl_tsDiagReadReportDM1 l_stActiveDM1;
					tU32  l_u32DTC =  pReportRecord->u32DTC();
					memcpy( &l_stActiveDM1.u32DTC, &l_u32DTC,
							sizeof(l_stActiveDM1.u32DTC));
					l_stActiveDM1.u16Priority =
							pReportRecord->u8GetSeverity();
					l_stActiveDM1.u32OccuranceCount =
							pReportRecord->u8GetFailOccurenceCounter();
					DM1List.push_back( l_stActiveDM1 );
				}
			}
			else{
				ETG_TRACE_COMP_THR (("!!! vdl_tclReportMemory::vProcessReadPriority => Error: pReportRecord NULL"));
			}
		}
	}
	FreeMemorySemaphore();
	return 0;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vProcessReadRequest
// 
// DESCRIPTION: process read request
//
// PARAMETER:  tU8 const& u8StatusMask - the status maskread request
//             midw_fi_tcl_CustomerReadResultList& roReadResult - request result goes in here
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vProcessReadRequest(tReadReportListArray&        roReadResult,
                                               tU32 const                   u32StatusMask,
                                               tU32 const                   u32DTCMask,
                                               vdl_teDiaglogOptions const   eOption) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vProcessReadRequest"));

   bool bAddToList = false;

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessReadRequest => Error: LOAD FAILED"));
      return;
   }


   LockMemorySemaphore();
   {
      // for each report
      for(tReportArrayCIt itReport =  m_oReports.begin();
          itReport != m_oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         if(NULL != pReportRecord)
         {
            // get trouble code depending on mode
            tU32 u32DTC             = pReportRecord->u32DTC();
            tU16 u16ITC             = pReportRecord->u16ITC();
            tU32 u32DTCStatus       = pReportRecord->u32GetStatus();
            tU8  u8CalibEnable      = pReportRecord->bIsDTCEnabled();
            tU8  u8CalibReadEnable  = pReportRecord->bIsDTCReadByCalibrationEnabled();
            tU8  u8DTCActive        = u8CalibEnable && u8CalibReadEnable;

            ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vProcessReadRequest => DTC:$%06x ITC:$%04x Status:$%04x, StatusMask:$%04x, ITCMask:$%04x CalEnable:%x CalReadEnable:%x", u32DTC,u16ITC, u32DTCStatus, u32StatusMask, u32DTCMask, u8CalibEnable, u8CalibReadEnable));

            if(u32DTC == 0xFFFFFF)
            {
               // ignore this DTC
               continue;
            }

            tBool bIsOptionOk = false;
            switch(eOption)
            {
               // search only for a unique DTC/ITC ID
               case OPTION_ITC_EQUAL:
               {
                  if(u16ITC == u32DTCMask)
                  {
                     bIsOptionOk = true;
                  }
                  break;
               }// case OPTION_ITC_EQUAL:

               case OPTION_DTC_EQUAL:
               {
                  if(u32DTC == u32DTCMask)
                  {
                     bIsOptionOk = true;
                  }
                  break;
               }// case OPTION_DTC_EQUAL:

               // use standard option (take valids)
               case OPTION_NONE:
               {
                  if(  (u8DTCActive == TRUE)
                     &&(u32StatusMask & u32DTCStatus)
                     &&(u32DTCMask & u16ITC)
                    )
                  {
                     bIsOptionOk = true;
                     if (u32StatusMask == 0x1000) //m_bIsNotCompleteSinceLastClear
                     {
                        for(tReportArrayCIt itReport2 =  m_oReports.begin();
                            itReport2 != m_oReports.end();
                            ++itReport2)
                        {
                           vdl_tclReportRecord* pReportRecord2 = (*itReport2); // get report record
                           if ((NULL != pReportRecord2) &&
                                 (u32DTC == pReportRecord2->u32DTC()) &&
                                 (u16ITC != pReportRecord2->u16ITC()) &&
                                 ((pReportRecord2->bGetIsFailedSinceDTCCleared()) ||
                                  (pReportRecord2->bGetIsPassedSinceDTCCleared())))
                           {
                              ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemoryBase::vProcessReadRequest => NotCompleteSinceLastClear override by ITC:$%04x, pass:%d, fail:%d",
                                    pReportRecord2->u16ITC(),
                                    pReportRecord2->bGetIsPassedSinceDTCCleared(),
                                    pReportRecord2->bGetIsFailedSinceDTCCleared()));
                              bIsOptionOk = false;
                              break;
                           }
                        }
                     }
                  }
                  break;
               }// case OPTION_NONE:

               default:
               {
                  ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::vProcessReadRequest => UNKNOWN switch(eOption:%x)",eOption));
                  NORMAL_M_ASSERT_ALWAYS();
                  break;
               }
            }// switch(eOption)

            if(bIsOptionOk)
            {
               // the DTC is valid to get
               bAddToList = true;
               // same DTC could be part of two different ITC's
               // now search for double
               tReadReportListArrayIt iPos = roReadResult.begin();
               while(iPos.operator!=(roReadResult.end()))
               {
                  // search for DTC in current LIST
                  if(iPos->u32DTC == u32DTC)
                  {
                     // found this DTC in the LIST, now check the status Bit-Field
                     tuRecordStatus uStatus;
                     uStatus.u32Status = iPos->u32Status;

                     // first active wins
                     if(  (pReportRecord->bGetIsActive() == true)
                        &&(uStatus.sStatus.m_bIsActive == false)
                       )
                     {
                        // this one is still in list, but is not active, so overwrite
                        ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemoryBase::vProcessReadRequest => DTC:$%06x, ITC:$%04x, Status:$%02x, StatusMask:$%02x, DTCMask:$%06x (overwrite by active)", u32DTC, u16ITC, u32DTCStatus, u32StatusMask,u32DTCMask));
                        //iPos->u32DTC    = u32DTC; // not necessary just the same
                        iPos->u16ITC    = u16ITC;
                        iPos->u32Status = u32DTCStatus;
                        bAddToList = false; // no need to add a new List item
                     }
                     // then history
                     else if(  (pReportRecord->bGetIsActive() == false)
                             &&(pReportRecord->bGetIsHistory() == true)
                             &&(uStatus.sStatus.m_bIsHistory == false)
                            )
                     {
                        // this one is still in list, but is not active, so overwrite
                        ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemoryBase::vProcessReadRequest => DTC:$%06x, ITC:$%04x, Status:$%02x, StatusMask:$%02x, DTCMask:$%06x (overwrite by history)", u32DTC, u16ITC, u32DTCStatus, u32StatusMask,u32DTCMask));
                        //iPos->u32DTC    = u32DTC; // not necessary just the same
                        iPos->u16ITC    = u16ITC;
                        iPos->u32Status = u32DTCStatus;
                        bAddToList = false; // no need to add a new List item
                     }
                     else
                     {
                        // this one is still in list, and both have same state, so do nothing
                        ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemoryBase::vProcessReadRequest => DTC:$%06x, ITC:$%04x, Status:$%02x, StatusMask:$%02x, DTCMask:$%06x (found same)", u32DTC, u16ITC, u32DTCStatus, u32StatusMask,u32DTCMask));
                        bAddToList = false; // no need to add a new List item
                     }
                     // continue maybe more ITC's have the same DTC
                  }// if(iPos->u32DTC == u32DTC)
                  ++iPos;
               }// while(iPos.operator!=(roReadResult.end()))

               if(bAddToList)
               {
                  // use this DTC/ITC
                  ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemoryBase::vProcessReadRequest => DTC:$%06x, ITC:$%04x, Status:$%02x, StatusMask:$%02x, DTCMask:$%06x", u32DTC, u16ITC, u32DTCStatus, u32StatusMask, u32DTCMask));
                  vdl_tsDiaglogReadReport oReadReport;

                  oReadReport.u32DTC      = u32DTC;
                  oReadReport.u16ITC      = u16ITC;
                  oReadReport.u32Status   = u32DTCStatus;

                  // add dtc to result
                  roReadResult.push_back (oReadReport);
               }// if(bAddToList)
            }// if(bUseReport)
         }// if(NULL != pReportRecord)
         else
         {
            ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessReadRequest => pReportRecord = NULL"));
         }
      }// for(tReportArrayCIt itReport =  m_oReports.begin();...
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vProcessReadRequest"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vProcessReadRequest
// 
// DESCRIPTION: process read request
//
// PARAMETER:  tU8 const& u8StatusMask - the status maskread request
//             midw_fi_tcl_CustomerReadResultList& roReadResult - request result goes in here
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::getItcInformation(tU16 ITC,vdl_tsDiaglogItcInformation& itcInformation)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::getItcInformation"));

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::getItcInformation => Error: LOAD FAILED"));
      return;
   }

   LockMemorySemaphore();
   {
      // for each report
      for(tReportArrayCIt itReport =  m_oReports.begin();
          itReport != m_oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         if(NULL != pReportRecord)
         {
            if(pReportRecord->u16ITC() == ITC)
            {
               // get trouble code depending on mode
               itcInformation.u32DTC               = pReportRecord->u32DTC();
               itcInformation.u16ITC               = pReportRecord->u16ITC();
               itcInformation.u32Status            = pReportRecord->u32GetStatus();
               itcInformation.u8AgeingCounter      = pReportRecord->u8GetAgeingCounter();
               itcInformation.u8OccurenceCounter   = pReportRecord->u8GetFailOccurenceCounter();

               ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::getItcInformation => DTC:$%06x ITC:$%04x Status:$%04x, AgeCtr;%02d,OccCtr:%02d", itcInformation.u32DTC,itcInformation.u16ITC,itcInformation.u32Status,itcInformation.u8AgeingCounter,itcInformation.u8OccurenceCounter));
               break; // found
            }// if(pReportRecord->u16ITC() == ITC)
         }// if(NULL != pReportRecord)
         else
         {
            ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::getItcInformation => pReportRecord = NULL"));
         }
      }// for(tReportArrayCIt itReport =  m_oReports.begin();...
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getItcInformation"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::pGetReportRecord
// 
// DESCRIPTION: get Report Recored by a given ITC
//
// PARAMETER:  tU16 ITC             : ITC ID
//
// RETURNVALUE: vdl_tclReportRecord*: pointer to record
// 
///////////////////////////////////////////////////////////////////////////////////
vdl_tclReportRecord* vdl_tclReportMemory::pGetReportRecord(tU16 u16ITC) const
{
   vdl_tclReportRecord* pReturn = NULL;

   for(tReportArrayCIt itReport =  m_oReports.begin();
       itReport != m_oReports.end();
       ++itReport)
   {
      vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
      if(NULL != pReportRecord)
      {
         ETG_TRACE_USR4_THR(( "--- vdl_tclReportMemory::pGetReportRecord => check ITC:%x <=> ListITC:%x",u16ITC,pReportRecord->u16ITC()));
         // serach for ITC in List
         if(pReportRecord->u16ITC() == u16ITC)
         {
            pReturn = pReportRecord;
            break;
         }
      }// if(NULL != pReportRecord)
   }// for(tReportArrayCIt itReport =  m_oReports.begin();

   return pReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vProcessSetToDefault
// 
// DESCRIPTION: Process SetToDefault
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tBool vdl_tclReportMemory::bProcessSetToDefault() const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vProcessSetToDefault"));
   tBool bReturn = false;

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessSetToDefault => Error: LOAD FAILED"));
   }
   else
   {
      // nothing to do here currently
      bReturn = true;
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vProcessSetToDefault"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::LockMemorySemaphore
// 
// DESCRIPTION: Lock the memory semaphre
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
inline void vdl_tclReportMemory::LockMemorySemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphoreWait(vdDiagLog_tclApp::m_hReportMemSem, OSAL_C_U32_INFINITE);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: LockSemaphore() => OSAL_s32SemaphoreWait failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::FreeMemorySemaphore
// 
// DESCRIPTION: free the memory semaphre
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
inline void vdl_tclReportMemory::FreeMemorySemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphorePost(vdDiagLog_tclApp::m_hReportMemSem);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: FreeSemapore() => OSAL_s32SemaphorePost failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::bIsAlreadyPassed
// 
// DESCRIPTION: check if already passed
//
// PARAMETER:  void
//
// RETURNVALUE: bool true if already passed 
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportMemory::bIsAlreadyPassed(vdl_tclReportRecord const* pReportRecord) const
{
   bool bReturn = false;

   if(  (pReportRecord->bGetIsActive() == false)
      &&(pReportRecord->bGetIsPassedSinceCurrentPowerUp() == true)
      &&(pReportRecord->bGetIsPassedSinceDTCCleared() == true)
      &&(pReportRecord->bGetIsNotCompleteThisPowerCycle() == false)
      &&(pReportRecord->bGetIsNotCompleteSinceCleared() == false)
      )
   {
      bReturn = true;
   }
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::bIsAlreadyFailed
// 
// DESCRIPTION: check if already failed
//
// PARAMETER:  void
//
// RETURNVALUE: bool true if already passed 
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportMemory::bIsAlreadyFailed(vdl_tclReportRecord const* pReportRecord) const
{
   bool bReturn = false;

   if(  (pReportRecord->bGetIsActive() == true)
      &&(pReportRecord->bGetIsFailedInCurrentPowerCycle() == true)
      &&(pReportRecord->bGetIsFailedSinceDTCCleared() == true)
      &&(pReportRecord->bGetIsNotCompleteThisPowerCycle() == false)
      &&(pReportRecord->bGetIsNotCompleteSinceCleared() == false)
     )
   {
      bReturn = true;
   }

   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask
// 
// DESCRIPTION: returns the Number of DTC by Statsus Mask
//
// PARAMETER:   tU32&        u32NumberOfDTC => Buffer to response
//              tU32 const   u32StatusMask  => Status Mask to filter for
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask( tU32&        u32NumberOfDTC,
                                                                tU32 const   u32StatusMask)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask"));

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask => Error: LOAD FAILED"));
      return;
   }

   LockMemorySemaphore();
   {
      // first set the number to zero
      u32NumberOfDTC = 0;

      // for each report
      for(tReportArrayCIt itReport =  m_oReports.begin();
          itReport != m_oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         if(NULL != pReportRecord)
         {
            // get trouble code depending on mode
            tU32 u32DTC       = pReportRecord->u32DTC();
            tU16 u16ITC       = pReportRecord->u16ITC();
            tU32 u32DTCStatus = pReportRecord->u32GetStatus();
            tU8  u8CalibEnable      = pReportRecord->bIsDTCEnabled();
            tU8  u8CalibReadEnable  = pReportRecord->bIsDTCReadByCalibrationEnabled();
            tU8  u8DTCActive        = u8CalibEnable && u8CalibReadEnable;

            ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask => DTC:$%06x ITC:$%04x Status:$%04x, StatusMask:$%04x, CalEnable:%x, CalReadEnable:%x", u32DTC,u16ITC, u32DTCStatus, u32StatusMask, u8CalibEnable, u8CalibReadEnable));
 
            if(  (u8DTCActive == TRUE)
               &&(u32StatusMask & u32DTCStatus)
              )
            {
               ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask => add DTC:$%06x ITC:$%04x Status:$%04x, StatusMask:$%04x", u32DTC,u16ITC, u32DTCStatus, u32StatusMask));
               u32NumberOfDTC++;
            }
         }// if(NULL != pReportRecord)
         else
         {
            ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask => pReportRecord = NULL"));
         }
      }// for(tReportArrayCIt itReport =  m_oReports.begin();...
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask"));

}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vProcessReadNumberOfDtcByStatusMask
// 
// DESCRIPTION: returns the last confirmed DTC
//
// PARAMETER:   tU32&        u32NumberOfDTC => Buffer to response
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vProcessReadLastConfirmedDTC(tReadReportListArray& roReadResult) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vProcessReadLastConfirmedDTC"));

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessReadLastConfirmedDTC => Error: LOAD FAILED"));
      return;
   }

   LockMemorySemaphore();
   {
      vdl_tsDiaglogReadReport oReadReport;

      oReadReport.u32DTC      = m_u32LastReportedDTC;
      oReadReport.u16ITC      = 0;
      oReadReport.u32Status   = m_u32StatOfLastReportedDTC;

      // add dtc to result
      roReadResult.push_back (oReadReport);

      ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::vProcessReadLastConfirmedDTC => DTC:$%06x Status:$%04x", oReadReport.u32DTC, oReadReport.u32Status));
   }            
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vProcessReadLastConfirmedDTC"));

}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportMemory::vProcessReadSupportedDTCs
// 
// DESCRIPTION: returns all supported DTCs
//
// PARAMETER:   tReadReportListArray&      roReadResult => Buffer to response
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vProcessReadSupportedDTCs(tReadReportListArray&      roReadResult)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vProcessReadSupportedDTCs"));

   bool bAddToList = false;

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessReadSupportedDTCs => Error: LOAD FAILED"));
      return;
   }

   LockMemorySemaphore();
   {
      // for each report
      for(tReportArrayCIt itReport =  m_oReports.begin();
          itReport != m_oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         if(NULL != pReportRecord)
         {
            // get trouble code depending on mode
            tU32 u32DTC             = pReportRecord->u32DTC();
            tU16 u16ITC             = pReportRecord->u16ITC();
            tU32 u32DTCStatus       = pReportRecord->u32GetStatus();
            tU8  u8CalibEnable      = pReportRecord->bIsDTCEnabled();
            bool dtcEnabled = true;

            if( vdDiagLog_tclApp::m_poInstance != NULL )
               {
            	   dtcEnabled = (vdDiagLog_tclApp::m_poInstance)->m_poDtcCfgChk->isDTCEnabled( u16ITC );

			   if(dtcEnabled)
			   {
            		   // DTC is not enabled; we will not process it
            		  // return false;


            ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::vProcessReadSupportedDTCs => DTC:$%06x ITC:$%04x Status:$%04x CalEnable:%x", u32DTC,u16ITC, u32DTCStatus,u8CalibEnable));

            if(u8CalibEnable == true)
            {
               // add only claibrated once

               // the DTC is valid to get
               bAddToList = true;
               // same DTC could be part of two different ITC's
               // now search for double
               tReadReportListArrayIt iPos = roReadResult.begin();
               while(iPos.operator!=(roReadResult.end()))
               {
                  // search for DTC in current LIST
                  if(iPos->u32DTC == u32DTC)
                  {
                     // found this DTC in the LIST, now check the status Bit-Field
                     tuRecordStatus uStatus;
                     uStatus.u32Status = iPos->u32Status;

                     // first active wins
                     if(  (pReportRecord->bGetIsActive() == true)
                        &&(uStatus.sStatus.m_bIsActive == false)
                       )
                     {
                        // this one is still in list, but is not active, so overwrite
                        ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::vProcessReadSupportedDTCs => DTC:$%06x, ITC:$%04x, Status:$%02x (overwrite by active)", u32DTC, u16ITC, u32DTCStatus));
                        //iPos->u32DTC    = u32DTC; // not necessary just the same
                        iPos->u16ITC    = u16ITC;
                        iPos->u32Status = u32DTCStatus;
                        bAddToList = false; // no need to add a new List item
                     }
                     // then history
                     else if(  (pReportRecord->bGetIsActive() == false)
                             &&(pReportRecord->bGetIsHistory() == true)
                             &&(uStatus.sStatus.m_bIsHistory == false)
                            )
                     {
                        // this one is still in list, but is not active, so overwrite
                        ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::vProcessReadSupportedDTCs => DTC:$%06x, ITC:$%04x, Status:$%02x(overwrite by history)", u32DTC, u16ITC, u32DTCStatus));
                        //iPos->u32DTC    = u32DTC; // not necessary just the same
                        iPos->u16ITC    = u16ITC;
                        iPos->u32Status = u32DTCStatus;
                        bAddToList = false; // no need to add a new List item
                     }
                     else
                     {
                        // this one is still in list, and both have same state, so do nothing
                        ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::vProcessReadSupportedDTCs => DTC:$%06x, ITC:$%04x, Status:$%02x (found same)", u32DTC, u16ITC, u32DTCStatus));
                        bAddToList = false; // no need to add a new List item
                     }
                     // continue maybe more ITC's have the same DTC
                  }// if(iPos->u32DTC == u32DTC)
                  ++iPos;
               }// while(iPos.operator!=(roReadResult.end()))

               if(bAddToList)
               {
                  // use this DTC/ITC
                  ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::vProcessReadSupportedDTCs => DTC:$%06x, ITC:$%04x, Status:$%02x", u32DTC, u16ITC, u32DTCStatus));
                  vdl_tsDiaglogReadReport oReadReport;

                  oReadReport.u32DTC      = u32DTC;
                  oReadReport.u16ITC      = u16ITC;
                  oReadReport.u32Status   = u32DTCStatus;

                  // add dtc to result
                  roReadResult.push_back (oReadReport);
               }// if(bAddToList)
            }// if(bUseReport)
           }
          }
         }// if(NULL != pReportRecord)
         else
         {
            ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vProcessReadSupportedDTCs => pReportRecord = NULL"));
         }
      }// for(tReportArrayCIt itReport =  m_oReports.begin();...
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vProcessReadSupportedDTCs"));

}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vPass
// 
// DESCRIPTION: 
//
// PARAMETER:  vdl_tclReportRecord* pReportRecord: Pointer to Record
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vPass(vdl_tclReportRecord* const pReportRecord)
{
   if(pReportRecord != NULL)
   {
      bool bIsActiveBefore = pReportRecord->bGetIsActive();
      pReportRecord->vPass();  // forward info to ReportRecord
      bool bIsActiveAfter = pReportRecord->bGetIsActive();

      if(  (pReportRecord->bIsChangeAllowedWihtoutDebounce()== true)
         &&(bIsActiveBefore == true)
         &&(bIsActiveAfter == false))
      {
         // we have a real change!!!
         if(NULL != vdDiagLog_tclApp::m_poInstance)
         {
            vdDiagLog_tclApp::m_poInstance->updateBlockingModeAfterPassed(pReportRecord);
         }
      }
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vFail
// 
// DESCRIPTION: 
//
// PARAMETER:  vdl_tclReportRecord* pReportRecord: Pointer to Record
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vFail(vdl_tclReportRecord* const pReportRecord)
{
   if(pReportRecord != NULL)
   {
      pReportRecord->vFail();  // forward info to ReportRecord
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vPassQualified
// 
// DESCRIPTION: 
//
// PARAMETER:  vdl_tclReportRecord* pReportRecord: Pointer to Record
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vPassQualified(vdl_tclReportRecord* const pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vPassQualified"));
   if(pReportRecord != NULL)
   {
      bool bIsActiveBefore = pReportRecord->bGetIsActive();
      pReportRecord->vPassQualified();  // forward info to ReportRecord
      bool bIsActiveAfter = pReportRecord->bGetIsActive();

      if(  (pReportRecord->bIsChangeAllowedWihtoutDebounce()== false)
         &&(bIsActiveBefore == true)
         &&(bIsActiveAfter == false))
      {
         // we have a real change!!!
         if(NULL != vdDiagLog_tclApp::m_poInstance)
         {
            vdDiagLog_tclApp::m_poInstance->updateBlockingModeAfterPassed(pReportRecord);
         }
      }

      (tVoid) bEventStep(EVENT_PASS_QUALIFIED, pReportRecord); // forward info to the PlugIns
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vPassQualified => pReportRecord = NULL"));
   }



   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vPassQualified"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vFailQualified
// 
// DESCRIPTION: 
//
// PARAMETER:  vdl_tclReportRecord* pReportRecord: Pointer to Record
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vFailQualified(vdl_tclReportRecord* const pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vFailQualified"));
   if(pReportRecord != NULL)
   {
      m_u32LastReportedDTC = pReportRecord->u32DTC();
      m_u32StatOfLastReportedDTC = pReportRecord->u32GetStatus();
      if(!pReportRecord->bGetIsHistory() && !this->isDTCActive(pReportRecord->u32DTC()))
         {
         ETG_TRACE_USR3_THR(( "!!! vdl_tclReportMemory::vFailQualified => adding to active dtc list"));
            m_activeDTCList.push_back(pReportRecord->u32DTC());
      }

      pReportRecord->vFailQualified();                // forward info to ReportRecord

      // forward to aging Module
      if(m_pAgeingAlgorithm != NULL)
      {
         m_pAgeingAlgorithm->onFailed(pReportRecord);
      }

      (tVoid) bEventStep(EVENT_FAIL_QUALIFIED, pReportRecord); // forward info to the PlugIns
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vFailQualified => pReportRecord = NULL"));
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vFailQualified"));
}

bool vdl_tclReportMemory::isDTCActive(tU32 dtc) const
{
      tReportArrayCIt iPos = m_oReports.begin();
      while(iPos != m_oReports.end())
      {
         vdl_tclReportRecord const* pThisReportRecord = (*iPos); // get report record
         if(pThisReportRecord->u32DTC() == dtc)
         {
            if(pThisReportRecord->bGetIsHistory())
            {
               ETG_TRACE_USR3_THR(( "!!! vdl_tclReportMemory::isDTCActive => DTC:$%06x ITC:$%04x active", pThisReportRecord->u32DTC(), pThisReportRecord->u16ITC()));
               return true;
            }
            ETG_TRACE_USR3_THR(( "!!! vdl_tclReportMemory::isDTCActive => DTC:$%06x ITC:$%04x not active", pThisReportRecord->u32DTC(), pThisReportRecord->u16ITC()));
         }
         ++iPos;
      }
      return false;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::bProcessControlCommand
// 
// DESCRIPTION: process control comannd in worklist
//
// PARAMETER:  vdl_teDiaglogControlMsg eControlCmd     : command
//             tDTCListArray oDTCList                  : List of DTC valid for this command
//
// RETURNVALUE: bool - true if data was sucessfully process
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportMemory::bProcessControlCommand(vdl_teDiaglogControlMsg eControlCmd, tDTCListArray const& oDTCList)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::bProcessControlCommand"));

   bool bReturn = true;

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::bProcessControlCommand => Error: LOAD FAILED"));
      return false;
   }

   LockMemorySemaphore();
   {
      size_t sSize = oDTCList.size();

      if(sSize != 0)
      {
         // change only requested
         for(tDTCListArrayCIt itDtcList =  oDTCList.begin();
             itDtcList != oDTCList.end();
             ++itDtcList)
         {
            tU32 u32DTCToControl = (*itDtcList);
            // for each report
            for(tReportArrayCIt itReport =  m_oReports.begin();
                itReport != m_oReports.end();
                ++itReport)
            {
               vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
               if(NULL != pReportRecord)
               {
                  tU32 u32DTC       = pReportRecord->u32DTC();
                  if(u32DTC == u32DTCToControl)
                  {
                     // get trouble code depending on mode
                     tU16 u16ITC       = pReportRecord->u16ITC();
                     switch(eControlCmd)
                     {
                        case DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_DISABLE:
                        {
                           ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_DISABLE", u32DTC,u16ITC));
                           pReportRecord->vSetDTCControlDisabled();
                           break;
                        }
                        case DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_ENABLE:
                        {
                           ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_ENABLE", u32DTC,u16ITC));
                           pReportRecord->vSetDTCControlEnabled();
                           break;
                        }
                        case DIAGLOG_CONTROL_CALIB_DISABLE:
                        {
                           ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CALIB_DISABLE", u32DTC,u16ITC));
                           pReportRecord->vSetDTCCalibrationDisabled();
                           break;
                        }
                        case DIAGLOG_CONTROL_CALIB_ENABLE:
                        {
                           ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CALIB_ENABLE", u32DTC,u16ITC));
                           pReportRecord->vSetDTCCalibrationEnabled();
                           break;
                        }
                        case DIAGLOG_CONTROL_READ_DISABLE:
                        {
                           ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_READ_DISABLE", u32DTC,u16ITC));
                           pReportRecord->vSetDTCReadByCalibrationDisabled();
                           break;
                        }
                        case DIAGLOG_CONTROL_READ_ENABLE:
                        {
                           ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_READ_ENABLE", u32DTC,u16ITC));
                           pReportRecord->vSetDTCReadByCalibrationEnabled();
                           break;
                        }
                        default:
                        {
                           ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::bProcessControlCommand => UNKNOWN switch(eControlCmd:%x)",eControlCmd));
                           NORMAL_M_ASSERT_ALWAYS();
                           bReturn = false;
                           break;
                        }
                     }// switch(eControlCmd)
                  }// if(u32DTC == u32DTCToControl)
               }// if(NULL != pReportRecord)
               else
               {
                  ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::bProcessControlCommand => NULL == pReportRecord"));
                  NORMAL_M_ASSERT_ALWAYS();
                  bReturn = false;
               }
            }// for(tReportArrayCIt itReport =  m_oReports.begin();...
         }// for(tDTCListArrayCIt itReport =  oDTCList.begin();
      }// if(u32Size != 0)
      else
      {
         // change all
         for(tReportArrayCIt itReport =  m_oReports.begin();
             itReport != m_oReports.end();
             ++itReport)
         {
            vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
            if(NULL != pReportRecord)
            {
               tU32 u32DTC       = pReportRecord->u32DTC();
               tU16 u16ITC       = pReportRecord->u16ITC();
               switch(eControlCmd)
               {
                  case DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_DISABLE:
                  {
                     ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_DISABLE", u32DTC,u16ITC));
                     pReportRecord->vSetDTCControlDisabled();
                     break;
                  }
                  case DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_ENABLE:
                  {
                     ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_ENABLE", u32DTC,u16ITC));
                     pReportRecord->vSetDTCControlEnabled();
                     break;
                  }
                  case DIAGLOG_CONTROL_CALIB_DISABLE:
                  {
                     ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CALIB_DISABLE", u32DTC,u16ITC));
                     pReportRecord->vSetDTCCalibrationDisabled();
                     break;
                  }
                  case DIAGLOG_CONTROL_CALIB_ENABLE:
                  {
                     ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::bProcessControlCommand => DTC:$%06x ITC:$%04x DIAGLOG_CONTROL_CALIB_ENABLE", u32DTC,u16ITC));
                     pReportRecord->vSetDTCCalibrationEnabled();
                     break;
                  }
                  default:
                  {
                     ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::bProcessControlCommand => UNKNOWN switch(eControlCmd:%x)",eControlCmd));
                     NORMAL_M_ASSERT_ALWAYS();
                     bReturn = false;
                     break;
                  }
               }// switch(eControlCmd)
            }// if(NULL != pReportRecord)
            else
            {
               ETG_TRACE_ERRMEM(( "!!! vdl_tclReportMemory::bProcessControlCommand => NULL == pReportRecord"));
               NORMAL_M_ASSERT_ALWAYS();
               bReturn = false;
            }
         }// for(tReportArrayCIt itReport =  m_oReports.begin();...
      }// else if(u32Size != 0)

      if( bReturn == true)
      {
         // save new status to flash
         vSaveNow();
      }
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::bProcessControlCommand"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord const& vdl_tclReportMemory::getReportRecord(tU32 aDTC) const
// 
// DESCRIPTION: get the Recored give by a DTC ID
//
// PARAMETER:  tU32 aDTC:  DTC ID
//
// RETURNVALUE: vdl_tclReportRecord: pointer to record
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportRecord const* vdl_tclReportMemory::getReportRecord(tU32 aDTC) const
{
   tReportArrayCIt itRecord = std::find_if(m_oReports.begin(), m_oReports.end(), VDD::equalsDTC(aDTC));
   return (itRecord!=m_oReports.end())?(*itRecord):NULL;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord* vdl_tclReportMemory::getReportRecordForDTC
// 
// DESCRIPTION: Get report record for DTC
//
// PARAMETER:  tU32 aDTC          : requested DTC
//
// RETURNVALUE: vdl_tclReportRecord *  : pointer to report record
// 
///////////////////////////////////////////////////////////////////////////////////

vdl_tclReportRecord* vdl_tclReportMemory::getReportRecordForDTC(tU32 aDTC) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::getReportRecordForDTC"));
   tReportArray reportRecordList;
   for(tReportArrayCIt itRecord = this->m_oReports.begin(); itRecord != this->m_oReports.end(); ++itRecord)
   {
	   if((*itRecord)->u32DTC() == aDTC)
	   {
		   reportRecordList.push_back(*itRecord);
	   }
   }
   vdl_tclReportRecord* pRecord = NULL;
   if(reportRecordList.size() == 1)
   {
       return reportRecordList.at(0);
   }
   else
   {
	   for(tReportArrayCIt itRecord = reportRecordList.begin(); itRecord != reportRecordList.end(); ++itRecord)
       {
		   if((*itRecord)->bGetIsActive() == true)
	       {
		      pRecord =(*itRecord);
		      break;
	       }
	       else if((*itRecord)->bGetIsHistory() == true)
	       {
		      pRecord =(*itRecord);
		      break;
	       }
	       else
	       {
		      pRecord =(*itRecord);
	       }
       }
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getReportRecordForDTC"));
   return pRecord;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: ExtendedData const* vdl_tclReportMemory::getExtendedData
// 
// DESCRIPTION: Get extended Data of a requested DTC
//
// PARAMETER:  tU32 dtc          : requested DTC
//             tU8 recordNumber  : 
//
// RETURNVALUE: ExtendedData     : Extended Data Object
// 
///////////////////////////////////////////////////////////////////////////////////
//
ExtendedData const* vdl_tclReportMemory::getExtendedData(tU32 dtc, tU8 recordNumber) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::getExtendedData"));

   ExtendedData const* pResult = NULL;
   if (!(this->isDTCActive(dtc)))
   {
      return pResult;
   }
   tReportArray ITCRecordList;
   for(tReportArrayCIt itRecord = m_oReports.begin(); itRecord != m_oReports.end(); ++itRecord)
   {
	   if((*itRecord)->u32DTC() == dtc)
	   {
		   ITCRecordList.push_back(*itRecord);
	   }
   }
   if(ITCRecordList.size() == 1)
   {
	   ETG_TRACE_USR3_THR(( "--- vdl_tclReportMemory::getExtendedData -- only one record found")); 
	   pResult = ITCRecordList.at(0)->getExtendedData(recordNumber);
   }
   else if(ITCRecordList.size() > 1)
   {
	   ETG_TRACE_USR3_THR(( "--- vdl_tclReportMemory::getExtendedData -- total records found: %d", ITCRecordList.size()));
	   pResult = vdl_tclReportRecord::extendedDataRecordListBasedOnMultiITC(ITCRecordList,recordNumber);
   }
   else
   {
      ETG_TRACE_ERR_THR(( "--- vdl_tclReportMemory::getExtendedData => DTC:%06x not Found",dtc));
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getExtendedData"));
   return pResult;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: ExtendedData const* vdl_tclReportMemory::getSnapShotData
// 
// DESCRIPTION: Get Snap Shot Data of a requested DTC
//
// PARAMETER:  tU32 dtc          : requested DTC
//             tU8 recordNumber  : 
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclReportMemory::getSnapShotData(tU32 dtc, tU8 recordNumber,tU32& status,tSnapShotDataList& SnapShotDataList) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::getSnapShotData"));

   // we have to search in the list for the most active DTC, because a DTC is not unique!!
   vdl_tclReportRecord const* reportRecord = getMostRecentReportRecordByDTC(dtc);

   if(reportRecord != NULL)
   {
      reportRecord->getSnapShotData(recordNumber,status,SnapShotDataList);
   }
   else
   {
      ETG_TRACE_ERR_THR(( "--- vdl_tclReportMemory::getSnapShotData => DTC:%06x not Found",dtc));
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getSnapShotData"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::bEventStep
//
// DESCRIPTION:  forward the EventStep to the PlugIns
//             
// PARAMETER:    vdl_teDiaglogEventStep enEventStep: Kind of EventStep
//               vdl_tclReportRecord* pReportRecord: pointer to Data
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tBool vdl_tclReportMemory::bEventStep(vdl_teDiaglogEventStep enEventStep, vdl_tclReportRecord* pReportRecord) const
{
   tBool bReturn = false;
   if(vdDiagLog_tclApp::m_poInstance != NULL)
   {
      bReturn = vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.bEventStep(enEventStep,pReportRecord);
   }
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::bCheckLoadStatus
//
// DESCRIPTION:  forward the EventStep to the Storge Handler
//             
// PARAMETER:     none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
bool vdl_tclReportMemory::bCheckLoadStatus() 
{
   return loadDataBaseIfNotLoaded();
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::vCheckAndSave
//
// DESCRIPTION:  forward the EventStep to the Storge Handler
//             
// PARAMETER:     none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vCheckAndSave()
{
   requestForSaveIfModified();
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::vUpdateActiveDTCList()
//
// DESCRIPTION:  update the list of active DTCs
//             
// PARAMETER:     none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vUpdateActiveDTCList()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vUpdateActiveDTCList"));
   if(m_activeDTCList.size() == 0)
   {
      return;
   }
   // update CCA
   if(  ((vdDiagLog_tclApp::m_poInstance) != NULL)
      &&((vdDiagLog_tclApp::m_poInstance)->m_poService != NULL))
   {
      (vdDiagLog_tclApp::m_poInstance)->m_poService->vSendActiveDTCListStatus(m_activeDTCList);
      m_activeDTCList.clear();
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vUpdateActiveDTCList => Can't send to CCA components"));
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vUpdateActiveDTCList"));
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::vSaveNow
//
// DESCRIPTION:  forward the EventStep to the Storge Handler
//             
// PARAMETER:     none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vSaveNow()
{
   saveNowIfModified();
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::vSetModified
//
// DESCRIPTION:  forward the EventStep to the Storge Handler
//             
// PARAMETER:    bool status => true: status changed
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclReportMemory::vSetModified()
{
   setModified();
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::getStateOfDTC
// 
// DESCRIPTION: get the current state of a given DTC
//
// PARAMETER:  tU32&        state            => reference to fill
//             tU32 const   u32DTCRequest    => DTC to serach for
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tBool vdl_tclReportMemory::getStateOfDTC( tU32 const u32DTCRequest, tU32& state) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::getStateOfDTC"));

   tBool found = false;
   tuRecordStatus uStatus;
   uStatus.u32Status = 0;

   // check we have loaded the database at start up
   if(!isDataBaseLoaded())
   {
      // if not return, trace info
      ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::getStateOfDTC => Error: LOAD FAILED"));
   }
   else
   {
      // for each report
      for(tReportArrayCIt itReport =  m_oReports.begin();
          itReport != m_oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         if(NULL != pReportRecord)
         {
            // get trouble code depending on mode
            tU32 u32DTC             = pReportRecord->u32DTC();
            tU16 u16ITC             = pReportRecord->u16ITC();
            tU32 u32DTCStatus       = pReportRecord->u32GetStatus();
            tU8  u8CalibEnable      = pReportRecord->bIsDTCEnabled();
            //tU8  u8CalibReadEnable  = pReportRecord->bIsDTCReadByCalibrationEnabled();
            //tU8  u8DTCActive        = u8CalibEnable && u8CalibReadEnable;

            if(   (u8CalibEnable)
               && (u32DTC == u32DTCRequest))
            {
               // we found a matching DTC
               found = true;

               // now check with maybe older found DTC's
               if(  (pReportRecord->bGetIsActive() == true)
                  &&(uStatus.sStatus.m_bIsActive == false)
                 )
               {
                  // this one is still found, but is not active, so overwrite
                  ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemoryBase::getStateOfDTC => DTC:$%06x, ITC:$%04x, StatusMask:$%08x", u32DTC, u16ITC, u32DTCStatus));
                  uStatus.u32Status = u32DTCStatus;
               }
               // then history
               else if(  (pReportRecord->bGetIsActive() == false)
                       &&(pReportRecord->bGetIsHistory() == true)
                       &&(uStatus.sStatus.m_bIsHistory == false)
                      )
               {
                  // this one is still in list, but is not active, so overwrite
                  ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemoryBase::getStateOfDTC => DTC:$%06x, ITC:$%04x, StatusMask:$%08x", u32DTC, u16ITC, u32DTCStatus));
                  uStatus.u32Status = u32DTCStatus;
               }
               else
               {
                  // this one is still in list, and both have same state, so do nothing
               }
               // continue maybe more ITC's have the same DTC
            }// if(u32DTC == u32DTCRequest)
         }// if(NULL != pReportRecord)
         else
         {
            ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::getStateOfDTC => pReportRecord = NULL"));
         }
      }// for(tReportArrayCIt itReport =  m_oReports.begin();...
   }

   // copy state to reference
   state = uStatus.u32Status;

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getStateOfDTC"));
   return found;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::deleteRecord
// 
// DESCRIPTION: delete the given record
//
// PARAMETER:  tReportArrayIt& iter => referenz to iterator
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tReportArrayIt vdl_tclReportMemory::deleteRecordByIter(tReportArrayIt iter)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::deleteRecordByIter"));
   vdl_tclReportRecord* pReportRecord = (*iter); // get report record

   if(NULL != pReportRecord)
   {
      ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::deleteRecordByIter => DTC:%06x",pReportRecord->u32DTC()));
      // first we have to remove the record everywhere it is used (stored)
      if(m_poQualificationObject != NULL)
      {
         m_poQualificationObject->clearQualiListByRecord(pReportRecord);
      }


      // check if this is a dynamical DTC
      if(pReportRecord->isDynamicalDTC())
      {
         // this is a dynamilcal DTC
         if(m_u16NumberOfDynamicITCs > 0)
         {
            m_u16NumberOfDynamicITCs--;
         }
         else
         {
            ETG_TRACE_ERR_THR(( "!!! vdl_tclReportMemory::vClearAll => fail to many DynamicITC deleted"));
         }
      }
      else
      {
         // currently only dynamical DTC's should be deleted
         NORMAL_M_ASSERT_ALWAYS();
      }

      //delete pReportRecord; PLA2HI: !!! WARNING !!!! actually it is strictly forbidden to remove a record
   }// if(NULL != pReportRecord)
   // now remove it from list
   iter = m_oReports.erase(iter);
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::deleteRecordByIter"));
   return iter;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::deleteRecord
// 
// DESCRIPTION: delete the given record
//
// PARAMETER:  pReportRecord          => pointer to record to delete
//
// RETURNVALUE: true = found; false = not found
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportMemory::searchAndDeleteRecord(const vdl_tclReportRecord* const pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::searchAndDeleteRecord"));
   bool bReturn = false;

   if(pReportRecord != NULL)
   {
      // search for record in the Report List
      tReportArrayIt iPos = m_oReports.begin();
      while(iPos != m_oReports.end())
      {
         vdl_tclReportRecord* pThisReportRecord = (*iPos); // get report record

         if(pThisReportRecord == pReportRecord)
         {
            // found
            (void) deleteRecordByIter(iPos);
            bReturn = true;
            break; // we found the recored, could stop now
         }// if(pThisReportRecord == pReportRecord)
         ++iPos;
      }// while(iPos != m_oReports.end())
   }// if(pReportRecord != NULL)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::searchAndDeleteRecord"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::vAgeingCompleted(vdl_tclReportRecord* const pReportRecord)
// 
// DESCRIPTION: forward ageing information to record
//
// PARAMETER:  pReportRecord          => pointer to record 
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclReportMemory::vAgeingCompleted(vdl_tclReportRecord*  pReportRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::vAgeingCompleted"));
   (tVoid) bEventStep(EVENT_AGEING_COMPLETED, pReportRecord); // forward info to the PlugIns
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::vAgeingCompleted"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::getMostRecentReportRecordByDTC(tU32 u32DTC)
// 
// DESCRIPTION: search for DTC and return most "active" one
//              1. active
//              2. history
//              3. inactive
//
// PARAMETER:  u32DTC => DTC to search for
//
// RETURNVALUE: found record (NULL if DTC not found)
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportRecord* vdl_tclReportMemory::getMostRecentReportRecordByDTC(tU32 u32DTC)const
{
   vdl_tclReportRecord* pRecord = NULL;
   vdl_tclReportRecord* pRecordActive = NULL;
   vdl_tclReportRecord* pRecordHistory = NULL;
   vdl_tclReportRecord* pRecordInactive = NULL;
   // same DTC could be part of two different ITC's
   // now search for double

   tReportArrayCIt iPos = m_oReports.begin();
   while(iPos!=(m_oReports.end()))
   {
      vdl_tclReportRecord* pReportRecord = (*iPos); // get report record
      // search for DTC in current LIST
      if(pReportRecord->u32DTC() == u32DTC)
      {
         // found this DTC in the LIST, now check the status Bit-Field
         // first active wins
         if(pReportRecord->bGetIsActive() == true)
         {
            pRecordActive = pReportRecord;
            break; // we found a active one, could stop here
         }
         // then history
         else if(pReportRecord->bGetIsHistory() == true)
         {
            if(pRecordHistory == NULL)
            {
               // set the first history to found
               pRecordHistory = pReportRecord;
            }
            // do not break, continue to found a active one
         }
         else
         {
            if(pRecordInactive == NULL)
            {
               // set the inactive one
               pRecordInactive = pReportRecord;
            }
            // do not break, continue to found a active one
         }
         // continue maybe more ITC's have the same DTC
      }// if(iPos->u32DTC == u32DTC)
      ++iPos;
   }// while(iPos.operator!=(roReadResult.end()))

   if(pRecordActive != NULL)
   {
      pRecord = pRecordActive;
   }
   else if(pRecordHistory != NULL)
   {
      pRecord = pRecordHistory;
   }
   else if(pRecordInactive != NULL)
   {
      pRecord = pRecordInactive;
   }

   return pRecord;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::getFirstActiveITC
// 
// DESCRIPTION: search for an active ITC
//
// PARAMETER:  none
//
// RETURNVALUE: ITC which is active
// 
///////////////////////////////////////////////////////////////////////////////////
//
tU16 vdl_tclReportMemory::getFirstActiveITC()
{
   tU16 ITC = 0;
   for(  tReportArrayCIt itReport =  m_oReports.begin();
         itReport != m_oReports.end();
         ++itReport)
   {
      vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
      if(NULL != pReportRecord)
      {
         if(  (pReportRecord->bGetIsActive())           //  active
            &&(pReportRecord->bIsDTCEnabled())          // is calibrated
            )
         {
            ITC = pReportRecord->u16ITC();
            ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::getFirstActiveITC => found ITC:%04x",ITC));
            break; // we can stop now, no ageing allowed
         }
      }// if(NULL != pReportRecord)
   }// for(tReportArrayCIt itReport =  m_oReports.begin()...

   return ITC;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclReportMemory::isAnyItcActiveOrHistory
// 
// DESCRIPTION: search for an active ITC
//
// PARAMETER:  none
//
// RETURNVALUE: true = found
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclReportMemory::isAnyItcActiveOrHistory()
{
   bool bReturn = false;
   for(  tReportArrayCIt itReport =  m_oReports.begin();
         itReport != m_oReports.end();
         ++itReport)
   {
      vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
      if(NULL != pReportRecord)
      {
         if(  (pReportRecord->bGetIsFailedSinceDTCCleared())   // history or active
            &&(pReportRecord->bIsDTCEnabled())                 // is calibrated
            )
         {
            tU16 ITC = pReportRecord->u16ITC();
            bReturn = true;
            ETG_TRACE_USR2_THR(( "--- vdl_tclReportMemory::isAnyItcActiveOrHistory => found ITC:%04x",ITC));
            break; // we can stop now, no ageing allowed
         }
      }// if(NULL != pReportRecord)
   }// for(tReportArrayCIt itReport =  m_oReports.begin()...
   return bReturn;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:      tVoid vdl_tclReportMemory::isBlockingModeActive
//
// DESCRIPTION:   forward to main
//
// PARAMETER:     pReportRecord     : pointer to caller record
//                blockingModeMask  : referenz for Block Mask to trace it out by caller
//
// RETURNVALUE:   true = Blocked
// 
///////////////////////////////////////////////////////////////////////////////////
//

bool vdl_tclReportMemory::isBlockingModeActive(vdl_tclReportRecord const* pReportRecord,tU32& blockingModeMask) const
{
   if(NULL != vdDiagLog_tclApp::m_poInstance)
   {
      return vdDiagLog_tclApp::m_poInstance->isBlockingModeActive(pReportRecord,blockingModeMask);
   }
   return false;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclReportMemory::getOccuranceCounterByDTC(tU32 u32DTC). Currently this is only used for AIVI project.
//
// DESCRIPTION: search for DTC and accumulate the occurance counter for different ITC's
//              1. active
//              2. history
//              3. inactive
//
// PARAMETER:  u32DTC => DTC to search for
//
// RETURNVALUE: found record (NULL if DTC not found)
//
///////////////////////////////////////////////////////////////////////////////////
//
tU8 vdl_tclReportMemory::getOccuranceCounterByDTC(tU32 u32DTC)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclReportMemory::getOccuranceCounterByDTC"));

   tU16 occuranceCounter = 0;

   tReportArrayCIt iPos = m_oReports.begin();
   while(iPos!=(m_oReports.end()))
   {
      const vdl_tclReportRecord* pReportRecord = (*iPos); // get report record

      ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::getOccuranceCounterByDTC  ITC=%04x",pReportRecord->u16ITC()));

      // search for DTC in current LIST
      if(pReportRecord->u32DTC() == u32DTC)
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclReportMemory::getOccuranceCounterByDTC  occuranceCounter=%d",occuranceCounter));

         occuranceCounter = (tU16)(occuranceCounter + (pReportRecord->u8GetFailOccurenceCounter()));

         if(occuranceCounter > 255)
         {
            occuranceCounter = 255;
         }
       // continue maybe more ITC's have the same DTC
      }// if(iPos->u32DTC == u32DTC)
      ++iPos;
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getOccuranceCounterByDTC"));
   return (tU8)occuranceCounter;
}
