//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_qualification.h
// PROJECT:      Diag Core
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog Qualification base typename
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 27.10.05  | ESA2 Barber                  | Initial version
// 28.10.14  | BSOT Plischke                | new Framework
//
//////////////////////////////////////////////////////////////////////////
#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_qualification
#define VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_qualification


typedef enum
{
   NONE_QUALIFICATION_RUNNING,
   PASS_QUALIFICATION_RUNNING,
   FAIL_QUALIFICATION_RUNNING
}teQualificationState;

// forward declaraiton
class vdl_tclReportRecord; //lint !e763


class /*interface*/ vdl_tclQualificationCallbackInterface
{
public:
   virtual ~vdl_tclQualificationCallbackInterface(){};

   // this functions are called from the qualification algorithm 
   virtual tVoid vFailQualified(vdl_tclReportRecord* pReportRecord)=0;
   virtual tVoid vPassQualified(vdl_tclReportRecord* pReportRecord)=0;
   virtual tVoid vQualificationStepFinshed()=0;
};



class vdl_tclQualificationRecord
{
public:
   vdl_tclQualificationRecord();
   virtual ~vdl_tclQualificationRecord();

   vdl_tclReportRecord*                   m_pReportRecord;
   vdl_tclQualificationCallbackInterface* m_pCallBackIf;

   // we need this two variables to know if we are still passing or failing, because the Record could be changed meanwhile
   tS32                       m_s32PassQualiyfingCounter; 
   tS32                       m_s32FailQualiyfingCounter;
};

//////////////////////////////////////////////////////////////////////////
// !!! IMPORTANT !!!
// THIS class has as interface Memory Objects
// which must be derivated from vdl_tclQualificationCallbackInterface
// the Memory object has to register and unregister 
// and allows only Records belonging to an registerd class
// otherwise exeptions could be possible
//
// created is this class only once by the framework, 
// do not derivate from this class!
//////////////////////////////////////////////////////////////////////////
class vdl_tclQualificationBase
{
private:
   typedef std::list<vdl_tclQualificationRecord>                          tQualificationList;
   typedef std::list<vdl_tclQualificationRecord>::iterator                tQualificationListIt;
   typedef std::list<vdl_tclQualificationRecord>::const_iterator          tQualificationListCIt;

   typedef std::list<vdl_tclQualificationCallbackInterface*>                          tCallBackList;
   typedef std::list<vdl_tclQualificationCallbackInterface*>::iterator                tCallBackListIt;
   typedef std::list<vdl_tclQualificationCallbackInterface*>::const_iterator          tCallBackListCIt;

   // member variables
public:
   static OSAL_tTimerHandle   m_hQualificationTimer;

private:

   tQualificationList            m_oQualificationList;
   tCallBackList                 m_poCallBackList;
   static OSAL_tSemHandle        m_hQualificationSem;

   // member functions
public:
   vdl_tclQualificationBase();
   virtual ~vdl_tclQualificationBase();

   void addToQualificationListFailed(vdl_tclQualificationCallbackInterface* const pCallBack, vdl_tclReportRecord* const pReportRecord);
   void addToQualificationListPassed(vdl_tclQualificationCallbackInterface* const pCallBack, vdl_tclReportRecord* const pReportRecord);
   void processQualificationStep();
   void clearQualiListByDtc(tU32 u32DtcMask);
   void clearQualiListByMemoryId(tU8 memoryId);
   void clearQualiListByRecord(const vdl_tclReportRecord* const pReportRecord);
   teQualificationState isQualifiying(const vdl_tclReportRecord* pReportRecord);
   void stopQualifiying(const vdl_tclReportRecord* pReportRecord);
   bool registerCallBack(vdl_tclQualificationCallbackInterface* const pCallBack);
   bool unregisterCallBack(vdl_tclQualificationCallbackInterface* const pCallBack);

private:
   bool bHasTimeQualifiers() const {return (m_oQualificationList.size() > 0);}
   void clearQualiListByCallBackIf(const vdl_tclQualificationCallbackInterface* const pCallBack);
   bool isCallBackRegistered(vdl_tclQualificationCallbackInterface* const pCallBack);
   void checkToStartQualiTimer() const;
   void checkToStopQualiTimer() const;
   void clearQualiList();
   inline void LockQualiSemaphore() const;
   inline void FreeQualiSemaphore() const;
   bool isBlockingModeActive(vdl_tclReportRecord const* pReportRecord,tU32& blockingModeMask) const;
};


#endif // #ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_qualification

