/*!
* \file       vd_diaglog_object.cpp
*
* \brief      DiagLog Object class - generic base class
*
* \details    used to persistently store and read the logging info persistent files.
*
* \component  Diaglog
*
* \ingroup    DiaglogFramework
*
* \copyright  (c) 2018 Robert Bosch GmbH
*
* The reproduction, distribution and utilization of this file as
* well as the communication of its contents to others without express
* authorization is prohibited. Offenders will be held liable for the
* payment of damages. All rights reserved in the event of the grant
* of a patent, utility model or design.
*/

#include <common/framework/vd_diaglog_settings.h>

#ifdef VARIANT_S_FTR_ENABLE_TRANSACTION_SAFETY
#ifndef __VD_DIAGLOG_INCLUDEGUARD_OBJECT_MANAGER__
#include "common/framework/vd_diaglog_object_manager.h"
#endif
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_object
   #include <common/framework/vd_diaglog_object.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_storage_interface
   #include <common/framework/vd_diaglog_storage_interface.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
   #include <common/framework/vd_diaglog_system_variables.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_external_data
   #include <project/externaldata/vd_diaglog_external_data.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#include <common/framework/utils/vd_diaglog_filedir.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_object.cpp.trc.h"
#endif

const char* dirName = "/var/opt/bosch/persistent/diagnosis";
#ifndef VARIANT_S_FTR_ENABLE_TRANSACTION_SAFETY
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::vdl_tclDiagLogObject
// 
// DESCRIPTION: default constructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclDiagLogObject::vdl_tclDiagLogObject():   
   m_bIsModified(false),
   m_u32DataBaseTimestamp(0),
   m_bCanSave(false),
   m_bIsLoaded(false),
   m_szFilename(0),
   m_poStorageUser(NULL),
   m_poStorageHandler(NULL)
{
   // should never called
   NORMAL_M_ASSERT_ALWAYS();
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::vdl_tclDiagLogObject
// 
// DESCRIPTION: default constructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclDiagLogObject::vdl_tclDiagLogObject(vdl_tclStorageHandler* const poStorageHandler,vdl_tclStorageUserInterface* const poStorageUser, tCString const szFilename):   
   m_bIsModified(false),
   m_u32DataBaseTimestamp(0),
   m_bCanSave(false),
   m_bIsLoaded(false),
   m_szFilename(szFilename),
   m_poStorageUser(poStorageUser),
   m_poStorageHandler(poStorageHandler)
{
   if(m_poStorageUser == NULL)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::vdl_tclDiagLogObject => NULL == m_poStorageUser"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   if(poStorageHandler == NULL)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::vdl_tclDiagLogObject => NULL == poStorageHandler"));
      NORMAL_M_ASSERT_ALWAYS();
   }
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::~vdl_tclDiagLogObject
// 
// DESCRIPTION: default denstructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclDiagLogObject::~vdl_tclDiagLogObject()
{
   _BP_TRY_BEGIN
   {
      (tVoid) bSave();
   }
   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::~vdl_tclDiagLogObject FAILED"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
   m_szFilename = 0;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::bReCreateFile
// 
// DESCRIPTION: Process SetToDefault
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bReCreateFile() 
{
   bool bReturn = false;

   // first remove if exist
    (tVoid) OSAL_s32IORemove(m_szFilename);

    // write Info to EM-Trace
    ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bReCreateFile] INFO Recreate: %s",m_szFilename));
    mode_t mode = (S_IRWXU | S_IRWXG);
    vdDiaglog_FileDir diaglogFileDir(dirName, mode);

   // create file Parent directory if not exist
   if (! (diaglogFileDir.createDirIfNotExist()))
   {
	   ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bReCreateFile] ERROR create Parent directory"));
   }

   // file does not exist therefore create and save an empty one
   OSAL_tIODescriptor fd = OSAL_IOCreate(m_szFilename, OSAL_EN_READWRITE);
   // if create succesfull
   if(fd != OSAL_ERROR )
   {
      ETG_TRACE_COMP_THR(( "--- vdl_tclDiagLogObject::bReCreateFile => OSAL_IOCreate successfully"));
      tS32 s32Err = OSAL_s32IOClose(fd);  
      if(OSAL_ERROR == s32Err)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_ERR_THR(("!!! vdl_tclDiagLogObject::bReCreateFile => ERROR: OSAL_s32IOClose ErrorCode:%x",u32ErrorCode));
         // write to EM-Trace
         ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bReCreateFile] ERROR Recreate: OSAL_s32IOClose ErrorCode:%x File:%s", static_cast<tUInt>(u32ErrorCode),m_szFilename));
      }
      // set the Timestemp of Diaglog.log
      tU32 TimeStamp = vdl_tclExternalData::u32GetCurrentTimestamp();
      vSetDataBaseTimestamp(TimeStamp);
      bReturn = bSave();
   }
   else
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERR_THR(( "!!! vdl_tclDiagLogObject::bReCreateFile => OSAL_IOCreate FAILED EC:%x File: %s",u32ErrorCode,m_szFilename));
      // write to EM-Trace
      ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bReCreateFile] ERROR Recreate: OSAL_IOCreate ErrorCode:%x File:%s", static_cast<tUInt>(u32ErrorCode), m_szFilename));
      NORMAL_M_ASSERT_ALWAYS();
   }

   return bReturn;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclDiagLogObject::bLoad
// 
// DESCRIPTION: load ReportMemory from flash
//
// PARAMETER:  void
//
// RETURNVALUE: bool true=succesfull / false=failed
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bLoad()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclDiagLogObject::bLoad"));

   bool bRecreate = false;
   bool bResult = false;

   if(m_szFilename == 0)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::bLoad => NULL == m_szFilename"));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }
   if(m_poStorageUser == NULL)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::bLoad => NULL == m_poStorageUser"));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }

   m_bIsLoaded = false;
    
   // open flash file
   OSAL_tIODescriptor fd = OSAL_IOOpen(m_szFilename, OSAL_EN_READONLY);
   if(OSAL_ERROR == fd)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_FATAL_THR(( "!!! vdl_tclDiagLogObject::bLoad => OSAL_IOOpen FAILED Error:%x File:%s", static_cast<tUInt>(u32ErrorCode),m_szFilename));
      // write to EM-Trace
      ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] OSAL_IOOpen FAILED Error:%x File:%s", static_cast<tUInt>(u32ErrorCode),m_szFilename));


      if(u32ErrorCode != OSAL_E_DOESNOTEXIST)
      {
         // only assert in case != "File not exist"
         NORMAL_M_ASSERT_ALWAYS();
      }

      m_bIsLoaded = bReCreateFile();
   }
   else
   {
      ETG_TRACE_COMP_THR(( "--- vdl_tclDiagLogObject::bLoad => Load %s",m_szFilename));
      tS32 s32Max, s32Min;
      m_poStorageUser->vGetSizeOfLoadBuffer(s32Max,s32Min);

      if(  (s32Max < 0)
         ||(s32Min < 0))
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::bLoad => OutOfRange s32Max:%d s32Min:%d",s32Max,s32Min));
         NORMAL_M_ASSERT_ALWAYS();
         s32Max = 0;
         s32Min = 0;
      }

      tStream oData;
      // add 2 Bytes, so we can read more and are possible to check if the data is reduced
      tU32 u32Size = (tU32)s32Max + DIAGLOG_MEMORY_HEADER_SIZE + 2;
      oData.resize(u32Size);

      // read whole file into vector
      tS32 s32BytesRead = OSAL_s32IORead(fd, &oData[0], u32Size); // get Data to the Bigger Buffer

      // check compatibility between code and log file
      if(OSAL_ERROR != s32BytesRead) // if compatible
      {
         if(true == bCheckHeader(oData))
         {
            // reduce size by Header Data
            s32BytesRead -= DIAGLOG_MEMORY_HEADER_SIZE;
            ETG_TRACE_USR4_THR(( "--- vdl_tclDiagLogObject::bLoad => data ok, continue"));
            // if file db same size as memory db load it else use fresh db
            if(  (s32Max == s32Min)
               &&(s32Max == s32BytesRead))
            {
               ETG_TRACE_USR4_THR(( "--- vdl_tclDiagLogObject::bLoad => size ok, continue"));
               // init class data from file
               oData.resize((tU32)s32BytesRead);
               bResult = m_poStorageUser->bLoad(oData);
               if(bResult == false)
               {
                  bRecreate = true;
               }
            }
            else if(  (s32Max != s32Min)
                    &&(s32Max >= s32BytesRead)
                    &&(s32Min <= s32BytesRead))
            {
               ETG_TRACE_USR4_THR(( "--- vdl_tclDiagLogObject::bLoad => size ok, continue"));
               // init class data from file
               oData.resize((tU32)s32BytesRead);
               bResult = m_poStorageUser->bLoad(oData);
               if(bResult == false)
               {
                  bRecreate = true;
               }
            }
            else // invalid data
            {
               ETG_TRACE_FATAL_THR(( "!!! vdl_tclDiagLogObject::bLoad => size nok, FAILED Max=%d Min=%d Read=%d",s32Max,s32Min,s32BytesRead));
               ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] DIAGLOG-File invalid => generate new one"));
               bRecreate = true;
            }
         }// if(true == bCheckHeader(oData))
         else // incompatible data
         {
            ETG_TRACE_FATAL_THR(( "!!! vdl_tclDiagLogObject::bLoad => incompatible data, FAILED"));
            ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] diaglog.log incompatible => generate new one"));
            bRecreate = true; 
         }
      }// if(OSAL_ERROR != s32BytesRead) // if compatible
      else // incompatible data
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_FATAL_THR(( "!!! vdl_tclDiagLogObject::bLoad => OSAL_s32IORead FAILED Error:%x File:%s",u32ErrorCode,m_szFilename));
         // write to EM-Trace
         ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] OSAL_s32IORead FAILED: Error=%x File:%s", static_cast<tUInt>(u32ErrorCode),m_szFilename));
         NORMAL_M_ASSERT_ALWAYS();
         bRecreate = true;
      }
      

      // close flash device
      tS32 s32Err = OSAL_s32IOClose(fd);  
      if (s32Err != OSAL_OK)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_FATAL_THR(( "!!! vdl_tclDiagLogObject::bLoad => OSAL_s32IOClose FAILED Error:%x File:%s",u32ErrorCode,m_szFilename));
         // write to EM-Trace
         ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] OSAL_s32IOClose FAILED: Error=%x File:%s", static_cast<tUInt>(u32ErrorCode),m_szFilename));
         NORMAL_M_ASSERT_ALWAYS();
      }

      // if a fault occured, we have to reload
      if(true == bRecreate)
      {
         m_bIsLoaded = bReCreateFile();
      }
      else
      {
         m_bIsLoaded = true;
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclDiagLogObject::bLoad"));
   return m_bIsLoaded; 
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclDiagLogObject::bSave
// 
// DESCRIPTION: save ReportMemory from flash
//
// PARAMETER:  tVoid
//
// RETURNVALUE: bool
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bSave() 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclDiagLogObject::bSave"));

   tS32 s32Err = 0;

   if(m_szFilename == 0)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::bSave => NULL == m_szFilename"));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }

   if(m_poStorageUser == NULL)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::bSave => NULL == m_poStorageUser"));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }

   ETG_TRACE_COMP_THR(( "--- vdl_tclDiagLogObject::bSave (%s)",m_szFilename));
   bool bResult = false;
    
   // open flash file
   OSAL_tIODescriptor fd = OSAL_IOOpen(m_szFilename, OSAL_EN_READWRITE);
   if(OSAL_ERROR == fd)
   {
      ETG_TRACE_ERR_THR(( "--- vdl_tclDiagLogObject::bSave => OSAL_IOOpen FAILED"));
      // write Info to EM-Trace
      ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bSave] OSAL_IOOpen FAILED"));

      return bResult;
   }

   tStream oData;
   // collect class data
   _BP_TRY_BEGIN
   {
      // save format + version
      vSaveVersionData(oData);
      m_poStorageUser->vSave(oData);
   }
   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclDiagLogObject::bSave => vSave FAILED"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
   // write whole vector into file
   tS32 s32Size = static_cast<tS32>(oData.size());
   if(s32Size == OSAL_s32IOWrite(fd, &oData[0], (tU32)s32Size))
   {
      ETG_TRACE_USR1_THR(( "--- vdl_tclDiagLogObject::bSave => OSAL_s32IOWrite OK (Size = %d)",s32Size));
      bResult = true;
      s32Err = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOFLUSH, 0);
      if(OSAL_ERROR == s32Err)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_FATAL_THR(( "!!! vdl_tclDiagLogObject::bSave => OSAL_s32IOControl FAILED Error:%x File:%s",u32ErrorCode,m_szFilename));
         // write to EM-Trace
         ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bSave] OSAL_s32IOControl FAILED Error:%x File:%s", static_cast<tUInt>(u32ErrorCode),m_szFilename));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclDiagLogObject::bSave => OSAL_C_S32_IOCTRL_FIOFLUSH succesfully"));
      }
   }
   else
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bSave => ERROR: OSAL_s32IOWrite Size:%d  FAILED: Error=%x",s32Size, static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
   }
   // close flash device
   s32Err = OSAL_s32IOClose(fd);  
   if(OSAL_ERROR == s32Err)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_FATAL_THR(( "!!! vdl_tclDiagLogObject::bSave => OSAL_s32IOClose FAILED Error:%x File:%s",u32ErrorCode,m_szFilename));
      // write to EM-Trace
      ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bSave] OSAL_s32IOClose FAILED Error:%x File:%s", static_cast<tUInt>(u32ErrorCode),m_szFilename));
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclDiagLogObject::bSave"));
   return bResult; 
}
#else //#ifndef VARIANT_S_FTR_ENABLE_TRANSACTION_SAFETY

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::vdl_tclDiagLogObject
// 
// DESCRIPTION: default constructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclDiagLogObject::vdl_tclDiagLogObject() :
    m_bIsModified(false),
    m_u32DataBaseTimestamp(0),
    m_bCanSave(false),
    m_bIsLoaded(false),
    m_szFilename(0),
    m_poStorageUser(NULL),
    m_poStorageHandler(NULL),
    m_poObjMgr(NULL)
{
    // should never called
    NORMAL_M_ASSERT_ALWAYS();
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::vdl_tclDiagLogObject
// 
// DESCRIPTION: default constructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclDiagLogObject::vdl_tclDiagLogObject(vdl_tclStorageHandler* const poStorageHandler, vdl_tclStorageUserInterface* const poStorageUser, tCString const szFilename) :
    m_bIsModified(false),
    m_u32DataBaseTimestamp(0),
    m_bCanSave(false),
    m_bIsLoaded(false),
    m_szFilename(szFilename),
    m_poStorageUser(poStorageUser),
    m_poStorageHandler(poStorageHandler),
    m_poObjMgr(new vdDiaglog_ObjManager(szFilename))
{
    if (m_poStorageUser == NULL)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::vdl_tclDiagLogObject => NULL == m_poStorageUser"));
        NORMAL_M_ASSERT_ALWAYS();
    }
    if (poStorageHandler == NULL)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::vdl_tclDiagLogObject => NULL == poStorageHandler"));
        NORMAL_M_ASSERT_ALWAYS();
    }
    if (m_poObjMgr == NULL)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::vdl_tclDiagLogObject => NULL == m_poObjMgr"));
        NORMAL_M_ASSERT_ALWAYS();
    }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::~vdl_tclDiagLogObject
// 
// DESCRIPTION: default denstructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclDiagLogObject::~vdl_tclDiagLogObject()
{
    _BP_TRY_BEGIN
    {
        (tVoid)bSave();
    }
    _BP_CATCH_ALL
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::~vdl_tclDiagLogObject FAILED"));
    NORMAL_M_ASSERT_ALWAYS();
    }
    _BP_CATCH_END

    if (m_poObjMgr != NULL)
    {
        delete m_poObjMgr;
        m_poObjMgr = NULL;
    }

    m_szFilename = 0;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclDiagLogObject::bReCreateFile
// 
// DESCRIPTION: Process SetToDefault
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bReCreateFile()
{
    bool bReturn = false;

    mode_t mode = (S_IRWXU | S_IRWXG);
    vdDiaglog_FileDir diaglogFileDir(dirName, mode);

    // create Parent directory if not exist
    if (!(diaglogFileDir.createDirIfNotExist()))
    {
        ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bReCreateFile] ERROR create Parent directory"));
    }

    // set the Timestemp of Diaglog.log
    tU32 TimeStamp = vdl_tclExternalData::u32GetCurrentTimestamp();
    vSetDataBaseTimestamp(TimeStamp);
    bReturn = bSave();

    return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclDiagLogObject::bLoad
// 
// DESCRIPTION: load ReportMemory from flash
//
// PARAMETER:  void
//
// RETURNVALUE: bool true=succesfull / false=failed
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bLoad()
{
    ETG_TRACE_USR3_THR(("--> vdl_tclDiagLogObject::bLoad"));

    bool bRecreate = false;
    bool bResult = false;

    if (m_szFilename == 0)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bLoad => NULL == m_szFilename"));
        NORMAL_M_ASSERT_ALWAYS();
        return false;
    }
    if (m_poStorageUser == NULL)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bLoad => NULL == m_poStorageUser"));
        NORMAL_M_ASSERT_ALWAYS();
        return false;
    }

    if (m_poObjMgr == NULL)
    {
        ETG_TRACE_FATAL_THR(("!!! vdl_tclDiagLogObject::bLoad - m_poObjMgr is NULL"));
        NORMAL_M_ASSERT_ALWAYS();
        return false;
    }

    m_bIsLoaded = false;

    ETG_TRACE_COMP_THR(("--- vdl_tclDiagLogObject::bLoad => Load %s", m_szFilename));

    tStream oData;

    // read whole file into vector
    ETG_TRACE_USR3_THR(("<-- vdl_tclDiagLogObject::bLoad - calling ObjectManager")); //gpu2kor - remove this.

    if (m_poObjMgr->loadData(oData) != FALSE) 
    {
        if (true == bCheckHeader(oData)) //check the header and extract the header info.
        {
            tS32 u32BytesRead =(tS32) oData.size();

            tS32 s32Max = 0;
            tS32 s32Min = 0; //these variables are re intilialised in the below function call.
            m_poStorageUser->vGetSizeOfLoadBuffer(s32Max, s32Min);

            if ((s32Max < 0) || (s32Min < 0))
            {
                ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bLoad => OutOfRange s32Max:%d s32Min:%d", s32Max, s32Min));
                NORMAL_M_ASSERT_ALWAYS();
                s32Max = 0;
                s32Min = 0;
            }
            
            // if file db same size or lesser as memory db load it else use fresh db
            if (((s32Max == s32Min) && (s32Max == u32BytesRead))
                || ((s32Max != s32Min) && (s32Max >= u32BytesRead) && (s32Min <= u32BytesRead)))
            {
                ETG_TRACE_USR4_THR(("--- vdl_tclDiagLogObject::bLoad => size ok, continue"));
                // init class data from file
                bResult = m_poStorageUser->bLoad(oData);
                if (bResult == false)
                {
                    bRecreate = true;
                }
            }
            else // invalid data
            {
                ETG_TRACE_FATAL_THR(("!!! vdl_tclDiagLogObject::bLoad => size nok, FAILED Max=%d Min=%d Read=%d", s32Max, s32Min, u32BytesRead));
                ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] DIAGLOG-File invalid => generate new one"));
                bRecreate = true;
            }
        }// if(true == bCheckHeader(oData))
        else // incompatible data
        {
            ETG_TRACE_FATAL_THR(("!!! vdl_tclDiagLogObject::bLoad => incompatible data, FAILED"));
            ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] File incompatible, File:%s", m_szFilename));
            bRecreate = true;
        }
    }// if(m_poObjMgr->loadData(oData) != FALSE)
    else // read error
    {

        ETG_TRACE_FATAL_THR(("!!! vdl_tclDiagLogObject::bLoad => Read from file FAILED File:%s", m_szFilename));
        // write to EM-Trace
        ETG_TRACE_ERRMEM(("DIAGLOG => [vdl_tclDiagLogObject::bLoad] Read from file FAILED File:%s", m_szFilename));
        NORMAL_M_ASSERT_ALWAYS();
        bRecreate = true;
    }

    // if a fault occured, we have to reload
    if (true == bRecreate)
    {
        m_bIsLoaded = bReCreateFile();
    }
    else
    {
        m_bIsLoaded = true;
    }
    ETG_TRACE_USR3_THR(("<-- vdl_tclDiagLogObject::bLoad"));

    return m_bIsLoaded;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclDiagLogObject::bSave
// 
// DESCRIPTION: save ReportMemory from flash
//
// PARAMETER:  tVoid
//
// RETURNVALUE: bool
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bSave()
{
    ETG_TRACE_USR3_THR(("--> vdl_tclDiagLogObject::bSave"));

    if (m_szFilename == 0)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bSave => NULL == m_szFilename"));
        NORMAL_M_ASSERT_ALWAYS();
        return false;
    }

    if (m_poStorageUser == NULL)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bSave => NULL == m_poStorageUser"));
        NORMAL_M_ASSERT_ALWAYS();
        return false;
    }

    if (m_poObjMgr == NULL)
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bSave => NULL == m_poObjMgr"));
        NORMAL_M_ASSERT_ALWAYS();
        return false;
    }

    ETG_TRACE_COMP_THR(("--- vdl_tclDiagLogObject::bSave (%s)", m_szFilename));
    bool bResult = false;

    tStream oData;
    // collect class data
    _BP_TRY_BEGIN
    {
        // save format + version
        vSaveVersionData(oData);
        m_poStorageUser->vSave(oData);
    }
    _BP_CATCH_ALL
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bSave => vSave FAILED"));
        NORMAL_M_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
       
    // write whole vector into file
    ETG_TRACE_USR3_THR(("<-- vdl_tclDiagLogObject::bSave - calling ObjectManager"));
    bResult = m_poObjMgr->updateData(oData);

    ETG_TRACE_USR3_THR(("<-- vdl_tclDiagLogObject::bSave"));

    return bResult;
}
#endif  //ifndef VARIANT_S_FTR_ENABLE_TRANSACTION_SAFETY
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclDiagLogObject::requestForSave
// 
// DESCRIPTION: Save Data if allowed
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclDiagLogObject::requestForSaveIfModified()
{
    // if database has changed
    if (bIsModified() && (true == m_bCanSave))
    {
        ETG_TRACE_COMP_THR(("--- vdl_tclDiagLogObject::requestForSaveIfModified => Save Now"));
        m_bIsModified = false; // reset flag
        m_bCanSave = false;
        if (bSave() == false) // save flash file
        {
            NORMAL_M_ASSERT_ALWAYS();
            m_bCanSave = true;
        }
        else
        {
            if (m_poStorageHandler != NULL)
            {
                m_poStorageHandler->saveDoneInformation();
            }
        }
    }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclDiagLogObject::saveNow
// 
// DESCRIPTION: Save Data NOW
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclDiagLogObject::saveNowIfModified()
{
    // if database has changed
    if (bIsModified())
    {
        ETG_TRACE_COMP_THR(("--- vdl_tclDiagLogObject::saveNowIfModified => Save Now"));
        m_bIsModified = false; // reset flag
        m_bCanSave = false;
        if (bSave() == false) // save flash file
        {
            NORMAL_M_ASSERT_ALWAYS();
            m_bCanSave = true;
        }
        else
        {
            if (m_poStorageHandler != NULL)
            {
                m_poStorageHandler->saveDoneInformation();
            }
        }
    }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclDiagLogObject::loadDataBaseIfNotLoaded
// 
// DESCRIPTION: load file if not loaded yet
//
// PARAMETER:  none
//
// RETURNVALUE: bool - true = loaded, false = not loaded
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::loadDataBaseIfNotLoaded()
{
    ETG_TRACE_USR3_THR(("--> vdl_tclDiagLogObject::loadDataBaseIfNotLoaded"));
    bool bResult = true;
    // if not already loaded
    if (false == m_bIsLoaded)
    {
        //  try to load, if successful
        if (true == bLoad())
        {
        }
        else  // couldn't load
        {
            bResult = false;
        }
    }
    ETG_TRACE_USR3_THR(("<-- vdl_tclDiagLogObject::loadDataBaseIfNotLoaded"));
    return bResult;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclDiagLogObject::deleteDataBase
// 
// DESCRIPTION: delete and create new file
//              beware: there is no backup!
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclDiagLogObject::deleteDataBase()
{
    ETG_TRACE_USR3_THR(("--> vdl_tclDiagLogObject::deleteDataBase"));
    bReCreateFile();
    ETG_TRACE_USR3_THR(("<-- vdl_tclDiagLogObject::deleteDataBase"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclDiagLogObject::bIsLogCompatible
// 
// DESCRIPTION: checks and Remove Header of loaded Data
//
// PARAMETER:  tStream& oData - Data vector
//
// RETURNVALUE: bool
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bCheckHeader(tStream& oData)
{
    ETG_TRACE_USR3_THR(("--> vdl_tclDiagLogObject::bCheckHeader"));

    if (oData.size()< DIAGLOG_MEMORY_HEADER_SIZE)
    {
        return false;
    }

    tU8 u8Version = static_cast<tU8>(oData.at(0));
    tU8 u8Format = static_cast<tU8>(oData.at(1));

    // read global timestamp
    tU8 u8Value = static_cast<tU8>(oData.at(2));
    tU32 u32Timestamp = (static_cast<tU32>(u8Value) << 24);
    u8Value = static_cast<tU8>(oData.at(3));
    u32Timestamp |= (static_cast<tU32>(u8Value) << 16);
    u8Value = static_cast<tU8>(oData.at(4));
    u32Timestamp |= (static_cast<tU32>(u8Value) << 8);
    u8Value = static_cast<tU8>(oData.at(5));
    u32Timestamp |= static_cast<tU32>(u8Value);
    vSetDataBaseTimestamp(u32Timestamp);

    ETG_TRACE_USR1_THR(("--- vdl_tclDiagLogObject::bCheckHeader => V:%02d.%02d Timestamp:%d", u8Version, u8Format, u32Timestamp));

    // remove Header Bytes
    oData.erase(oData.begin(), oData.begin() + DIAGLOG_MEMORY_HEADER_SIZE);

    return bIsLogCompatible(u8Version, u8Format);
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclDiagLogObject::bIsLogCompatible
// 
// DESCRIPTION: checks format and version of diaglog for compatibility with code
//
// PARAMETER:  u8 Version, u8 Format
//
// RETURNVALUE: bool
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclDiagLogObject::bIsLogCompatible(const tU8 fileVersion, const tU8 fileFormat) const
{
    tU8 swVersion = u8GetMemoryVersion();
    tU8 swFormat = 0;
    bool result = false;

    if (m_poStorageUser != NULL)
    {
        swFormat = m_poStorageUser->u8GetDataFormatVerision();
        if ((fileVersion == swVersion)
            && (fileFormat == swFormat))
        {
            result = true;
        }
    }
    else
    {
        ETG_TRACE_ERRMEM(("!!! vdl_tclDiagLogObject::bIsLogCompatible => NULL == m_poStorageUser"));
        NORMAL_M_ASSERT_ALWAYS();
    }

    return result;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclDiagLogObject::vSaveVersionData
// 
// DESCRIPTION: saves version and format codes into diaglog data stream
//
// PARAMETER:  vdl_tclDiagLogObject::tStreamCIterator& roData - diaglog data stream
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclDiagLogObject::vSaveVersionData(tStream& roData) const
{
    ETG_TRACE_USR3_THR(("--> vdl_tclDiagLogObject::vSaveVersionData"));

    // version number
    tU8 u8Version = u8GetMemoryVersion();
    roData.push_back(static_cast<tS8>(u8Version));

    tU8 u8Format = 0;
    // format code
    if (m_poStorageUser != NULL)
    {
        u8Format = m_poStorageUser->u8GetDataFormatVerision();
    }
    roData.push_back(static_cast<tS8>(u8Format));

    // timestamp
    tU32 u32Timestamp = u32GetDataBaseTimestamp();
    roData.push_back(static_cast<tS8>((u32Timestamp >> 24) & 0xff));
    roData.push_back(static_cast<tS8>((u32Timestamp >> 16) & 0xff));
    roData.push_back(static_cast<tS8>((u32Timestamp >> 8) & 0xff));
    roData.push_back(static_cast<tS8>(u32Timestamp & 0xff));

    ETG_TRACE_USR2_THR(("--- vdl_tclDiagLogObject::vSaveVersionData: V%02d.%02d Timestamp = %d", u8Version, u8Format, u32Timestamp));

    ETG_TRACE_USR3_THR(("<-- vdl_tclDiagLogObject::vSaveVersionData"));
}

