/************************************************************************
* FILE:         vd_diaglog_blockingmode.cpp
* PROJECT:      ALL
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION: DiagLog main application class
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author             | Modification
* 23.01.13  | BSOT Plischke      | initial version
*************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
// first include diaglog settings
#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#include <common/framework/vd_diaglog_settings.h>

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_blockingmode
   #include <common/framework/vd_diaglog_blockingmode.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record
   #include <common/framework/vd_diaglog_report_record.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include <common/framework/vd_diaglog_report_memory.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
   #include <common/framework/vd_diaglog_system_variables.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_blockingmode.cpp.trc.h"
#endif


OSAL_tSemHandle vdDiagLog_BlockingMode::m_hBlockSem = OSAL_C_INVALID_HANDLE;
#define DIAGLOG_BLOCKING_SEM_NAME   "DIAGLOG_BLOCKINGSEM"

tU32 vdDiagLog_BlockingMode::m_u32BlockingModeTimerCallAdd =  0;
tU32 vdDiagLog_BlockingMode::m_u32BlockingModeTimerCallRemove =  0;
bool vdDiagLog_BlockingMode::m_isBlockingModeUpdateTrigger =  false;


// security mechanism for memory overrun because of many updates
const tU32 BLOCKING_MODE_UPDATE_LIMIT = 5;
const tU32 BLOCKING_MODE_UPDATE_TIMER = 500; //[msec]

/*************************************************************************
*
* FUNCTION: vdDiagLog_BlockingMode::vdDiagLog_BlockingMode()
* 
* DESCRIPTION: 
*
*  Constructor: - Initialize AMT, create Sem, create notification table 
* 
*
* PARAMETER: void
*
* RETURNVALUE: none
*
*************************************************************************/
vdDiagLog_BlockingMode::vdDiagLog_BlockingMode():
   m_hStartupBlockTimer(OSAL_C_INVALID_HANDLE),
   m_hPowerCrankBlockTimer (OSAL_C_INVALID_HANDLE),
   m_hPowerRunBlockTimer(OSAL_C_INVALID_HANDLE),
   m_hMostBusBlockTimer(OSAL_C_INVALID_HANDLE),
   m_hVoltageBlockTimer(OSAL_C_INVALID_HANDLE),
   m_hBlockUpdateTimer(OSAL_C_INVALID_HANDLE),
   m_u32BlockingMode(DIAGLOG_BLOCKINGMODES_DEFAULT),
   m_u32BlockingModeLast(DIAGLOG_BLOCKINGMODES_DEFAULT),
   m_u32BlockingModeSetSinceLastUpdate(0),
   m_u32BlockingModeGoneSinceLastUpdate(0),
   m_u32ValidSinceLastSendBlockingMode(0),
   m_isMostBusTimerRunning (false),
   m_u32BlockingModeCounter(0),
   m_isBlockingModeUpdateAllowed(true),
   m_isBlockingModeUpdateRequested(false)


{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::vdDiagLog_BlockingMode"));

   // collect DTC Blocking List map
   // Create semaphore to avoid parallel processing
   tS32 s32Result = OSAL_s32SemaphoreCreate(DIAGLOG_BLOCKING_SEM_NAME, &m_hBlockSem, 1); 
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_BlockingMode::vdDiagLog_BlockingMode => OSAL_s32SemaphoreCreate DIAGLOG_BLOCKING_SEM_NAME failed: Result=%x, Error=%x",s32Result, static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
      m_hBlockSem = OSAL_C_INVALID_HANDLE;
   }

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::vdDiagLog_BlockingMode"));
}

/*************************************************************************
*
* FUNCTION: vdDiagLog_BlockingMode::~vdDiagLog_BlockingMode()
* 
* DESCRIPTION: destructor, frees all ressources 
*             ( if not done by vOnApplicationClose before )
*
* PARAMETER: void
* 
* RETURNVALUE: none
*
*************************************************************************/
vdDiagLog_BlockingMode::~vdDiagLog_BlockingMode()
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::~vdDiagLog_BlockingMode"));

   tS32 s32Result = OSAL_s32SemaphoreClose(m_hBlockSem);   
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_BlockingMode::~vdDiagLog_BlockingMode => OSAL_s32SemaphoreClose(m_hBlockSem) failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }

   s32Result = OSAL_s32SemaphoreDelete(DIAGLOG_BLOCKING_SEM_NAME);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_BlockingMode::~vdDiagLog_BlockingMode => OSAL_s32SemaphoreDelete(DIAGLOG_BLOCKING_SEM_NAME) failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
   m_hBlockSem = OSAL_C_INVALID_HANDLE;

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::~vdDiagLog_BlockingMode"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdDiagLog_BlockingMode::LockSemaphore
// 
// DESCRIPTION: Lock the semaphre
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
inline void vdDiagLog_BlockingMode::LockSemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphoreWait(m_hBlockSem, OSAL_C_U32_INFINITE);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: LockSemaphore() => m_hBlockSem failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdDiagLog_BlockingMode::FreeSemaphore
// 
// DESCRIPTION: free the semaphre
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
inline void vdDiagLog_BlockingMode::FreeSemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphorePost(m_hBlockSem);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: FreeSemapore() => m_hBlockSem failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

/*************************************************************************
*
* FUNCTION: bool vdDiagLog_BlockingMode::bOnInit ()
* 
* DESCRIPTION: is called after the initialisation of the framework is finished
*
* PARAMETER: void
*
* RETURNVALUE: true
*
*************************************************************************/
bool vdDiagLog_BlockingMode::bOnInit()
{
   ETG_TRACE_USR3_THR(( "-->  vdDiagLog_BlockingMode::bOnInit()"));
   tS32 s32Result;

   // BLOCKINGMODE create timer
#ifdef DIAGLOG_BLOCKING_MODE_STARTUP_TIMER
   ETG_TRACE_USR1_THR(( "---  vdDiagLog_BlockingMode::bOnInit => Create BLOCKINGMODE-TIMER m_hStartupBlockTimer"));
   s32Result = OSAL_s32TimerCreate(vBlockStartupTimerHandler, (tPVoid)this, &m_hStartupBlockTimer);    
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bOnInit => ERROR: OSAL_s32TimerCreate m_hStartupBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }
   // start the Timer now
   (tVoid)bStartStartupBlockTimer(DIAGLOG_BLOCKING_TIME_DEFAULT_STARTUP);
#endif

#ifdef DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER
   ETG_TRACE_USR1_THR(( "---  vdDiagLog_BlockingMode::bOnInit => Create BLOCKINGMODE-TIMER m_hPowerCrankBlockTimer"));
   s32Result = OSAL_s32TimerCreate(vBlockPowerCrankTimerHandler, (tPVoid)this, &m_hPowerCrankBlockTimer);    
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bOnInit => ERROR: OSAL_s32TimerCreate m_hPowerCrankBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }
#endif

#ifdef DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER
   ETG_TRACE_USR1_THR(( "---  vdDiagLog_BlockingMode::bOnInit => Create BLOCKINGMODE-TIMER m_hPowerRunBlockTimer"));
   s32Result = OSAL_s32TimerCreate(vBlockPowerRunTimerHandler, (tPVoid)this, &m_hPowerRunBlockTimer);    
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bOnInit => ERROR: OSAL_s32TimerCreate m_hPowerRunBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));

      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }
#endif

#ifdef DIAGLOG_BLOCKING_MODE_MOST_TIMER
   ETG_TRACE_USR1_THR(( "---  vdDiagLog_BlockingMode::bOnInit => Create BLOCKINGMODE-TIMER m_hMostBusBlockTimer"));
   s32Result = OSAL_s32TimerCreate(vBlockMostBusTimerHandler, (tPVoid)this, &m_hMostBusBlockTimer);    
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bOnInit => ERROR: OSAL_s32TimerCreate m_hMostBusBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }
#endif

#ifdef DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER
   ETG_TRACE_USR1_THR(( "---  vdDiagLog_BlockingMode::bOnInit => Create BLOCKINGMODE-TIMER m_hVoltageBlockTimer"));
   s32Result = OSAL_s32TimerCreate(vBlockVoltageTimerHandler, (tPVoid)this, &m_hVoltageBlockTimer);    
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bOnInit => ERROR: OSAL_s32TimerCreate m_hVoltageBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }
#endif

   ETG_TRACE_USR1_THR(( "---  vdDiagLog_BlockingMode::bOnInit => Create BLOCKINGMODEUPDATE TIMER"));
   s32Result = OSAL_s32TimerCreate(vBlockUpdateTimerHandler, (tPVoid)this, &m_hBlockUpdateTimer);    
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bOnInit => ERROR: OSAL_s32TimerCreate m_hBlockUpdateTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
      return false;
   }

   ETG_TRACE_USR3_THR(("<--- vdDiagLog_BlockingMode::bOnInit()"));
   return  true;
}


/*************************************************************************
*
* FUNCTION: tVoid vdDiagLog_BlockingMode::vOnClose()
* 
* DESCRIPTION: releases all resources and sends close - message
*
* PARAMETER: void
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdDiagLog_BlockingMode::vOnClose()
{  
   ETG_TRACE_USR3_THR(( "-->  vdDiagLog_BlockingMode::vOnClose"));

   tS32 s32Result;

   // kill timers
#ifdef DIAGLOG_BLOCKING_MODE_STARTUP_TIMER
   s32Result = OSAL_s32TimerDelete(m_hStartupBlockTimer);
   m_hStartupBlockTimer = OSAL_C_INVALID_HANDLE;
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vOnClose => ERROR: OSAL_s32TimerDelete m_hStartupBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
   }
#endif

#ifdef DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER
   s32Result = OSAL_s32TimerDelete(m_hPowerCrankBlockTimer);
   m_hPowerCrankBlockTimer = OSAL_C_INVALID_HANDLE;
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vOnClose => ERROR: OSAL_s32TimerDelete m_hPowerCrankBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
   }
#endif

#ifdef DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER
   s32Result = OSAL_s32TimerDelete(m_hPowerRunBlockTimer);
   m_hPowerRunBlockTimer = OSAL_C_INVALID_HANDLE;
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vOnClose => ERROR: OSAL_s32TimerDelete m_hPowerRunBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
   }
#endif

#ifdef DIAGLOG_BLOCKING_MODE_MOST_TIMER
   s32Result = OSAL_s32TimerDelete(m_hMostBusBlockTimer);
   m_hMostBusBlockTimer = OSAL_C_INVALID_HANDLE;
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vOnClose => ERROR: OSAL_s32TimerDelete m_hMostBusBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
   }
#endif

#ifdef DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER
   s32Result = OSAL_s32TimerDelete(m_hVoltageBlockTimer);
   m_hVoltageBlockTimer = OSAL_C_INVALID_HANDLE;
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vOnClose => ERROR: OSAL_s32TimerDelete m_hVoltageBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
   }
#endif

   s32Result = OSAL_s32TimerDelete(m_hBlockUpdateTimer);
   m_hBlockUpdateTimer = OSAL_C_INVALID_HANDLE;
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32_OsalError = OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vOnClose => ERROR: OSAL_s32TimerDelete m_hBlockUpdateTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR3_THR(( "<--  vdDiagLog_BlockingMode::vOnClose"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: OSAL_tpfCallback vdDiagLog_BlockingMode::vBlockStartupTimerHandler
// 
// DESCRIPTION: timer callback handler for Blocking Timer (Blocking Timer at Startup)
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_BlockingMode::vBlockStartupTimerHandler(tPVoid /*pvArg*/)
{
#ifdef DIAGLOG_BLOCKING_MODE_STARTUP_TIMER
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::vBlockStartupTimerHandler"));
   // free BlockingMode Timer
   vResetBlockingModeTimerCall(DIAGLOG_BLOCKING_MODE_STARTUP_TIMER);
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::vBlockStartupTimerHandler"));
#endif
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: OSAL_tpfCallback vdDiagLog_BlockingMode::vBlockPowerCrankTimerHandler
// 
// DESCRIPTION: timer callback handler for Blocking Timer. Blocking x ms after getting out from CRANK Mode
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_BlockingMode::vBlockPowerCrankTimerHandler(tPVoid /*pvArg*/)
{
#ifdef DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::vBlockPowerCrankTimerHandler"));
   // free BlockingMode Timer
   vResetBlockingModeTimerCall(DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER);
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::vBlockPowerCrankTimerHandler"));
#endif
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: OSAL_tpfCallback vdDiagLog_BlockingMode::vBlockPowerRunTimerHandler
// 
// DESCRIPTION: timer callback handler for Blocking Run Timer. Blockign for 5s after getting into RUN.
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_BlockingMode::vBlockPowerRunTimerHandler(tPVoid /*pvArg*/)
{
#ifdef DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER
   // free BlockingMode Timer
   ETG_TRACE_USR3(( "-->  vdDiagLog_BlockingMode::vBlockPowerRunTimerHandler"));
   vResetBlockingModeTimerCall(DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER);
   ETG_TRACE_USR3(( "<--  vdDiagLog_BlockingMode::vBlockPowerRunTimerHandler"));
#endif
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: OSAL_tpfCallback vdDiagLog_BlockingMode::vBlockMostBusTimerHandler
// 
// DESCRIPTION: timer callback handler for Blocking U0029 Timer
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_BlockingMode::vBlockMostBusTimerHandler(tPVoid pvArg)
{
#ifdef DIAGLOG_BLOCKING_MODE_MOST_TIMER
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::vBlockMostBusTimerHandler"));
   // free BlockingMode Timer
   vResetBlockingModeTimerCall(DIAGLOG_BLOCKING_MODE_MOST_TIMER);
   ((vdDiagLog_BlockingMode*)(pvArg))->m_isMostBusTimerRunning = false;
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::vBlockMostBusTimerHandler"));
#endif
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: OSAL_tpfCallback vdDiagLog_BlockingMode::vBlockVoltageTimerHandler
// 
// DESCRIPTION: timer callback handler for Blocking Timer. Blockign for x sec after getting Voltage out Of Range Conditions
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_BlockingMode::vBlockVoltageTimerHandler(tPVoid /*pvArg*/)
{
#ifdef DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::vBlockVoltageTimerHandler"));
   // free BlockingMode Timer
   vResetBlockingModeTimerCall(DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER);
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::vBlockVoltageTimerHandler"));
#endif
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: OSAL_tpfCallback vdDiagLog_BlockingMode::vBlockUpdateTimerHandler
// 
// DESCRIPTION: timer callback handler for Blocking Mode Update trigger
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_BlockingMode::vBlockUpdateTimerHandler(tPVoid /*pvArg*/)
{
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::vBlockUpdateTimerHandler"));
   m_isBlockingModeUpdateTrigger = true;   // set flag for Event

   if(vdDiagLog_tclApp::m_hEventHandle != OSAL_C_INVALID_HANDLE) 
   {
      tS32 s32Result = OSAL_s32EventPost(vdDiagLog_tclApp::m_hEventHandle, DIAGLOG_EVENT_TRIGGER_UPDATE_AFTER_BM, OSAL_EN_EVENTMASK_OR);
      if(OSAL_ERROR == s32Result)
      {
         tU32 u32_OsalError = OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vBlockUpdateTimerHandler => ERROR: OSAL_s32EventPost(DIAGLOG_EVENT_TRIGGER_UPDATE_AFTER_BM) FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vBlockUpdateTimerHandler => ERROR: OSAL_C_INVALID_HANDLE"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::vBlockUpdateTimerHandler"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::bStartStartupBlockTimer
//
// DESCRIPTION:  Start Block Timer
//             
// PARAMETER:    None
//
// RETURNVALUE:  true if Successfull
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::bStartStartupBlockTimer(tU16 u16Time)
{
   bool bReturn = true;

#ifdef DIAGLOG_BLOCKING_MODE_STARTUP_TIMER

   if(u16Time == 0)
   {
       resetBlockingMode(DIAGLOG_BLOCKING_MODE_STARTUP_TIMER);
       bReturn = false;
   }
   else // if(u16Time == 0)
   {
      if(m_hStartupBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         setBlockingMode(DIAGLOG_BLOCKING_MODE_STARTUP_TIMER);
         tS32 s32Result = OSAL_s32TimerSetTime (m_hStartupBlockTimer, u16Time, 0);

         if(OSAL_ERROR == s32Result)
         {
            tU32 u32_OsalError = OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartStartupBlockTimer => ERROR: OSAL_s32TimerSetTime m_hStartupBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
            // free blocking mode if timer not possible
            resetBlockingMode(DIAGLOG_BLOCKING_MODE_STARTUP_TIMER);
            NORMAL_M_ASSERT_ALWAYS();
            bReturn = false;
         }
         else
         {    
            ETG_TRACE_USR4_THR(("--- vdDiagLog_BlockingMode::bStartStartupBlockTimer => Main Blocking Timer Set: %u", u16Time));
            bReturn = true;
         }  
      }
      else // if(m_hStartupBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         resetBlockingMode(DIAGLOG_BLOCKING_MODE_STARTUP_TIMER);
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartStartupBlockTimer => ERROR: OSAL_C_INVALID_HANDLE"));
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
      }
   }

#endif

   return bReturn;
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::bStartPowerBlockTimer
//
// DESCRIPTION:  Start Power Block Timer: Blocking Mode after Getting out from Crank
//             
// PARAMETER:    None
//
// RETURNVALUE:  true if Successfull
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::bStartPowerBlockTimer(tU16 u16Time)
{
   bool bReturn = true;

#ifdef DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::bStartPowerBlockTimer"));

   if(u16Time == 0)
   {
       resetBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER);
       bReturn = false;
   }
   else // if(u16Time == 0)
   {
      if(m_hPowerCrankBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         setBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER);
         tS32 s32Result = OSAL_s32TimerSetTime (m_hPowerCrankBlockTimer, u16Time, 0);

         if(OSAL_ERROR == s32Result)
         {
            tU32 u32_OsalError = OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartPowerBlockTimer => ERROR: OSAL_s32TimerSetTime m_hPowerCrankBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
            // free blocking mode if timer not possible
            resetBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER);
            NORMAL_M_ASSERT_ALWAYS();
            bReturn = false;
         }
         else
         {    
            ETG_TRACE_USR4_THR(("--- vdDiagLog_BlockingMode::bStartPowerBlockTimer => Power Blocking Timer Set: %u", u16Time));
            bReturn = true;
         }  
      }
      else // if(m_hPowerCrankBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         resetBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER);
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartPowerBlockTimer => ERROR: m_hPowerCrankBlockTimer => OSAL_C_INVALID_HANDLE"));
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
      }
   }
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::bStartPowerBlockTimer"));
#endif

   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::bStartPowerRunBlockTimer
//
// DESCRIPTION:  Start RUN Block Timer (some DTC are blocked 5s after coming into RUN Mode )
//             
// PARAMETER:    None
//
// RETURNVALUE:  true if Successfull
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::bStartPowerRunBlockTimer(tU16 u16Time)
{
   bool bReturn = true;

#ifdef DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::bStartPowerRunBlockTimer"));

   if(u16Time == 0)
   {
       resetBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER);
       bReturn = false;
   }
   else // if(u16Time == 0)
   {
      if(m_hPowerRunBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         ETG_TRACE_USR1_THR(( "--- vdDiagLog_BlockingMode::bStartPowerRunBlockTimer => Setting Blocking Mode POWER RUN TIMER"));
         setBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER);
         tS32 s32Result = OSAL_s32TimerSetTime (m_hPowerRunBlockTimer, u16Time, 0);

         if(OSAL_ERROR == s32Result)
         {
            tU32 u32_OsalError = OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartPowerRunBlockTimer => ERROR: OSAL_s32TimerSetTime m_hPowerRunBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
            // free blocking mode if timer not possible
            resetBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER);
            NORMAL_M_ASSERT_ALWAYS();
            bReturn = false;
         }
         else
         {  
            ETG_TRACE_USR4_THR(("--- vdDiagLog_BlockingMode::bStartPowerRunBlockTimer => Power Run Blocking Timer Set: %u", u16Time));
            bReturn = true;
         }  
      }
      else // if(m_hPowerRunBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         resetBlockingMode(DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER);
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartPowerRunBlockTimer => ERROR: m_hPowerRunBlockTimer => OSAL_C_INVALID_HANDLE"));
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
      }
   }

   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::bStartPowerRunBlockTimer"));
#endif

   return bReturn;
}



///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::bStartMostBusBlockTimer
//
// DESCRIPTION:  Start U0029 Block Timer 
//             
// PARAMETER:    None
//
// RETURNVALUE:  true if Successfull
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::bStartMostBusBlockTimer(tU16 u16Time)
{
   bool bReturn = true;

#ifdef DIAGLOG_BLOCKING_MODE_MOST_TIMER
   ETG_TRACE_USR3_THR (("--> vdDiagLog_BlockingMode::bStartMostBusBlockTimer"));

   if(u16Time == 0)
   {
       resetBlockingMode(DIAGLOG_BLOCKING_MODE_MOST_TIMER);
       bReturn = false;
   }
   else // if(u16Time == 0)
   {
      if(m_hMostBusBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         if (!m_isMostBusTimerRunning)
         {
           tS32 s32Result = OSAL_s32TimerSetTime (m_hMostBusBlockTimer, u16Time, 0);

           if(OSAL_ERROR == s32Result)
           {
             tU32 u32_OsalError = OSAL_u32ErrorCode();
             ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartMostBusBlockTimer => ERROR: OSAL_s32TimerSetTime m_hMostBusBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
             // free blocking mode if timer not possible
             resetBlockingMode(DIAGLOG_BLOCKING_MODE_MOST_TIMER);
             NORMAL_M_ASSERT_ALWAYS();
             bReturn = false;
           }
           else
           {    
              ETG_TRACE_USR4_THR(("--- vdDiagLog_BlockingMode::bStartMostBusBlockTimer => U0029 Blocking Timer Set: %u", u16Time));
              m_isMostBusTimerRunning = true;
              bReturn = true;
              setBlockingMode(DIAGLOG_BLOCKING_MODE_MOST_TIMER);
           }  
         }
      }
      else // if(m_hMostBusBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         resetBlockingMode(DIAGLOG_BLOCKING_MODE_MOST_TIMER);
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartMostBusBlockTimer => ERROR: m_hMostBusBlockTimer => OSAL_C_INVALID_HANDLE"));
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
      }
   }// else // if(u16Time == 0)

   ETG_TRACE_USR3_THR (("<-- vdDiagLog_BlockingMode::bStartMostBusBlockTimer"));
#endif

   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::bEndMostBusBlockTimer
//
// DESCRIPTION:  End U0029 Block Timer 
//             
// PARAMETER:    None
//
// RETURNVALUE:  true if Successfull
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::bEndMostBusBlockTimer()
{
   ETG_TRACE_USR3_THR(( "-->  vdDiagLog_BlockingMode::bEndMostBusBlockTimer"));
   bool bReturn = false;

   if(m_hMostBusBlockTimer != OSAL_C_INVALID_HANDLE)
   {
      
      tS32 s32Result = OSAL_s32TimerSetTime (m_hMostBusBlockTimer, 0, 0);

      if(OSAL_ERROR == s32Result)
      {
         tU32 u32_OsalError = OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bEndMostBusBlockTimer => ERROR: OSAL_s32TimerSetTime m_hMostBusBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {    
         ETG_TRACE_USR4(("U0029 Block Timer Reset"));
         m_isMostBusTimerRunning = false;
         bReturn = true;
      }  
   }
   else // if(m_hMostBusBlockTimer != OSAL_C_INVALID_HANDLE)
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bEndMostBusBlockTimer => ERROR: m_hMostBusBlockTimer => OSAL_C_INVALID_HANDLE"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   resetBlockingMode(DIAGLOG_BLOCKING_MODE_MOST_TIMER);

   ETG_TRACE_USR3_THR(( "<--  vdDiagLog_BlockingMode::bEndMostBusBlockTimer"));
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::bStartVoltageBlockTimer
//
// DESCRIPTION:  Start Voltage Block Timer. Some DTCs are Blocked some tim after getting out of a Voltage Out Of Range Condition
//             
// PARAMETER:    None
//
// RETURNVALUE:  true if Successfull
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::bStartVoltageBlockTimer(tU16 u16Time)
{
   bool bReturn = true;

#ifdef DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER

   if(u16Time == 0)
   {
       resetBlockingMode(DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER);
       bReturn = false;
   }
   else // if(u16Time == 0)
   {
      if(m_hVoltageBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         setBlockingMode(DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER);
         tS32 s32Result = OSAL_s32TimerSetTime (m_hVoltageBlockTimer, u16Time, 0);

         if(OSAL_ERROR == s32Result)
         {
            tU32 u32_OsalError = OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartVoltageBlockTimer => ERROR: OSAL_s32TimerSetTime m_hVoltageBlockTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
            // free blocking mode if timer not possible
            resetBlockingMode(DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER);
            NORMAL_M_ASSERT_ALWAYS();
            bReturn = false;
         }
         else
         {    
            ETG_TRACE_USR4_THR(("--- vdDiagLog_BlockingMode::bStartVoltageBlockTimer => Voltage Blocking Timer Set: %u", u16Time));
            bReturn = true;
         }  
      }
      else // if(m_hVoltageBlockTimer != OSAL_C_INVALID_HANDLE)
      {
         resetBlockingMode(DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER);
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::bStartVoltageBlockTimer => ERROR: m_hVoltageBlockTimer => OSAL_C_INVALID_HANDLE"));
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
      }
   }
#endif

   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::vSetResetBlockingMode
//
// DESCRIPTION:  change Blocking Mode request
//             
// PARAMETER:    tU32 u32ModeSet    = Blocking Mode activate
//               tU32 u32ModeReset  = Blocking Mode deactivate
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////

tVoid vdDiagLog_BlockingMode::vSetResetBlockingMode(tU32 u32ModeSet, tU32 u32ModeReset)
{
   ETG_TRACE_USR3_THR(("--> vdDiagLog_BlockingMode::vSetResetBlockingMode => Set:%x Reset:%x", u32ModeSet,u32ModeReset));
   // first check if this is a Timer (multiple choice)
   // set conditions
   #ifdef DIAGLOG_BLOCKING_MODE_STARTUP_TIMER
   if((u32ModeSet & DIAGLOG_BLOCKING_MODE_STARTUP_TIMER) == DIAGLOG_BLOCKING_MODE_STARTUP_TIMER)
   {
      u32ModeSet&=~DIAGLOG_BLOCKING_MODE_STARTUP_TIMER; // remove FLAG will be done by Timer
      (tVoid) bStartStartupBlockTimer(DIAGLOG_BLOCKING_TIME_DEFAULT_STARTUP);
   }
   #endif

   #ifdef DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER
   if((u32ModeSet & DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER) == DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER)
   {
      u32ModeSet&=~DIAGLOG_BLOCKING_MODE_POWER_CRANK_TIMER; // remove FLAG will be done by Timer
      (tVoid)bStartPowerBlockTimer(vdl_tclGlobalSystemVariables::u16GetCrankDelayTime());
   }
   #endif

   #ifdef DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER
   if((u32ModeSet & DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER) == DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER)
   {
      u32ModeSet&=~DIAGLOG_BLOCKING_MODE_POWER_RUN_TIMER; // remove FLAG will be done by Timer
      (tVoid)bStartPowerRunBlockTimer(DIAGLOG_BLOCKING_TIME_DEFAULT_POWER_RUN);
   }
   #endif

   #ifdef DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER
   if((u32ModeSet & DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER) == DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER)
   {
      u32ModeSet&=~DIAGLOG_BLOCKING_MODE_VOLTAGE_TIMER; // remove FLAG will be done by Timer
      (tVoid)bStartVoltageBlockTimer(DIAGLOG_BLOCKING_TIME_DEFAULT_VOLTAGE);
   }
   #endif

   #ifdef DIAGLOG_BLOCKING_MODE_MOST_TIMER
   if((u32ModeSet & DIAGLOG_BLOCKING_MODE_MOST_TIMER) == DIAGLOG_BLOCKING_MODE_MOST_TIMER)
   {
      u32ModeSet&=~DIAGLOG_BLOCKING_MODE_MOST_TIMER; // remove FLAG will be done by Timer
      (tVoid)bStartMostBusBlockTimer(vdl_tclGlobalSystemVariables::u16GetMostBusDelayTimer());
   }
   #endif

   // reset conditions
   #ifdef DIAGLOG_BLOCKING_MODE_MOST_TIMER
   if((u32ModeReset & DIAGLOG_BLOCKING_MODE_MOST_TIMER) == DIAGLOG_BLOCKING_MODE_MOST_TIMER)
   {
      u32ModeReset&=~DIAGLOG_BLOCKING_MODE_MOST_TIMER; // remove FLAG will be done by Timer
      (tVoid)bEndMostBusBlockTimer();
   }
   #endif

    ETG_TRACE_USR4_THR(("--- vdDiagLog_BlockingMode::vSetResetBlockingMode => after Timer Set:%x Reset%x", u32ModeSet,u32ModeReset));
   // forward all other
   changeBlockingMode(u32ModeSet,u32ModeReset);

    ETG_TRACE_USR3_THR(("<-- vdDiagLog_BlockingMode::vSetResetBlockingMode"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::vResetBlockingModeTimerCall
//
// DESCRIPTION:  vResetBlockingMode
//             
// PARAMETER:    u32Mode Mode : Bitmask Blockingmode
//
// RETURNVALUE:  none
//
// HINT: called by Timer Event
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_BlockingMode::vResetBlockingModeTimerCall(tU32 u32Mode)
{ 
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::vResetBlockingModeTimerCall"));

   m_u32BlockingModeTimerCallRemove |= u32Mode;

   if(vdDiagLog_tclApp::m_hEventHandle != OSAL_C_INVALID_HANDLE) 
   {
      tS32 s32Result = OSAL_s32EventPost(vdDiagLog_tclApp::m_hEventHandle, DIAGLOG_EVENT_TRIGGER_UPDATE_AFTER_BM, OSAL_EN_EVENTMASK_OR);
      NORMAL_M_ASSERT(OSAL_ERROR != s32Result);
   }
   else
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vResetBlockingModeTimerCall => ERROR: m_hEventHandle => OSAL_C_INVALID_HANDLE"));
      NORMAL_M_ASSERT_ALWAYS();
   }
 
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::vResetBlockingModeTimerCall"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::vSetResetBlockingModeEventUpdate
//
// DESCRIPTION:  vSetResetBlockingModeEventUpdate
//             
// PARAMETER:    u32Mode Mode : Bitmask Blockingmode
//
// RETURNVALUE:  none
//
// HINT: called from Diaglog Thread
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_BlockingMode::vSetResetBlockingModeEventUpdate()
{ 
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::vSetResetBlockingModeEventUpdate"));

   // this Event is used for 2 different things
   // 1st check for change of BlockingMode by Timer
   if(  (m_u32BlockingModeTimerCallRemove != 0)
      ||(m_u32BlockingModeTimerCallAdd != 0))
   {
      // yes we have an Update
      tU32 u32ModeReset = m_u32BlockingModeTimerCallRemove;
      m_u32BlockingModeTimerCallRemove = 0;

      tU32 u32ModeSet = m_u32BlockingModeTimerCallAdd;
      m_u32BlockingModeTimerCallAdd = 0;

      ETG_TRACE_USR4_THR(( "--- vdDiagLog_BlockingMode::vSetResetBlockingModeEventUpdate Set:%x Reset:%x",u32ModeSet,u32ModeReset));
      vSetResetBlockingMode(u32ModeSet,u32ModeReset);
   }// if(  (m_u32BlockingModeTimerCallRemove != 0)

   // 2nd check for Blocking Mode Update request
   if(m_isBlockingModeUpdateTrigger == true)
   {
      // we can reset the request flag and send a new updated request
      m_isBlockingModeUpdateTrigger = false;
      m_isBlockingModeUpdateAllowed = true;
      if(m_isBlockingModeUpdateRequested == true)
      {
         ETG_TRACE_USR1_THR(( "--- vdDiagLog_BlockingMode::vSetResetBlockingModeEventUpdate => Timer run out -> send Update now"));
         vUpdateAfterBlockingMode();
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::vSetResetBlockingModeEventUpdate"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::vUpdateAfterBlockingMode
//
// DESCRIPTION:  vUpdateAfterBlockingMode
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_BlockingMode::vUpdateAfterBlockingMode()
{ 
   if (vdDiagLog_tclApp::m_poInstance != NULL)
   {
      // the following m_u32BlockingModeCounter test is only for checking the number of updates, 
      // if it is to high it could result in massive slowing the system
      if(m_u32BlockingModeCounter < BLOCKING_MODE_UPDATE_LIMIT)
      {
         ++m_u32BlockingModeCounter;
      }
      else
      {
         ETG_TRACE_ERRMEM(("DIAGLOG => BLOCKING MODE UPDATE OVERRRUN more then %d/%d[ms]!!!",static_cast<tInt>(m_u32BlockingModeCounter), static_cast<tInt>(BLOCKING_MODE_UPDATE_TIMER)));
      }
      
      if(m_isBlockingModeUpdateAllowed == true)
      {
         m_isBlockingModeUpdateAllowed = false;
         m_isBlockingModeUpdateRequested = false;

         m_u32ValidSinceLastSendBlockingMode = m_u32BlockingModeLast | m_u32BlockingModeSetSinceLastUpdate;

         // save the last send value now
         m_u32BlockingModeLast = m_u32BlockingMode;

         m_u32BlockingModeSetSinceLastUpdate    = 0;
         m_u32BlockingModeGoneSinceLastUpdate   = 0;


         // send update to requested
         update();

         // start update Block Timer
          m_u32BlockingModeCounter = 0;
         tS32 s32Result = OSAL_s32TimerSetTime (m_hBlockUpdateTimer, BLOCKING_MODE_UPDATE_TIMER, 0);
         if(OSAL_ERROR == s32Result)
         {
            tU32 u32_OsalError = OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_BlockingMode::vUpdateAfterBlockingMode => ERROR: OSAL_s32TimerSetTime m_hBlockUpdateTimer FAILED ErrorCode =%x", static_cast<tUInt>(u32_OsalError)));
            NORMAL_M_ASSERT_ALWAYS();
         }
      }// if(m_isBlockingModeUpdateAllowed)
      else
      {
         m_isBlockingModeUpdateRequested = true;
         ETG_TRACE_USR1_THR(( "--- vdDiagLog_BlockingMode::vUpdateAfterBlockingMode => Update forbidden now, wait for retrigger"));
      }
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::changeBlockingMode
//
// DESCRIPTION:  Blocking Mode has changed, internal call to change the Mode self
//             
// PARAMETER:    tU32 u32ModeSet    = Blocking Mode activate
//               tU32 u32ModeReset  = Blocking Mode deactivate
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_BlockingMode::changeBlockingMode(tU32 u32ModeSet, tU32 u32ModeReset)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::changeBlockingMode"));
   bool updateBlockingMode = false;
   bool changed = false;

   // we are allowed to send now if changed
   m_u32BlockingModeCounter = 0;

   // check for change
   if(  (u32ModeSet != 0)
      ||(u32ModeReset != 0))
   {
      //m_u32BlockingModeLast = m_u32BlockingMode;

      if (!((m_u32BlockingMode & u32ModeSet)==u32ModeSet))
      {
         m_u32BlockingMode |= u32ModeSet;                   // store the current state
         m_u32BlockingModeSetSinceLastUpdate |= u32ModeSet; // store the change state since last update
         changed = true;
      }

      if (m_u32BlockingMode & u32ModeReset)
      {
         m_u32BlockingMode &= ~u32ModeReset;                   // store the current state
         m_u32BlockingModeGoneSinceLastUpdate |= u32ModeReset; // store the change state since last update
         updateBlockingMode = true;
         changed = true;
      }

      if(changed == true)
      {
    	  ETG_TRACE_ERR_THR(("--- vdl_tclTrace::vTraceRx => m_u32BlockingMode:%x",m_u32BlockingMode));
      }

      // update only if requiered
      if(updateBlockingMode == true)
      {
         vUpdateAfterBlockingMode();
      }
   }// if(  (u32ModeSet != 0)...

   
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::changeBlockingMode"));
}




///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdDiagLog_BlockingMode::isBlockingModeActive
// check if blocking mode is active
//
// PARAMETER:  pReportRecord     : pointer to caller record (include memory ptr)
//             blockingModeMask  : referenz for Block Mask to trace it out by caller
//
// RETURNVALUE: true = Blocked
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_BlockingMode::isBlockingModeActive(vdl_tclReportRecord const* pReportRecord,tU32& blockingModeMask) const
{
   bool bReturn = false;

   if(pReportRecord != NULL)
   {
      // first check BlockingMode List
      bReturn = isBlockingModeByMaskActive(pReportRecord,blockingModeMask);

      // second check Block DTC List
      if(bReturn == false)
      {
         bReturn = isBlockingModeByDTCListActive(pReportRecord,blockingModeMask);
      }// if(bReturn == false)

      ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::isBlockingModeActive => DTC:%x is %d",pReportRecord->u32DTC(),ETG_ENUM(BLOCKINGMODE_CUR_STATE, bReturn)));
   }// if(pReportRecord != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdDiagLog_BlockingMode::isBlockingModeByMaskActive
// check if blocking mode is active
//
// PARAMETER:  pReportRecord     : pointer to caller record (include memory ptr)
//             blockingModeMask  : referenz for Block Mask to trace it out by caller
//
// RETURNVALUE: true = Blocked
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_BlockingMode::isBlockingModeByMaskActive(vdl_tclReportRecord const* pReportRecord,tU32& blockingModeMask) const
{
   bool bReturn = false;

   if(pReportRecord != NULL)
   {
      if(NULL != vdDiagLog_tclApp::m_poInstance)
      {
         // first check BlockingMode List

         // compare System with DTC Blockmode
         blockingModeMask = (vdDiagLog_tclApp::m_poInstance)->u32GetSystemBlockingMode() & pReportRecord->u32GetDTCBlockingMode();
         ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::isBlockingModeByMaskActive => BM-Sys=%x BM-DTC=%x",(vdDiagLog_tclApp::m_poInstance)->u32GetSystemBlockingMode(),pReportRecord->u32GetDTCBlockingMode()));
         if(blockingModeMask)
         {
            // blocked by BlockingMode
            ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::isBlockingModeByMaskActive => BM=%x",blockingModeMask));
            bReturn = true;
         }// if(blockingModeMask)
      }// if(NULL != vdDiagLog_tclApp::m_poInstance)
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// if(pReportRecord != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdDiagLog_BlockingMode::isBlockingModeByDTCListActive
// check if blocking mode is active
//
// PARAMETER:  pReportRecord     : pointer to caller record (include memory ptr)
//             blockingModeMask  : referenz for Block Mask to trace it out by caller
//
// RETURNVALUE: true = Blocked
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_BlockingMode::isBlockingModeByDTCListActive(vdl_tclReportRecord const* pReportRecord,tU32& blockingModeMask) const
{
   bool bReturn = false;

   if(pReportRecord != NULL)
   {
      const tU32* const DTCBlockListPtr = pReportRecord->ptrGetBlockList();   
      for (int i=0; i<DIAGLOG_MAX_BLOCKDTCLIST;i++)
      {
         // over each element in Block list
         tU32 blockDTC = DTCBlockListPtr[i]; 
         if(blockDTC != 0)
         {
            const vdl_tclReportMemory* pMemory = pReportRecord->u8GetMemoryObject();
            if(pMemory != NULL)
            {
               vdl_tclReportRecord const* pBlockRecord = pMemory->getMostRecentReportRecordByDTC(blockDTC);
               // check if this DTC is active
               if(pBlockRecord != NULL)
               {
                  if(pBlockRecord->bGetIsActive() == true)
                  {
                     ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::isBlockingModeActive => BM DTC:%x is active",pBlockRecord->u32DTC()));
                     bReturn = true;
                     blockingModeMask = DIAGLOG_BLOCKING_MODE_DTC_LIST + pBlockRecord->u32DTC();
                     // found one, can stop here
                     break;
                  }// if(pRecord->bGetIsActive() == true)
               }// if(pRecord != NULL)
            }// if(pMemory != NULL)
            else
            {
               NORMAL_M_ASSERT_ALWAYS();
            }// else //if(pMemory != NULL)
         }// if(blockDTC != 0)
      }// for (int i=0; i<DIAGLOG_MAX_BLOCKDTCLIST;i++)
   }// if(pReportRecord != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::addDTCToBlockList
//
// DESCRIPTION:  register a ITC for the DTCBlockList
//               HINT: must be used in LockSemaphore
//             
// PARAMETER:    tU8  MemoryId      = Memory ID
//               tU16 requestingITC = ITC who have this blocking reason 
//               tU32 blockDTC      = DTC which is a Blocking reason
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdDiagLog_BlockingMode::addDTCToBlockList(tU8 MemoryId, tU16 requestingITC, tU32 blockDTC)
{
   // create a pair of MemoryId and blockDTC as unique identifier
   tIdDtcPair idDtcIdentifier = std::make_pair(MemoryId,blockDTC);
   
   // search if still exist
   tDTCBlockingMapIt it = this->m_DTCBlockingMap.find(idDtcIdentifier);
   if(it!= m_DTCBlockingMap.end())
   {
      // still in map
      tITCVector& ITCList = it->second;  // reference to element, only for better understanding

      bool found = false;
      for (tITCVectorCIt itVector = ITCList.begin(); itVector != ITCList.end(); ++itVector)     
      {
         if(*itVector == requestingITC)
         {
            found = true;
            // still exist in vector
            ETG_TRACE_FATAL_THR(( "!!! vdDiagLog_BlockingMode::addDTCToBlockList => Still exist pair<%x,%x> ITC:%04x",MemoryId,blockDTC,requestingITC));
            // should never happen
            NORMAL_M_ASSERT_ALWAYS();
         }
      }
      if(found == false)
      {
         // add to vecor
         ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::addDTCToBlockList => add too existing pair<%x,%x> ITC:%04x",MemoryId,blockDTC,requestingITC));
         ITCList.push_back(requestingITC);
      }
   }// if(it!= m_DTCBlockingMap.end())
   else
   {
      ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::addDTCToBlockList => create new pair<%x,%x> ITC:%04x",MemoryId,blockDTC,requestingITC));
      // add to map
      tITCVector ITCList;
      ITCList.push_back(requestingITC);
      m_DTCBlockingMap.insert(std::make_pair(idDtcIdentifier,ITCList));
   }// else // if(it!= m_DTCBlockingMap.end())
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::updateBlockingModeAfterPassed
//
// DESCRIPTION:  the given ITC swich to PASSED, 
//               so we have to check if an updated is requiered
//               HINT: do not store any of the given pointer, they could be invalid!!!
//             
// PARAMETER:    vdl_tclReportRecord const* pReportRecord = pointer to record
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdDiagLog_BlockingMode::updateBlockingModeAfterPassed(vdl_tclReportRecord const* pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::updateBlockingModeAfterPassed"));
   tU32 DTC = pReportRecord->u32DTC();
   tU8  MemoryId = pReportRecord->u8GetMemoryId();
   const vdl_tclReportMemory* const pMemory = pReportRecord->u8GetMemoryObject();
   bool doUpdate = false;

   LockSemaphore();
   {
      // check if DTC is in blocking List
      tIdDtcPair idDtcIdentifier = std::make_pair(MemoryId,DTC);
      tDTCBlockingMapCIt it = this->m_DTCBlockingMap.find(idDtcIdentifier);
      if(it!= m_DTCBlockingMap.end())
      {
         // found it
         if(pMemory != NULL)
         {
            // check if the DTC is also PASSED, because THIS is an ITC call
            tuRecordStatus state;
            if(pMemory->getStateOfDTC(DTC, state.u32Status))
            {
               if(state.sStatus.m_bIsActive == false)    
               {
                  // ok this DTC is now PASSED

                  // get the ITC List
                  tITCVector ITCList = it->second;

                     for (tITCVectorCIt iter = ITCList.begin(); iter != ITCList.end(); ++iter)
                     {
                        // add each ITC to List
                        tIdItcPair idItcPair = std::make_pair(MemoryId,*iter);
                        ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::updateBlockingModeAfterPassed => add pair<%02x,%04x> to update List",MemoryId,*iter));
                        m_ITCUpdateSet.insert(idItcPair);
                        doUpdate = true;
                     }

               }// if(state.sStatus.m_bIsActive == false) 
            }// if(pMemory->getStateOfDTC( DTC, state.u32Status);
         }// if(pMemory != NULL)
      }// if(it!= m_DTCBlockingMap.end())
   }
   FreeSemaphore();
   
   // now handle the update
   if(doUpdate)
   {
      update();
   }

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::updateBlockingModeAfterPassed"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::registerBlockModeCallback
//
// DESCRIPTION:  register the callback functuion for an given ITC
//             
// PARAMETER:    tU8  ITC           = ITC ID
//               tU32 callbackId    = unique callback ID (e.g. CCA) 
//               tU32 blockingMask  = Blocking Reason (Bit Mask)
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::registerBlockModeCallback(tU8 MemoryId, tU16 ITC, tU32 callbackId, tU32 blockingMask, std::vector<tU32> blockDTCList)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::registerBlockModeCallback"));
   bool bReturn = false;
   if(   (ITC != 0) 
      && (callbackId != 0))
   {
      LockSemaphore();
      {
         // create a pair of MemoryId and blockDTC as unique identifier
         tIdItcPair idItcIdentifier = std::make_pair(MemoryId,ITC);
         tsBlockingConfig sBlockingConfig;
         sBlockingConfig.blockingMask   = blockingMask;
         sBlockingConfig.callbackId     = callbackId;
         sBlockingConfig.dtcBlockList   = blockDTCList;

         // the second parameter of pair retVal gives the info if it is added or still exist
         std::pair<tITCCallbackMapIt,bool> retVal;
         retVal = m_ITCCallbackMap.insert(std::make_pair(idItcIdentifier,sBlockingConfig));
         if(retVal.second == false)
         {
            bReturn = false;
            ETG_TRACE_ERR_THR(( "!!! vdDiagLog_BlockingMode::registerBlockModeCallback => Element <%x,%x> still exist!",idItcIdentifier.first,idItcIdentifier.second));
         }
         else
         {
            // add now to BlockList
            for (tDTCVectorCIt iterVector = blockDTCList.begin(); iterVector != blockDTCList.end(); ++iterVector)
            {
               addDTCToBlockList(MemoryId, ITC, *iterVector);
            }
         }
      }
      FreeSemaphore();
   }
   else
   {
      // invalid
   }

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::registerBlockModeCallback"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::unregisterBlockModeCallback
//
// DESCRIPTION:  register the callback functuion for an given ITC
//             
// PARAMETER:    tU8  ITC           = ITC ID
//               tU32 callbackId    = unique callback ID (e.g.. CCA) 
//               tU32 blockingMask  = Blocking Reason (Bit Mask)
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
bool vdDiagLog_BlockingMode::unregisterBlockModeCallback(tU8 MemoryId, tU16 ITC)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::unregisterBlockModeCallback"));
   bool bReturn = false;
   if(ITC != 0) 
   {
      // create a pair of MemoryId and blockDTC as unique identifier
      tIdItcPair idItcIdentifier = std::make_pair(MemoryId,ITC);
      size_t result = m_ITCCallbackMap.erase(idItcIdentifier);
      bReturn = (result>=1)?true:false;
   }
   else
   {
      // invalid
   }

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::unregisterBlockModeCallback"));
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_BlockingMode::update
//
// DESCRIPTION:  check if we have to update an ITC
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdDiagLog_BlockingMode::update()
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_BlockingMode::update"));

   tServiceUpdateMap updateMap;

   LockSemaphore();
   {
      // over all entries
      for (tITCCallbackMapCIt iter = m_ITCCallbackMap.begin(); iter != m_ITCCallbackMap.end(); ++iter)
      {
         bool updateByBlockignMode = false;
         bool updateByDTCList      = false;
         tIdItcPair idItcPair      = iter->first;

         tU32 dtcBlockingMode       = iter->second.blockingMask;
         tU32 lastBlockingMode      = u32GetValidSinceLastSendBlockingMode();
         tU32 currentBlockingMode   = u32GetSystemBlockingMode();

         // continue only if DTC was blocked and after the last ResetBlockingMode not blocked any more
         if(   (!(currentBlockingMode & dtcBlockingMode)) 
            && (lastBlockingMode & dtcBlockingMode))
         {  
            // not blocked any more by BlockingMode
            updateByBlockignMode = true;
         }
         
         if(updateByBlockignMode == false)
         {
            // and also check if the ITC is not in update list by DTC block, only if not still valid to update
            for (tITCUpdateSetCIt iterUpdateSet = m_ITCUpdateSet.begin(); iterUpdateSet != m_ITCUpdateSet.end(); ++iterUpdateSet)
            {
               if(*iterUpdateSet == idItcPair)
               {
                  updateByDTCList = true;
                  m_ITCUpdateSet.erase(iterUpdateSet);
                  break;
               }
            }
         }// if(updateByBlockignMode == false)

         // do we have to update?
         if(updateByDTCList || updateByBlockignMode)
         {
            // this ITC has to update

            // check first if currently blocked
            if( isBlockingModeActive(idItcPair)== false)
            {
               // yes we can update this one
               tU32 callbackId = iter->second.callbackId;
               tU16 ITC        = idItcPair.second;
               tITCVector itcList;
               itcList.push_back(ITC);

               std::pair<tServiceUpdateMapIt,bool> retVal;
               retVal = updateMap.insert(std::make_pair(callbackId,itcList));
               if(retVal.second == false)
               {
                  // still exist
                  tITCVector&          itcListOfMap = retVal.first->second;   // reference to element, only for better understanding
                  bool                 bFound       = false;

                  ETG_TRACE_USR4_THR(( "--- vdDiagLog_BlockingMode::update => callback:%x still exist! ITC-List size=%d",callbackId,itcListOfMap.size()));

                  for (tITCVectorCIt iterItcList = itcListOfMap.begin(); iterItcList != itcListOfMap.end(); ++iterItcList)
                  {
                     if(*iterItcList == ITC)
                     {
                        // still exist, do nothing and continue
                        bFound = true;
                        break;
                     }
                  }
                  if(bFound == false)
                  {
                     // add new element
                     itcListOfMap.push_back(ITC);
                     ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::update => callback:%x add ITC:%04x",callbackId,ITC));
                  }
               }// if(retVal.second == false)
               else
               {
                  ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::update => callback:%x created",callbackId));
                  ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::update => callback:%x add ITC:%04x",callbackId,ITC));
               }// else // if(retVal.second == false)
            }// if( isBlockingModeActive(idItcPair)== false)
         }// if(updateByDTCList || updateByBlockignMode)
      }// for (tITCCallbackMapCIt iter = m_ITCCallbackMap.begin(); iter != m_ITCCallbackMap.end(); iter++)

      // m_ITCUpdateSet should be empty now
      for (tITCUpdateSetCIt iterUpdateSet = m_ITCUpdateSet.begin(); iterUpdateSet != m_ITCUpdateSet.end(); ++iterUpdateSet)
      {
         tU8 MemoryId   = iterUpdateSet->first;
         tU16 ITC       = iterUpdateSet->second;
         ETG_TRACE_ERR_THR(( "!!! vdDiagLog_BlockingMode::update => No update rule found for pair <%x,%x>:",MemoryId,ITC));
      }

      // delete update list
      m_ITCUpdateSet.clear();
   }
   FreeSemaphore();

   // do we have something to update?
   if(updateMap.size() >0)
   {
      if(NULL != vdDiagLog_tclApp::m_poInstance)
      {
         (vdDiagLog_tclApp::m_poInstance)->m_oMemoryMaster.vSendUpdateStatusToRequested(updateMap);
      }
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_BlockingMode::update"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:      tVoid vdDiagLog_BlockingMode::isBlockingModeActive
//
// DESCRIPTION:   check if blocking mode is active
//
// PARAMETER:     idItcPair     : unique idetifier of pair <Memory ID, ITC>
//
// RETURNVALUE:   true = Blocked
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_BlockingMode::isBlockingModeActive(const tIdItcPair& idItcPair) const
{
   bool bReturn = false;

   tITCCallbackMapCIt it = this->m_ITCCallbackMap.find(idItcPair);
   if(it!= m_ITCCallbackMap.end())
   {
      // found id in Map
      if(NULL != vdDiagLog_tclApp::m_poInstance)
      {
         // first check BlockingMode List

         // compare System with DTC Blockmode
         tU32 blockModeDTC    = it->second.blockingMask;
         tU32 blockModeSystem = (vdDiagLog_tclApp::m_poInstance)->u32GetSystemBlockingMode();
         tU32 blockMode       = blockModeSystem & blockModeDTC;
         ETG_TRACE_USR2_THR(( "--- vdDiagLog_BlockingMode::isBlockingModeActive => BM-Sys=%x BM-DTC=%x",blockModeSystem,blockModeDTC));
         if(blockMode)
         {
            // blocked by BlockingMode
            ETG_TRACE_USR1_THR(( "--- vdDiagLog_BlockingMode::isBlockingModeActive => pair<%x,%x> block by BM=%x",idItcPair.first, idItcPair.second, blockMode));
            bReturn = true;
         }// if(blockingModeMask)

         if(bReturn == false)
         {
            // second check Block DTC List
            tU8      memoryId     = idItcPair.first;
            tDTCVector blockDTCList = it->second.dtcBlockList;
            for (tDTCVectorCIt iterDTCList = blockDTCList.begin(); iterDTCList != blockDTCList.end(); ++iterDTCList)
            {
               tU32 DTC = *iterDTCList;
               vdl_tclReportRecord const* pBlockRecord = (vdDiagLog_tclApp::m_poInstance)->m_oMemoryMaster.getMostRecentReportRecordByDTC(memoryId,DTC);
               // check if this DTC is active
               if(pBlockRecord != NULL)
               {
                  if(pBlockRecord->bGetIsActive() == true)
                  {
                     // found one
                     ETG_TRACE_USR1_THR(( "--- vdDiagLog_BlockingMode::isBlockingModeActive => pair<%x,%x> block by DTC=%x",idItcPair.first, idItcPair.second, DTC));
                     bReturn = true;
                  }// if(pRecord->bGetIsActive() == true)
               }// if(pBlockRecord != NULL)
            }// for (tDTCVectorCIt iterDTCList = blockDTC.begin(); iterDTCList != blockDTC.end(); iterDTCList++)
         }// if(bReturn == false)
      }// if(NULL != vdDiagLog_tclApp::m_poInstance)
   }// if(it!= m_DTCBlockingMap.end())

   return bReturn;
}
